/*
ScPl - A plotting library for .NET

PhysicalAxis.cs
Copyright (C) 2003
Matt Howlett, Paolo Pierini

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.
   
2. Redistributions in binary form must reproduce the following text in 
   the documentation and / or other materials provided with the 
   distribution: 
   
   "This product includes software developed as part of 
   the ScPl plotting library project available from: 
   http://www.netcontrols.org/scpl/" 

------------------------------------------------------------------------

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

$Id: PhysicalAxis.cs,v 1.11 2004/05/05 11:07:37 mhowlett Exp $

*/

using System;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Collections;

namespace scpl
{
	/// <summary>
	/// C# doesn't have templates. If it did, I might derive PhysicalAxis from the 
	/// templated Axis type (LinearAxis etc). Instead, use a has-a relationship with an
	/// Axis superclass.
	/// Class containing information about the physical axis.
	/// </summary>
	public class PhysicalAxis
	{
		#region init
		/// <summary>
		/// Helper function for constructors.
		/// </summary>
		private void init()
		{
			this.axis_ = null;
			this.physicalMax_ = null;
			this.physicalMin_ = null;
		}
		#endregion

		#region constructors
		/// <summary>
		/// Parameterless constructor.
		/// </summary>
		private PhysicalAxis()
		{
			init();
		}

		/// <summary>
		/// Constructor, using an existing axis and information on physical coordinates.
		/// </summary>
		/// <param name="a">An Axis instance.</param>
		/// <param name="physicalMin">The physical coordinates of the axis minimum.</param>
		/// <param name="physicalMax">The physical coordinates of the axis maximum.</param>
		public PhysicalAxis( Axis a, PointF physicalMin, PointF physicalMax )
		{
			this.axis_ = a;
			this.physicalMin_ = physicalMin;
			this.physicalMax_ = physicalMax;
		}
		#endregion

		#region GetBoundingBox
		/// <summary>
		/// Get the bounding box of the physical axis.
		/// </summary>
		/// <returns></returns>
		public virtual RectangleF GetBoundingBox( )
		{
			System.Drawing.Bitmap b = new System.Drawing.Bitmap(1,1);
			Graphics g = Graphics.FromImage(b);
			return (RectangleF)this.Draw(g);
		}
		#endregion

		#region Draw
		/// <summary>
		/// Draws the Axis connected to this PhysicalAxis.
		/// </summary>
		/// <param name="g">The Graphics surface.</param>
		/// <returns>The bounding box.</returns>
		public virtual object Draw( System.Drawing.Graphics g )
		{
			return this.axis_.Draw( g, (PointF)physicalMin_, (PointF)physicalMax_ );
		}
		#endregion

		#region WorldToPhysical
		/// <summary>
		/// Transformation from world (user) coordinates to physical (screen) coordinates.
		/// </summary>
		/// <param name="coord">The coordinate value to transform.</param>
		/// <param name="clip">Flag to indicate clipping.</param>
		/// <returns>The transformed coordinates.</returns>
		public PointF WorldToPhysical( double coord, bool clip )
		{
			return axis_.WorldToPhysical( coord, (PointF)physicalMin_, (PointF)physicalMax_, clip );
		}
		#endregion

		#region PhysicalToWorld
		/// <summary>
		/// Transformation from physical (screen) coordinates to  world (user) coordinates.
		/// </summary>
		/// <param name="p">A physical coordinate.</param>
		/// <param name="clip">Flag to indicate clipping.</param>
		/// <returns>The transformed coordinates.</returns>
		public double PhysicalToWorld( PointF p, bool clip )
		{
			return axis_.PhysicalToWorld(p, (PointF)physicalMin_, (PointF)physicalMax_, clip );
		}
		#endregion

		#region SetWorldLimits
		/// <summary>
		/// This sets new world limits for the axis from two points
		/// selected within the plot area.
		/// </summary>
		/// <param name="min">The upper left point of the selection.</param>
		/// <param name="max">The lower right point of the selection.</param>
		public void SetWorldLimits(Point min, Point max)
		{
			double minc;
			double maxc;
			if (axis_ != null)
			{
				minc = axis_.WorldMin;
				maxc = axis_.WorldMax;
				if ( !axis_.Reversed ) 
				{
					double tmp = this.PhysicalToWorld(min,true);
					axis_.WorldMax = this.PhysicalToWorld(max,true);
					axis_.WorldMin = tmp;
				}
				else
				{
					double tmp = this.PhysicalToWorld(min,true);
					axis_.WorldMin = this.PhysicalToWorld(max,true);
					axis_.WorldMax = tmp;
				}
				// need to trap somehow if the user selects an 
				// arbitrarily small range. Otherwise the GDI+ 
				// drawing routines lead to an overflow in painting 
				// the picture. This may be not the optimal solution,
				// but if the GDI+ draw leads to an overflow the
				// graphic surface becomes unusable anymore and I
				// had difficulty to trap the error.
				double half = (axis_.WorldMin + axis_.WorldMax)/2;
				double width = axis_.WorldMax - axis_.WorldMin;
				if (Math.Abs(half/width) > 1.0e12)
				{
					axis_.WorldMin = minc;
					axis_.WorldMax = maxc;
				}
			}
		}
		#endregion

		#region get/set PhysicalMin
		/// <summary>
		/// Accessor for the physical coordinates of the axis minimum.
		/// </summary>
		public PointF PhysicalMin
		{
			get
			{
				return (PointF)physicalMin_; // unbox.
			}
			set
			{
				physicalMin_ = value; // box.
			}
		}
		#endregion

		#region get/set PhysicalMax
		/// <summary>
		/// Accessor for the physical coordinates of the axis maximum.
		/// </summary>
		public PointF PhysicalMax
		{
			get
			{
				return (PointF)physicalMax_; // unbox.
			}
			set
			{
				physicalMax_ = value; // box.
			}
		}
		#endregion

		#region get/set Axis
		/// <summary>
		/// Accessor for the Axis which the PhysicalAxis refers to.
		/// </summary>
		public Axis Axis
		{
			get
			{
				return this.axis_;
			}
			set
			{
				this.axis_ = value;
			}
		}
		#endregion

		#region get Length
		public double Length
		{
			get
			{
				double xLen = Math.Abs(this.PhysicalMax.X - this.PhysicalMin.X);
				double yLen = Math.Abs(this.PhysicalMax.Y - this.PhysicalMin.Y);

				double physicalLength = Math.Sqrt( xLen*xLen + yLen*yLen );
				return physicalLength;
			}
		}
		#endregion

		#region get PixelWorldLength
		public double PixelWorldLength
		{
			get
			{
				return this.Axis.Length / this.Length;
			}
		}
		#endregion

		private Axis axis_;
		/// <summary>
		/// Protected member containing the physical coordinates of the axis minimum.
		/// </summary>
		protected object physicalMin_;  // Boxed. Defaults to null.
		/// <summary>
		/// Protected member containing the physical coordinates of the axis maximum.
		/// </summary>
		protected object physicalMax_;
	}
}
