/*
ScPl - A plotting library for .NET

TextPlot.cs
Copyright (C) 2003-2004
Paolo Pierini, Matt Howlett 

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.
   
2. Redistributions in binary form must reproduce the following text in 
   the documentation and / or other materials provided with the 
   distribution: 
   
   "This product includes software developed as part of 
   the ScPl plotting library project available from: 
   http://www.netcontrols.org/scpl/" 

------------------------------------------------------------------------

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

using System;
using System.Drawing;
using System.Data;

namespace scpl
{
	/// <summary>
	/// Summary description for TextPlot.
	/// </summary>
	public class TextPlot : PointPlot
	{

		#region class TextDataAdapter
		private class TextDataAdapter
		{
			private object data_;
			private object dataSource_;
			private string dataMember_;

			#region Constructor
			public TextDataAdapter( object dataSource, string dataMember, object data )
			{
				this.data_ = data;
				this.dataSource_ = dataSource;
				this.dataMember_ = dataMember;
			}
			#endregion
			#region this[int i]
			public string this[int i]
			{
				get
				{
					// this is inefficient [could set up delegates in constructor].

					if (data_ is string[])
					{
						return ((string[])data_)[i];
					}
					
					if (data_ is string)
					{
						if (dataSource_ == null)
						{
							throw new System.Exception( "Error: DataSource null" );
						}

						System.Data.DataRowCollection rows;

						if ( dataSource_ is System.Data.DataSet )
						{
							if (dataMember_ != null)
							{
								// TODO error check
								rows = ((DataTable)((DataSet)dataSource_).Tables[dataMember_]).Rows;
							}
							else
							{
								// TODO error check
								rows = ((DataTable)((DataSet)dataSource_).Tables[0]).Rows;
							}
						}

						else if (dataSource_ is System.Data.DataTable )
						{
							rows = ((DataTable)dataSource_).Rows;
						}

						else
						{
							throw new System.Exception ( "not implemented yet" );
						}

						return (string)((System.Data.DataRow)(rows[i]))[(string)data_];
					}

					if (data_ is System.Collections.ArrayList)
					{
						object dataPoint = ((System.Collections.ArrayList)data_)[i];
						if (dataPoint is string)
							return (string)dataPoint;
						throw new System.Exception( "TextDataAdapter: data not in recognised format" );
					}
					
					if (data_ == null)
					{
						return "text";
					}

					throw new System.Exception( "Text data not of recognised type" );
				}
			}
			#endregion
			#region Count
			public int Count
			{
				get
				{
					// this is inefficient [could set up delegates in constructor].

					if (data_ == null)
					{
						return 0;
					}
					if (data_ is string[])
					{
						return ((string[])data_).Length;
					}
					if (data_ is System.Collections.ArrayList)
					{
						return ((System.Collections.ArrayList)data_).Count;
					}
					throw new System.Exception( "Text data not in correct format" );
				}
			}
			#endregion
		}
		#endregion

		#region TextLabelPositions
		/// <summary>
		/// Enumeration for Text label position with respect to datapoint
		/// </summary>
		public enum TextLabelPositions
		{
			/// <summary>
			/// Above
			/// </summary>
			Above,
			/// <summary>
			/// Below
			/// </summary>
			Below,
			/// <summary>
			/// To the left.
			/// </summary>
			Left,
			/// <summary>
			/// To the Right.
			/// </summary>
			Right
		}
		#endregion

		#region Constructors
		/// <summary>
		/// Constructor for TexPlot
		/// </summary>
		public TextPlot()
		{
		}
		/// <summary>
		/// Constructor for TexPlot
		/// </summary>
		/// <param name="marker">The desired marker.</param>
		public TextPlot( Marker marker ) : base( marker )
		{
		}
		#endregion

		#region TextLabelPosition
		/// <summary>
		/// Accessor to the position of the label.
		/// </summary>
		public TextLabelPositions TextLabelPosition
		{
			get
			{
				return textLabelPosition_;
			}
			set
			{
				textLabelPosition_ = value;
			}
		}
		private TextLabelPositions textLabelPosition_ = TextLabelPositions.Above;
		#endregion

		#region TextData
		public object TextData
		{
			get
			{
				return textData_;
			}
			set
			{
				textData_ = value;
			}
		}
		object textData_;
		#endregion

		#region Font
		/// <summary>
		/// The Font used to write text.
		/// </summary>
		public Font Font
		{
			get
			{
				if ( font_ == null)
				{
					FontFamily fontFamily = new FontFamily("Arial");
					font_ = new Font(fontFamily, 10, FontStyle.Regular, GraphicsUnit.Pixel);
				}
				return font_;
			}
			set
			{
				font_ = value;
			}
		}
		private Font font_;
		#endregion

		#region Draw
		/// <summary>
		/// Renders the marker plot and the label.
		/// </summary>
		/// <param name="g">The Graphics surface.</param>
		/// <param name="xAxis">The X axis where the trace plot is attached to.</param>
		/// <param name="yAxis">The Y axis where the trace plot is attached to.</param>
		public override void Draw( Graphics g, PhysicalAxis xAxis, PhysicalAxis yAxis )
		{
			SequenceAdapter data = 
				new SequenceAdapter( this.DataSource, this.DataMember, this.ValueData, this.AbscissaData );

			TextDataAdapter textData = new TextDataAdapter( this.DataSource, this.DataMember, this.TextData );

			// first plot the marker
		    // we can do this cast, since the constructor accepts only this type!
			for (int i=0; i<data.Count; ++i)
			{
				try
				{
					PointD pt = data[i];
					if ( !Double.IsNaN(pt.X) && !Double.IsNaN(pt.Y) )
					{
						PointF xPos = xAxis.WorldToPhysical( pt.X, false);
						PointF yPos = yAxis.WorldToPhysical( pt.Y, false);
						marker_.Draw(g,(int)xPos.X, (int)yPos.Y);
						if ( textData[i] != "" )
						{
							SizeF size = g.MeasureString( textData[i], this.Font );
							switch (textLabelPosition_)
							{
								case TextLabelPositions.Above:
									g.DrawString( textData[i], font_, Brushes.Black, new PointF(xPos.X-size.Width/2,yPos.Y-size.Height-marker_.Size*2/3));
									break;
								case TextLabelPositions.Below:
									g.DrawString( textData[i], font_, Brushes.Black, new PointF(xPos.X-size.Width/2,yPos.Y+marker_.Size*2/3));
									break;
								case TextLabelPositions.Left:
									g.DrawString( textData[i], font_, Brushes.Black, new PointF(xPos.X-size.Width-marker_.Size*2/3,yPos.Y-size.Height/2));
									break;
								case TextLabelPositions.Right:
									g.DrawString( textData[i], font_, Brushes.Black, new PointF(xPos.X+marker_.Size*2/3,yPos.Y-size.Height/2));
									break;
							}
						}
					}
				}
				catch
				{
					throw new Exception("Error in TextPlot.Draw");
				}
			}

		}
		#endregion

	}
}
