/*
ScPl - A plotting library for .NET

Web.PlotSurface2d.cs
Copyright (C) 2003 
Paolo Pierini, Matt Howlett

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.
   
2. Redistributions in binary form must reproduce the following text in 
   the documentation and / or other materials provided with the 
   distribution: 
   
   "This product includes software developed as part of 
   the ScPl plotting library project available from: 
   http://www.netcontrols.org/scpl/" 

------------------------------------------------------------------------

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/
using System;
using System.Drawing;
using System.Drawing.Imaging;
using System.Web.UI;
using System.Web.UI.WebControls;
using System.ComponentModel;
using System.IO;

namespace scpl
{
	namespace web
	{
		/// <summary>
		/// The class implementing the web control.
		/// </summary>
		[
		DefaultProperty("Title"), 
		ToolboxData("<{0}:PlotSurface2D runat=server></{0}:PlotSurface2D>"),
		Designer(typeof(scpl.web.Design.PlotSurface2D))
		]
		public class PlotSurface2D : System.Web.UI.WebControls.WebControl, IPlotSurface2D 
		{
			private scpl.Bitmap.PlotSurface2D bps_;

			#region Empty Constructor
			/// <summary>
			/// Parameterless constructor.
			/// </summary>
			public PlotSurface2D() : base()
			{
			}
			#endregion
			#region Inner
			/// <summary>
			/// The inner scpl.Bitmap.PlotSurface2D object.
			/// </summary>
			[
			Browsable(false)
			]
			protected scpl.Bitmap.PlotSurface2D Inner
			{
				get
				{
					if (bps_ == null)
					{
						bps_ = new scpl.Bitmap.PlotSurface2D(1,1);
					}
					return bps_;
				}
			}
			#endregion
			#region Width/Height set/get
			/// <summary>
			/// The Width of the control.
			/// </summary>
			public override Unit Width
			{
				get
				{
					return Inner.Width;
				}
				set
				{
					if (Inner.Width != value)
					{
						Unit xs = value;
						if ( xs == 0 ) xs=100;
						Unit ys = this.Height;
						if ( ys == 0 ) ys=100;
						Inner.Bitmap = new System.Drawing.Bitmap((int)xs.Value,(int)ys.Value);
					}
				}
			}
			/// <summary>
			/// The Height of the control.
			/// </summary>
			public override Unit Height
			{
				get
				{
					return Inner.Height;
				}
				set
				{
					if (Inner.Height != value)
					{
						Unit xs = this.Width;
						if ( xs == 0 ) xs=100;
						Unit ys = value;
						if ( ys == 0 ) ys=100;
						Inner.Bitmap = new System.Drawing.Bitmap((int)xs.Value,(int)ys.Value);
					}
				}
			}
			#endregion

			/// <summary>
			/// The URL to redirect for the plot.
			/// </summary>
			protected string PlotUrl;

			#region the prefix used for the session variables
			private string prefix()
			{
				string toReturn = "__PlotSurface2D_";
				toReturn += this.ID;
				toReturn += "_";
				toReturn += this.Page.ToString();
				toReturn += "_";
				return toReturn;
			}
			#endregion

			// These only delegate to the inner bitmap.PlotSurface2D object
			// Will need to do further work to line up with the latest 
			// modifications of the Windows.PlotSurface2D
			#region Clear
			/// <summary>
			/// Clears the plot.
			/// </summary>
			public void Clear()
			{
				Inner.Clear();
			}
			#endregion
			#region Add(IDrawable p)
			/// <summary>
			/// Add a plot to the plot surface.
			/// </summary>
			/// <param name="p">the plot to add.</param>
			public void Add( IDrawable p )
			{
				Inner.Add( p );
			}
			#endregion
			#region Add( IDrawable, XAxisPosition, YAxisPosition )
			/// <summary>
			/// Adds a trace plot to the plot surface.
			/// </summary>
			/// <param name="p">The trace plot to add.</param>
			/// <param name="xp">The position of the X axis.</param>
			/// <param name="yp">The position of the Y axis.</param>
			public void Add( IDrawable p, scpl.PlotSurface2D.XAxisPosition xp, scpl.PlotSurface2D.YAxisPosition yp )
			{
				Inner.Add( p, xp, yp );
			}
			#endregion
			#region get/set Title
			/// <summary>
			/// The plot title.
			/// </summary>
			public string Title
			{
				get 
				{
					return Inner.Title;
				}
				set 
				{
					Inner.Title = value;
				}
			}
			#endregion
			#region get/set TitleFont
			/// <summary>
			/// The plot title font.
			/// </summary>
			public System.Drawing.Font TitleFont 
			{
				get 
				{
					return Inner.TitleFont;
				}
				set 
				{
					Inner.TitleFont = value;
				}
			}
			#endregion
			#region get/set Padding
			/// <summary>
			/// The padding space (in graphics units) for the plot.
			/// </summary>
			public int Padding
			{
				get
				{
					return Inner.Padding;
				}
				set
				{
					Inner.Padding = value;
				}
			}
			#endregion
			#region get/set XAxis1
			/// <summary>
			/// The first abscissa axis.
			/// </summary>
			public Axis XAxis1
			{
				get
				{
					return Inner.XAxis1;
				}
				set
				{
					Inner.XAxis1 = value;
				}
			}
			#endregion
			#region get/set YAxis1
			/// <summary>
			/// The first ordinate axis.
			/// </summary>
			public Axis YAxis1
			{
				get
				{
					return Inner.YAxis1;
				}
				set
				{
					Inner.YAxis1 = value;
				}
			}
			#endregion
			#region get/set XAxis2
			/// <summary>
			/// The second abscissa axis.
			/// </summary>
			public Axis XAxis2
			{
				get
				{
					return Inner.XAxis2;
				}
				set
				{
					Inner.XAxis2 = value;
				}
			}
			#endregion
			#region get/set YAxis2
			/// <summary>
			/// The second ordinate axis.
			/// </summary>
			public Axis YAxis2
			{
				get
				{
					return Inner.YAxis2;
				}
				set
				{
					Inner.YAxis2 = value;
				}
			}
			#endregion
			#region get/set PlotBackColor
			/// <summary>
			/// The plot background color.
			/// </summary>
			public System.Drawing.Color PlotBackColor
			{
				get
				{
					return Inner.PlotBackColor;
				}
				set
				{
					Inner.PlotBackColor = value;
				}
			}
			#endregion
			#region get/set BackColor
			/// <summary>
			/// The background color.
			/// </summary>
			public override System.Drawing.Color BackColor
			{
				get
				{
					return base.BackColor;
				}
				set
				{
					base.BackColor = value;
					Inner.BackColor = value;
				}
			}
			#endregion
			#region get/set Legend
			public scpl.Legend Legend
			{
				get
				{
					return Inner.Legend;
				}
				set
				{
					Inner.Legend = value;
				}
			}
			#endregion
			#region get/set SmoothingMode
			public System.Drawing.Drawing2D.SmoothingMode SmoothingMode 
			{ 
				get
				{
					return Inner.SmoothingMode;
				}
				set
				{
					Inner.SmoothingMode = value;
				}
			}
			#endregion

			#region get/set inner Bitmap
			/// <summary>
			/// The inner bitmap object used for rendering the web control.
			/// </summary>
			[
			Browsable(false)
			]
			public System.Drawing.Bitmap Bitmap
			{
				get
				{
					return Inner.Bitmap;
				}
				set
				{
					Inner.Bitmap = value;
				}
			}
			#endregion
			#region Refresh
			/// <summary>
			/// Refreshes the plot.
			/// </summary>
			public void Refresh()
			{
				Inner.Refresh();
			}
			#endregion
			#region ToBrowser
			/// <summary>
			/// Renders the inner bitmap to a MemoryStream.
			/// </summary>
			/// <returns>The MemoryStream object.</returns>
			public System.IO.MemoryStream ToBrowser() 
			{
				return Inner.ToBrowser();
			}
			#endregion

			#region OnInit override
			/// <summary>
			/// Initialization event.
			/// </summary>
			/// <param name="e"></param>
			protected override void OnInit(EventArgs e)
			{
				System.Web.HttpRequest request = Context.Request;
				System.Web.HttpResponse response = Context.Response;
				if (request.Params["PlotSurface2D_" + this.ID] != null) 
				{
					// retrieve the bitmap and display
					response.Clear();
					try
					{
						response.ContentType = "Image/Png"; 
						System.Drawing.Bitmap bmp = (System.Drawing.Bitmap) Context.Session[prefix()+"PNG"];
                        // don't ask why, but if I write directly to the response
						// I have a GDI+ error, if I first write to a MemoryStream and 
						// then to the response.OutputStream I don't get an error.
						System.IO.MemoryStream s = new System.IO.MemoryStream();
						bmp.Save( s, System.Drawing.Imaging.ImageFormat.Png);
						s.WriteTo(response.OutputStream);
					}
					catch (Exception ex)
					{
						response.ContentType = "Text/HTML";
						response.Write(	ex.Message);
					}
					finally
					{
						response.Flush();
						response.End(); 
					}
				}
				else
				{
					////
					// CHANGE / Felix Livni / 2004.04.07 
					// instances the new plot object and stores it
					// in context.
					bps_ = new scpl.Bitmap.PlotSurface2D((int)Width.Value, (int)Height.Value);
					// END CHANGE / Felix Livni 
					////
				}
				string requestUrl = request.Url.ToString();
				if ( requestUrl.IndexOf("?") == -1 ) 
				{
					this.PlotUrl = requestUrl + "?PlotSurface2D_" + this.ID + "=1";
				} 
				else
				{ 
					this.PlotUrl = requestUrl + "&PlotSurface2D_" + this.ID + "=1"; 
				} 
				base.OnInit (e);
			}
			#endregion

			/*
			#region DataSource
			public object DataSource
			{
				get
				{
					return Inner.DataSource;
				}
				set
				{
					Inner.DataSource = value;
				}
			}
			#endregion
			#region ValueData
			public object ValueData
			{
				get
				{
					return Inner.ValueData;
				}
				set
				{
					Inner.ValueData = value;
				}
			}
			#endregion		
			#region AbscissaData
			public object AbscissaData
			{
				get
				{
					return Inner.AbscissaData;
				}
				set
				{
					Inner.AbscissaData = value;
				}
			}
			#endregion
			*/

			#region Render
			/// <summary> 
			/// Render this control as an HTML stream.
			/// </summary>
			/// <param name="output">The HTML writer to write out to.</param>
			protected override void Render(HtmlTextWriter output)
			{
				// here the Inner Bitmap.PlotSurface2D is rendered
				// and the image saved in a session variable
				Inner.Refresh();
				Context.Session[prefix()+"PNG"] = Inner.Bitmap;
				if (this.BorderStyle == BorderStyle.None)
				{
					output.AddAttribute("border","0");
				}
				else
				{
					output.AddAttribute("border",this.BorderWidth.ToString());
					output.AddAttribute("borderColor",this.BorderColor.ToKnownColor().ToString());
				}
				output.AddAttribute("cellSpacing","0");
				output.AddAttribute("cellPadding","0");
				output.RenderBeginTag("table");
				output.RenderBeginTag("tr");
				output.AddAttribute("vAlign","center");
				output.AddAttribute("align","middle");
				output.RenderBeginTag("td");
				output.RenderBeginTag("P");
				output.AddAttribute("src",this.PlotUrl);
				output.AddAttribute("alt",this.PlotUrl);
				output.RenderBeginTag("img");
				output.RenderEndTag();
				output.RenderEndTag();
				output.RenderEndTag();
				output.RenderEndTag();
				output.RenderEndTag();
				output.Flush();
			}
			#endregion

			#region AddAxesConstraint
			public void AddAxesConstraint( AxesConstraint c )
			{
				Inner.AddAxesConstraint( c );
			}
			#endregion
		}
	}
}
