/* 
	defines constants and functions to access problem area and LS (local store) of SPU
*/

#ifndef __PPU_H
#define __PPU_H
#pragma once

#if defined(PS3)

#if !defined(__SPU__)
	#include <sys/sys_time.h>
	#include <sys/time_util.h>

	#if defined(__cplusplus)
		extern "C" 
		{
	#endif
			void InvokeOnLinkedStack(void (*proc)(void *), void *arg, void *stack, size_t stackSize);
	#if defined(__cplusplus)
		} // extern "C"
	#endif
#endif //__SPU__

namespace NPPU
{
	static const unsigned int scPCRawSPUOffset			= 0x00100000;	//offset to the address space of a raw SPU
	static const unsigned int scPCRawSPUBaseAddr		= 0xE0000000;	//base address of a raw SPU
	static const unsigned int scPCRawSPULSOffset		= 0x00000000;	//offset to the local store address of a raw SPU from the base address
	static const unsigned int scPCRawSPUProbOffset	= 0x00040000;	//offset to the problem space(address of the memory mapped registers for an SPU) of a raw SPU

	static const unsigned int scPCIntStatMailbox		= 0x01;				//interrupt status: mailbox
	static const unsigned int scPCIntStatStopSignal = 0x02;				//interrupt status: stop signal
	static const unsigned int scPCIntStatHaltSignal = 0x04;				//interrupt status: halt signal
	
	static const unsigned int scPCDMALSA						= 0x3004;	//offset for the load store address register
	static const unsigned int scPCDMAEAH						= 0x3008;	//offset for the high part of the effective address register
	static const unsigned int scPCDMAEAL						= 0x300C;	//offset for the low part of the effective address register
	static const unsigned int scPCDMASizeTag				= 0x3010;	//offset for the DMA size register
	static const unsigned int scPCDMAClassCMD				= 0x3014;	//offset for the DMA class register
	static const unsigned int scPCDMACMDStatus			= 0x3014;	//offset for the DMA command status register
	static const unsigned int scPCDMAQStatus				= 0x3104;	//offset for the DMA queue register
	static const unsigned int scPCPrxyQueryType			= 0x3204;	//offset for the proxy tag query type register
	static const unsigned int scPCPrxyQueryMask			= 0x321C;	//offset for the proxy tag query mask register
	static const unsigned int scPCPrxyTagStatus			= 0x322C;	//offset for the proxy tag status register
	static const unsigned int scPCPPUMB							= 0x4004;	//offset for the PPU mailbox register
	static const unsigned int scPCSPUMB							= 0x400C;	//offset for the SPU mailbox register
	static const unsigned int scPCMBStat						= 0x4014;	//offset for the mailbox status register
	static const unsigned int scPCSPURunCntl				= 0x401C;	//offset for the SPU run/control register
	static const unsigned int scPCSPUStatus					= 0x4024;	//offset for the SPU status register
	static const unsigned int scPCSPUNPC						= 0x4034;	//offset for the next program count register
	static const unsigned int scPCSigNotify1				= 0x1400C;//offset for the first notify signal register
	static const unsigned int scPCSigNotify2				= 0x1C00C;//offset for the second notify signal register
	static const unsigned int scPCPPUStackSize			= (32*1024);		//PPU stack size for thread creation
	static const unsigned int scPCMFCGetCMD					= 0x40;		//MFC GET command
	static const unsigned int scPCMFCGetFSCMD				= 0x4A;		//MFC GET[fs] command (fence and start spu execution)
	//number of jobs are allocated statically and put between the 128 byte aligned push/pull pointers, thats why 32-1
	static const unsigned int scMaxSPU							= 5;			//maximum number of SPUs to manage
	static const unsigned int scDebugCallbackPort   = 0xFF;		//callback indicating return from debug state

	typedef union SSpuStatusRegister
	{
		struct
		{
			unsigned int sc										: 16;
			unsigned int reserved2							: 5;
			unsigned int isolateExitStatus			: 1;
			unsigned int isolateLoadStatus			: 1;
			unsigned int reserved1							: 1;
			unsigned int isolationStatus				: 1;
			unsigned int illegalChannelInstructionDetected	: 1;
			unsigned int invalidInstructionDetected		: 1;
			unsigned int singleStepStatus			: 1;
			unsigned int waitStatus						: 1;
			unsigned int haltStatus						: 1;
			unsigned int programStopAndSignalStatus		: 1;
			unsigned int runStatus							: 1;
		};
		unsigned int	val;
	} 
	SSpuStatusRegister;

	//eieio operation (synchronizes all active io-operations on spu, ppu)
	inline void Eieio()
	{
		asm ("eieio");
	}

#if !defined(__SPU__)
	inline const unsigned int GetTimeBaseFrequency()
	{
		return sys_time_get_timebase_frequency();
	}

	inline const unsigned int GetSPUTimeBaseFrequency()
	{
		//currently the same
		return sys_time_get_timebase_frequency();
	}

	inline const unsigned int GetCPUFrequency()
	{
		return sys_time_get_timebase_frequency() * 40;
	}

	inline void GetTimeTB(uint64_t& rTime)
	{
		SYS_TIMEBASE_GET(rTime);
	}

	inline void GetTimeTB(volatile uint64_t& rTime)
	{
		SYS_TIMEBASE_GET(rTime);
	}

	inline const uint64_t GetTimeTB()
	{
		uint64_t t;
		SYS_TIMEBASE_GET(t);
		return t;
	}

	//returns local store (LS) address for a SPU for a given SPU id
	inline const unsigned int GetSPULSBaseAddr(const unsigned int cID)
	{
		return (scPCRawSPUOffset * cID + scPCRawSPUBaseAddr + scPCRawSPULSOffset);
	}

	//returns problem space address for a SPU for a given SPU id
	//problem space is the address of the memory mapped registers for an SPU
	inline const unsigned int GetSPUProbBaseAddr(const unsigned int cID)
	{
		return (scPCRawSPUOffset * cID + scPCRawSPUBaseAddr + scPCRawSPUProbOffset);
	}

	//returns local store (LS) address for a SPU for a given SPU id
	inline const unsigned int ReadSPULS(const unsigned int cID, const unsigned int cOffset)
	{
		return (*(volatile unsigned int*)(GetSPULSBaseAddr(cID) + cOffset));
	}

	//returns local store (LS) address for a SPU for a given SPU id
	inline void WriteSPULS(const unsigned int cID, const unsigned int cOffset, const unsigned int cValue)
	{
		*(unsigned int*)(GetSPULSBaseAddr(cID) + cOffset) = cValue;
	}

	//returns the address of a register (given by an offset)  from problem space for a given SPU id
	inline const volatile unsigned int GetSPURegAddr(const unsigned int cID, const unsigned int cOffset)
	{
		return (const volatile unsigned int)(GetSPUProbBaseAddr(cID) + cOffset);
	}

	//write a value to a register in problem space for a given SPU id and register offset
	inline void WriteSPUProbReg(const unsigned int cID, const unsigned int cOffset, const int cValue)
	{
		*(unsigned int*)GetSPURegAddr(cID, cOffset) = cValue;
	}

	//read a value from a register in problem space for a given SPU id and register offset
	inline const unsigned int ReadSPUProbReg(const unsigned int cID, const unsigned int cOffset)
	{
		return (*(volatile unsigned int*)GetSPURegAddr(cID, cOffset));
	}
#endif//__SPU__

	//struct covers DMA data setup common to all jobs and packets
	class CCommonDMABase
	{
	public:
		__attribute__((always_inline))
			inline CCommonDMABase(){}

		__attribute__((always_inline))
			inline void SetJobParamData(void* pParamData)
		{
			m_pParamData = pParamData;
		}

		__attribute__((always_inline))
			inline const void* GetJobParamData() const
		{
			return m_pParamData;
		}

	protected:
		void*	m_pParamData;					//parameter struct, not initialized to 0 for performance reason
	};

}//NPPU

#if !defined(__SPU__)
	__attribute__((always_inline))
	inline const unsigned long long rdtsc()
	{
		unsigned long long tb;
		NPPU::GetTimeTB(tb);
		return tb;
	}
#endif//__SPU__

#endif //PS3
#endif //__PPU_H
