/* 
	definition and implementation for the spu memory area manager
	used for:
		- memory allocation which cannot be handler by individual bucket memory manager
		- destination buffer for SPU's printf result
		- destination buffer for SPU's cache profiling data
	1 line for each SPU, so no need for synchronizations
	furthermore it provides an area for each SPU where memory allocation requests are transferred:
		- SPU sends requests via a couple of SPPUMemRequests
		- SPU toggles mailbox interrupt and the data are passed to the SPUMemAreaMan
		- a mailbox result is being written to SPU (which waits for it): 0 is successful, 1 otherwise
	also contains bucket headers transferred once to SPU
*/

#ifndef __SPU_MEM_AREA_MAN_H
#define __SPU_MEM_AREA_MAN_H
#pragma once

#if defined(PS3)

#include "../SPU/SPUMemManagerBase.h"
#include <IJobManSPU.h>
#include <platform.h>
#if defined(SUPP_SN) && !defined(__SPU__)
	#include <lib/libsn.h>
#endif
#include "../SPU/JobStructs.h"

namespace NPPU
{
	// forward declaration of printf wrapper
	int PrintOut( const char *fmt, ... );

	//type of memory request
	enum PPUMemRequestType
	{
		eMR_Delete = 0,
		eMR_ReAlloc = 1,
		eMR_Alloc = 2,
		eMR_Unknown
	};

	//data for one memory request and bucket headers
	//do not make it larger than 128 bytes, static number in SPUDriver back transfer
	struct SPPUMemRequestData
	{
		static const uint32 scPPUMemRequestDataTransferSize = 16; //transfer size of memory request setup

		uint32 address;					//address where to allocate into or delete from
		PPUMemRequestType type;	//request type
		uint32 size;						//size of request (for delete it does not matter)
		volatile uint32 valid;	//1 if valid, 0 if not (fast check)
	
		NSPU::SBucketInfo bucketInfo _ALIGN(16);	//bucket headers containing size and location

		__attribute__((always_inline))
		inline SPPUMemRequestData() : address(0), type(eMR_Unknown), size(0), valid(0)
		{
			assert(sizeof(SPPUMemRequestData) == 128);
		}
	} _ALIGN(128);

	#define SIZEOF_SPPUMEMREQUESTDATA 128

#if !defined(__SPU__)
	class CSPUMemAreaMan
	{
	public:
		CSPUMemAreaMan();

		~CSPUMemAreaMan()
		{}

#if defined(SUPP_SPU_FRAME_STATS)
		void ResetStats(const uint32 cPivotID = 0);
		void GetSPUFrameStats(NPPU::SSPUFrameStats& rStats) const;
#endif

		__attribute__((always_inline))
		void* GetSPUMemArea(const uint32 cSPUIndex) const
		{
			return (void*)&m_Requests[cSPUIndex];
		}
#if defined(DO_SPU_PROFILING)
		void HandleProfRequest(const uint32 cSPUIndex, const uint32 cProfFileCntr, const uint32 cJobId);
#endif
		void HandleMemRequest(const uint32 cSPUIndex);
		const char* HandlePrintfRequest(const uint32 cSPUIndex, const char* pJobName)
		{
			char *pBuf = m_PrintfBufs + cSPUIndex * SPU_PRINTF_BUF_SIZE;
			PrintOut("SPU%d(%s): %s",cSPUIndex,pJobName, pBuf);
#if !defined(SUPP_SN)
			//if(memcmp(SPU_ASSERT_STRING, (void*)pBuf, strlen(SPU_ASSERT_STRING)) == 0)
			//{
			//	assert(0);
			//	return NULL;
			//}
#endif
			return (const char*)pBuf;
		}
		
		const NPPU::SCallback& GetCallbackData(const uint32 cSPUIndex)
		{
			m_CustomCallbackDMAData[cSPUIndex].Wait();
			return m_CustomCallbackDMAData[cSPUIndex].callbackData;
		}

		SPPUMemRequestData m_Requests[NPPU::scMaxSPU] _ALIGN(128);//requests for SPUs, one for each SPU
		char m_PrintfBufs[SPU_PRINTF_BUF_SIZE * NPPU::scMaxSPU] _ALIGN(128);//buffer for printf's
		char m_CacheProfBufs[MAX_PROF_ID * (4*3) * NPPU::scMaxSPU] _ALIGN(128);//buffer for cache profiling data (12 byte per id)
		NPPU::SDMACallbackData m_CustomCallbackDMAData[NPPU::scMaxSPU] ;//callback data for each spu
#if defined(SUPP_SPU_FRAME_STATS)
		SSingleSPUStat m_SPUStats;//running stats for SPUs, one for each SPU
		uint64_t m_LastTime;	//time of last reset call
#endif
	};
#endif// __SPU__

	static const uint32 scPrintfBufDiff		= SIZEOF_SPPUMEMREQUESTDATA * NPPU::scMaxSPU;//diff on SPU
	static const uint32 scProfBufDiff			= scPrintfBufDiff + SPU_PRINTF_BUF_SIZE * NPPU::scMaxSPU;
	static const uint32 scCallbackBufDiff	= scProfBufDiff + (MAX_PROF_ID * (4*3) * NPPU::scMaxSPU);
#if defined(SUPP_SPU_FRAME_STATS)
	static const uint32 scStatsDiff				= (scCallbackBufDiff + NPPU::scMaxSPU * sizeof(NPPU::SDMACallbackData) + 127) & ~127;
#endif
}//NPPU

#if !defined(__SPU__)
inline NPPU::CSPUMemAreaMan::CSPUMemAreaMan()
{
#if defined(SUPP_SPU_FRAME_STATS)
	ResetStats();
#endif//SUPP_SPU_FRAME_STATS
}

#if defined(SUPP_SPU_FRAME_STATS)
inline void NPPU::CSPUMemAreaMan::ResetStats(const uint32 cPivotID)
{
	m_SPUStats.lock = 1;
//	m_SPUStats.curSPUPivot = cPivotID;
	for(uint32 i=0; i<NPPU::scMaxSPU; ++i)	
		m_SPUStats.count[i]	= 0;
	NPPU::GetTimeTB(m_LastTime);
	m_SPUStats.lock = 0;
}
#endif//SUPP_SPU_FRAME_STATS
#endif//__SPU__

#endif //PS3
#endif //__SPU_MEM_AREA_MAN_H
