/* 
	definitions for SPU page mechanism
*/

#ifndef __SPU_PAGES_H
#define __SPU_PAGES_H
#pragma once

#if defined(PS3)

//do not change to anything but 4
#define scMaxSPUPageCount 4

#include <IJobManSPU.h>

namespace NSPU
{
	//page info each SPUDriver retrieves to initializes the page logic
	//serves also as storage for SPU global data during startup
	struct SPageDirInfo
	{
		int32	 globalVarBaseAddr;				//global variable base address
		uint32 pageNum;									//number of pages
		uint32 pageDirEA;								//ea of page directory
		uint32 ppuSyncEA;								//ea of m_SPUJobQueue, spin lock follows, required for SPU job spawning
		uint32 gcmGlobalPPUContext;			//ea of gCellGcmCurrentContext
		uint32 gcmGlobalPPUControlReg;	//ea of rsx control register
		uint32 gcmCmdAddressBase;				//base ea for main command buffer for cellGcmAddressToOffset
		uint32 reportEA;								//base ea for main memory mapped reports
		uint32 gcmRsxBaseAddress;				//ea of rsx local memory
		uint32 gcmCountDeviceThreadEA;	//ea of CCryDXPSGCMSyncMan::m_CountDeviceThread
		uint8  profilingEnabled;				//1 if profiling info needs to be retrieved
		uint8  pad0[3];
		uint32 funcProfTimingEA;				//base ea of function profiler timings
		uint32 jobStorageLS _ALIGN(16);	//spu driver address, written by spus
		uint32 gcmCmdResetOffset;				//gcm offset it gets reset to after wrap
		uint32 gcmInjectBufOff;					//offset of injection buffer for RSXMemcpy
		uint32 pad;
	} _ALIGN(16);

	//keep size in sync with JobGen
	struct SPageInfo
	{
		uint32 ea;						//effective address of page
		uint32 size;					//transfer size of page
	};

#if defined(__SPU__)

	//do not change without adapting in MissHandler_spu.S
	#define RETURN_STACK_MAX_ENTRIES 48
	//entry of the return stack
	//16 byte to load and save quickly
	//written into via vec_ushort8, do not reorder 
	//g_ReturnStackTopIndex is the current offset to its top entry, always incremented by 16 byte == sizeof(SReturnStackEntry)
	struct SReturnStackEntry
	{
		uint16 pad0;					//was: uint16 pageTextOff;	//link register offset relative to page start
		uint16 pageID;				//id of page calling the cross page function
		uint16 pageSlot;			//slot the page is currently transferred to and calling from
		uint16 pad [5];
		static const uint32 cIsJobPage = 0xFFFF;	//identifier for jobs, set into pageID as base stack entry
	} _ALIGN(16);

	//possible page states
	//do not change values, changed and accessed via asm in MissHandler_spu.S too
	#define PAGE_STATE_INACTIVE 0		//initial state
	#define PAGE_STATE_STREAMING 1	//page is streaming (transfer from main mem to LS in progress)
	#define PAGE_STATE_READY 2			//page is ready to be used (DMA transfer has been finished)
	
	//page state stored for each page on SPU
	//do not change order, gets accessed in miss handler asm code
	struct SPageState
	{
		uint16 curIndex;				//index of page (serves as ID, 0...), duplicated in other directory for fast comp's
		uint16 curState;				//current state of page
		uint32 pad0;
		uint32 transDecrEnd;		//decrementer value of page transfer end (estimate) (to not stall on it too early), 0 is reset value
		uint32 pad;

		__attribute__((always_inline))
		inline void Reset()
		{
			curIndex				= 0xFFFF;
			curState				= PAGE_STATE_INACTIVE;
			transDecrEnd		= 0xFFFFFFFF;//max val so any check with cur dec. value expect it finished
		}
	} _ALIGN(16);

	//macro definitions for page intrinsic functions	
	#define ResetPageLRUCounter() g_PageLRUCounter	= spu_splats((uint32)0)
	#define ResetPageLRU() g_SPUPageLRUDir	= spu_splats((uint32)0)
	#define IncrPageLRU()  g_PageLRUCounter = spu_add(g_PageLRUCounter, 1)
	#define GetPageLRU()	spu_extract(g_PageLRUCounter, 0)
	#define UpdatePageLRUByIndex(cPageIndex, cLRUVal) g_SPUPageLRUDir = spu_insert(cLRUVal, g_SPUPageLRUDir, cPageIndex)
	#define UpdatePageLRUByID(cPageIDSplat4) g_SPUPageLRUDir = spu_sel(g_SPUPageLRUDir, g_PageLRUCounter, spu_cmpeq(cPageIDSplat4, g_SPUPageDir))
	#define ResetPageDir() NSPU::g_SPUPageDir			= spu_splats(-1)
	#define SetPageIndex(cPageIndex, cSlot) g_SPUPageDir = spu_insert(cPageIndex, g_SPUPageDir, cSlot)
	#define GetPageIndex(cSlot) spu_extract(g_SPUPageDir, cSlot)
	#define IsPagePresent(cPageIDSplat4) (spu_extract(spu_orx(spu_cmpeq(g_SPUPageDir, cPageIDSplat4)), 0) != 0)
	#define GetPageReplIndex() GetReplIndex(g_SPUPageLRUDir)
	#define GetEmptyPageSlot() spu_extract(spu_add(spu_cntlz(spu_gather(spu_cmpeq(g_SPUPageDir, spu_splats(-1)))), -28), 0)
	#define GetPageSlot(cPageIDSplat4) spu_extract(spu_add(spu_cntlz(spu_gather(spu_cmpeq(g_SPUPageDir, cPageIDSplat4))), -28), 0)
	#define NO_EMPTY_PAGE_SLOT 4
#endif//__SPU__
}

#endif //PS3
#endif //__SPU_PAGES_H
