/* 
	structures used for elf parsing
*/

#ifndef __ELF_H
#define __ELF_H
#pragma once

#if defined(PS3)

//represents 64 bit values
typedef union
{
	unsigned long long ull;	//64 bit representation.
	unsigned int ui[2];			//32 bit representation.
	void *address;					//pointer representation
} TAddr64;

namespace NSPU
{
	namespace NElf
	{
		//elf header
		struct SElf32Ehdr
		{
			unsigned char eIdent[16];  
			unsigned short    elfType;         
			unsigned short    elfMachine;      
			unsigned int    elfVersion;      
			unsigned int    elfEntry;        
			unsigned int    elfPhoff;        
			unsigned int    elfShoff;        
			unsigned int    elfFlags;        
			unsigned short    elfEhsize;       
			unsigned short    elfPhentsize;    
			unsigned short    elfPhnum;        
			unsigned short    elfShentsize;    
			unsigned short    elfShnum;        
			unsigned short    elfShstrndx;     
		};

		//elf program header
		struct SElf32Phdr
		{
			unsigned int    progType;         
			unsigned int    progOffset;       
			unsigned int    progVaddr;        
			unsigned int    progPaddr;        
			unsigned int    progFilesz;       
			unsigned int    progMemsz;        
			unsigned int    progFlags;        
			unsigned int    progAlign;        
		};

		//elf section header
		struct SElf32Shdr
		{
			unsigned int    shName;        
			unsigned int    shType;        
			unsigned int    shFlags;       
			unsigned int    shAddr;        
			unsigned int    shOffset;      
			unsigned int    shSize;        
			unsigned int    shLink;        
			unsigned int    shInfo;        
			unsigned int    shAddralign;   
			unsigned int    shEntsize;     
		};

		static const unsigned int scSHFAlloc	= 0x2;	//to check for sections that occupy memory during execution
		static const unsigned int scSHTNoBits = 8;		//to check for bss section

		//represents information about a parsed ELF file
		struct SElfInfo
		{
			TAddr64 spuElfEA;						//effective address of SPU ELF
			TAddr64 spuImageEA;					//effective address of start of loadable section inside SPU ELF

			unsigned int LSDestination;	//destination SPU local store address for SPU ELF
			unsigned int imageSize;			//total size of loadable section inside SPU ELF (bssSize is added)
			unsigned int entry;					//program entry point of SPU ELF
			unsigned int entryOff;			//program entry point offset for SPU ELF (entry - LSDestination)
			unsigned int bssStart;			//bss start address of SPU ELF
			unsigned int bssSize;				//bss size of SPU ELF

			SElfInfo() : bssStart(0), bssSize(0)
			{}
		};

		//return results for ParseElf
		enum EParseResult
		{
			ePR_Success,			//success of parsing
			ePR_NoElf,				//no elf file
			ePR_NoSPUElf,			//its no spu elf file
			ePR_ElfTooBig,		//elf is too big
			ePR_NoQWAddress		//image start is not on a quadword address
		};

		//parses spu elf file and overwrites elf header with PElfInfo structure
		//rpElfInfo will point at elf into inside pSpuElf
		const EParseResult ParseElf(void* __restrict pSpuElf, SElfInfo*& __restrict rpElfInfo);
	}//NElf
}//NSPU

#endif //PS3
#endif //__ELF_H
