/* 
	all required headers for a job
	this must never be included from a header file or non SPU job (_SPU_JOB will be defined from the makefile)
*/
#ifndef __SPU_FUNC_H
#define __SPU_FUNC_H
#pragma once

#if defined(PS3) && defined(__SPU__)

#if defined(_SPU_JOB)
	#define __spu_get_cur_page_id SPUGetCurPageID
	#define __spu_get_cur_page_addr SPUGetCurPageAddr
#endif

namespace NSPU
{
	//struct holding all information for function pointer and virtual function calls
	struct SPageFuncPtr
	{
		vec_ushort8 vecData;
		SPageFuncPtr(){};
		__attribute__((always_inline))
		inline SPageFuncPtr(const SPageFuncPtr& cFrom) : vecData(cFrom.vecData){};
		__attribute__((always_inline))
		inline SPageFuncPtr(vec_ushort8 cVecData) : vecData(cVecData){}
		__attribute__((always_inline))
		inline SPageFuncPtr(const int cCurID, const unsigned int cFuncID, const unsigned int cEA)
		{
			vecData = spu_insert(cCurID, vecData, 1);
			//retrieve function data, cEA is passed for debug reason
			const unsigned int cFuncTableBaseAddr = spu_extract(g_SetMaskSL4, 3);
			assert(cFuncTableBaseAddr > 0 && cFuncTableBaseAddr < 256 * 1024);
			//obtain dest.page ID and its function offset, put into the expectd short-slots
			unsigned int cFuncEntry = ((unsigned int*)cFuncTableBaseAddr)[cFuncID];
			assert(cFuncEntry != 0xFFFFFFFF);//must not be valid
			//insert scWeak-flag if it destPageID equals current page
			const uint32 cFuncEntryMasked = (cFuncEntry & 0x0000FFFF);
			cFuncEntry |= (cFuncEntryMasked == cCurID)?0x8000 : 0;//mark as weak
			vecData = spu_insert((unsigned short)(cFuncEntry & 0x0000FFFF), vecData, 4);//set destPageID
			vecData = spu_insert((unsigned short)((cFuncEntry & 0xFFFF0000) >> 16), vecData, 3);//set destOffset, keep stored as multiple of 4
	#if !defined(_NO_SPU_ASSERT)
			//check against expected PPU address
			const unsigned int cDebugFuncTableBaseAddr = spu_extract(g_SetMaskSL4, 2);
			assert(cDebugFuncTableBaseAddr == 0 || ((unsigned int*)cDebugFuncTableBaseAddr)[cFuncID] == cEA);
	#endif
		}
		__attribute__((always_inline))
		inline const int GetDestPageID() const
		{
			return (int)spu_extract(vecData, 4);
		}
		__attribute__((always_inline))
		inline void SetDestPageID(const int cDestID)
		{
			vecData = spu_insert(cDestID, vecData, 4);
		}
		__attribute__((always_inline))
		inline void SetDestPageOff(const int cDestOff)
		{
			vecData = spu_insert(cDestOff, vecData, 3);
		}
	};

	//function pointer history table of 4 entries
	//do not reorder or add members, asm accesses it, JobGen too
	struct SFuncHistoryTable
	{
		vec_uint4 ppuEA;						//PPU addresses
		vec_uint4 lru;							//corresponding LRU data
		vec_ushort8 funcData[4];		//corresponding cached spu func data, must not be SPageFuncPtr to avoid static ctors
	};

	//local version of history table, function pointers are only redirected within same page
	struct SFuncHistoryTableLocal
	{
		vec_uint4 ppuEA;						//PPU addresses
		vec_uint4 lru;							//corresponding LRU data
		vec_uint4 funcData;					//4 absolute page addresses
	};
}//NSPU

#if defined(_SPU_JOB)
	//to be used in pages
	#define __spu_func_hist_table_name(hist_name) __spu_hist_table##hist_name
	#define __spu_init_func_hist_table(hist_name) static NSPU::SFuncHistoryTable __spu_func_hist_table_name(hist_name) __attribute__((nocommon))
	#define __spu_func_local_hist_table_name(hist_name) __spu_local_hist_table##hist_name
	#define __spu_init_func_hist_table_local(hist_name) static NSPU::SFuncHistoryTableLocal __spu_func_local_hist_table_name(hist_name) __attribute__((nocommon))

	//cache history query version
//	#define __spu_init_cached_func_ptr(hist_name, func_name, FUNC, ea) NSPU::SPageFuncPtr __spu_fnct_ptr_##func_name (SPUHistFuncLookup(&__spu_func_hist_table_name(hist_name), (unsigned int)ea, FUNC))
	#define __spu_init_cached_func_ptr(hist_name, FUNC, ea) __spu_init_cached_func_ptr_reg((NSPU::SFuncHistoryTable*)(hist_name), (uint32)ea)
	#define __spu_init_cached_func_ptr_local(hist_name, FUNC, ea) __spu_init_cached_func_ptr_reg_local((NSPU::SFuncHistoryTableLocal*)(hist_name), (uint32)ea)

	#define DECL_HIST_TABLE(hist_name, page_name) NSPU::SFuncHistoryTable __spu_hist_table##hist_name ## _page_id_ ##page_name  __attribute__((nocommon)) __attribute__((section(".data")))
	#define DECL_HIST_TABLE_LOCAL(hist_name, page_name) NSPU::SFuncHistoryTableLocal __spu_local_hist_table##hist_name ## _page_id_ ##page_name  __attribute__((nocommon)) __attribute__((section(".data")))
	#define DECL_HIST_TABLE_LOCAL_DUMMY(hist_name, page_name) extern NSPU::SFuncHistoryTableLocal* __spu_local_hist_table##hist_name ## _page_id_ ##page_name;

	//do not change prefixes, accessed in PageGen too
	#define HIST_TABLE(hist_name, page_name) __spu_hist_table##hist_name ## _page_id_ ##page_name
	#define HIST_TABLE_LOCAL(hist_name, page_name) __spu_local_hist_table##hist_name ## _page_id_ ##page_name
	#define HIST_TABLE_PAGE_PREFETCH(hist_name, page_name) __spu_set_active_pages_from_hist_table((NSPU::SFuncHistoryTable*)HIST_TABLE(hist_name, page_name))

	#define INIT_HIST_TABLE \
		__asm__ ("ori $70,%0, 0" : : "r" ((uint32)pHistTable));\
		__asm__ ("ori $71,%0, 0" : : "r" (cEA));
	
	//need both version to avoid aliasing
	__attribute__((always_inline))
	inline void __spu_init_cached_func_ptr_reg(NSPU::SFuncHistoryTable* const __restrict pHistTable,  const unsigned int cEA)
	{
		INIT_HIST_TABLE
	}

	__attribute__((always_inline))
	inline void __spu_init_cached_func_ptr_reg_local(NSPU::SFuncHistoryTableLocal* const __restrict pHistTable,  const unsigned int cEA)
	{
		INIT_HIST_TABLE
	}

	#define __spu_init_resolve_func(ea) *(unsigned int*)(void*)G_SPU_JOB_RESOLVE_FUNC_ADDR = (unsigned int)(ea)
	#define __spu_report_resolve_func_mismatch(ea)	SPUFuncPtrAssertFunc(0xFFFFFFFF, ea)

	//direct query version
	#define __spu_init_func_ptr(name, curPageId, funcId, ea) NSPU::SPageFuncPtr __spu_fnct_ptr_##name (curPageId, funcId, (unsigned int)ea)
	//explicit data query
	#define __spu_get_func_ptr_page(name) __spu_fnct_ptr_##name.GetDestPageID()
	//register query
	#define __spu_get_func_ptr_page_reg() spu_extract(g_CrossPageData, 4)
	#define	__spu_init_func_ptr_call(name) __spu_init_cross_call(__spu_fnct_ptr_##name.vecData)
	#define	__spu_init_func_ptr_call_by_id(name) __spu_init_cross_call(name.vecData)

	//used in fn_resolve to indicate empty function body targets
	#define FLAG_NON_EMPTY_FNCT\
		__asm__ ("il $72,1" :  : );
	#define FLAG_EMPTY_FNCT\
		__asm__ ("il $72,0" :  : );


#endif	//_SPU_JOB
#endif	//__SPU__
#endif	//__SPU_JOB_H
