#include "DeVirt.h"
#include "Util.h"

#include <iostream>
#include <fstream>
#include <sstream>
#include <cstring>
#include <cstdlib>
#include <cassert>

#include <sys/types.h>
#include <sys/stat.h>
#include <cstring>
#include <errno.h>

bool DeVirt::m_Verbose = false;

extern DeVirt deVirt;

std::string Param::Decl(const std::string &paramName) const
{
  std::ostringstream out;

  if (IsEllipsis())
    out << "...";
  else
  {
    const std::string &type = m_Type;
    size_t namePos = type.find('%');
    assert(namePos != std::string::npos);

    // skip param name for void
    if( type.substr(0, namePos) == "void" )
    {
      out << type.substr(0, namePos);
		} 
    else 
    {
			out << type.substr(0, namePos) << ' '
          << paramName << type.substr(namePos + 1);
		}

  }
  return out.str();
}

const Param &Param::Ellipsis()
{
  static const Param ellipsis;

  return ellipsis;
}

void Param::Dump() const { std::cout << *this << std::endl; }

std::ostream &operator<< (std::ostream &out, const Param &param)
{
  if (param.IsEllipsis())
    return out << "...";

  size_t pos = param.m_Type.find('%');
  assert(pos != std::string::npos);
  out << param.m_Type.substr(0, pos) << " %" << param.m_Name
    << "% " << param.m_Type.substr(pos + 1);
  if (!param.m_Default.empty())
    out << " = " << param.m_Default;
  return out;
}

std::string Method::QualifiedName() const
{
  if (const Iface *iface = GetIface())
    return iface->Name() + "::" + Name();
  else if (const IClass *iclass = GetIClass())
    return iclass->Name() + "::" + Name();
  else
    return std::string("<unknown>::") + Name();
}

std::string Method::FullName() const
{
  std::ostringstream out;

  if (const Iface *iface = GetIface())
    out << iface->Name();
  else if (const IClass *iclass = GetIClass())
    out << iclass->Name();
  else
    out << "<unknown>";
  out << "::" << Name() << '(';
  const std::vector<Param> &params = Params();
  size_t nParams = params.size();
  for (size_t i = 0; i < nParams; ++i)
  {
    const Param &param = params[i];
    if (i > 0)
      out << ", ";
    out << param.Decl("");
  }
  out << ')';
  if (IsConst())
    out << " const";
  return out.str();
}

Method *Method::GetImplMethod() const
{
  Iface *iface = GetIface();
  if (iface == NULL)
    return NULL;
  const IClass *iclass = iface->GetIClass();
  if (iclass == NULL)
    return NULL;
  const std::vector<Method *> &methods = iclass->Methods();
  Method *ifaceMethod = NULL;
  const std::string &name = Name();
  // First pass - try to identify the matching method based on the name and
  // the number of parameters.  If multiple matches are found, then these are
  // collected in the 'matchList'.
  std::vector<Method *> matchList;
  bool isConst = m_IsConst;
  size_t numParams = m_Params.size();
  for (
      std::vector<Method *>::const_iterator
	it = methods.begin(), itEnd = methods.end();
      it != itEnd;
      ++it)
  {
    Method *method = *it;
    if (isConst != method->IsConst())
      continue;
    if (numParams != method->Params().size())
      continue;
    if (name != method->Name())
      continue;
    if (ifaceMethod != NULL || !matchList.empty())
    {
      // Not unique.
      if (ifaceMethod != NULL)
	matchList.push_back(ifaceMethod);
      matchList.push_back(method);
      ifaceMethod = NULL;
    }
    else
      ifaceMethod = method;
  }
  if (ifaceMethod != NULL)
    return ifaceMethod;
  // Second pass - try to identify the matching method by comparing the
  // parameter types.
  if (!matchList.empty())
  {
    for (
	std::vector<Method *>::const_iterator
	  it = matchList.begin(), itEnd = matchList.end();
	it != itEnd;
	++it)
    {
      Method *method = *it;
      assert(numParams == method->Params().size());
      bool match = true;
      for (size_t i = 0; i < numParams; ++i)
      {
	const std::string &paramType = m_Params[i].Type();
	const std::string &ifaceParamType = method->Params()[i].Type();
	if (paramType != ifaceParamType)
	{
	  match = false;
	  break;
	}
      }
      if (match)
      {
	// Multiple matches may be due to #ifdef'ing, we'll simply return the
	// first match.
	ifaceMethod = method;
	break;
      }
    }
  }
  return ifaceMethod;
}

void Method::Dump() const { std::cout << *this << std::endl; }

std::ostream &operator<< (std::ostream &out, const Method &method)
{
  out << method.m_Type << ' ' << method.m_Name << '(';
  const char *comma = "";
  for (
      std::vector<Param>::const_iterator
	it = method.m_Params.begin(), itEnd = method.m_Params.end();
      it != itEnd;
      ++it)
  {
    out << comma << *it;
    comma = ", ";
  }
  out << ')';
  if (method.m_IsConst)
    out << " const";
  if (!method.m_DefaultImpl.empty())
    out << " { ... }";
  else if (method.m_IsPure)
    out << " = 0;";
  else
    out << ';';
  out << " [" << method.m_Pos.first << '-' << method.m_Pos.second << ']';
  if (method.GetIface() != NULL && method.m_HaveImpl)
    out << " (have implementation)";
  return out;
}

Iface::~Iface()
{
  for (
      std::vector<Method *>::const_iterator
	it = m_Methods.begin(), itEnd = m_Methods.end();
      it != itEnd;
      ++it)
    delete *it;
  m_Methods.clear();
}

Method *Iface::GetMethod(const char *name) const
{
  for (
      std::vector<Method *>::const_iterator
	it = m_Methods.begin(), itEnd = m_Methods.end();
      it != itEnd;
      ++it)
  {
    Method *method = *it;
    if (method->Name() == name)
      return method;
  }
  return NULL;
}

void Iface::Dump() const { std::cout << *this << std::endl; }

std::ostream &operator<< (std::ostream &out, const Iface &iface)
{
  out << "interface " << iface.m_Name;
  const char *comma = " : ";
  for (
      std::vector<Iface *>::const_iterator
	it = iface.m_Bases.begin(), itEnd = iface.m_Bases.end();
      it != itEnd;
      ++it)
  {
    out << comma << "public " << (*it)->m_Name;
    comma = ", ";
  }
  out << std::endl << '{' << std::endl;
  for (
      std::vector<Method *>::const_iterator
	it = iface.m_Methods.begin(), itEnd = iface.m_Methods.end();
      it != itEnd;
      ++it)
    out << "  " << **it << std::endl;
  out << "};" << std::endl;
  return out;
}

IClass::~IClass()
{
  for (
      std::vector<Method *>::const_iterator
	it = m_Methods.begin(), itEnd = m_Methods.end();
      it != itEnd;
      ++it)
    delete *it;
  m_Methods.clear();
}

void IClass::Dump() const { std::cout << *this << std::endl; }

std::ostream &operator<< (std::ostream &out, const IClass &iclass)
{
  out << "implementation class " << iclass.m_Name << std::endl
    << '{' << std::endl;
  for (
      std::vector<Method *>::const_iterator
	it = iclass.m_Methods.begin(), itEnd = iclass.m_Methods.end();
      it != itEnd;
      ++it)
    out << "  " << **it << std::endl;
  out << "};" << std::endl;
  return out;
}

std::string File::BaseName() const
{
  const char *sep = std::max(
      std::strrchr(m_FileName.c_str(), '/'),
      std::strrchr(m_FileName.c_str(), '\\'));
  const char *baseName = sep != NULL ? sep + 1 : m_FileName.c_str();
  const char *dot = std::strrchr(baseName, '.');

  if (dot != NULL)
    return std::string(baseName, dot - baseName);
  else
    return baseName;
}

bool File::Load() const
{
  if (m_Buffer != NULL)
    return true;
  assert(Util::CheckPath(m_FileName));
  m_Buffer = DeVirt::ReadFile(m_FileName);
  if (m_Buffer == NULL)
    return false;
  return true;
}

int File::OffsetToLine(size_t offset) const
{
  const char *buffer = Buffer();
  int line = 1;

  for (size_t i = 0; i < offset; ++i)
  {
    char c = buffer[i];
    if (c == 0)
      return -1;
    else if (c == '\n')
      ++line;
  }
  return line;
}

std::string File::LineMessage(size_t offset) const
{
  std::ostringstream out;

  out << deVirt.CodeRoot() << "\\" << m_FileName << '(' << OffsetToLine(offset) << ')';
  return out.str();
}

Iface *DeVirt::GetIface(const char *name)
{
  for (
      std::list<Iface>::iterator
	it = m_IfaceList.begin(), itEnd = m_IfaceList.end();
      it != itEnd;
      ++it)
  {
    Iface *iface = &*it;
    if (iface->m_Name == name)
      return iface;
  }
  m_IfaceList.push_back(Iface(name));
  return &m_IfaceList.back();
}

IClass *DeVirt::GetIClass(const char *name)
{
  for (
      std::list<IClass>::iterator
	it = m_IClassList.begin(), itEnd = m_IClassList.end();
      it != itEnd;
      ++it)
  {
    IClass *iclass = &*it;
    if (iclass->m_Name == name)
      return iclass;
  }
  m_IClassList.push_back(IClass(name));
  return &m_IClassList.back();
}

bool DeVirt::IsIface(const char *name) const
{
  for (
      std::list<Iface>::const_iterator
	it = m_IfaceList.begin(), itEnd = m_IfaceList.end();
      it != itEnd;
      ++it)
  {
    const Iface *iface = &*it;
    if (iface->m_Name == name)
      return !iface->m_Methods.empty();
  }
  return false;
}

File *DeVirt::CreateFile(const char *fileName, File::Type type)
{
  assert(Util::CheckPath(fileName));
  for (
      std::list<File>::iterator
	it = m_FileList.begin(), itEnd = m_FileList.end();
      it != itEnd;
      ++it)
  {
    File *file = &*it;
    if (file->FileName() == fileName)
      return NULL;
  }
  std::string filePath = fileName;
  Util::ConvertPath(filePath);
  m_FileList.push_back(File(filePath, type));
  return &m_FileList.back();
}

File *DeVirt::GetFile(const char *fileName, File::Type type)
{
  assert(Util::CheckPath(fileName));
  for (
      std::list<File>::iterator
	it = m_FileList.begin(), itEnd = m_FileList.end();
      it != itEnd;
      ++it)
  {
    File *file = &*it;
    if (file->FileName() == fileName)
    {
      if (file->GetType() != type)
	Warning(
	    "file '%s' added as interface _and_ implementation file",
	    fileName);
      return file;
    }
  }
  std::string filePath = fileName;
  Util::ConvertPath(filePath);
  m_FileList.push_back(File(filePath, type));
  return &m_FileList.back();
}

std::string DeVirt::TempDir()
{
	/*
  const char *tempDir = std::getenv("TEMP");

  if (tempDir == NULL)
    tempDir = std::getenv("TMP");
  if (tempDir == NULL)
    tempDir = "/tmp";
  return tempDir;
	*/
	return m_CacheDir + Util::PathSeperator + std::string("Temp");
}

void DeVirt::InfoLoc(const char *loc, const char *format, ...)
{
  std::va_list ap;

  if (!m_Verbose)
    return;
  va_start(ap, format);
  InfoLocV(loc, format, ap);
  va_end(ap);
}

void DeVirt::InfoLocV(const char *loc, const char *format, std::va_list ap)
{
  std::va_list ap1;
  size_t length;
  static const char prefix[] = "";

  if (!m_Verbose)
    return;
  va_copy(ap1, ap);
  length = vsnprintf(NULL, 0, format, ap1);
  va_end(ap1);
  char *buffer = new char[sizeof prefix + length];
  std::memcpy(buffer, prefix, sizeof prefix - 1);
  vsnprintf(buffer + sizeof prefix - 1, length + 1, format, ap);
  buffer[length + sizeof prefix - 1] = 0;
  Message(loc, buffer);
  delete[] buffer;
}

void DeVirt::Info(const char *format, ...)
{
  std::va_list ap;

  va_start(ap, format);
  InfoLocV(NULL, format, ap);
  va_end(ap);
}

void DeVirt::WarningLoc(const char *loc, const char *format, ...)
{
  std::va_list ap;

  va_start(ap, format);
  WarningLocV(loc, format, ap);
  va_end(ap);
}

void DeVirt::WarningLocV(const char *loc, const char *format, std::va_list ap)
{
  std::va_list ap1;
  size_t length;
  static const char prefix[] = "warning: ";

  va_copy(ap1, ap);
  length = vsnprintf(NULL, 0, format, ap1);
  va_end(ap1);
  char *buffer = new char[sizeof prefix + length];
  std::memcpy(buffer, prefix, sizeof prefix - 1);
  vsnprintf(buffer + sizeof prefix - 1, length + 1, format, ap);
  buffer[length + sizeof prefix - 1] = 0;
  Message(loc, buffer);
  delete[] buffer;
}

void DeVirt::Warning(const char *format, ...)
{
  std::va_list ap;

  va_start(ap, format);
  WarningLocV(NULL, format, ap);
  va_end(ap);
}

void DeVirt::ErrorLoc(const char *loc, const char *format, ...)
{
  std::va_list ap;

  va_start(ap, format);
  ErrorLocV(loc, format, ap);
  va_end(ap);
}

void DeVirt::ErrorLocV(const char *loc, const char *format, std::va_list ap)
{
  std::va_list ap1;
  size_t length;
  static const char prefix[] = "error: ";

  va_copy(ap1, ap);
  length = vsnprintf(NULL, 0, format, ap1);
  va_end(ap1);
  char *buffer = new char[sizeof prefix + length];
  std::memcpy(buffer, prefix, sizeof prefix - 1);
  vsnprintf(buffer + sizeof prefix - 1, length + 1, format, ap);
  buffer[length + sizeof prefix - 1] = 0;
  Message(loc, buffer);
  delete[] buffer;
}

void DeVirt::Error(const char *format, ...)
{
  std::va_list ap;

  va_start(ap, format);
  ErrorLocV(NULL, format, ap);
  va_end(ap);
}

void DeVirt::Message(const char *loc, const char *message)
{
  if (loc != NULL)
    std::cerr << loc << ": " << message << std::endl;
  else
    std::cerr << "devirt: " << message << std::endl;
}

char *DeVirt::ReadFile(const char *fileName)
{
  assert(Util::CheckPath(fileName));
  struct stat sb;

  if (stat(fileName, &sb) == -1)
  {
    Warning("can not access file '%s': %s", fileName, std::strerror(errno));
    return NULL;
  }
  assert(fileName && Util::CheckPath(fileName));
  std::ifstream in(fileName);
  if (!in)
  {
    Error("error opening file '%s' for reading", fileName);
    return NULL;
  }
  char *buffer = new char[sb.st_size + 1];
  ::memset(buffer,0,sizeof(char)*(sb.st_size+1));
  in.read(buffer, sb.st_size);
  if (!in.eof())
    if (!in)
    {
      Error("error reading file '%s'", fileName);
      delete[] buffer;
      return NULL;
    }
  buffer[sb.st_size] = 0;
  in.close();
  return buffer;
}

bool DeVirt::UpdateFile(
    const std::string &fileName,
    const std::string &content
    )
{
  assert(Util::CheckPath(fileName));
  struct stat sb;
  char *buffer = NULL;

  if (stat(fileName.c_str(), &sb) == -1)
  {
    if (errno != ENOENT)
    {
      Warning("can not access file '%s': %s",
	  fileName.c_str(), std::strerror(errno));
      return false;
    }
  }
  else
  {
    assert(Util::CheckPath(fileName));
    buffer = ReadFile(fileName);
    if (buffer == NULL)
      return false;
    if (content == buffer)
    {
      delete[] buffer;
      Info("file '%s' unchanged", fileName.c_str());
      return true;
    }
  }
  delete[] buffer;
  assert(Util::CheckPath(fileName));
  std::remove(fileName.c_str());
  std::ofstream out(fileName.c_str());
  if (!out)
  {
    Error("can not open file '%s' for writing", fileName.c_str());
    return false;
  }
  out << content;
  out.close();
  return true;
}

void DeVirt::Dump() const { std::cout << *this << std::endl; }

std::ostream &operator<< (std::ostream &out, const DeVirt &deVirt)
{
  out << "<DeVirt>" << std::endl
    << "iface keyword = '" << deVirt.m_IfaceKw << "'" << std::endl
    << "unique prefix keyword = '" << deVirt.m_UniquePrefixKw << "'"
      << std::endl
    << "unique virtual keyword = '" << deVirt.m_UniqueVirtualKw << "'"
      << std::endl
    << std::endl;
  for (
      std::list<Iface>::const_iterator
	it = deVirt.m_IfaceList.begin(), itEnd = deVirt.m_IfaceList.end();
      it != itEnd;
      ++it)
  {
    const Iface &iface = *it;
    out << iface << std::endl;
    if (iface.GetFile() == NULL)
      continue;
    out << "--- BEGIN DECL ---" << std::endl;
    iface.WriteDecl(out) << std::endl;
    out << "--- END DECL ---" << std::endl;
    out << "--- BEGIN DEF ---" << std::endl;
    iface.WriteDef(out) << std::endl;
    out << "--- END DEF ---" << std::endl;
  }
  for (
      std::list<IClass>::const_iterator
	it = deVirt.m_IClassList.begin(), itEnd = deVirt.m_IClassList.end();
      it != itEnd;
      ++it)
    out << *it << std::endl;
  out << "</DeVirt>" << std::endl;
  return out;
}

bool Method::operator ==(const Method &other) const
{
	if( m_Class != other.m_Class ) return false;
	if( m_Name != other.m_Name ) return false;
	if( m_MangledName != other.m_MangledName ) return false;
	if( m_Type != other.m_Type ) return false;
	if( m_DefaultImpl != other.m_DefaultImpl ) return false;	
	if( m_IsConst != other.m_IsConst ) return false;
	if( m_IsPure != other.m_IsPure ) return false;
	if( m_HaveImpl != other.m_HaveImpl ) return false;

	if( m_Params.size() != other.m_Params.size() ) return false;
	
	for(  size_t i = 0 ; i < m_Params.size() ; ++i )
	{
		if( m_Params[i] != other.m_Params[i] )
			return false;
	}
		
	return true;
}

bool Param::operator ==(const Param &other) const
{
	if( m_Type != other.m_Type ) return false;
	if( m_Name != other.m_Name ) return false;
	if( m_Default != other.m_Default ) return false;

	return true;
}


bool Param::operator !=(const Param &other) const
{
	return !(*this==other);
}

