#ifndef _DeVirt_h_
#define _DeVirt_h_ 1

#include <iostream>
#include <list>
#include <vector>
#include <map>
#include <string>
#include <cstdarg>
#include <cstddef>
#include <cassert>
#include <string>

#include "Util.h"

#if defined __GNUG__
  #define PRINTF(...) __attribute__ ((format (printf, __VA_ARGS__)))
#else
  #define PRINTF(...)
#endif

class Iface;
class IClass;
class Class;
class File;
class DeVirt;

// Define this to enable ASM level symbol remapping.
//
// If this option is enabled, then the patched header files will contain ASM
// statements that map the mangled names of the interface methods to the
// mangled names of the implementation methods.
#define DEVIRT_ASM_SYMBOL_MAPPING 1
//#undef DEVIRT_ASM_SYMBOL_MAPPING

// Define this the write a symbol mapping file to the output directory.
//
// The symbol mapping file contains linker options to define/rename the
// interface method symbols to the corresponding implementation methods.
#define DEVIRT_LD_SYMBOL_MAPPING 1
//#undef DEVIRT_LD_SYMBOL_MAPPING

#if defined(_MSC_VER)
  #include "stdint.h"
  #include <iostream>
  #include <algorithm>
  #include <fstream>
  #include <cctype>
	#include <direct.h>
  #include <wchar.h>

  #define __PRETTY_FUNCTION__ __FUNCTION__

  namespace std
  {
    static inline unsigned __int64 strtoui64(
	const char *const nptr,
	char **const endptr,
	const int base
	)
    {
      return ::_strtoui64(nptr, endptr, base);
    }
  }
  #define strtoll strtoui64

  static inline int access(const char *const fileName, const int)
  {
    assert(fileName && Util::CheckPath(fileName));
    return std::ifstream(fileName).is_open() ? 0 : -1;
  }
  #define R_OK 1
  #include <stdio.h>
  #include <stdlib.h>
  #define snprintf _snprintf
  typedef signed __int64 ssize_t;
  #pragma warning( disable : 4996 )
  #define va_copy(a,b) a=b
  #define S_ISDIR(mode) ((mode & _S_IFDIR) != 0)

	static inline int mkdir(const char *const dirName, const int)
	{
    assert(dirName && Util::CheckPath(dirName));
		return ::_mkdir(dirName);
	}

  // microsoft character validator assertion workaround when checking
  // unicode or multibyte characters. As a whitespace in unicode and
  // multibyte is the same as in ascii, microsoft still wants the
  // character in question to be ascii
  namespace std 
  {
    static inline int my_isspace(int c) 
    {
      return ::iswspace(c);
    }
  };
  #define isspace my_isspace


#else
  using std::snprintf;
  using std::vsnprintf;
#endif

// Type representing a range of offsets within the original source file.
typedef std::pair<unsigned, unsigned> PosT;

// A parameter of an method.
class Param
{
  friend class DeVirt;

  // The parameter type.
  //
  // This is a string containing the complete type declaration including
  // qualifiers.  The string contains a single '%' character as a placeholder
  // for the parameter name.  An empty parameter type indicates an ellipsis.
  std::string m_Type;

  // The parameter name, empty for none.
  std::string m_Name;

  // The default initializer, empty for none.
  std::string m_Default;

public:
	Param(){}
	Param( const Param &other) :
		m_Type(other.m_Type),
		m_Name(other.m_Name),
		m_Default(other.m_Default) {}

  bool IsEllipsis() const { return m_Type.empty(); }

  bool IsVoid() const
  {
    return m_Type == "void%" && m_Name.empty() && m_Default.empty();
  }

  const std::string &Type() const { return m_Type; }
  const std::string &Name() const { return m_Name; }
  const std::string &Default() const { return m_Default; }

  std::string Decl(const std::string &paramName) const;
  std::string Decl() const { return Decl(m_Name); }

  static const Param &Ellipsis();

  void Dump() const;
  friend std::ostream &operator<< (std::ostream &, const Param &);
	bool operator==( const Param & ) const;
	bool operator!=( const Param & ) const;
};

// An interface method.
class Method
{
  friend class DeVirt;

  // Link to the containing class.
  Class *m_Class;

  // Unqualified method name.
  const std::string m_Name;

  // The mangled name.
  //
  // Empty for unknown.
  std::string m_MangledName;

  // The method type.
  std::string m_Type;

  // Default implementation, empty for none.
  std::string m_DefaultImpl;

  // Parameter list.
  std::vector<Param> m_Params;

	// possibly extra macro which was appended to the function
	std::string m_extMacro;

  // The position within the source file.
  PosT m_Pos;

  // Flag indicating if the method is pure virtual.
  bool m_IsPure;

  // Flag indicating if the method is const qualified.
  //
  // No other qualifications are supported for interface methods.
  bool m_IsConst;

  // Flag indicating if a matching implementation method was found.
  //
  // Method matching is checked only based on the method name.  The first
  // method in the associated implementation class that matches the name will
  // set this flag.
  bool m_HaveImpl;

public:
  Method(const std::string &name)
    : m_Class(NULL),
      m_Name(name),
      m_IsPure(false),
      m_IsConst(false),
      m_HaveImpl(false)
  { }

	Method(const Method &other)
    : m_Class(other.m_Class),
      m_Name(other.m_Name),
      m_IsPure(other.m_IsPure),
      m_IsConst(other.m_IsConst),
      m_HaveImpl(other.m_HaveImpl)
  { }

  const std::string &Name() const { return m_Name; }
  std::string QualifiedName() const;
  std::string FullName() const;
  const std::string &MangledName() const { return m_MangledName; }
  const std::string &Type() const { return m_Type; }
  const std::string &DefaultImpl() const { return m_DefaultImpl; }
  const std::vector<Param> &Params() const { return m_Params; }
	const std::string &ExtMacro() const { return m_extMacro; }

  bool IsPure() const { return m_IsPure; }
  bool IsConst() const { return m_IsConst; }
  bool HaveImpl() const { return m_HaveImpl; }

  inline const char *Buffer() const;
  inline const char *Begin() const;
  inline const char *End() const;

  inline Iface *GetIface() const;
  inline IClass *GetIClass() const;

  // For an interface method, get the implementation method.
  //
  // Note that multiple implementation methods might match the interface
  // method (typically because of #ifdef'ed code).
  //
  // If no unique matching implementation method is found, the method returns
  // NULL.
  Method *GetImplMethod() const;

  // Write the method declaration to the specified output stream.
  //
  // If the method has an inline default implementation, then the inline code
  // is also written to the output stream.
  std::ostream &WriteDecl(std::ostream &) const;

	// Write only the declaration witouth extra macros or default implementations
	std::ostream &WriteInClassDecl(std::ostream &) const;

  // Write the method definition to the specified output stream.
  //
  // Only the declarative part of the definition is written - writing the
  // method body is up to the caller.
  std::ostream &WriteDef(std::ostream &) const;

  // Get the definition parameter name of the parameter at the specified
  // index.
  //
  // The parameter names have the form "$param<N>" where <N> is the parameter
  // index plus 1.
  std::string ParamName(unsigned index) const;

  void Dump() const;
  friend std::ostream &operator<< (std::ostream &, const Method &);
	bool operator==( const Method & ) const;
};

// Common base for interfaces and implementation classes.
class Class
{
	friend class DeVirt;
protected:
	std::string m_Alignment;
	
public:	
  virtual ~Class() { }
};

// An interface.
//
// Interfaces may form a hierarchy.  Default implementations of unique methods
// are permitted, but these must be inline and only one implementation per
// method is used.  Non-unique methods are kept virtual in the de-virtualized
// version of the class.
class Iface : public Class
{
  friend class DeVirt;

  // The interface name.
  std::string m_Name;

  // List of bases.
  std::vector<Iface *> m_Bases;

  // List of interface methods.
  //
  // The interface instance owns the methods in this list and will take care
  // of destroying them.
  std::vector<Method *> m_Methods;

  // Link to the containing source file.
  const File *m_File;

  // Link to the implementation class.
  const IClass *m_IClass;

  // Position within the source file.
  PosT m_Pos;

  // Flag indicating if all virtual methods should be considered unique.
  //
  // If this flag is not set, then only methods with the unique-prefix
  // (DeVirt::m_UniquePrefixKw) or with the alternative virtual keyword
  // (DeVirt::m_UniqueVirtualKw) are considered unique.
  //
  // If this flag is set, then the unique-prefix and/or keyword are still
  // accepted, just not required.
  bool m_AllUnique;


  // flag to indicate if a destructor was definied
  bool m_hasDstr;

public:
  Iface() : m_File(NULL), m_IClass(NULL), m_AllUnique(false), m_hasDstr(false) { }

  Iface(const std::string &name)
    : m_Name(name), m_File(NULL), m_IClass(NULL), m_AllUnique(false), m_hasDstr(false) 
  { }

  virtual ~Iface();

  // Write the patched declaration of the interface to the specified stream.
  std::ostream &WriteDecl(std::ostream &) const;

  // Write the wrapper method definitions to the specified stream.
  std::ostream &WriteDef(std::ostream &) const;

	// Write the wrapper method definitions in the devirtualized header
	std::ostream &WriteDefInplace(std::ostream &) const;
	
  // Write out the ASM code mapping the interface methods to the
  // implementation methods.
  std::ostream &WriteAsm(std::ostream &) const;

  // Write out the LD options mapping.
  std::ostream &WriteLdMap(std::ostream &) const;

  inline const char *Buffer() const;
  inline const char *Begin() const;
  inline const char *End() const;

  // Check if the interface contains a method with the specified name.
  bool HaveMethod(const char *name) const { return GetMethod(name) != NULL; }
  bool HaveMethod(const std::string &name) const
  { return HaveMethod(name.c_str()); }

  // Get the method with the specified name.
  //
  // If multiple methods with the specified name are defined, then only the
  // first match is returned.  The method returns NULL if no matching method
  // is found.
  Method *GetMethod(const char *name) const;
  Method *GetMethod(const std::string &name) const
  { return GetMethod(name.c_str()); }

  // Get all methods with the specified name.
  template <typename InserterT>
  size_t GetMethods(const char *name, InserterT inserter) const
  {
    size_t count = 0;

    for (
	std::vector<Method *>::const_iterator
	  it = m_Methods.begin(), itEnd = m_Methods.end();
	it != itEnd;
	++it)
    {
      Method *method = *it;
      if (method->Name() == name)
      {
	*inserter = method;
	++inserter;
	++count;
      }
    }
    return count;
  }
  template <typename InserterT>
  size_t GetMethods(const std::string &name, InserterT inserter) const
  {
    return GetMethods(name.c_str(), inserter);
  }

  const std::vector<Method *> &Methods() const { return m_Methods; }

  const std::string &Name() const { return m_Name; }
  const File *GetFile() const { return m_File; }
  const IClass *GetIClass() const { return m_IClass; }

  bool IsEmpty() const { return m_Methods.empty(); }

  void Dump() const;
  friend std::ostream &operator<< (std::ostream &, const Iface &);
};

// A class implementing an interface.
//
// We're not interested in the hierarchy here.  All pure virtual methods of
// the interface _must_ be implemented in the class.  Virtual methods with
// default implementations may be omitted.
//
// All implemented method _must_ be declared with the
// 'DeVirt::m_UniqueVirtualKw' or no virtual keyword at all.  We'll emit a
// warning if this is not the case (because the implementation header are
// _not_ replaced, a 'virtual' keyword would introduce an unneeded virtual
// method).
class IClass : public Class
{
  friend class DeVirt;

  // The class name.
  std::string m_Name;

  // The interface implemented by the implementation class.
  Iface * m_Iface;

  // List of implemented methods.
  std::vector<Method *> m_Methods;

  // Link to the containing source file.
  const File *m_File;

public:
  IClass() : m_Iface(NULL), m_File(NULL) { }

  IClass(const std::string &name)
    : m_Name(name), m_Iface(NULL), m_File(NULL)
  { }

  virtual ~IClass();

  const std::string &Name() const { return m_Name; }
  Iface * GetIface() const { return m_Iface; }
  const File *GetFile() const { return m_File; }
  const std::vector<Method *> &Methods() const { return m_Methods; }

  void Dump() const;
  friend std::ostream &operator<< (std::ostream &, const IClass &);
};

inline Iface *Method::GetIface() const
{
  return dynamic_cast<Iface *>(m_Class);

}
inline IClass *Method::GetIClass() const
{
  return dynamic_cast<IClass *>(m_Class);
}

// Structure representing in interface declaration header file.
class File
{
  friend class DeVirt;

public:
  enum Type { IFACE_FILE, IMPL_FILE };

private:
  // The name of the file.
  std::string m_FileName;

  // The file type.
  Type m_Type;

  // List of interfaces defined in the file.
  //
  // The Iface instances are owned by the DeVirt instance.
  std::vector<Iface *> m_IfaceList;

  // List of implementation classes in the file.
  //
  // The IClass instances are owned by the DeVirt instance.
  std::vector<IClass *> m_IClassList;

  // Buffer holding the contents of the file.
  //
  // The buffer is null-terminated.  (Allocated size is one byte more than the
  // file size.)
  mutable const char *m_Buffer;

public:
  File(const std::string &fileName, Type type)
    : m_FileName(fileName),
      m_Type(type),
      m_Buffer(NULL)
  { 
    assert(Util::CheckPath(m_FileName));
  }

  ~File() { Unload(); }

  // Get the file name.
  const std::string &FileName() const { return m_FileName; }

  // Get the base name of the file name, not including the suffix.
  std::string BaseName() const;

  // Get the file type.
  Type GetType() const { return m_Type; }

  // Load the file into the buffer.
  //
  // If the buffer is already loaded, then the method has no effect and
  // returns 'true'.
  //
  // Returs 'true' on success and 'false' in case of an error.
  bool Load() const;

  // Un-load the buffer.
  void Unload() const { delete[] m_Buffer; m_Buffer = NULL; }

  // Get the buffer pointer.
  //
  // This method implies a call to Load().  In case of an error, a NULL
  // pointer is returned.
  const char *Buffer() const { Load(); return m_Buffer; }

  // Convert an offset to a line number.
  //
  // The method returns -1 if the offset is out of range.
  int OffsetToLine(size_t offset) const;

  // Get a file, line message for the specified offset within the file.
  std::string LineMessage(size_t offset) const;
};

const char *Method::Buffer() const
{
  return dynamic_cast<const Iface *>(m_Class)->Buffer();
}
const char *Method::Begin() const { return Buffer() + m_Pos.first; }
const char *Method::End() const { return Buffer() + m_Pos.second; }

const char *Iface::Buffer() const { return m_File->Buffer(); }
const char *Iface::Begin() const { return Buffer() + m_Pos.first; }
const char *Iface::End() const { return Buffer() + m_Pos.second; }

// De-virtualizer.
class DeVirt
{
  // The interface keyword.
  //
  // Structures and/or classes using this prefix keyword are recognized as
  // interfaces and all methods are assumed to be unique.
  //
  // Structures/classes are also recognized as interfaces if they contain at
  // least one virtual interface method.
  std::string m_IfaceKw;

  // The unique keyword.
  //
  // This keyword may be used as a prefix keyword for unique virtual methods.
  std::string m_UniquePrefixKw;

  // The unique virtual keyword.
  //
  // Alternative virtual keyword for marking unique virtual methods.
  std::string m_UniqueVirtualKw;

	// base directory for all implementations
	std::string m_BaseDir;

  // List of interfaces.
  std::list<Iface> m_IfaceList;

  // List of implementation classes.
  std::list<IClass> m_IClassList;

  // List of interface files.
  std::list<File> m_FileList;

  // Flag indicating verbose operation.
  static bool m_Verbose;

  // The output directory for patched interface files.
  std::string m_OutputDir;

  // The directory holding the mangled names cache.
  std::string m_CacheDir;

  // The compilation command.
  std::string m_CompileCommand;

	// code Root path
	std::string m_CodeRoot;

	// util function to write out a textsegment while patching the include path of
	// #include "file", was necessary since the devirtualized header resists in a new directory
	void WriteWithPatchingInclude( std::ostream &out, char *p, size_t num, const std::string &path ) const;

	// util function to find out if a header was devirtualized
	bool IsDevirtualizedHeader( const std::string &file ) const;

  Iface *GetIface(const char *name);
  Iface *GetIface(const std::string &name) { return GetIface(name.c_str()); }
  IClass *GetIClass(const char *name);
  IClass *GetIClass(const std::string &name)
  { return GetIClass(name.c_str()); }

  bool IsIface(const char *name) const;
  bool IsIface(const std::string &name) const
  { return IsIface(name.c_str()); }

  File *CreateFile(const char *fileName, File::Type type);
  File *CreateFile(const std::string &fileName, File::Type type)
  { return CreateFile(fileName.c_str(), type); }

  File *GetFile(const char *fileName, File::Type type);
  File *GetFile(const std::string &fileName, File::Type type)
  { return GetFile(fileName.c_str(), type); }

  Class *ParseClass(char **code, char *base, const File *);

  Class *ParseClass(char *&code, char *base, const File *file)
  {
    char *p = code;
    Class *clazz = ParseClass(&p, base, file);
    code = p;
    return clazz;
  }

  Method *ParseMethod(char **code, char *base, const File *file,
      bool haveIfaceKw, Iface *iface) const;

  Method *ParseMethod(
      char *&code,
      char *base,
      const File *file,
      bool haveIfaceKw,
      Iface *iface
      ) const
  {
    char *p = code;
    Method *method = ParseMethod(&p, base, file, haveIfaceKw, iface);
    code = p;
    return method;
  }

  static bool ParseParam(char **code, Method *method);

  static bool ParseParam(char *&code, Method *method)
  {
    char *p = code;
    bool returnValue = ParseParam(&p, method);
    code = p;
    return returnValue;
  }

  std::string GenCompileCommand(const char *cppName, const char *asmName, const char *path)
    const;
  std::string GenCompileCommand(const std::string &cppName,
      const std::string &asmName,
			const std::string &path) const
  {
    return GenCompileCommand(cppName.c_str(), asmName.c_str(), path.c_str());
  }

  // Load mangled names from the specified symbol list.
  bool LoadMangledNames(const std::map<std::string, std::string> &, File *,
      bool reportMissing);

  // Load mangled names from the cache.
  bool LoadMangledNames(std::istream &, File *);

  // Save mangled names to the cache.
  void SaveMangledNames(std::ostream &, const File *) const;

  // Load mangled names from an ASM file.
  bool LoadMangledNamesFromAsm(std::istream &, File *);

  // Get the name of the symbol cache file for the specified interface file.
  std::string CacheFileName(const File *) const;

  // Load the mangled names of the specified interface file from the cache.
  //
  // The method checks if the cache file is newer than the interface file.
  void LoadMangledNamesFromCache(File *);

  // Save the mangled names to the cache.
  void SaveMangledNamesToCache(const File *) const;

  // Obtain all mangled names required for generating the output files.
  bool GenMangledNames();

	void GenMangledNames( File *file, int &nErrors );

  bool WriteFile(std::ostream &, const File &,
      std::vector<const Iface *> &, bool bIsDummyHeader ) const;
  bool WriteFile(std::ostream &out, const File &file) const
  {
    std::vector<const Iface *> ifaceList;
    return WriteFile(out, file, ifaceList, false);
  }

	void WriteUeberFileIncludes();

public:
  DeVirt()
    : m_IfaceKw("UNIQUE_IFACE"),
      m_UniquePrefixKw("UNIQUE"),
      m_UniqueVirtualKw("VIRTUAL")
  { }

  static void SetVerbose(bool enable = true) { m_Verbose = enable; }
  static bool Verbose() { return m_Verbose; }

  void SetOutputDir(const char *dir) 
  { m_OutputDir = dir; Util::ConvertPath(m_OutputDir); }

	void SetBaseDir( const std::string &dir )
	{ m_BaseDir = dir; }
  void SetOutputDir(const std::string &dir) 
  { m_OutputDir = dir; Util::ConvertPath(m_OutputDir); }
  const std::string &OutputDir() const { return m_OutputDir; }

  void SetCacheDir(const char *dir) 
  { 
    m_CacheDir = dir; Util::ConvertPath(m_CacheDir);
  }
  void SetCacheDir(const std::string &dir) 
  { 
    m_CacheDir = dir; Util::ConvertPath(m_CacheDir);
  }
  void SetDefaultCacheDir() 
  { 
	  std::string str = TempDir();
    m_CacheDir = Util::ConvertPath(str)
      + Util::PathSeperator + "mangled_cache"; 
  }
  const std::string &CacheDir() const { return m_CacheDir; }

  void SetCompileCommand(const char *cmd) { m_CompileCommand = cmd; }
  void SetCompileCommand(const std::string &cmd) { m_CompileCommand = cmd; }
  const std::string &CompileComand() const { return m_CompileCommand; }

	void SetCodeRoot( const std::string &code_root ) { m_CodeRoot = code_root; }
	const std::string &CodeRoot() { return m_CodeRoot; }

  std::string TempDir();

  // Parse the specified header file.
  bool ParseFile(const char *fileName, File::Type fileType);
  bool ParseFile(const std::string &fileName, File::Type fileType)
  { return ParseFile(fileName.c_str(), fileType); }

  // Write the output file.
  bool Write();

  // Messages.
  PRINTF(2, 3) static void InfoLoc(
      const char *loc, const char *format, ...);
  PRINTF(2, 0) static void InfoLocV(
      const char *loc, const char *format, std::va_list);
  PRINTF(1, 2) static void Info(const char *format, ...);
  PRINTF(2, 3) static void WarningLoc(
      const char *loc, const char *format, ...);
  PRINTF(2, 0) static void WarningLocV(
      const char *loc, const char *format, std::va_list);
  PRINTF(1, 2) static void Warning(const char *format, ...);
  PRINTF(2, 3) static void ErrorLoc(
      const char *loc, const char *format, ...);
  PRINTF(2, 0) static void ErrorLocV(
      const char *loc, const char *format, std::va_list);
  PRINTF(1, 2) static void Error(const char *format, ...);
  static void Message(const char *loc, const char *message);
  static void Message(const char *loc, const std::string &message)
  {
    return Message(loc, message.c_str());
  }

  static char *ReadFile(const char *fileName);
  static char *ReadFile(const std::string &fileName)
  { return ReadFile(fileName.c_str()); }

  static bool UpdateFile(const std::string &fileName,
      const std::string &content);

  void Dump() const;
  friend std::ostream &operator<< (std::ostream &, const DeVirt &);
};

#endif

// vim:expandtab

