#include "DeVirt.h"
#include "Getopt.h"

#include <iostream>
#include <fstream>
#include <cstdlib>
#include <sys/stat.h>
#include <set>

// Global de-virtualizer instance.
DeVirt deVirt;


void ParseFiles( const std::vector<std::string> &rFileList, std::string prefix, std::vector<std::string> &failedFiles, File::Type type )
{
	for (
    std::vector<std::string>::const_iterator
      it = rFileList.begin(), itEnd = rFileList.end();
    it != itEnd;
    ++it)
  {
    std::string sFile = prefix + *it;
    if (!deVirt.ParseFile(Util::ConvertPath( sFile ), type))
    {
			failedFiles.push_back( *it );
    }
  }
}

int main(int argc, char **argv)
{	
	if( argc < 2 )
	{
		deVirt.Error("Please specify options file to use as first parameter");
		std::abort();
	}
	
	std::vector<char*> args;

	// open option file
	std::fstream options_file( argv[1] );
	if( !options_file.is_open() )
	{
		deVirt.Error("Error opening the devirt options file: %s", argv[1] );
		std::abort();
	}

	std::string options;
	std::getline( options_file, options );

	options_file.close();
	
	// build args array	
	args.push_back( argv[0] );
	args.push_back( &options[0] );
	for( std::string::iterator cur = options.begin() ; cur != options.end() ; ++cur )
	{		
		if( *cur == '\'' )
		{
			++cur;
			while( *cur != '\'' )
			{
				*(cur-1) = *cur;
				if( cur == options.end() )
				{
					deVirt.Error("Error parsing the config options");
					std::abort();
				}
				++cur;				
			}
			*(cur-1) = '\0';
			*cur = '\0';
		}

		if( *cur == ' ' )
		{
			*cur = '\0';
			if( cur + 1 != options.end() )
				args.push_back( &(*(cur + 1)) );
		}
			
	}

  // Parse the command line. 
	std::vector<std::string> ihFileList;
  std::vector<std::string> chFileList;

	std::vector<std::string> ihGameFileList;
  std::vector<std::string> chGameFileList;

	std::set<std::string> devirtDirectoryList;
	std::vector<std::string> failedFiles;
	std::string baseDir;

	//TODO find better names instead of g and k..
	static const char opt[] = "hvi:c:o:m:C:B:g:k:R:j:";
  static const char helpText[] =
    "devirt: Interface Devirtualization Tool\n"
    "Options:\n"
    "-i INPUT_IHFILE\n"
    "      Specify an input header file containing interface definitions.\n"
    "-c INPUT_CHFILE\n"
    "      Specify an input header file containing class declarations of\n"
    "      classes implementing the interfaces.\n"
    "-o OUTPUT_DIR\n"
    "      Specify the output directory.  For every input header file\n"
    "      containing interfaces, a patched output header file with the same\n"
    "      name is created in the specified output directory.\n"
    "-m CACHE_DIR\n"
    "      Specify the directory holding the mangled names cache.\n"
    "-C COMPILE_COMMAND\n"
    "      Specify the command to compile the generated C++ file for mapping\n"
    "      methods to mangled names from C++ to assembly.  This command should\n"
    "      be close to the standard compilation command of the project.  The\n"
    "      command should _not_ include a -o option (this option will be\n"
    "      appended by the tool).\n"
    "-v    Verbose operation.\n"
    "-h    Display this help screen and exit.\n";
  Getopt::Reset();
  while (true)
  {
    int c = Getopt::Getopt( (int)args.size(), &args[0], opt);
    if (c == -1)
      break;
    switch (c)
    {
    case 'h':
      std::cout << helpText << std::endl;
      std::exit(EXIT_SUCCESS);
    case 'v':
      deVirt.SetVerbose();
      break;
    case 'i':
			ihFileList.push_back( Util::ConvertPath( std::string(Getopt::Optarg) ) );
			devirtDirectoryList.insert( deVirt.OutputDir() + Util::PathSeperator + 
																	baseDir + Util::PathSeperator + 
																	Util::GetPath( Util::ConvertPath(  std::string(Getopt::Optarg) ) ) );
      break;
    case 'c':
      chFileList.push_back( Util::ConvertPath(  std::string(Getopt::Optarg) ));
      break;
		case 'g':
			ihGameFileList.push_back( Util::ConvertPath(  std::string(Getopt::Optarg) ) );
			devirtDirectoryList.insert( deVirt.OutputDir() + Util::PathSeperator + 
																	baseDir + Util::PathSeperator + 
																	Util::GetPath( Util::ConvertPath( std::string(Getopt::Optarg) ) ) );
      break;
    case 'k':
      chGameFileList.push_back( Util::ConvertPath( std::string(Getopt::Optarg) ));
      break;
    case 'o':
      deVirt.SetOutputDir(  Util::ConvertPath( std::string(Getopt::Optarg)) );
      break;
    case 'm':
      deVirt.SetCacheDir( Util::ConvertPath( std::string(Getopt::Optarg)) );
      break;
    case 'C':
      deVirt.SetCompileCommand(Getopt::Optarg);
      break;
		case 'B':
			baseDir = Util::ConvertPath( std::string(Getopt::Optarg));
			deVirt.SetBaseDir( baseDir );
			break;
		case 'R':
			deVirt.SetCodeRoot( Util::ConvertPath( std::string(Getopt::Optarg)) );
			break;
		case 'j':
			devirtDirectoryList.insert( deVirt.OutputDir() + Util::PathSeperator + Util::ConvertPath(  std::string(Getopt::Optarg) ) );
			break;
    case '?':
      std::exit(EXIT_FAILURE);		
    default:
      std::abort();
    }
  }
  if (deVirt.CacheDir().empty())
    deVirt.SetDefaultCacheDir();

	devirtDirectoryList.insert( deVirt.CacheDir() );	
	devirtDirectoryList.insert( deVirt.TempDir() );
	devirtDirectoryList.insert( deVirt.OutputDir() + Util::PathSeperator + std::string("CryEngine"));	
	
	if( !Util::CreateDirectories( devirtDirectoryList ) )
	{
		deVirt.Error("Error while Creating needed directories");
		std::abort();
	}

	ParseFiles( ihFileList, baseDir, failedFiles, File::IFACE_FILE );
	ParseFiles( ihGameFileList, "", failedFiles, File::IFACE_FILE );
 
  
	ParseFiles( chFileList, baseDir, failedFiles, File::IMPL_FILE );
	ParseFiles( chGameFileList, "", failedFiles, File::IMPL_FILE );
 

	for( size_t i = 0 ; i < 	failedFiles.size() ; ++i)
  {
    std::string failedFileS = failedFiles[i];
    DeVirt::Warning("error accessing/parsing file %s", 
                   Util::ConvertPath(failedFileS).c_str());

  }

  if (!deVirt.Write())
    std::exit(EXIT_FAILURE);

  return 0;
}

