#include "DeVirt.h"
#include "Util.h"

#include <iostream>
#include <sstream>
#include <memory>
#include <cstring>
#include <cctype>
#include <cassert>

// Magic number to indicate that a destructor was found
Method* FOUND_DESTRUCTOR = (Method*)0xFFFFFFF;

namespace
{
  inline bool IsInitId(char c)
  {
    return std::isalpha(c) || c == '_' || c == '$';
  }

  inline bool IsId(char c)
  {
    return std::isalnum(c) || c == '_' || c == '$';
  }

  inline bool ParseKw(char *&code, const char *kw)
  {
    size_t kwLen = std::strlen(kw);

    if (std::strncmp(code, kw, kwLen))
      return false;
    char c = code[kwLen];
    if (IsId(c))
      return false;
    code += kwLen;
    return true;
  }

  inline bool ParseKw(char *&code, const std::string &kw)
  {
    return ParseKw(code, kw.c_str());
  }

  inline bool ParseId(char *&code, std::string &name)
  {
    char *p = code;

    if (!IsInitId(*p))
      return false;
    for (++p; IsId(*p); ++p);
    name.assign(code, p - code);
    code = p;
    return true;
  }

  inline bool ParseSimpleType(char *&code)
  {
    char *p = code;
    bool match = false;

    // Maybe subject to optimization.  We'll accept any sequence of recognized
    // type keyword, without checking if that sequence makes sense in C.  The
    // sequence 'signed long long int' is accepted as well as
    // 'float unsigned long double').
    while (ParseKw(p, "char") || ParseKw(p, "int") 
      || ParseKw(p, "signed") || ParseKw(p, "unsigned")
      || ParseKw(p, "short") || ParseKw(p, "long")
      || ParseKw(p, "float") || ParseKw(p, "double"))
    {
      match = true;
      for (; std::isspace(*p); ++p);
    }
    if (match)
    {
      code = p;
      return true;
    }
    return false;
  }

  inline bool IsDelim(char delim)
  {
    return delim == '(' || delim == '[' || delim == '{';
  }

  inline char InvDelim(char delim)
  {
    switch (delim)
    {
    case '(': return ')';
    case '[': return ']';
    case '{': return '}';
    }
    return delim;
  }

  // Skip a matched pair of delimiters.
  char *SkipMatched(char *code, char delim)
  {
    char *p = code;

    while (true)
    {
      char c = *p;
      if (c == 0) return p;
      if (c == delim) return p + 1;
      ++p;
      if (IsDelim(c))
        p = SkipMatched(p, InvDelim(c));
    }
  }

  // Skip up to the specified character.  Characters in nested delimiter pairs
  // are skipped.
  char *SkipTo(char *code, char to)
  {
    char *p = code;

    while (true)
    {
      char c = *p;
      if (c == 0 || c == to) return p;
      ++p;
      if (IsDelim(c))
        p = SkipMatched(p, InvDelim(c));
    }
  }

  // Skip a complete statement.
  char *SkipStmt(char *code)
  {
    char *p = code;

    //special skip for macros, since they don't end with ';'
    if( code[-1] == '\n' && code[0] == '#' )
    {
      return SkipTo( p, '\n' );
    }

    while (true)
    {
      char c = *p;
      if (c == 0) return p;
      ++p;
      if (IsDelim(c))
      {
        p = SkipMatched(p, InvDelim(c));
        if (c == '{')
        {
          // This is the end of the statement.  If this is followed by a
          // semicolon, then we'll assume that the semicolon is part of the
          // statement, otherwise the statement ends here.
          char *q = p;
          for (; std::isspace(*p); ++p);
          return *p == ';' ? p + 1 : q;
        }
      }
      else if (c == ';')
        return p;
    }
  }

}

bool DeVirt::ParseFile(const char *fileName, File::Type fileType)
{
  assert(Util::CheckPath(fileName));
  File *file = CreateFile(fileName, fileType);
  if (file == NULL)
  {
    Warning("input file '%s' specified more than once", fileName);
    return true;
  }

  const char *fileBuffer = file->Buffer();
  if (fileBuffer == NULL) return false;
  size_t bufferLen = std::strlen(fileBuffer);
  char *buffer = new char[bufferLen + 1];
  std::copy(fileBuffer, fileBuffer + bufferLen + 1, buffer);

  // Copy literal text to the literal buffer.  This is done to move parts of
  // the code that are not of interest out of the way (replace them by
  // whitespace).  The following elements are moved to the literal buffer:
  // - Comments.
  // - Preprocessor directives and #pragma.
  // - String and character constants - both are filled with spaces (the
  //   quotes remain in the buffer).
  char *literalBuffer = new char[bufferLen + 1];
  std::memset(literalBuffer, 0, bufferLen + 1);
  char *p = buffer;
  bool lineStart = true;
  while (true)
  {
    for (; std::isspace(*p); ++p);
    if (*p == 0)
      break;
    if (lineStart && *p == '#')
    {
      char *q = literalBuffer + (p - buffer);
      while (true)
      {
        char c = *p;
        if (c == '\r' || c == '\n' || c == 0)
          break;
        *p++ = ' ';
        *q++ = c;
        if (c == '\\')
        {
          if (p[0] == '\r' && p[1] == '\n')
          {
            q[0] = '\r', q[1] = '\n';
            q += 2;
            p += 2;
          }
          else if (p[0] == '\n')
          {
            q[0] = '\n';
            ++q;
             ++p;
          }
        }
      }
      continue;
    }
    if (*p == '"' || *p == '\'')
    {
      char *q = literalBuffer + (p - buffer);
      char quote = *p++;
      *q++ = quote;
      while (true)
      {
        int c = *p;
        if (c == quote || c == 0) 
        {
          *q++ = quote;
          ++p;
          break;
        }
        *p++ = ' ';
        *q++ = c;
        if (c == '\\' && p[0] != 0)
        {
          q[0] = p[0];
          if (p[0] != '\n' && p[0] != '\r')
          p[0] = ' ';
          ++q;
          ++p;
        }
      }
      lineStart = false;
      continue;
    }
    if (p[0] == '/' && p[1] == '/')
    {
      char *q = literalBuffer + (p - buffer);
      while (true)
      {
        char c = *p;
        if (c == '\r' || c == '\n' || c == 0)
          break;
        *p++ = ' ';
        *q++ = c;
      }
      lineStart = true;
      continue;
    }
    if (p[0] == '/' && p[1] == '*' && p[2] != 0)
    {
      char *q = literalBuffer + (p - buffer);
      q[0] = '/', q[1] = '*', q[2] = p[2];
      p[0] = ' ', p[1] = ' ', p[2] = ' ';
      q += 3;
      p += 3;
      while (true)
      {
        char c = *p;
        if (c == 0)
          break;
        *p++ = ' ';
        *q++ = c;
        if (c == '*' && p[0] == '/')
        {
          q[0] = '/';
          p[0] = ' ';
          p += 1;
          break;
        }
      }
      lineStart = false;
//       continue;
    }
    if (*p == '\n')
    {
      ++p;
      lineStart = true;
      continue;
    }
    ++p;
  }

  // Merge string literals.  When the string contents are restored from the
  // literal buffer, then this will also bring back the quoting characters and
  // the result will be the same string literals as in the original source.
  p = buffer;
  char *merge = NULL;
  while (true)
  {
    for (; std::isspace(*p); ++p);
    if (*p == 0)
      break;
    if (*p == '"')
    {
      if (merge != NULL)
      {
        *merge = ' ';
        *p = ' ';
      }
      p = std::strchr(p + 1, '"');
      if (p == NULL)
        break;
      merge = p;
    }
    else if (!std::isspace(*p))
      merge = NULL;
    ++p;
  }

  // Parse the file.
  p = buffer;
  while (true)
  {
    for (; std::isspace(*p); ++p);

    if (*p == 0)
      break;

    Class *clazz = ParseClass(p, buffer, file);
    if (clazz != NULL)
    {
      Iface *iface = dynamic_cast<Iface *>(clazz);
      if (iface != NULL)
        file->m_IfaceList.push_back(iface);
      else
        file->m_IClassList.push_back(dynamic_cast<IClass *>(clazz));
      continue;
    }
    p = SkipStmt(p);
  }

  delete[] buffer;
  delete[] literalBuffer;
  return true;
}

Class *DeVirt::ParseClass(char **code, char *base, const File *file)
{
  char *p = *code;
  bool haveIfaceKw = false;
  bool foundDestr = false;
	std::string allignment;
	
  for (; std::isspace(*p); ++p);
  char *classBegin = p;

  if (ParseKw(p, m_IfaceKw))
  {
    if (file->GetType() == File::IMPL_FILE)
      WarningLoc(
        file->LineMessage(classBegin - base).c_str(),
        "interface keyword '%s' ignore in implementation file",
        m_IfaceKw.c_str());
    else
      haveIfaceKw = true;
  }

  for (; std::isspace(*p); ++p);

  if (!ParseKw(p, "struct") && !ParseKw(p, "class"))
    return NULL;

  for (; std::isspace(*p); ++p);
  std::string name;

  if (!ParseId(p, name))
    return NULL;

  for (; std::isspace(*p); ++p);
  bool haveNonPublicBases = false;
  std::list<std::string> bases;
  Iface *iface = NULL;

  if (*p == ':')
  {
    bool first = true;
    while (true)
    {
      for (++p; std::isspace(*p); ++p);

      if (ParseKw(p, "private") || ParseKw(p, "protected"))
        haveNonPublicBases = true;
      else
        ParseKw(p, "public"); // Skip 'public' if present.

      for (; std::isspace(*p); ++p);
      std::string baseId;

      if (!ParseId(p, baseId))
        return NULL;

      if( first && IsIface(baseId) )
      {
        iface = GetIface(baseId);
      }

      for (; std::isspace(*p); ++p);

      if (*p != ',')
        break;

      first =false;

    }
  }

  if (haveNonPublicBases && iface == NULL)
  {
    if (haveIfaceKw)
      WarningLoc(
        file->LineMessage(static_cast<size_t>(p - base)).c_str(),
        "interface '%s': "
        "base classes of interfaces must be public",
        name.c_str());
    return NULL;
  }
  if (*p != '{')
    return NULL;
  ++p;
  std::list<Method *> methods;
  while (true)
  {
    Method *method = NULL;

    for (; std::isspace(*p); ++p);
    if (*p == 0)
      return NULL;
    if (*p == '}')
      break;

    method = ParseMethod(p, base, file, haveIfaceKw, iface);
    if( method == FOUND_DESTRUCTOR )
    {
      foundDestr = true;
      p = SkipStmt(p);
      continue;
    }
    else if (method == NULL)
    {
      p = SkipStmt(p);
      continue;
    }
    methods.push_back(method);

  }

  assert(*p == '}');
  for (++p; std::isspace(*p); ++p);
	
	//skip _ALIGN(x) macro
	if( ParseKw(p, "_ALIGN") )
	{
		char *start, *end;
// 		p += 6; // == strlen("_ALIGN")
		assert( *p == '(' );
		start = p + 1;
		p = SkipMatched( p+1, ')' );
		end = p - 1;
				
		allignment = std::string( start, end - start );
	}
	
  if (*p != ';' )
  {
    for (
      std::list<Method *>::const_iterator
      it = methods.begin(), itEnd = methods.end();
      it != itEnd;
      ++it)
        delete *it;
    methods.clear();
    return NULL;
  }
  ++p;
  *code = p;

  if ( iface != NULL )
  {
    IClass *iclass = GetIClass(name);
    iclass->m_File = file;
    iclass->m_Iface = iface;
    for (
      std::list<Method *>::const_iterator
      it = methods.begin(), itEnd = methods.end();
      it != itEnd;
      ++it)
        (**it).m_Class = iclass;
    std::copy(
      methods.begin(),
      methods.end(),
      std::back_inserter(iclass->m_Methods));

    if ( iface->m_IClass != NULL )
    {
			// since we don't check different compile paths here, we emit this warning only when the class name differs
			if( iface->m_IClass->Name() != iclass->Name() )
			{
				WarningLoc(
					file->LineMessage(static_cast<size_t>(classBegin - base)).c_str(),
					"multiple implementations of interface '%s'",
					iface->Name().c_str());
			}
    }
    else
      iface->m_IClass = iclass;

	  iclass->m_Alignment = allignment;
    return iclass;
  }
  else
  {
    iface = GetIface(name);
    iface->m_File = file;
    iface->m_hasDstr = foundDestr;

    for (
      std::list<std::string>::const_iterator
      it = bases.begin(), itEnd = bases.end();
      it != itEnd;
      ++it)
    {
      Iface *baseIface = GetIface(*it);
      iface->m_Bases.push_back(baseIface);
    }
    if (haveIfaceKw)
      iface->m_AllUnique = true;
    for (
      std::list<Method *>::const_iterator
      it = methods.begin(), itEnd = methods.end();
      it != itEnd;
      ++it)
        (**it).m_Class = iface;
    std::copy(
      methods.begin(),
      methods.end(),
      std::back_inserter(iface->m_Methods));

    char *classEnd = p;
    unsigned classBeginOffset = static_cast<unsigned>(classBegin - base);
    unsigned classEndOffset = static_cast<unsigned>(classEnd - base);
    iface->m_Pos = std::make_pair(classBeginOffset, classEndOffset);
	  
	  iface->m_Alignment = allignment;
    return iface;
  }
}

Method *DeVirt::ParseMethod(
    char **code,
    char *base,
    const File *file,
    bool haveIfaceKw,
    Iface *iface
    ) const
{
  char *p = *code;
  File::Type fileType = file->GetType();
  bool isUnique = haveIfaceKw;

  for (; std::isspace(*p); ++p);
  // Ignore 'public:', 'protected:', 'private:'.
  if (ParseKw(p, "public")
      || ParseKw(p, "protected")
      || ParseKw(p, "private"))
  {
    for (; std::isspace(*p); ++p);
    if (*p != ':')
      return NULL;
    for (++p; std::isspace(*p); ++p);
  }
  char *methodDeclBegin = p;
  bool haveVirtualKw = false;
  if (ParseKw(p, m_UniquePrefixKw))
  {
    for (; std::isspace(*p); ++p);
    if (!ParseKw(p, "virtual"))
		{
			// handle case where the virtual is missing in front of the destructor
			if( *p == '~' || ParseKw(p, "CRYINTERFACE_DECLARE" ) )
				return FOUND_DESTRUCTOR;
			else
				return NULL;
		}
    else if (fileType == File::IMPL_FILE)
      haveVirtualKw = true;
    isUnique = true;
  }
  if (ParseKw(p, m_UniqueVirtualKw))
    isUnique = true;
  else if (!ParseKw(p, "virtual"))
	{
		// handle case where the virtual is missing in front of the destructor
		if( *p == '~' || ParseKw(p, "CRYINTERFACE_DECLARE" ) )
			return FOUND_DESTRUCTOR;

    isUnique = false;
	}
  else
    haveVirtualKw = true;
  if (!isUnique && iface == NULL)
    return NULL;
  // We assume that the return type of the interface method is represented
  // in a simple syntax where the method name appears at the end.  For
  // syntactically complex types (like function pointers), a typedef for the
  // return type must be used.
  for (; std::isspace(*p); ++p);
  char *methodBegin = p;
  p = SkipTo(p, '(');
  if (*p != '(')
    return NULL;
  char *q = p - 1;
  for (; q > methodBegin && std::isspace(*q); --q);
  char *nameEnd = q + 1;
  // Operator check.  We'll need a temporary null-termination to check for
  // the 'operator' keyword.
  char nameEndChar = *nameEnd;
  *nameEnd = 0;
  char *opKw = std::strstr(methodBegin, "operator");
  *nameEnd = nameEndChar;
  if (opKw != NULL
      && ((opKw > methodBegin && IsId(opKw[-1])) || IsId(opKw[8])))
    opKw = NULL;
  if (opKw != NULL && SkipTo(opKw, '(') != p)
    opKw = NULL;
  char *nameBegin = opKw;
  if (nameBegin == NULL)
  {
    for (
      nameBegin = nameEnd;
      nameBegin > methodBegin && IsId(nameBegin[-1]);
      --nameBegin);

    if (nameBegin == nameEnd || !IsInitId(nameBegin[0]))
      return NULL;
  }
  // Check for garbage between methodBegin and nameBegin.
  if (std::memchr(methodBegin, ';', nameBegin - methodBegin) != NULL)
    return NULL;
  // Check if this is a virtual destructor.
  q = nameBegin;
  for (; q > methodBegin && std::isspace(q[-1]); --q);
  if (q > methodBegin && q[-1] == '~')
  {
  if( iface == NULL ) 
  {
    return FOUND_DESTRUCTOR;
  }
    // Virtual destructor, ignore.
    return NULL;
  }
  std::string methodName(nameBegin, nameEnd - nameBegin);

  // don't devirtualize ref counting methods
  if( 	methodName == "AddRef" || 
    methodName == "Release" ||
    methodName == "ReleaseForce" ||
    methodName == "GetRefCounter" )
      return NULL;

  if (iface != NULL)
  {
    std::list<Method *> ifaceMethods;
    iface->GetMethods(methodName, std::back_inserter(ifaceMethods));

    if (ifaceMethods.empty())
      return NULL;

    InfoLoc(
      file->LineMessage(methodBegin - base).c_str(),
      "found unique implementation for method '%s::%s'",
      iface->Name().c_str(), methodName.c_str());

    for (
      std::list<Method *>::const_iterator
      it = ifaceMethods.begin(), itEnd = ifaceMethods.end();
      it != itEnd;
      ++it)
        (**it).m_HaveImpl = true;

    if (fileType == File::IMPL_FILE && haveVirtualKw)
    {
      WarningLoc(
        file->LineMessage(methodBegin - base).c_str(),
        "'virtual' keyword used in implementation of de-virtualized "
        "method '%s::%s' (use '%s' macro instead)",
        iface->Name().c_str(), methodName.c_str(),
        m_UniqueVirtualKw.c_str());
    }
  }
  std::auto_ptr<Method> method(new Method(methodName));
  if (iface != NULL)
    method->m_HaveImpl = true;
  char *typeBegin = methodBegin, *typeEnd = nameBegin;
  for (; typeEnd > typeBegin && std::isspace(typeEnd[-1]); --typeEnd);
  method->m_Type.assign(typeBegin, typeEnd - typeBegin);
  assert(*p == '(');
  char *paramListEnd = SkipTo(p + 1, ')');
  if (*paramListEnd != ')')
    return NULL;
  // We'll place a temporary null-termination at the end of the parameter list
  // to simplify parsing.
  while (true)
  {
    for (++p; std::isspace(*p); ++p);
    if (p == paramListEnd)
      break;
    *paramListEnd = 0;
    bool parseParamRV = ParseParam(p, &*method);
    *paramListEnd = ')';
    if (!parseParamRV)
      return NULL;
    assert(p <= paramListEnd);
    if (p == paramListEnd)
      break;
    if (*p != ',')
      return NULL;
  }
  assert(*p == ')');
  // Check for (void).
  if (method->m_Params.size() == 1 && method->m_Params.front().IsVoid())
    method->m_Params.clear();
  for (++p; std::isspace(*p); ++p);
	
  if (ParseKw(p, "const"))
  {
    method->m_IsConst = true;
    for (; std::isspace(*p); ++p);
  }

  //check for macros after function decl
  if( IsInitId(*p))
  {
    char *macroStart = p;
    for (; IsId(*p); ++p);
    assert(*p == '(');
    p = SkipMatched( p + 1, ')' );
    for (; std::isspace(*p); ++p);
    char *macroEnd = p;

    method->m_extMacro = std::string( macroStart, macroEnd - macroStart );
  }

  bool isPure = false;
  if (*p == '=')
  {
    for (++p; std::isspace(*p); ++p);
    if (*p != '0')
    {
      // Eh?  What's this?
      return NULL;
    }
    for (++p; std::isspace(*p); ++p);
    isPure = true;
  }
  char *defaultImplBegin = NULL, *defaultImplEnd = NULL;
  if (*p == '{')
  {
    defaultImplBegin = p;
    p = SkipMatched(p + 1, '}');
    if (*p == 0)
      return NULL;
    defaultImplEnd = p;
  }
  else if (*p != ';')
    return NULL;
  if (defaultImplBegin != NULL)
    method->m_DefaultImpl.assign(
      defaultImplBegin,
      defaultImplEnd - defaultImplBegin);
  else
    method->m_IsPure = isPure;
  ++p;
  *code = p;
  char *methodDeclEnd = p;
  unsigned beginOffset = static_cast<unsigned>(methodDeclBegin - base);
  unsigned endOffset = static_cast<unsigned>(methodDeclEnd - base);
  method->m_Pos = std::make_pair(beginOffset, endOffset);

  return method.release();
}

bool DeVirt::ParseParam(char **code, Method *method)
{
  char *p = *code;
  bool isSimpleType = false;
  bool isPointerOrRef = false;
  char *paramEnd = SkipTo(p, ',');
  // Very simplistic parameter parser.  For syntactically complicated
  // parameter declarations in interfaces, typedefs must be used.
  for (; std::isspace(*p); ++p);

  if (p[0] == '.' && p[1] == '.' && p[2] == '.')
  {
    // This is an ellipsis.
    method->m_Params.push_back(Param::Ellipsis());
    *code = paramEnd;
    return true;
  }
  char *typeBegin = p;
  if (ParseKw(p, "const"))
    for (; std::isspace(*p); ++p);
  if (ParseSimpleType(p))
  {
    isSimpleType = true;
    if (ParseKw(p, "const"))
      for (; std::isspace(*p); ++p);
  }

	// skip possible spaces between Type and *(or &), so that these are always stored without spaces
	for (; std::isspace(*p); ++p);

  while (*p == '*' || *p == '&')
  {
    isPointerOrRef = true;
    for (++p; std::isspace(*p); ++p);
    if (ParseKw(p, "const"))
      for (; std::isspace(*p); ++p);
  }

  if (!isSimpleType && !isPointerOrRef)
  {
    if (!IsInitId(*p)) return false;

    while(true)
    {
      // fetch all characters from id
      for (++p; IsId(*p); ++p);
      for (; std::isspace(*p); ++p);

      // is it a template?
      if (*p == '<')
      {
        int bracketPairs = 0;			
        // check for some unusuabl templates which are not parsed (eg foo< a>b > )
        for( char *q = p ; *q != ';' ; ++q )
        {
          if( *q == '<' ) ++bracketPairs;
          if( *q == '>' ) --bracketPairs;
        }

        if( bracketPairs != 0 )
        {
          std::cerr <<"No devirtualization support for advanced template function parameter" << std::endl;
          return NULL;
        }

        // skip first '<' to ensure correct counting
        ++p;

        // since we assume that the underlying code is correct, just parse till the matching closing '>'
        for( int count = 1;  count > 0 ; ++p )
        {
          if( *p == '<' ) ++count;
          if( *p == '>' ) --count;			
        }

        // adjust paramEnd (for example templates with two parameters)			
        if( paramEnd < p ) paramEnd = SkipTo(p, ',');

        // stop type parsing if no scope qualifier follows
        if( p[0] != ':' && p[1] != ':' ) break;

      }
      else if( p[0] == ':' && p[1] ==':' ) // check for name qualifiers
      {
        p +=2;
        if (!IsInitId(*p)) return false;
      }
      else
      {
        break;
      }
    }

    for (; std::isspace(*p); ++p);
      if (ParseKw(p, "const"))
        for (; std::isspace(*p); ++p);

    while (*p == '*' || *p == '&')
    {
      isPointerOrRef = true;
      for (++p; std::isspace(*p); ++p);
      if (ParseKw(p, "const"))
      for (; std::isspace(*p); ++p);
    }
  }

  char *typeEnd = p;
  for (; typeEnd > typeBegin && std::isspace(typeEnd[-1]); --typeEnd);
  char *nameBegin = NULL, *nameEnd = NULL;
  if (IsInitId(*p))
  {
    nameBegin = p;
    for (++p; IsId(*p); ++p);
    nameEnd = p;
    for (; std::isspace(*p); ++p);
  }
  char *typeAfterId = NULL, *typeAfterIdEnd = NULL;
  if (*p == '[')
  {
    typeAfterId = p;
    while (true)
    {
      p = SkipMatched(p + 1, ']');
      if ( p[-1] != ']')		//Since SkipMatched returns p pointing to the elemet *after* the delimeter, we must check one previous elemnt
        return false;
      for (; std::isspace(*p); ++p);
      if (*p != '[')
        break;
    }
    typeAfterIdEnd = p;
    while (typeAfterIdEnd > typeAfterId && std::isspace(typeAfterIdEnd[-1]))
      --typeAfterIdEnd;
  }
  char *defaultBegin = NULL, *defaultEnd = NULL;
  if (*p == '=')
  {
    for (++p; std::isspace(*p); ++p);
    defaultBegin = p;
    defaultEnd = paramEnd;
    p = defaultEnd;
    while (defaultEnd > defaultBegin && std::isspace(defaultEnd[-1]))
      --defaultEnd;
  }
  if (*p != 0 && *p != ',')
    return false;
  method->m_Params.push_back(Param());
  Param *param = &method->m_Params.back();
  if (typeAfterId == NULL)
  {
    char typeEndChar = *typeEnd;
    *typeEnd = '%';
    param->m_Type.assign(typeBegin, typeEnd - typeBegin + 1);
    *typeEnd = typeEndChar;
  }
  else
  {
    std::ostringstream typeOut;
    typeOut.write(typeBegin, (std::streamsize)(typeEnd - typeBegin));
    typeOut.put('%');
    typeOut.write(typeAfterId, (std::streamsize)(typeAfterIdEnd - typeAfterId));
    param->m_Type = typeOut.str();
  }
  if (nameBegin != NULL)
    param->m_Name.assign(nameBegin, nameEnd - nameBegin);
  if (defaultBegin != NULL)
    param->m_Default.assign(defaultBegin, defaultEnd - defaultBegin);
  *code = paramEnd;
	
	// remove all spaces from the parameter
	for( unsigned int i = 1 ; i < param->m_Type.size() ; )
	{
		if( param->m_Type[i-1] == ' ' && (param->m_Type[i] == '*' || param->m_Type[i] == '&') )
		{
			for( unsigned int j = i ; j < param->m_Type.size() ; ++ j )
				param->m_Type[j-1] = param->m_Type[j];

			param->m_Type.resize( param->m_Type.size() - 1 );
		}
		else
			++i;
	}


  return true;
}

