#include "DeVirt.h"
#include "Util.h"

#include <iostream>
#include <map>
#include <fstream>
#include <sstream>
#include <cstring>
#include <cassert>
#include <cstdlib>

#include <sys/types.h>
#include <sys/stat.h>

namespace
{
  void CopyDirectives(
		      std::ostream &out,
		      const char *p,
		      ssize_t count
		      )
  {		
    const char *end = p + count;
    bool lineStart = false;

    while (p < end)
    {
      char c = *p++;
      if (c == '\n')
	lineStart = true;
      else if (c == '#' && lineStart)
      {
	const char *q = p;
	for (; std::isspace(*q); ++q);
	if (!std::strncmp(q, "define", 6) && std::isspace(q[6]))
	{
	  for (q += 7; std::isspace(*q); ++q);
	  const char *r = q;
	  for (; std::isalnum(*q) || *q == '_' || *q == '$'; ++q);
	  if (r > q)
	  {
	    out << "#undef ";
	    out.write(q, (std::streamsize)(r - q));
	    out << std::endl;
	  }
	}

	while (true)
	{
	  for (; *q != '\n' && *q != '\r'; ++q);
	  if (q[-1] != '\\')
	    break;
	  if (q[0] == '\n')
	    ++q;
	  else if (q[0] == '\r' && q[1] == '\n')
	    q += 2;
	}
	out << std::endl << '#';
	out.write(p, (std::streamsize)(q - p));
	out << std::endl;
      }
      else if (!std::isspace(c))
	lineStart = false;
    }
  }

  // predicate function to check if parameters contain an ellipsis
  bool HasEllipsis( const std::vector<Param> &params )
  {
    size_t nParams = params.size();
    for (size_t i = 0; i < nParams; ++i)
    {
      const Param &param = params[i];
      if( param.IsEllipsis() )
	return true;
    }

    return false;
  }

  // predicate function to check the parameters for a specific type
  bool ContainsType( const std::vector<Param> &params, const std::string &type )
  {
    size_t nParams = params.size();
    for (size_t i = 0; i < nParams; ++i)
    {
      const Param &param = params[i];
      if( param.Type().find(type) != std::string::npos )
	return true;
    }

    return false;
  }

  // util function to remove the first occuriance of a substring from another string
  void RemoveFromString( std::string &str, const std::string &remove )
  {
    size_t pos = str.find(remove);
    if( pos != std::string::npos )
      str.erase(pos, remove.length());

  }

  // util function to remove each found instance of c in the string str
  void RemoveFromString( std::string &str, char c )
  {
    size_t j = 0;
    size_t i = 0;
    for(   ; i < str.size() ;  )
    {
      while( i < str.size() && str[i] == c )
      {			
	++i;
      }

      str[j] = str[i];
      ++i;
      ++j;
    }
    str.resize( str.size() - (i-j) );
  }

  // predicate function, checks the scope of a declaration name
  // to determine if this is really a nested declaration and not 
  // a forward declaration
  bool CheckForNestedDecl( const std::string &tmp, size_t pos, size_t offset )
  {
    if( pos == std::string::npos ) return false;

    char cNextChar = tmp[ pos + offset ];

    // catch overlapping names
    if( std::isalpha(cNextChar) ) return false;

    // catch inplace forward declarations
    if( cNextChar == '*' ) return false;
    if( cNextChar == '&' ) return false;
		
    return true;
  }

  // predicate function to determine if a class is nested in another
  bool IsNested( std::string sClass, const char *pStart, const char *pEnd )
  {
    std::string tmp( pStart, pEnd-pStart );
    RemoveFromString( tmp, ' ' );
    RemoveFromString( tmp, '\t' );

    // strip some unneeded stuff
    RemoveFromString( sClass, "const" );
    RemoveFromString( sClass, ' ' );
    RemoveFromString( sClass, '&' );
    RemoveFromString( sClass, '*' );
    RemoveFromString( sClass, '[' );
    RemoveFromString( sClass, ']' );
    std::string classDecl = "class" + sClass;
    std::string structDecl = "struct" + sClass;
    std::string enumDecl = "enum" + sClass;
    size_t hasClassPos = tmp.find(classDecl);
    size_t hasStructPos = tmp.find(structDecl);
    size_t hasEnumPos = tmp.find(enumDecl);
    bool bHasClass = CheckForNestedDecl( tmp, hasClassPos, classDecl.size() );
    bool bHasStruct = CheckForNestedDecl( tmp, hasStructPos, structDecl.size() );
    bool bHasEnum = CheckForNestedDecl( tmp, hasEnumPos, enumDecl.size() );

    bool hasTypedef = false;
    std::string::size_type pos = tmp.find("typedef");

    while (pos != std::string::npos) 
    {
      size_t lineStart = tmp.find(sClass, pos );
      size_t lineEnd = tmp.find(";", pos );
      if( lineStart < lineEnd )
      {
	std::string line = tmp.substr( pos, lineEnd - lineStart );
	if( line.find("(") != std::string::npos )
	{
	  if( tmp[lineStart - 1] == '*' )
	  {	
	    hasTypedef = true;
	    break;
	  }
	}else {
	  hasTypedef = true;
	  break;
	}			
      }
      pos = tmp.find("typedef", pos + 1);
    }

    bool IsPrimitive =	sClass == "int" ||
      sClass == "float" ||
      sClass == "double" ||
      sClass == "int32" ||
      sClass == "uint32"||
      sClass == "char" ||
      sClass == "void" ||
      sClass == "uint8" ||
      sClass == "int8" ||
      sClass == "short";

    if( bHasClass || bHasStruct || bHasEnum || hasTypedef )
      return !IsPrimitive;
    else
      return false;	
  }

  void GetDirectoryContent( std::string dir, std::vector<std::string> &files);

  // predicate to check if a filename can be found in a list of file names
  bool ContainesFile( const std::string &file, const std::vector<std::string> &files )
  {
    for( std::vector<std::string>::const_iterator it = files.begin() ;
	 it != files.end() ; ++it )
    {
      if( file == (*it) )
      {	
	return true;
      }	
    }

    return false;
  }
	
  static const char CryCGVTableList[] = "crycg_virtual_functions_list";	
}

std::ostream &Method::WriteDecl(std::ostream &out) const
{	
  Iface *iface = GetIface();
  // sanity check if this method is  really INLINE devirtualizeable
  // skip if implementation file resists in a cpp file, it has ellipse parameters(not inlineable)
  // and skip functions with TSerialize(strange typedef, would need some investigation
  bool bIsDevirtualizeable = !(		(iface && iface->GetIClass() && iface->GetIClass()->GetFile()->FileName().find(".cpp") != std::string::npos) ||
					HasEllipsis( Params() ) ||
					ContainsType( Params(), "TSerialize") );

  out << "  ";
  if( !m_DefaultImpl.empty() || !bIsDevirtualizeable ) 
  {
    // if there is a default implementation in the header file, or if the function is not devirtualizeable, use virtual dispatch
    out << "virtual ";
  }
  else
  {
    // this macro is used to prefix the function declarations with ILINE in case of compilation with ueber-files
    out << "DEVIRTUALIZE_INLINE_FUNCTION_PREFIX ";			
  }

  out << Type() << ' ' << Name() << '(';
	
  const std::vector<Param> &params = Params();
  size_t nParams = params.size();
  for (size_t i = 0; i < nParams; ++i)
  {
    const Param &param = params[i];
    if (i > 0){
      out << ", ";
    }
    if (param.IsEllipsis())
    {
      assert(i == nParams - 1);
      out << "...";
    }
    else
    {
      out << param.Decl();			
      const std::string &defaultValue = param.Default();
      if (!defaultValue.empty())
	out << " = " << defaultValue;
    }
  }
  out << ')';

  if (IsConst())
  {
    out << " const";		
  }

  out << " " << m_extMacro << " ";

  if( !bIsDevirtualizeable )
  {
    // mark skipped function without impl as pure virtual
    out << " =0";
  }

  const std::string &defaultImpl = DefaultImpl();
  if (!defaultImpl.empty() )
  {
    out << defaultImpl;		
  }else {
    out << ';' << std::endl;		
  }	
  return out;
}

std::ostream &Method::WriteInClassDecl(std::ostream &out) const
{
  out << "  ";
  out << Type() << ' ' << Name() << '(';
	
  const std::vector<Param> &params = Params();
  size_t nParams = params.size();
  for (size_t i = 0; i < nParams; ++i)
  {
    const Param &param = params[i];
    if (i > 0){
      out << ", ";
    }
    if (param.IsEllipsis())
    {
      assert(i == nParams - 1);
      out << "...";
    }
    else
    {
      out << param.Decl();			
      const std::string &defaultValue = param.Default();
      if (!defaultValue.empty())
	out << " = " << defaultValue;
    }
  }
  out << ')';

  if (IsConst())
  {
    out << " const";		
  }

  out << ';' << std::endl;		

  return out;
}

std::ostream &Method::WriteDef(std::ostream &out) const
{
  std::string className;
  if (const Iface *iface = GetIface())
    className = iface->Name();
  else
    className = GetIClass()->Name();
  out << Type() << ' ' << className << "::" << Name() << '(';
  const std::vector<Param> &params = Params();
  size_t nParams = params.size();
  for (size_t i = 0; i < nParams; ++i)
  {
    const Param &param = params[i];
    if (i > 0)
      out << ", ";
    if (param.IsEllipsis())
    {
      assert(i == nParams - 1);
      out << "...";
    }
    else
      out << param.Decl(ParamName(static_cast<unsigned>(i)));
  }
  out << ')';
  if (IsConst())
    out << " const";
  return out;
}

std::string Method::ParamName(unsigned paramIndex) const
{
  assert(paramIndex < Params().size());
  char paramName[64];
  snprintf(paramName, sizeof paramName, "$param%u", paramIndex + 1);
  paramName[sizeof paramName - 1] = 0;
  return paramName;
}


std::ostream &Iface::WriteDecl(std::ostream &out) const
{	
  const char *ifaceBegin = Begin(), *ifaceEnd = End();
  const char *p = ifaceBegin;

  for (
       std::vector<Method *>::const_iterator
	 it = m_Methods.begin(), itEnd = m_Methods.end();
       it != itEnd;
       ++it)
  {		

    const Method *method = *it;
    if (!method->HaveImpl())
      continue;
    out.write(p, (std::streamsize)(method->Begin() - p));
    p = method->End();
    assert(p < ifaceEnd);
    method->WriteDecl(out);    
  }      
  out.write(p, (std::streamsize)( (ifaceEnd - 2) - p));

  // append a virtual destructor if no was found
  if( !m_hasDstr && m_AllUnique && p > ifaceBegin )
  {
    out << "public: \nvirtual ~" << m_Name << "(){ /*Dummy for devirtualization */ }\n";
  }	

  out << "\n}";
  // add _ALIGN macro if needed
  if( m_Alignment.size() > 0 )
  {
    out << "_ALIGN( " << m_Alignment << ")";
  }
  out << "\n;";
  p +=2;
  return out;
}

// free util function used to write some helper code, which forces the compiler to generate assembler code
// for an inline function by taking its address
void WriteHelperToGenerateInlineCode( std::ostream &out, const Method *method, const std::string &sClass,  const std::string &sInterface, int counter, const char *pBegin, const char *pEnd )
{
  // skip these for SPU since there they are not needed, and they generate lots of crycg warnings
  out << "#if !defined(__SPU__)" << std::endl;

  const std::vector<Param> &params = method->Params();
  size_t nParams = params.size();

  // to get the compiler to output code for inline functions
  // use a weak dummy function to take its address
  out << "typedef ";
  std::string returnType =method->Type();
  RemoveFromString( returnType, "inline" );
  RemoveFromString( returnType, "ILINE" );
  if( IsNested( returnType, pBegin, pEnd ) )
  {
    if(returnType.find("const") != std::string::npos )
    {
      RemoveFromString(returnType, "const");
      RemoveFromString(returnType, ' ');
      out << " const ";
    }
    out << sInterface << "::";
  }		
  out << returnType;
  out << " (" << sClass << "::*";				
  std::stringstream sCounter;
  sCounter << counter;
  std::string typedefName = sClass + "_" + method->Name() + "_devirt_helper_" + std::string(sCounter.str());
  out << typedefName << ")(";
  for (size_t i = 0; i < nParams; ++i)
  {      
    if (i > 0)
      out << ", "; 
		
    std::string tmp = params[i].Type();			
    RemoveFromString(tmp, '%' );
    if( IsNested( tmp, pBegin, pEnd ) )
    {
      if(tmp.find("const") != std::string::npos )
      {
	RemoveFromString(tmp, "const");
	RemoveFromString(tmp, ' ');
	out << " const ";
      }
      out << sClass << "::";
    }
    if( tmp.find("[") != std::string::npos )
    {
      tmp.erase(tmp.find("["), tmp.size()-tmp.find("["));
      out << tmp;
      out << "*";
    }
    else
    {
      out << tmp;
    }
		
  }
	
  out << ")";
  if (method->IsConst())
    out << " const ";

  out << ";" << std::endl;

  out << typedefName << " " << typedefName << "_function() __attribute__((weak));" << std::endl;
  out << typedefName << " " << typedefName << "_function() { return &"
      << sClass << "::" << method->Name() << "; }" << std::endl;

  out << "#endif // __SPU__" << std::endl;

}

std::ostream &Iface::WriteDef(std::ostream &out) const
{
  const IClass *iclass = GetIClass();
  if (iclass == NULL)
    return out;

  const char *ifaceBegin = Begin(), *ifaceEnd = End();
  const char *p = ifaceBegin;
  int counter = 0;
  for (
       std::vector<Method *>::const_iterator
	 it = m_Methods.begin(), itEnd = m_Methods.end();
       it != itEnd;
       ++it)
  {
    const Method *method = *it;

    bool bIsDevirtualizeable = !(		(iclass->GetFile()->FileName().find(".cpp") != std::string::npos) ||
						HasEllipsis( method->Params() ) ||
						ContainsType( method->Params(), "TSerialize") );

    
    CopyDirectives(out, p, method->Begin() - p);

    p = method->End();
    assert(p < ifaceEnd);
    if (!method->HaveImpl())
      continue;
    Method *implMethod = method->GetImplMethod();
    if (implMethod == NULL)
      continue;
    if (!method->DefaultImpl().empty())
      continue;
    if( !bIsDevirtualizeable )
      continue;

    method->WriteDef(out);

    out << " { ";
    if (method->Type() != "void")
      out << "return ";
    out << "static_cast<";
    if (method->IsConst())
      out << "const ";
    out << iclass->Name() << " *>(this)->"
	<< iclass->Name() << "::" << method->Name() << '(';
    const std::vector<Param> &params = method->Params();
    size_t nParams = params.size();
    for (size_t i = 0; i < nParams; ++i)
    {
      if (params[i].IsEllipsis())
	break;
      if (i > 0)
	out << ", "; 

      //dont forward void params
      if( !params[i].IsVoid() )
	out << method->ParamName(static_cast<unsigned>(i));

    }
    out << "); }" << std::endl;

    WriteHelperToGenerateInlineCode( out, method, this->Name(), this->Name(), counter, Begin(), End() );

    if( !implMethod->DefaultImpl().empty() )
    {
      // if the implementation method is defined inline, also make sure to keep these functions
      ++counter;
      WriteHelperToGenerateInlineCode( out, implMethod, iclass->Name(), this->Name(), counter, Begin(), End() );
    }

    out << std::endl;
    ++counter;
  }
  CopyDirectives(out, p, ifaceEnd - p);
  return out;
}

std::ostream &Iface::WriteDefInplace(std::ostream &out) const
{
  const IClass *iclass = GetIClass();
  if (iclass == NULL)
    return out;

  const char *ifaceBegin = Begin(), *ifaceEnd = End();
  const char *p = ifaceBegin;
  for (
       std::vector<Method *>::const_iterator
	 it = m_Methods.begin(), itEnd = m_Methods.end();
       it != itEnd;
       ++it)
  {
    const Method *method = *it;

    bool bSkipMethod = false;
    for( std::vector<Param>::const_iterator param = method->Params().begin() ;
	 param != method->Params().end() ; ++param )
    {
      if( param->Type().find("TSerialize") != std::string::npos ||
	  param->IsEllipsis() )
      {
	bSkipMethod = true;
	break;
      }
    }

    if( bSkipMethod )
      continue;

    CopyDirectives(out, p, method->Begin() - p);
			
    p = method->End();
    assert(p < ifaceEnd);
    if (!method->HaveImpl())
      continue;
    Method *implMethod = method->GetImplMethod();
    if (implMethod == NULL)
      continue;
    if (!method->DefaultImpl().empty())
      continue;
		
    if( HasEllipsis( method->Params() ) || ContainsType( method->Params(), "TSerialize") )
      out << "#if !defined(USING_UEBER_FILE_FOR_DEVIRTUALIZER)" << std::endl;
    	
    method->WriteDef(out);
    out << " { ";
    if (method->Type() != "void")
      out << "return ";
    out << "static_cast<";
    if (method->IsConst())
      out << "const ";
    out << iclass->Name() << " *>(this)->"
	<< iclass->Name() << "::" << method->Name() << '(';
    const std::vector<Param> &params = method->Params();
    size_t nParams = params.size();
    for (size_t i = 0; i < nParams; ++i)
    {
      if (params[i].IsEllipsis())
	break;
      if (i > 0)
	out << ", "; 

      //dont forward void params
      if( !params[i].IsVoid() )
	out << method->ParamName(static_cast<unsigned>(i));

    }
    out << "); }" << std::endl;

    if( HasEllipsis( method->Params() ) || ContainsType( method->Params(), "TSerialize") )
      out << "#if !defined(USING_UEBER_FILE_FOR_DEVIRTUALIZER)" << std::endl;

  }
  CopyDirectives(out, p, ifaceEnd - p);
  return out;
}

std::ostream &Iface::WriteAsm(std::ostream &out) const
{
  const char *ifaceBegin = Begin(), *ifaceEnd = End();
  const char *p = ifaceBegin;
  for (
       std::vector<Method *>::const_iterator
	 it = m_Methods.begin(), itEnd = m_Methods.end();
       it != itEnd;
       ++it)
  {
    const Method *ifaceMethod = *it;
    CopyDirectives(out, p, ifaceMethod->Begin() - p);
    p = ifaceMethod->End();
    assert(p < ifaceEnd);
    if (ifaceMethod->HaveImpl())
    {
      Method *implMethod = ifaceMethod->GetImplMethod();
      if (implMethod == NULL)
	continue;
      if (!implMethod->DefaultImpl().empty())
	continue;
      // Note: missing mangled names are OK, since this method is also being
      // called for generating the ASM to extract the mangled names from.
      const std::string &ifaceMangledName = ifaceMethod->MangledName();
      if (ifaceMangledName.empty())
      {
#if 0
	DeVirt::Warning(
			"missing mangled name for interface method '%s'",
			ifaceMethod->QualifiedName().c_str());
#endif
	continue;
      }
      const std::string &implMangledName = implMethod->MangledName();
      if (implMangledName.empty())
      {
#if 0
	DeVirt::Warning(
			"missing mangled name for implementation method '%s'",
			implMethod->QualifiedName().c_str());
#endif
	continue;
      }
#if 0
      out << "__asm__ __volatile__ (\".set " << ifaceMangledName
	  << ", " << implMangledName << "\\n.set ." << ifaceMangledName
	  << ", ." << implMangledName << "\\n\");\n";
#else
      /*
	out
	<< "\n"
	<< "#if defined(PS3) && !defined(CRYCG_CM)\n"
	<< "__asm__\n"
	<< "#ifndef __SNC__\n"
	<< "__volatile__\n"
	<< "#endif\n"
	<< "(\".set ." << ifaceMangledName
	<< ", ." << implMangledName << "\\n\");\n"
	<< "__asm__\n"
	<< "#ifndef __SNC__\n"
	<< "__volatile__\n"
	<< "#endif\n"
	<< "(\".set " << ifaceMangledName
	<< ", " << implMangledName << "\\n\");\n"
	<< "#endif\n";
      */
#endif
    }
  }
  CopyDirectives(out, p, ifaceEnd - p);
  return out;
}

std::ostream &Iface::WriteLdMap(std::ostream &out) const
{
  std::map<std::string, std::string> symbolMap;

  for (
       std::vector<Method *>::const_iterator
	 it = m_Methods.begin(), itEnd = m_Methods.end();
       it != itEnd;
       ++it)
  {
    const Method *ifaceMethod = *it;
    if (ifaceMethod->HaveImpl())
    {
      Method *implMethod = ifaceMethod->GetImplMethod();
      if (implMethod == NULL)
	continue;
      if (!implMethod->DefaultImpl().empty())
	continue;
      if( !ifaceMethod->DefaultImpl().empty() )
	continue;

      const std::string &ifaceMangledName = ifaceMethod->MangledName();
      if (ifaceMangledName.empty()  )
      {
	DeVirt::Warning(
			"missing mangled name for interface method '%s'",
			ifaceMethod->QualifiedName().c_str());
	continue;
      }
      const std::string &implMangledName = implMethod->MangledName();
      if (implMangledName.empty())
      {
	DeVirt::Warning(
			"missing mangled name for implementation method '%s'",
			implMethod->QualifiedName().c_str());
	continue;
      }
      symbolMap[ifaceMangledName] = implMangledName;
    }
  }
  for (
       std::map<std::string, std::string>::const_iterator
	 it = symbolMap.begin(), itEnd = symbolMap.end();
       it != itEnd;
       ++it)
  {
    const std::string &ifaceMangledName = it->first;
    const std::string &implMangledName = it->second;
    out
      << "--defsym " << ifaceMangledName << '=' << implMangledName << '\n'
      << "--defsym ." << ifaceMangledName << "=." << implMangledName << '\n';
  }
  return out;
}

namespace
{
  std::string MakeIncludeGuard(const std::string &fileName)
  {
    std::ostringstream out;
    out << '_';
    for (const char *p = fileName.c_str(); *p != 0; ++p)
    {
      char c = *p;
      if (std::isalnum(c) || c == '_' || c == '$')
	out << c;
      else
	out << '_';
    }
    out << '_';
    return out.str();
  }

  const char *BaseName(const char *fileName)
  {
    assert(Util::CheckPath(fileName));
    const char *sep = std::max(
			       std::strrchr(fileName, '/'),
			       std::strrchr(fileName, '\\'));
    return sep != NULL ? sep + 1 : fileName;
  }

  const char *BaseName(const std::string &fileName)
  {
    assert(Util::CheckPath(fileName));
    return BaseName(fileName.c_str());
  }

  std::string IfaceOutputFileName(
				  const std::string &inputFileName,
				  const std::string &outputDir
				  )
  {
    assert(Util::CheckPath(inputFileName));
    std::string result = outputDir + Util::PathSeperator +
      inputFileName;
    assert(Util::CheckPath(result));
    return result;
  }

  std::string LdMapOutputFileName(const std::string &outputDir)
  {
    return outputDir + Util::PathSeperator + "devirt.ld";
  }
}



bool DeVirt::IsDevirtualizedHeader( const std::string &file ) const
{
  for( std::list<File>::const_iterator it = m_FileList.begin() ;
       it != m_FileList.end() ; ++it )
  {
    if( file == it->BaseName() )
      return true;
  }

  return false;
}

void DeVirt::WriteWithPatchingInclude( std::ostream &out, char *p, size_t num, const std::string &path ) const
{
  for( size_t i = 0 ; i < num ; ++i )
  {
    if( p[i] == '#' && p[i+1] == 'i' && p[i+2] == 'n' && p[i+3] == 'c' && p[i+4] == 'l' && p[i+5] == 'u' && p[i+6] == 'd' && p[i+7] == 'e' )
    {	
      bool bSkipPatching = false;
      for( size_t j = 0 ; p[i+j] != '\n' ; ++ j )
      {
	if( p[i+j] == '<' )
	{
	  bSkipPatching = true;
	  break;
	}
      }

      if( !bSkipPatching )
      {
	std::string headerFile;
	size_t j = 0; 
	while( p[i+j] != '\"' ) ++j;
	++j;

	while( p[i+j] != '\"' )
	{
	  headerFile += p[i+j];
	  ++j;
	}

	// check files in the iface directory, if we need to patch
	std::vector<std::string> files;
	GetDirectoryContent( m_CodeRoot + Util::PathSeperator + path + Util::PathSeperator, files );
			
	if( !ContainesFile( headerFile, files ) || IsDevirtualizedHeader( headerFile ) )
	{
	  bSkipPatching = true;
	}
				
      }

      if( !bSkipPatching )
      {
	while( p[i] != '\"' )
	{
	  out << p[i];
	  ++i;
	}
	out << "\"" << path << "/";
      }
      else
	out << p[i];
			
    }
    else
      out << p[i];
  }
}

bool DeVirt::WriteFile(
		       std::ostream &out,
		       const File &file,
		       std::vector<const Iface *> &ifaceList,
		       bool bIsDummyHeader
		       ) const
{
  const char *base = file.Buffer();
  /*const*/ char *p = const_cast<char*>(base);
  bool first = true;
  for (
       std::vector<Iface *>::const_iterator
	 it = file.m_IfaceList.begin(), itEnd = file.m_IfaceList.end();
       it != itEnd;
       ++it)
  {
    const Iface *iface = *it;
    PosT pos = iface->m_Pos;
    const char *ifaceBegin = base + pos.first;
    const char *ifaceEnd = base + pos.second;
    assert(ifaceBegin >= p);

    if(first)
    {
      for( int i = 0 ; i < ifaceBegin - p ; ++i)
      {	
	// Remove DEVIRTUALIZATION INCLUDE GUARD
	if( 	p[i] == '#' && p[i+1] == 'i' && p[i+2] == 'n' && p[i+3] == 'c' && p[i+4] == 'l' && p[i+5] == 'u' &&  p[i+6] == 'd' &&
		p[i+7] == 'e' && p[i+8] == ' ' && p[i+9] == 'D' && p[i+10] == 'E' && p[i+11] == 'V' && p[i+12] == 'I' &&
		p[i+13] == 'R' && p[i+14] == 'T' && p[i+15] == 'U' )
	{	
	  while( p[i] != '\n' )	p[i++] = ' ';
	}
  
      }
      out << "#ifndef DEVIRTUALIZE_INLINE_FUNCTION_PREFIX" << std::endl;
      out << "#define DEVIRTUALIZE_INLINE_FUNCTION_PREFIX " << std::endl;
      out << "#endif" << std::endl;			
			
    }
    
    WriteWithPatchingInclude( out, p, (std::streamsize)(ifaceBegin - p), 
			      Util::GetPath(iface->GetFile()->FileName()) );

    if (!iface->IsEmpty())
    {
      out << "#undef $IFACEWRAP_" << iface->Name() << std::endl;
      out << "#define $IFACEWRAP_" << iface->Name() << " "
	  << iface->Name() << "_Wrapper.h" << std::endl;
      ifaceList.push_back(iface);
    }
    iface->WriteDecl(out);
    out << std::endl;
#if defined DEVIRT_ASM_SYMBOL_MAPPING
    iface->WriteAsm(out);
#endif
    p = const_cast<char*>(ifaceEnd);
    first = false;

  }

  out << p << std::endl;

  // in case this is a dervirtualized header, write out dummy implementations
  // currently only inheriences the first parent
  if( bIsDummyHeader )
  {	
    std::string headerGuard = file.BaseName();
    for (
	 std::vector<Iface *>::const_iterator
	   it = file.m_IfaceList.begin(), itEnd = file.m_IfaceList.end();
	 it != itEnd;
	 ++it)
    {
      headerGuard.append("_");
      headerGuard.append( (*it)->Name() );
    }

    out << "#if !defined(" << headerGuard << ")" << std::endl;
    out << "#define " << headerGuard << std::endl;;
		
    for (
	 std::vector<Iface *>::const_iterator
	   it = file.m_IfaceList.begin(), itEnd = file.m_IfaceList.end();
	 it != itEnd;
	 ++it)
    {
      if( (*it)->GetIClass() == NULL )
	continue;
      const IClass *iclass = (*it)->GetIClass();

      if( (*it)->GetFile()->FileName() == iclass->GetFile()->FileName() )
	continue;

      if( iclass->GetFile()->FileName().find(".cpp") != std::string::npos )
	continue;

      // start with dummy guard which says we only use these for ueber files
      out << "#if defined(USING_UEBER_FILE_FOR_DEVIRTUALIZER)" << std::endl;
      out << "#if !defined(COMPLING_MODULE_CONTAINING_";
      out << iclass->Name();
      out << "_CLASS)" << std::endl;

			

      out << "class " << iclass->Name() << " : public " << (*it)->Name() << std::endl;
      out << "{" << std::endl;
      out << "public:" << std::endl;

      const char *ifaceBegin = (*it)->Begin(), *ifaceEnd = (*it)->End();
      const char *pp = ifaceBegin;
      for( std::vector<Method *>::const_iterator method = (*it)->Methods().begin() ;
	   method != (*it)->Methods().end() ; ++method )
      {				

	CopyDirectives(out, pp, (*method)->Begin() - pp);
	pp = (*method)->End();

	(*method)->WriteInClassDecl(out);

				
      }
      CopyDirectives(out, pp, ifaceEnd - pp);
      out << "};" << std::endl;

      out << std::endl;

      (*it)->WriteDefInplace(out);

      out << std::endl;

      out << "#endif" << std::endl;
      out << "#endif" << std::endl;
			
      out << std::endl;
    }

    out << "#endif // " <<headerGuard << std::endl;
  }

  return true;
}

bool DeVirt::Write()
{
  const std::string &outputDir = m_OutputDir;

  if (outputDir.empty())
  {
    Error("no output directory specified");
    return false;
  }

  struct stat sbuf;
  std::memset(&sbuf, 0, sizeof sbuf);
  if (stat(outputDir.c_str(), &sbuf) == -1)
  {
    Error("output directory '%s' does not exist", outputDir.c_str());
    return false;
  }
  if (!S_ISDIR(sbuf.st_mode))
  {
    Error("output directory '%s' is not a directory", outputDir.c_str());
    return false;
  }

  if (!GenMangledNames())
    return false;

  std::vector<const Iface *> ifaceList;
  for (
       std::list<File>::const_iterator
	 it = m_FileList.begin(), itEnd = m_FileList.end();
       it != itEnd;
       ++it)
  {
    const File &file = *it;
    if (file.GetType() != File::IFACE_FILE || file.m_IfaceList.empty())
      continue;

    // Generate the output file names.
    const std::string &outputFileName
      = IfaceOutputFileName(file.FileName(), outputDir);

    // Write the patched interface file.
    std::ostringstream out;
    if (!WriteFile(out, file, ifaceList, true))
      return false;
    if (!UpdateFile(outputFileName, out.str()))
      return false;
  }

  // Write the interface wrapper files.
  for (
       std::vector<const Iface *>::const_iterator
	 it = ifaceList.begin(), itEnd = ifaceList.end();
       it != itEnd;
       ++it)
  {
    const Iface *iface = *it;
    const std::string &wrapperFileName
      = outputDir + Util::PathSeperator + Util::GetPath(iface->GetFile()->FileName()) + Util::PathSeperator + iface->Name() + "_Wrapper.h";
    assert(Util::CheckPath(wrapperFileName));
    std::ostringstream out(wrapperFileName.c_str());
    const std::string &includeGuard = MakeIncludeGuard(wrapperFileName);
    out << "#ifndef " << includeGuard << std::endl;
    out << "#define " << includeGuard << " 1" << std::endl;
    iface->WriteDef(out);
    out << "#endif // defined " << includeGuard << std::endl << std::endl;
    if (!UpdateFile(wrapperFileName, out.str()))
      return false;
  }

#if defined DEVIRT_LD_SYMBOL_MAPPING
  {
    // Write the symbol mapping linker options file.
    const std::string &outputFileName = LdMapOutputFileName(outputDir);
    std::ostringstream out;
    for (
	 std::vector<const Iface *>::const_iterator
	   it = ifaceList.begin(), itEnd = ifaceList.end();
	 it != itEnd;
	 ++it)
    {
      const Iface *iface = *it;
      iface->WriteLdMap(out);
    }
    if (!UpdateFile(outputFileName, out.str()))
      return false;
  }
#endif // defined DEVIRT_LD_SYMBOL_MAPPING

  // ---------------------------------------------------------//
  // Dump the crycg virtual method table file
  // ---------------------------------------------------------//
  std::string crycgMangleFile(m_CacheDir + Util::PathSeperator 
			      + CryCGVTableList);
  std::ofstream crycgMangleList(crycgMangleFile.c_str());
  if (!crycgMangleList)
  {
    Error(
	  "could not open crycgMangleFile %s",
	  crycgMangleFile.c_str());
    std::abort();
  }
  for (
       std::list<File>::const_iterator
	 it = m_FileList.begin(), itEnd = m_FileList.end();
       it != itEnd;
       ++it)
  {
    const File *file = &(*it);
    for (        
	 std::vector<Iface *>::const_iterator
	   fit = file->m_IfaceList.begin(), fitEnd = file->m_IfaceList.end();
	 fit != fitEnd;
	 ++fit)
    {
      const Iface *iface = *fit;
      if (iface->IsEmpty())
	continue;
      const std::vector<Method *> &methods = iface->Methods();
      for (
	   std::vector<Method *>::const_iterator
	     mit = methods.begin(), mitEnd = methods.end();
	   mit != mitEnd;
	   ++mit)
      {
	Method *ifaceMethod = *mit;
	if (!ifaceMethod->HaveImpl())
	  continue;
	if( !ifaceMethod->DefaultImpl().empty() )
	  continue;

	Method *implMethod = ifaceMethod->GetImplMethod();
      
	if (implMethod == NULL)
	  continue;
      
	if (!implMethod->DefaultImpl().empty())
	{
	  Info("implementation method for virtual method %s has no name!",
	       ifaceMethod->FullName().c_str());
	  continue;
	}

	crycgMangleList << file->FileName() << "\t";
	if (!ifaceMethod->FullName().empty() &&
	    !ifaceMethod->MangledName().empty())
	{
	  crycgMangleList << "::" << ifaceMethod->FullName() << '\t'
			  << ifaceMethod->MangledName(); 
	}
	else 
	{ 
	  crycgMangleList << "UNKNOWN"; 
	} 
	crycgMangleList << '\t'; 
	if (!implMethod->FullName().empty() &&
	    !implMethod->MangledName().empty())
	{
	  crycgMangleList << "::" << implMethod->FullName() << '\t'
			  << implMethod->MangledName();
	}
	else 
	{ 
	  crycgMangleList << "UNKNOWN"; 
	} 
	crycgMangleList << std::endl; 
      }
    }
  }
  crycgMangleList.close();

  WriteUeberFileIncludes();

  return true;
}

std::string GetModule( const std::string &path, const std::string &baseDir )
{
  size_t pos = std::string::npos;

  std::string module = path;;

  if( module.find( baseDir ) != std::string::npos )
  {
    pos = module.find( baseDir ) + baseDir.size();
    module = module.substr( pos );
  }

  pos = module.find( Util::PathSeperator );
  module = module.substr( 0, pos );

  return module;
}

void DeVirt::WriteUeberFileIncludes()
{
  // the key is the name of the module
  // the value.first it the content for the define file for the ueberfiles, which tell which
  // implementations resits in this module
  // the value.second field is the wrapper include for the ueber files
  typedef std::map<std::string, std::pair<std::string, std::string> > TfoundModules;
  TfoundModules foundModules;

  // collect all modules from impl files
  // build list per module where the implementation files are
  for( std::list<IClass>::iterator implFile = m_IClassList.begin() ;
       implFile != m_IClassList.end() ; ++implFile )
  {		
    std::string module = GetModule( implFile->GetFile()->FileName(), m_BaseDir ); 	
	
    foundModules[ module ] = std::make_pair<std::string, std::string>("#define USING_UEBER_FILE_FOR_DEVIRTUALIZER\n#define DEVIRTUALIZE_INLINE_FUNCTION_PREFIX __attribute__((always_inline)) inline\n", "" );
  }
	
  for( std::list<IClass>::iterator implFile = m_IClassList.begin() ;
       implFile != m_IClassList.end() ; ++implFile )
  {		
    std::string module = GetModule( implFile->GetFile()->FileName(), m_BaseDir );
			
    for( TfoundModules::iterator it = foundModules.begin() ;
	 it != foundModules.end() ; ++it )
    {				
      std::string &define_collection = it->second.first;
      std::string &wrapper_file = it->second.second;

      if( it->first == module )
      {								
	define_collection.append("#define COMPLING_MODULE_CONTAINING_");
	define_collection.append( implFile->Name() );
	define_collection.append( "_CLASS\n" );

	// dont write something in the wrapper include file in case of impl in cpp file
	if( implFile->GetFile()->FileName().find(".cpp") == std::string::npos )
	{
	  wrapper_file.append("#include \"");
	  wrapper_file.append( implFile->GetFile()->FileName() );
	  wrapper_file.append( "\"\n" );

	  wrapper_file.append("#include <");
	  const Iface *iface = implFile->GetIface();
	  const std::string wrapperFileName = iface->Name() + "_Wrapper.h";
	  wrapper_file.append( wrapperFileName );
	  wrapper_file.append( ">\n" );
	}
      }
    }	
  }
	
  // dump all files
  for( TfoundModules::iterator it = foundModules.begin() ;
       it != foundModules.end() ; ++it )
  {	
    std::string defineFileName = m_OutputDir + Util::PathSeperator + 
      m_BaseDir + it->first + 
      Util::PathSeperator + it->first + "_devirt_defines.h";

    std::string wrapperFileName = m_OutputDir + Util::PathSeperator + 
      m_BaseDir + it->first + 
      Util::PathSeperator + it->first + "_wrapper_includes.h";

    std::ofstream defineFile(  defineFileName.c_str() );
    std::ofstream wrapperFile(  wrapperFileName.c_str() );

    if( !defineFile.is_open() )
    {
      DeVirt::Error("Could not create file %s\n", defineFileName.c_str() );
      exit(-1);
    }

    if( !wrapperFile.is_open() )
    {
      DeVirt::Error("Could not create file %s\n", wrapperFileName.c_str() );
      exit(-1);
    }

    defineFile << it->second.first;
    wrapperFile << it->second.second;

    defineFile.close();
    wrapperFile.close();
  }

}


// need platform specific code to get directoy content
#ifdef WIN32
#include <windows.h>
#else
#include <stdio.h>
#include <sys/types.h>
#include <dirent.h>
#endif

namespace {

  void GetDirectoryContent( std::string dir, std::vector<std::string> &files)
  {    

#if defined WIN32
    WIN32_FIND_DATA findFileData;

    dir += "*"; 
    HANDLE hFind = FindFirstFile( dir.c_str(), &findFileData);
    if(hFind  == INVALID_HANDLE_VALUE) 
    {
      std::cout <<"No files found in dir " << dir <<std::endl;
      return;
    } else 
    {
      //std::cout <<"Files found." <<std::endl;
    }
		
    int fileNumber = 0;
    files.push_back( findFileData.cFileName );
    while(FindNextFile(hFind, &findFileData)) {
      fileNumber++;
      files.push_back( findFileData.cFileName );
    }

    FindClose(hFind);
#else
    dirent *ep=NULL; DIR *dp = opendir(dir.c_str());
    if (dp == NULL) { 
      char token[1024]; 
      sprintf(token, "could not open directory %s", dir.c_str());
      perror(token); return; 
    }
    while ((ep = readdir(dp))) files.push_back( ep->d_name ); 
    closedir(dp); 
#endif

  }

}
