// Copyright: (c) by Crytek GmbH
#pragma once


#include <D3dx8math.h>										// D3DXMATRIX
#include "3DShaderManager.h"							// C3DShaderManager
#include "3DLight.h"											// C3DLight

struct IDirect3D8;
struct IDirect3DDevice8;
struct IDirect3DVertexBuffer8;
struct IDirect3DTexture8;
class C3DTargetOrientation;



enum eRenderMode
{
	eRM_Wireframe=0,										// every gfx card
	eRM_Gouraud,												// every gfx card
	eRM_GouraudSpecular,								// every gfx card
	eRM_TexturedFullbright,							// every gfx card
	eRM_TexturedLit,										// every gfx card, currently not implemented
	eRM_NormalmapFullbright,						// every gfx card
	eRM_PolybumpDiffusePixelShader,			// every gfx card with pixelshader
	eRM_PolybumpSpecPixelShader,				// every gfx card with pixelshader
	eRM_PolybumpSpecPixelShader_CM,			// every gfx card with pixelshader, normalizing per pixel shader, 2Pass
	eRM_Endmarker,											// used for cycling

	// not available through cycling ------------------------------------

	eRM_PolybumpDiffuseNoPixelShader		// every gfx card with dot3, currently not implemented
};


class C3DRenderer
{
public:

	//! constructor
	C3DRenderer();

	//! destructor
	virtual ~C3DRenderer();

	//!
	void BeginRender( const C3DTargetOrientation &inCamera, float infAspectRatio=1.0f );

	//! 
	void EndRender();

	//! show the urrent frame (switch backbuffer or blit)
	void Present( const char *inszStatusText );

	//! initialization
	//! \param handle to the output window
	//! /return true=success, false otherwise
	bool Init( HWND inhHWND );

	//! deinitialization
	void DeInit();

	//! output text on the backbuffer (could be optimized)
	//! /param infX x position in logical coordinates
	//! /param infY y position in logical coordinates
	//! /param inszText pointer to the zero terminated string
	void DrawText( float infX, float infY, const char *inszText );

	// *****************************************************************

	//! do we have pixel shader support
	//! /return true=yes we have, false otherwise
	bool AskForPixelShaderSupport();

	//! should be use wireframe (depends on the current rendermode)
	//! /return true=use wireframe, false otherwise
	bool AskWireframe();

	//! should we use textures (depends on the current rendermode)
	//! /return true=use textures, false otherwise
	bool AskTextured();

	//! get the PixelShader ID for various kinds of Polybumpmapping (depends on the current rendermode)
	//! /return 0=no shader, otherwise this value is the PixelShder ID
	DWORD AskPixelShader( const DWORD indwPass, const bool inbMissingAssets );

	//! get the VertexShader ID for variour kinds of Polybumpmapping (depends on the current rendermode)
	//! /return 0=no shader, otherwise this value is the VertexShder ID
	DWORD AskVertexShader( const DWORD indwPass, const bool inbMissingAssets );

	//!
	//! /return true=use specular, false=don't use specular
	bool AskSpecularEnable();

	//! Should the material use the non pixel shader (dot3 = GF1,GF2,GF4MX) fallback
	//! (depends on the current rendermode)
	//! /return true=use the fallback, false otherwise
	bool AskPolyBumpFallback();

	//! return a human readable shader name
	//! /return pointer to the ASCII name of the shader
	const char *AskShaderName();

	//!
	bool AskDrawLightSources();

	//!
	bool AskShowNormalmap();

	//! /return true=1 Pass for each Lightsource, false=1 Pass for all Lightsources
	bool AskEachLightOnePass();

	//!
	bool AskFullbright();

	//!
	DWORD AskAmountOfPasses();

	//!
	//! /return true=successfull, false otherwise
	bool Resize3DEnvironment();

	//! use a normalization cubemap in texture unit 2 and 3
	bool AskUseNormalizeCubemapT2T3();

	//! used for MeasureOverdraw
	//! /return true=one,one, false=zero,one
	bool AskDestBlendOneOne();


	//! /return D3DCULL_NONE | D3DCULL_CW | D3DCULL_CCW
	DWORD AskCullMode();

	//!
	bool AskReadZBuffer();

	// Attributes ******************************************************

	//! get attribute
	IDirect3DDevice8 *GetDirectXDevice();

	//! get attribute
	enum eRenderMode GetRenderMode();

	//! set attribute
	void SetRenderMode( const enum eRenderMode ineRenderMode );

	//! get attribute
	HWND GetWindowHWND() const;

	//! perspective, world and view
	D3DXMATRIX GetWVPMatrix() const;

	//! view and world
	D3DXMATRIX GetWVMatrix() const;

	//! world
	D3DXMATRIX GetWMatrix() const;

	//! get eye position in worldspace
	D3DXVECTOR3 GetEyePos() const;

	//! get eye direction
	D3DXVECTOR3 GetEyeDir() const;

	//! /return frametime in 1/100 ms
	DWORD GetFrameTime100() const;

	//! /return blended frametime in 1/100 ms
	DWORD GetBlendedFrameTime100() const;

	bool GetShowVertexBaseSpace() const;
	void SetShowVertexBaseSpace( const bool inbValue );

	bool GetShowTriBaseSpace() const;
	void SetShowTriBaseSpace( const bool inbValue );

	bool GetShowNormals() const;
	void SetShowNormals( const bool inbValue );

	bool GetForceDirLight() const;
	void SetForceDirLight( const bool inbValue );

	float GetDebugScale() const;
	void SetDebugScale( const float infValue );

	//! /param outdwWidth 0.. , could be 0 if window is minimized
	//! /param outdwHeight 0.. , could be 0 if window is minimized
	void GetBackbufferSize			( DWORD &outdwWidth, DWORD &outdwHeight ) const;


	C3DPixelShaderManager			m_ShaderManager;						//!< pixel shaders 

private:
	HWND											m_hWindow;								//!< handle to the output window
	IDirect3D8 *							m_pD3D;										//!< direct3d interface
	IDirect3DDevice8 *				m_pd3dDevice;							//!< direct3ddevice interface
	ID3DXFont *								m_pd3dFont;								//!< 3d font interface

	DWORD											m_BackbufferWidth;				//!< width
	DWORD											m_BackbufferHeight;				//!< height

	D3DXMATRIX								m_mWVPTransform;					//!< World,View,Persp Matrix
	D3DXMATRIX								m_mWVTransform;						//!< World,View Matrix
	D3DXMATRIX								m_mWTransform;						//!< World Matrix
	D3DXVECTOR3								m_vEyePos;								//!< eye position
	D3DXVECTOR3								m_vEyeDir;								//!< eye direction

	LARGE_INTEGER							m_dwRenderStartTime;			//!< in processor clock ticks, time after BeginRender (used for calculating frametime)
	DWORD											m_dwLastFrameTime100;			//!< in 1/100 ms (used for advancing physics and animation)
	DWORD											m_dwBlendedFrameTime100;	//!< in 1/100ms
	DWORD											m_dwBlendedSum100;				//!< used for m_dwBlendedFrameTime calculation, in 1/100ms
	DWORD											m_dwBlendedSamples;				//!< used for m_dwBlendedFrameTime calculation

	enum eRenderMode					m_eCurrentRenderMode;			//!< the actual rendermode (wireframe, textures, polybump, ..)
	bool											m_bShowVertexBaseSpace;		//!< show colored vectors of local space of every vertex
	bool											m_bShowTriBaseSpace;			//!< show colored vectors of local space of every triangle
	bool											m_bShowNormals;						//!< show grey normal vectors
	bool											m_bForceDirLight;					//!< true=directional lights, false=point lights
	float											m_fDebugScale;						//!< for base vectors and normals (default:1.0f), in percent of the original size e.g. 10=10% of the object


private:

	void InitFont();
	void DeInitFont();
};



