////////////////////////////////////////////////////////////////////////////
//
//  CryEngine Source File.
//  Copyright (C), Crytek.
// -------------------------------------------------------------------------
//  File name:   ExcelReport.cpp
//  Created:     19/03/2008 by Timur.
//  Description: Implementation of the CryEngine Unit Testing framework
// -------------------------------------------------------------------------
//  History:
//
////////////////////////////////////////////////////////////////////////////

#include "StdAfx.h"
#include "ExcelReport.h"
#include "CryPath.h"
#include "IResCompiler.h"

//////////////////////////////////////////////////////////////////////////
struct FilesComparePredicate
{
	bool operator()( const CFileStats *fs1,const CFileStats *fs2 )
	{
		if (fs1->m_bSuccess != fs2->m_bSuccess)
			return fs1->m_bSuccess < fs2->m_bSuccess;
		return fs1->m_DstFileSize > fs2->m_DstFileSize;
	}
};

//////////////////////////////////////////////////////////////////////////
static __int64 ComputeSizeInKB(__int64 sz)
{
	return (sz < 0) ? 0 : (sz + 512) / 1024;
}

//////////////////////////////////////////////////////////////////////////
static string GetSourceControlString( CFileStats &fs )
{
	const char *newln = "\r\n";
	string str;
	str += string(fs.m_sc.user) + "@" + fs.m_sc.workspace; str += newln;
	str += fs.m_sc.changeDescription; str += newln;
	char tempbuf[128];
	str += string("@") + itoa(fs.m_sc.change,tempbuf,10); str += newln;
	str += string("#") + itoa(fs.m_sc.revision,tempbuf,10); str += newln;

	char timebuf[128];
	time_t t = (time_t)fs.m_sc.time;
	if(ctime_s(timebuf, sizeof(timebuf), &t) != 0)
	{
		timebuf[0] = 0;
	}
	str += timebuf;

	return str;
}

//////////////////////////////////////////////////////////////////////////
void CExcelReport::Export( IResourceCompiler *pRC,const char *filename,std::vector<CFileStats*> &files )
{
	m_pRC = pRC;
	XmlNodeRef WorkBook = NewWorkbook();

	std::sort( files.begin(),files.end(),FilesComparePredicate() );

	ExportSummary(files);
	ExportTextures(files);
	ExportCGF(files);
	ExportCHR(files);
	ExportCAF(files);

	SaveToFile(filename);
}

//////////////////////////////////////////////////////////////////////////
void CExcelReport::ExportSummary( std::vector<CFileStats*> &Files )
{
	NewWorksheet( "Summary" );

	SFileVersion fv = m_pRC->GetFileVersion();

	AddColumn( "",150 );
	AddColumn( "",150 );

	AddRow();
	AddCell( "RC Version",CELL_BOLD );
	char buffer[1024];
	sprintf( buffer,"%d.%d.%d",fv.v[2],fv.v[1],fv.v[0] );
	AddCell( buffer );

	AddRow();
	AddCell( "RC Compile Time",CELL_BOLD );
	sprintf( buffer,"%s %s", __DATE__, __TIME__ );
	AddCell( buffer );

	AddRow();
	AddRow();

	sprintf( buffer,"%d files processed",(int)Files.size() );
	AddCell( buffer );

}

//////////////////////////////////////////////////////////////////////////
void CExcelReport::ExportTextures( std::vector<CFileStats*> &files )
{
	NewWorksheet( "Textures" );

	FreezeFirstRow();
	AutoFilter(1,10);

	BeginColumns();
	AddColumn( "Ok",40 );
	AddColumn( "File",400 );
	AddColumn( "User",80 );
	AddColumn( "Size (KB)",80 );
	AddColumn( "Width",50 );
	AddColumn( "Height",50 );
	AddColumn( "Mips",50 );
	AddColumn( "Format",80 );
	AddColumn( "Type",80 );
	AddColumn( "Alpha",50 );
	AddColumn( "Sides",50 );
	AddColumn( "Perforce",100 );
	AddColumn( "Error",100 );
	EndColumns();

	for (uint32 i = 0; i < files.size(); i++)
	{
		CFileStats &fs = *files[i];
		if (fs.m_type != CFileStats::eTexture)
			continue;

		AddRow();

		AddCell( fs.m_bSuccess ? "OK" : "FAIL", (fs.m_bSuccess ? 0 : CELL_HIGHLIGHT) );
		AddCell( (fs.m_sc.bValid) ? fs.m_sc.depotFile : fs.m_sSourceFilename );
		AddCell( fs.m_sc.user,CELL_CENTERED );

		AddCell( ComputeSizeInKB(fs.m_DstFileSize) );
		AddCell( fs.m_textureInfo.w );
		AddCell( fs.m_textureInfo.h );
		AddCell( fs.m_textureInfo.nNumMips );
		AddCell( fs.m_textureInfo.format,CELL_CENTERED  );

		const char *type = "2D";
		if (fs.m_textureInfo.nDepth > 1)
			type = "3D";
		else if (fs.m_textureInfo.nSides > 1)
			type = "Cubemap";
		AddCell( type,CELL_CENTERED );
		AddCell( (fs.m_textureInfo.bAlpha) ? "Yes" : "",CELL_CENTERED );
		AddCell( fs.m_textureInfo.nSides );

		string ssstr = (fs.m_sc.bValid) ? GetSourceControlString(fs) : "";
		AddCell( ssstr.c_str() );
		AddCell( fs.m_sErrorLog );
	}
}

//////////////////////////////////////////////////////////////////////////
void CExcelReport::ExportCGF( std::vector<CFileStats*> &files )
{
	NewWorksheet( "Geometry" );

	FreezeFirstRow();
	AutoFilter(1,15);

	BeginColumns();
	AddColumn( "Ok",40 );
	AddColumn( "File",400 );
	AddColumn( "User",80 );
	AddColumn( "File Size",80 );
	AddColumn( "Mesh Size (KB)",80 );
	AddColumn( "Mesh Size Lod0 (KB)",80 );
	AddColumn( "LODs",50 );
	AddColumn( "Sub Meshes",50 );
	AddColumn( "Vertices",50 );
	AddColumn( "Tris",50 );
	AddColumn( "Phys Tris",80 );
	AddColumn( "Phys Size (KB)",80 );
	AddColumn( "Phys Proxies",80 );
	AddColumn( "LODs Tris",80 );
	AddColumn( "Split LODs",80 );
	AddColumn( "Perforce",100 );
	AddColumn( "Error",100 );
	EndColumns();

	for (uint32 i = 0; i < files.size(); i++)
	{
		CFileStats &fs = *files[i];
		if (fs.m_type != CFileStats::eCGF)
			continue;

		int nMeshSizeTotal = 0;
		for (int lod = 0; lod < MAX_CGF_LODS; lod++)
			nMeshSizeTotal += fs.m_geomInfo.nMeshSizePerLod[lod];

		AddRow();

		AddCell( fs.m_bSuccess ? "OK" : "FAIL", (fs.m_bSuccess ? 0 : CELL_HIGHLIGHT) );
		AddCell( (fs.m_sc.bValid) ? fs.m_sc.depotFile : fs.m_sSourceFilename );
		AddCell( fs.m_sc.user,CELL_CENTERED );
		AddCell( ComputeSizeInKB(fs.m_DstFileSize) );
		AddCell( (nMeshSizeTotal+512)/1024 );
		AddCell( (fs.m_geomInfo.nMeshSize+512)/1024 );
		AddCell( fs.m_geomInfo.nLods );
		AddCell( fs.m_geomInfo.nSubMeshCount );
		AddCell( fs.m_geomInfo.nVertices );
		AddCell( fs.m_geomInfo.nIndices/3 );
		AddCell( fs.m_geomInfo.nPhysTriCount );
		AddCell( (fs.m_geomInfo.nPhysProxySize+512)/1024 );
		AddCell( (fs.m_geomInfo.nPhysProxyCount) );

		if (fs.m_geomInfo.nLods > 1)
		{
			// Print lod1/lod2/lod3 ...
			char tempstr[256];
			char numstr[32];
			tempstr[0] = 0;
			int numlods = 0;
			for (int lod = 0; lod < MAX_CGF_LODS; lod++)
			{
				if (fs.m_geomInfo.nIndicesPerLod[lod] != 0)
				{
					sprintf_s(numstr,sizeof(numstr),"%d",(fs.m_geomInfo.nIndicesPerLod[lod]/3) );
					if (numlods > 0)
						strcat_s(tempstr,sizeof(tempstr)," / ");
					strcat_s(tempstr,sizeof(tempstr),numstr);
					numlods++;
				}
			}
			AddCell(tempstr,CELL_CENTERED);
		}
		else
		{
			AddCell("");
		}

		AddCell( (fs.m_geomInfo.bSplitLods) ? "Yes" : "" );
		
		string ssstr = (fs.m_sc.bValid) ? GetSourceControlString(fs) : "";
		AddCell( ssstr.c_str() );
		AddCell( fs.m_sErrorLog );
	}
}


//////////////////////////////////////////////////////////////////////////
void CExcelReport::ExportCHR( std::vector<CFileStats*> &files )
{
	NewWorksheet( "Characters" );

	FreezeFirstRow();
	AutoFilter(1,15);

	BeginColumns();
	AddColumn( "Ok",40 );
	AddColumn( "File",400 );
	AddColumn( "User",80 );
	AddColumn( "File Size",80 );
	AddColumn( "Mesh Size (KB)",80 );
	AddColumn( "Mesh Size Lod0 (KB)",80 );
	AddColumn( "LODs",50 );
	AddColumn( "Sub Meshes",50 );
	AddColumn( "Vertices",50 );
	AddColumn( "Tris",50 );
	AddColumn( "Phys Tris",80 );
	AddColumn( "Phys Size (KB)",80 );
	AddColumn( "Phys Proxies",80 );
	AddColumn( "LODs Tris",80 );
	AddColumn( "Perforce",100 );
	AddColumn( "Error",100 );
	EndColumns();

	for (uint32 i = 0; i < files.size(); i++)
	{
		CFileStats &fs = *files[i];
		if (fs.m_type != CFileStats::eCHR)
			continue;

		int nMeshSizeTotal = 0;
		for (int lod = 0; lod < MAX_CGF_LODS; lod++)
			nMeshSizeTotal += fs.m_geomInfo.nMeshSizePerLod[lod];

		AddRow();

		AddCell( fs.m_bSuccess ? "OK" : "FAIL", (fs.m_bSuccess ? 0 : CELL_HIGHLIGHT) );
		AddCell( (fs.m_sc.bValid) ? fs.m_sc.depotFile : fs.m_sSourceFilename );
		AddCell( fs.m_sc.user,CELL_CENTERED );
		AddCell( ComputeSizeInKB(fs.m_DstFileSize) );
		AddCell( (nMeshSizeTotal+512)/1024 );
		AddCell( (fs.m_geomInfo.nMeshSize+512)/1024 );
		AddCell( fs.m_geomInfo.nLods );
		AddCell( fs.m_geomInfo.nSubMeshCount );
		AddCell( fs.m_geomInfo.nVertices );
		AddCell( fs.m_geomInfo.nIndices/3 );
		AddCell( fs.m_geomInfo.nPhysTriCount );
		AddCell( (fs.m_geomInfo.nPhysProxySize+512)/1024 );
		AddCell( (fs.m_geomInfo.nPhysProxyCount) );

		if (fs.m_geomInfo.nLods > 1)
		{
			// Print lod1/lod2/lod3 ...
			char tempstr[256];
			char numstr[32];
			tempstr[0] = 0;
			int numlods = 0;
			for (int lod = 0; lod < MAX_CGF_LODS; lod++)
			{
				if (fs.m_geomInfo.nIndicesPerLod[lod] != 0)
				{
					sprintf_s(numstr,sizeof(numstr),"%d",(fs.m_geomInfo.nIndicesPerLod[lod]/3) );
					if (numlods > 0)
						strcat_s(tempstr,sizeof(tempstr)," / ");
					strcat_s(tempstr,sizeof(tempstr),numstr);
					numlods++;
				}
			}
			AddCell(tempstr,CELL_CENTERED);
		}
		else
		{
			AddCell("");
		}

		string ssstr = (fs.m_sc.bValid) ? GetSourceControlString(fs) : "";
		AddCell( ssstr.c_str() );

		AddCell( fs.m_sErrorLog );
	}
}



//////////////////////////////////////////////////////////////////////////
void CExcelReport::ExportCAF( std::vector<CFileStats*> &files )
{
	NewWorksheet( "Animations" );

	FreezeFirstRow();
	AutoFilter(1,5);

	BeginColumns();
	AddColumn( "Ok",40 );
	AddColumn( "File",400 );
	AddColumn( "User",80 );
	AddColumn( "File Size",80 );
	
	AddColumn( "Perforce",100 );
	AddColumn( "Error",100 );
	EndColumns();

	for (uint32 i = 0; i < files.size(); i++)
	{
		CFileStats &fs = *files[i];
		if (fs.m_type != CFileStats::eCAF)
			continue;

		int nMeshSizeTotal = 0;
		for (int lod = 0; lod < MAX_CGF_LODS; lod++)
			nMeshSizeTotal += fs.m_geomInfo.nMeshSizePerLod[lod];

		AddRow();

		AddCell( fs.m_bSuccess ? "OK" : "FAIL", (fs.m_bSuccess ? 0 : CELL_HIGHLIGHT) );
		AddCell( (fs.m_sc.bValid) ? fs.m_sc.depotFile : fs.m_sSourceFilename );
		AddCell( fs.m_sc.user,CELL_CENTERED );
		AddCell( ComputeSizeInKB(fs.m_DstFileSize) );

		string ssstr = (fs.m_sc.bValid) ? GetSourceControlString(fs) : "";
		AddCell( ssstr.c_str() );
		AddCell( fs.m_sErrorLog );
	}
}