////////////////////////////////////////////////////////////////////////////
//
//  CryEngine Source File.
//  Copyright (C), Crytek GmbH, 2010.
// -------------------------------------------------------------------------
//  File name:   ListFile.cpp
//  Version:     v1.00
//  Created:     02/02/2010 by Timur.
//  Description: 
// -------------------------------------------------------------------------
//  History:
//
////////////////////////////////////////////////////////////////////////////

#include "StdAfx.h"
#include "ListFile.h"
#include "StringHelpers.h"
#include "TempFilePakExtraction.h"
#include "PathHelpers.h"
#include "IPakSystem.h"
#include "IResCompiler.h"
#include "IRCLog.h"

//////////////////////////////////////////////////////////////////////////
CListFile::CListFile( IResourceCompiler *pRC )
{
	m_pRC = pRC;
}

//////////////////////////////////////////////////////////////////////////
bool CListFile::Process( const string &inListFile,const string &inListFormat,const string &wildcard,std::vector<string> &outFiles )
{
	if (!inListFile.empty() && inListFile[0] == '@')
	{
		int splitter = inListFile.find_first_of( "|;," );
		if (splitter < 0)
			return false;

		string zipFilename = inListFile.substr(1,splitter-1);
		string listFilename = inListFile.substr(splitter+1);
		zipFilename.Trim();
		listFilename.Trim();

		ParseListFileInZip( zipFilename,listFilename,inListFormat,wildcard,outFiles );
		return true;
	}

	std::vector<string> lines;
	// Parse List File.
	if (!ReadLines(inListFile,lines))
		return false;

	for (size_t i = 0; i < lines.size(); ++i)
	{
		string line = lines[i];

		// Line can either be normal file
		// or a z zip file + list file (ex: @Levels\AlienVessel\Level.pak|resourcelist.txt)
		if (line[0] == '@')
		{
			// If Line starts with a @ sign, this means zip file
			
			line = line.substr(1); // skip @ character

			const size_t splitter = line.find_first_of( "|;," );
			if (splitter == line.npos)
				continue;

			string zipFilename = line.substr(0,splitter);
			string listFilename = line.substr(splitter+1);
			zipFilename.Trim();
			listFilename.Trim();

			ParseListFileInZip( zipFilename,listFilename,inListFormat,wildcard,outFiles );
		}
		else
		{
			if (!ProcessLine( line,inListFormat,wildcard,outFiles ))
			{
				return false;
			}
		}
	}

	return true;
}

//////////////////////////////////////////////////////////////////////////
bool CListFile::ReadLines( const string &inListFile,std::vector<string> &lines )
{
	FILE *f = fopen(inListFile,"rt");
	if (!f)
		return false;

	char line[1024];
	while (fgets(line,sizeof(line),f) != NULL)
	{
		if (line[0])
		{
			string strLine = line;
			strLine.Trim();
			if (!strLine.empty())
			{
				lines.push_back(strLine);
			}
		}
	}
	fclose(f);

	return true;
}

//////////////////////////////////////////////////////////////////////////
void CListFile::ParseListFileInZip( const string &zipFilename,const string &listFilename,const string &inListFormat,const string &wildcard,std::vector<string> &outFiles )
{
	// Open zip file
	IPakSystem* pPakSystem = m_pRC->GetPakSystem();

	string sFileInPak = string("@") + zipFilename + "|" + listFilename;
	TempFilePakExtraction fileProxy( sFileInPak.c_str(), pPakSystem );

	std::vector<string> lines;
	// Parse List File.
	if (!ReadLines(fileProxy.GetTempName(),lines))
	{
		RCLogWarning("List file %s not found in zip file %s",listFilename.c_str(),zipFilename.c_str());
		return;
	}

	for (size_t i = 0; i < lines.size(); ++i)
	{
		if (!ProcessLine( lines[i],inListFormat,wildcard,outFiles ))
		{
			return;
		}
	}
}

//////////////////////////////////////////////////////////////////////////
bool CListFile::ProcessLine( const string &line,const string &format,const string &wildcard,std::vector<string> &outLines )
{
	if (!StringHelpers::MatchesWildcardsIgnoreCase(line, wildcard))
	{
		return true;
	}

	std::vector<string> tokens;
	if (!StringHelpers::MatchesWildcardsIgnoreCaseExt(line, wildcard, tokens))
	{
		RCLogError("Unexpected failure in ProcessLine()");
		return false;
	}

	string str = format;

	for (;;)
	{
		const size_t startpos = str.find('{');
		if (startpos == str.npos)
		{
			break;
		}

		const size_t endpos = str.find('}', startpos + 1);
		if (endpos == str.npos)
		{
			break;
		}

		const string indexStr = str.substr(startpos + 1, endpos - startpos - 1);

		bool bBadSyntax = indexStr.empty();
		for (size_t i = 0; i < indexStr.length(); ++i)
		{
			if (!isdigit(indexStr[i]))
			{
				 bBadSyntax = true;
			}
		}
		if (bBadSyntax)
		{
			RCLogError("Syntax error in element {%s} in input string %s", indexStr.c_str(), format.c_str());
			return false;
		}

		const int index = atoi(indexStr.c_str());
		if ((index < 0) || (index > tokens.size()))
		{
			RCLogError("Bad index specified in {%s} in input string %s", indexStr.c_str(), format.c_str());
			return false;
		}

		str = str.replace(startpos, endpos - startpos + 1, ((index == 0) ? line : tokens[index - 1]));
	}

	str.replace( '/','\\' );
	outLines.push_back(str);
}
