// ResourceCompiler.cpp : Defines the entry point for the console application.
//

#include "stdafx.h"
#include <time.h>
#include <DbgHelp.h>
#include <io.h>
#include "ResourceCompiler.h"
#include "CmdLine.h"
#include "Config.h"
#include "CfgFile.h"
#include "FileUtil.h"
#include "IConvertor.h"
#include "ICrySourceControl.h"
#include "CrashHandler.h"
#include "CpuInfo.h"
#include "Mailer.h"
#include "StringHelpers.h"
#include "ListFile.h"
#include "ICryXML.h"
#include "IXmlSerializer.h"
#include "crc32.h"
#include "PropertyVars.h"
#include "ExcelReport.h"

#pragma comment( lib, "Version.lib" )

static const char* const RC_FILENAME_LOG              = "rc_log.log";
static const char* const RC_FILENAME_WARNINGS         = "rc_log_warnings.log";
static const char* const RC_FILENAME_ERRORS           = "rc_log_errors.log";
static const char* const RC_FILENAME_CRASH_DUMP       = "rc_crash.dmp";
static const char* const RC_FILENAME_FILEDEP          = "rc_stats_filedependencies.log";
static const char* const RC_FILENAME_PRESETUSAGE      = "rc_stats_presetusage.log";
static const char* const RC_FILENAME_XLS_FILESIZES    = "rc_stats_filesizes_xls.log";  // in format that can be easily read from Excel
static const char* const RC_FILENAME_XLS_REPORT       = "rc_report.xml";               // in Excel xml format that can be easily read from Excel
static const char* const RC_FILENAME_OUTPUT_FILE_LIST = "rc_outputfiles.txt";          // list of source=target filenames that rc processed, used for cleaning target folder



//////////////////////////////////////////////////////////////////////////
// Globals.
//////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////
// If not in static library.
//#ifdef CRY_STRING
//int sEmptyStringBuffer[] = { -1, 0, 0, 0 };
//string::StrHeader* string::m_emptyStringData = (string::StrHeader*)&sEmptyStringBuffer;
//wstring::StrHeader* wstring::m_emptyStringData = (wstring::StrHeader*)&sEmptyStringBuffer;
//#endif //CRY_STRING

// Determines whether a path to a file system object such as a file or directory is valid

BOOL RCPathFileExists (const char* szPath)
{
	DWORD dwAttr = GetFileAttributes (szPath);
	return (dwAttr != 0xFFFFFFFF);
}

//////////////////////////////////////////////////////////////////////////
class COutputFileList
{
public:
	struct SFile
	{
		string outputFile;
		string inputFile;
	
		bool operator==( const SFile &f ) const { return inputFile == f.inputFile && outputFile == f.outputFile; }
		bool operator!=( const SFile &f ) const { return inputFile != f.inputFile || outputFile != f.outputFile; }
		bool operator<( const SFile &f ) const { return inputFile < f.inputFile; }
	};
	std::vector<SFile> m_outputFiles;
public:
	void Add( SFile &of )
	{
		of.inputFile = PathHelpers::ToDosPath(of.inputFile);
		of.inputFile.MakeLower();
		of.outputFile = PathHelpers::ToDosPath(of.outputFile);
		of.outputFile.MakeLower();
		m_outputFiles.push_back(of);
	}
	void RemoveInputFile( const string &inputFile )
	{
		SFile searchFile;
		searchFile.inputFile = inputFile;
		std::vector<SFile>::iterator it = std::lower_bound( m_outputFiles.begin(),m_outputFiles.end(),searchFile );
		while (it != m_outputFiles.end())
		{
			if (it->inputFile == inputFile)
			{
				std::vector<SFile>::iterator next = it;
				next++;
				m_outputFiles.erase(it);
				it = next;
			}
			else
			{
				break;
			}
		}
	}
	void Sort()
	{
		struct output_files_compare : public std::binary_function<string,string,bool> 
		{
			bool operator()( const SFile &left,const SFile &right ) const
			{
				if (left.inputFile < right.inputFile)
					return true;
				if (left.inputFile > right.inputFile)
					return false;
				return left.outputFile < right.outputFile;
			}
		};
		if (!m_outputFiles.empty())
		{
			// Sort output files.
			std::sort( m_outputFiles.begin(),m_outputFiles.end(),output_files_compare() );
			// Remove duplicates
			std::vector<SFile>::iterator end = std::unique(m_outputFiles.begin(), m_outputFiles.end());
			if (end != m_outputFiles.end())
			{
				m_outputFiles.erase( end,m_outputFiles.end() );
			}
		}
	}
	void Save( const char *filename )
	{
		// store output depends file list.
		FILE *f = fopen( filename,"wt" );
		if (f)
		{
			for (size_t i = 0; i < m_outputFiles.size(); i++)
			{
				fprintf( f,"%s=%s\n",m_outputFiles[i].inputFile.c_str(),m_outputFiles[i].outputFile.c_str() );
			}
			fclose(f);
		}
	}
	void Load( const char *filename )
	{
		FILE *file = fopen( filename,"rt" );
		if (!file)
			return;

		char linebuf[MAX_PATH*4];
		while (!feof(file))
		{
			char *line = fgets(linebuf, sizeof(linebuf), file);
			if (line && line[0])
			{
				char *pos = strchr(line,'=');
				if (pos)
				{
					*pos = 0;
					SFile of;
					of.inputFile = line;
					of.outputFile = pos+1;
					of.inputFile.Trim();
					of.outputFile.Trim();
					if (!of.inputFile.empty() && !of.outputFile.empty())
					{
						m_outputFiles.push_back(of);
					}
				}
			}
		}
		fclose(file);
		
		// Sort results and remove duplicates.
		Sort();
	}
};


//////////////////////////////////////////////////////////////////////////
// ResourceCompiler implementation.
//////////////////////////////////////////////////////////////////////////
ResourceCompiler::ResourceCompiler()
{
	m_bWarningHeaderLine=false;
	m_bErrorHeaderLine=false;
	m_bStatistics = false;
	m_bQuiet = false;
	m_maxThreads = 0;
	m_numWarnings = 0;
	m_numErrors = 0;
	m_bSourceControlCreated = false;
	m_bLogRecordingEnabled = false;
	m_pOutputFileList = new COutputFileList;
	InitializeThreadIds();
}

ResourceCompiler::~ResourceCompiler()
{
	delete m_pOutputFileList; 
	if(m_bSourceControlCreated)
	{
		if(m_pSourceControl)
		{
			m_fnDestroySourceControl(m_pSourceControl);
			m_pSourceControl = 0;
		}
		m_bSourceControlCreated = false;
	}
	for (size_t i = 0; i < m_StatsFiles.size(); i++)
	{
		delete m_StatsFiles[i];
	}
}

//////////////////////////////////////////////////////////////////////////
void ResourceCompiler::RegisterConvertor( IConvertor *conv )
{
	m_extensionManager.RegisterConvertor( conv, this );
}

//////////////////////////////////////////////////////////////////////////
FILE*	ResourceCompiler::OpenFile( const char *filename,const char *mode )
{
	FILE *file = fopen(filename,mode);
	// check if read only.
	return file;
}

IRCLog *ResourceCompiler::GetIRCLog()
{
	return(this);
}

//////////////////////////////////////////////////////////////////////////
IPakSystem* ResourceCompiler::GetPakSystem()
{
	return &m_pakSystem;
}

//////////////////////////////////////////////////////////////////////////
const char* ResourceCompiler::GetSectionName( EPlatform platform ) const
{
	switch (platform)
	{
	case ePlatform_PC:				return "PC";
//	case ePlatform_XBOX:			return "XBOX";
//	case ePlatform_PS2:			return "PS2";
//	case ePlatform_GAMECUBE:	return "GAMECUBE";
//	case ePlatform_WII:			return "WII";
	case ePlatform_PS3:			return "PS3";
	case ePlatform_X360:			return "X360";
	default:
		// unknown platform.
		MessageBoxError( _T("Section name requested for unknown platform") );
		assert(0);
	}
	return "";
}

void ResourceCompiler::RemoveOutputFiles()
{
	DeleteFile(FormLogFileName(RC_FILENAME_FILEDEP));
	DeleteFile(FormLogFileName(RC_FILENAME_PRESETUSAGE));
	DeleteFile(FormLogFileName(RC_FILENAME_XLS_FILESIZES));
	DeleteFile(FormLogFileName(RC_FILENAME_XLS_REPORT));
}



class FilesToConvert
{
public:
	std::vector<RcFile> m_allFiles;
	std::vector<RcFile> m_inputFiles;
	std::vector<RcFile> m_outOfMemoryFiles;
	std::vector<RcFile> m_failedFiles;
	std::vector<RcFile> m_convertedFiles;
private:
	CryCriticalSectionRC m_lock;

public:
	void lock()
	{
		m_lock.Lock();
	}

	void unlock()
	{
		m_lock.Unlock();
	}
};


struct ThreadData
{
	ResourceCompiler* rc;
	FilesToConvert *pFilesToConvert;
	unsigned long threadIdTLSIndex;
	int threadId;
	EPlatform platform;
	Config config;
	IConvertor* convertor;
	ICompiler* compiler;
};


unsigned int WINAPI ThreadFunc(void* threadDataMemory);
	

static void CompileFilesMultiThreaded(
	ResourceCompiler* pRC,
	unsigned long a_threadIdTLSIndex,
	FilesToConvert& a_files,
	int threadCount,
	EPlatform platform,
	IConfig* config,
	IConvertor* convertor	)
{
	if (threadCount <= 0)
	{
		return;
	}

	while (!a_files.m_inputFiles.empty())
	{
		// Never create more threads than needed
		if (threadCount > a_files.m_inputFiles.size())
		{
			threadCount = a_files.m_inputFiles.size();
		}

		RCLog("Spawning %d thread%s", threadCount, ((threadCount > 1) ? "s" : ""));
		RCLog("");

		char szRCPath[1000];
		GetModuleFileName(NULL, szRCPath, sizeof(szRCPath));
		string exePath = PathHelpers::GetDirectory(szRCPath);

		// Initialize the convertor
		convertor->Init(config, exePath.c_str());

		// Initialize the thread data for each thread.
		std::vector<ThreadData> threadData(threadCount);
		for (int threadIndex = 0; threadIndex < threadCount; ++threadIndex)
		{
			threadData[threadIndex].rc = pRC;
			threadData[threadIndex].platform = platform;
			threadData[threadIndex].config.SetConfigKeyRegistry(pRC);
			threadData[threadIndex].config.Merge(config);
			threadData[threadIndex].convertor = convertor;
			threadData[threadIndex].compiler = convertor->CreateCompiler();
			threadData[threadIndex].threadIdTLSIndex = a_threadIdTLSIndex;
			threadData[threadIndex].threadId = threadIndex + 1;
			threadData[threadIndex].pFilesToConvert = &a_files;
		}

		// Spawn the threads.
		std::vector<HANDLE> threads(threadCount);
		for (int threadIndex = 0; threadIndex < threadCount; ++threadIndex)
		{
			threads[threadIndex] = (HANDLE)_beginthreadex( 
				0,                               //void *security,
				0,                               //unsigned stack_size,
				ThreadFunc,                      //unsigned ( *start_address )( void * ),
				&threadData[threadIndex],        //void *arglist,
				0,                               //unsigned initflag,
				0);                              //unsigned *thrdaddr 
		}

		// Wait until all the threads have exited
		int lastCountShown = -1;
		while(WaitForMultipleObjects(threads.size(), &threads[0], TRUE, 500) == WAIT_TIMEOUT)
		{
			// Show progress

			a_files.lock();

			if(!a_files.m_inputFiles.empty())
			{
				const int processedFileCount = a_files.m_outOfMemoryFiles.size() + a_files.m_failedFiles.size() + a_files.m_convertedFiles.size();

				if(processedFileCount != lastCountShown)
				{
					lastCountShown = processedFileCount;
					const float fPercentage = (100.f * processedFileCount) / a_files.m_allFiles.size();
					char str[1024];
					_snprintf_s(str, sizeof(str), sizeof(str) - 1,"Progress: %.1f%% %s", fPercentage, a_files.m_inputFiles.back().m_sourceInnerPathAndName.c_str());
					SetConsoleTitle(str);
				}
			}

			a_files.unlock();
		}

		assert(a_files.m_inputFiles.empty());

		// Release all the compiler objects.
		for (int threadIndex = 0; threadIndex < threadCount; ++threadIndex)
		{
			threadData[threadIndex].compiler->Release();
		}

		// Clean up the converter.
		convertor->DeInit();

		if(!a_files.m_outOfMemoryFiles.empty())
		{
			if(threadCount > 1)
			{
				// If we ran out of memory when processing files, we will process the files again in one thread (since we may
				// have run out of memory just because we had multiple threads).
				a_files.m_inputFiles.insert( a_files.m_inputFiles.end(), a_files.m_outOfMemoryFiles.begin(), a_files.m_outOfMemoryFiles.end() );
				threadCount = 1; 
			}
			else
			{
				a_files.m_failedFiles.insert( a_files.m_failedFiles.end(), a_files.m_outOfMemoryFiles.begin(), a_files.m_outOfMemoryFiles.end() );
			}

			a_files.m_outOfMemoryFiles.resize(0);
		}
	}
}


//////////////////////////////////////////////////////////////////////////
// Returns true if successfully converted at least one file
bool ResourceCompiler::Compile( EPlatform platform,IConfig *config,const char *filespec )
{
	SetupMaxThreads(config);

	RemoveOutputFiles();		// to remove old files for less confusion

	m_bStatistics = config->HasKey("statistics");

	FilesToConvert filesToConvert;	// paths in filesToConvert.m_allFiles are relative

	const bool bVerbose = config->HasKey("verbose");
	const bool bRecursive = config->GetAs<bool>("recursive", true);
	
	m_presets = new CfgFile();
	string presetcfg;

	if(!config->Get("presetcfg", presetcfg))
	{
		char str[512];

		sprintf(str,"No preset configuration defined (e.g. presetcfg=rc_presets_pc.ini)");

		RCLog("%s",str);

		MessageBox(0,str,"ResourceCompiler Error",MB_OK|MB_ICONERROR);
	
		// it's better to have resource not working without that info
		return false;
	}
	else if(!m_presets->Load( PathHelpers::Join(m_exePath,presetcfg)) )
	{
		char str[512];

		sprintf(str,"Failed to read preset configuration %s, exiting...", presetcfg.c_str());

		RCLog("%s",str);

		MessageBox(0,str,"ResourceCompiler Error",MB_OK|MB_ICONERROR);

		// it's better to have resource not working without that info
		return false;
	}

	RCLog("Used preset configuration: presetcfg=%s",presetcfg.c_str());

	// RC expects source filenames in command line in form 
	// "<source left path><mask for recursion>" or "<source left path><non-masked name>".
	// After recursive subdirectory scan we will have a list with source filenames in 
	// form "<source left path><source inner path><name>".
	// The target filename can be written as "<target left path><source inner path><name>".

	// Determine the target output path (may be a different directory structure).
	// If none is specified, the target path is the same as the <source left path>.
	string targetLeftPath;
	config->Get("targetroot", targetLeftPath);
	targetLeftPath = PathHelpers::RemoveSeparator(targetLeftPath);

	string sourceRoot;
	config->Get("sourceroot", sourceRoot);
	sourceRoot = PathHelpers::RemoveSeparator(sourceRoot);

	bool bUseListFile = false;
	string listFile;
	config->Get( "listfile",listFile );
	string listFormat;
	config->Get( "listformat",listFormat );
	if (!listFile.empty())
	{
		if (listFormat.empty())
		{
			listFormat = "{0}";
		}
		bUseListFile = true;
		CListFile lf(this);
		std::vector<string> filenames;
		lf.Process( listFile,listFormat,filespec,filenames );
		if (filenames.empty())
		{
			RCLogError("No files to convert found in list file %s", listFile.c_str());
			return false;
		}
		if (bVerbose)
		{
			RCLog("Contents of the list file \"%s\" (filter is \"%s\"):", listFile.c_str(), filespec);
			for (size_t i = 0; i < filenames.size(); ++i)
			{
				RCLog(" %3d: \"%s\"", i, filenames[i].c_str());
			}
		}
		for (size_t i = 0; i < filenames.size(); ++i)
		{
			filesToConvert.m_allFiles.push_back(RcFile(sourceRoot, filenames[i], targetLeftPath));
		}
	}

	if (!bUseListFile)
	{
		const DWORD dwFileSpecAttr = GetFileAttributes(PathHelpers::Join(sourceRoot, filespec));

		if (dwFileSpecAttr == 0xFFFFFFFF)
		{
			// There's no such file
			if (string(filespec).find_first_of("*?") == string::npos)
			{
				// It's not a mask (path\*.mask). Allow later part of the code to open it from the .pak (see PakSystem::Open()).
				RCLog("RC can't find file '%s'. Will try to find the file in .pak files.", filespec);
				if (sourceRoot.empty())
				{
					filesToConvert.m_allFiles.push_back(RcFile(PathHelpers::GetDirectory(filespec), PathHelpers::GetFilename(filespec), targetLeftPath));
				}
				else
				{
					filesToConvert.m_allFiles.push_back(RcFile(sourceRoot, filespec, targetLeftPath));
				}
			}
			else
			{
				// It's a mask (path\*.mask). Scan directory and accumulate matching filenames in the list.
				// Multiple masks allowed, for example path\*.xml;*.dlg;path2\*.mtl

				std::vector<string> tokens;
				StringHelpers::Split(filespec, ";", false, tokens);

				for (size_t t = 0; t < tokens.size(); ++t)
				{
					// Scan directory and accumulate matching filenames in the list.
					const string path = PathHelpers::Join(sourceRoot, PathHelpers::GetDirectory(tokens[t]));
					const string pattern = PathHelpers::GetFilename(tokens[t]);
					RCLog("Scanning directory '%s' for '%s'...", path.c_str(), pattern.c_str());
					std::vector<string> filenames;
					FileUtil::ScanDirectory(path, pattern, filenames, bRecursive, targetLeftPath);
					for (size_t i = 0; i < filenames.size(); ++i)
					{
						string sourceLeftPath;
						string sourceInnerPathAndName;
						if (sourceRoot.empty())
						{
							sourceLeftPath = PathHelpers::GetDirectory(tokens[t]);
							sourceInnerPathAndName = filenames[i];
						}
						else
						{
							sourceLeftPath = sourceRoot;
							sourceInnerPathAndName = PathHelpers::Join(PathHelpers::GetDirectory(tokens[t]), filenames[i]);
						}
						filesToConvert.m_allFiles.push_back(RcFile(sourceLeftPath, sourceInnerPathAndName, targetLeftPath));
					}
				}

				if (filesToConvert.m_allFiles.empty())
				{
					// We failed to find any file matching the mask specified by user.
					// Using mask (say, *.cgf) usually means that user doesn't know if
					// the file exists or not, so it's better to return "success" code.
					RCLog("RC can't find files matching '%s', 0 files converted", filespec);
					return true;
				}
			}
		}
		else
		{
			// The file exists

			if (dwFileSpecAttr & FILE_ATTRIBUTE_DIRECTORY)
			{
				// We found a file, but it's a directory, not a regular file.
				// Let's assume that the user wants to export every file in the 
				// directory (with subdirectories if bRecursive == true) or
				// that he wants to export a file specified in /file option.
				const string path = PathHelpers::Join(sourceRoot, filespec);
				const string pattern = config->GetAs<string>("file", "*.*");
				RCLog("Scanning directory '%s' for '%s'...", path.c_str(), pattern.c_str());
				std::vector<string> filenames;
				FileUtil::ScanDirectory(path, pattern, filenames, bRecursive, targetLeftPath);
				for (size_t i = 0; i < filenames.size(); ++i)
				{
					string sourceLeftPath;
					string sourceInnerPathAndName;
					if (sourceRoot.empty())
					{
						sourceLeftPath = filespec;
						sourceInnerPathAndName = filenames[i];
					}
					else
					{
						sourceLeftPath = sourceRoot;
						sourceInnerPathAndName = PathHelpers::Join(filespec, filenames[i]);
					}
					filesToConvert.m_allFiles.push_back(RcFile(sourceLeftPath, sourceInnerPathAndName, targetLeftPath));
				}

				if (filesToConvert.m_allFiles.empty())
				{
					if (pattern.find_first_of("*?") == string::npos)
					{
						// Failed to find the file specified by user.
						// Using a filename without mask (say, abcde.cgf) usually means that user
						// expects that the file exists, so it's better to return "failure" code.
						RCLog("RC can't find file %s, 0 files converted", filespec);
						return false;
					}
					else
					{
						// Failed to find any file matching the mask specified by user.
						// Using mask (say, *.cgf) usually means that user doesn't know if
						// the file exists or not, so it's better to return "success" code.
						RCLog("RC can't find files matching %s, 0 files converted", filespec);
						return true;
					}
				}
			}
			else
			{	
				// we found a regular file
				if (sourceRoot.empty())
				{
					filesToConvert.m_allFiles.push_back(RcFile(PathHelpers::GetDirectory(filespec), PathHelpers::GetFilename(filespec), targetLeftPath));
				}
				else
				{
					filesToConvert.m_allFiles.push_back(RcFile(sourceRoot, filespec, targetLeftPath));
				}
			}
		}

		if (filesToConvert.m_allFiles.empty())
		{
			RCLogError("No Files to Convert.");
			return false;
		}
	}

	if (config->GetAs<bool>("copyonly", false))
	{
		if (targetLeftPath.empty())
		{
			RCLogError("/copyonly: you must specify /targetroot.");
			return false;
		}
		CopyFiles(config, filesToConvert.m_allFiles);
		return true;
	}

	{
		string pakFilename;
		if (config->Get("zip", pakFilename))
		{
			string folderInPak;
			config->Get("FolderInZip", folderInPak);
			return CreatePakFile(config, filesToConvert.m_allFiles, folderInPak, pakFilename, true);
		}
	}

	// these are the files that couldn't be converted
	std::vector<string> arrNonConvertedFiles;

	// Split up the files based on the convertor they are to use.
	typedef std::map<IConvertor*, std::vector<RcFile> > FileConvertorMap;
	FileConvertorMap fileConvertorMap;
	for (size_t i = 0; i < filesToConvert.m_allFiles.size(); ++i)
	{
		string filenameForConvertorSearch;
		{
			string sOverWriteExtension;
			if (config->Get("overwriteextension", sOverWriteExtension))
			{
				filenameForConvertorSearch = string("filename.") + sOverWriteExtension;
			}
			else
			{
				filenameForConvertorSearch = filesToConvert.m_allFiles[i].m_sourceInnerPathAndName;
			}
		}

		IConvertor* convertor = m_extensionManager.FindConvertor(platform, filenameForConvertorSearch.c_str());

		FileConvertorMap::iterator convertorIt = fileConvertorMap.find(convertor);
		if (convertorIt == fileConvertorMap.end())
		{
			convertorIt = fileConvertorMap.insert(std::make_pair(convertor, std::vector<RcFile>())).first;
		}
		(*convertorIt).second.push_back(filesToConvert.m_allFiles[i]);
	}

	if (bVerbose)
	{
		RCLog("%d file%s to convert:", filesToConvert.m_allFiles.size(), ((filesToConvert.m_allFiles.size() != 1) ? "s" : ""));
		for (size_t i = 0; i < filesToConvert.m_allFiles.size(); ++i)
		{
			RCLog("  \"%s\"  \"%s\" -> \"%s\"",
				filesToConvert.m_allFiles[i].m_sourceLeftPath.c_str(), 
				filesToConvert.m_allFiles[i].m_sourceInnerPathAndName.c_str(), 
				filesToConvert.m_allFiles[i].m_targetLeftPath.c_str());
		}
		RCLog("");
	}

	// Loop through all the convertors that we need to invoke.
	for (FileConvertorMap::iterator convertorIt = fileConvertorMap.begin(); convertorIt != fileConvertorMap.end(); ++convertorIt)
	{
		assert(filesToConvert.m_inputFiles.empty());
		assert(filesToConvert.m_outOfMemoryFiles.empty());

		IConvertor* convertor = (*convertorIt).first;
		if (!convertor)
		{
			continue;
		}

		// Check whether this convertor is thread-safe.
		assert(m_maxThreads >= 1);
		int threadCount = m_maxThreads;
		if ((threadCount > 1) && (!convertor->SupportsMultithreading()))
		{
			RCLog("/threads specified, but convertor does not support multi-threading. Falling back to single-threading.");
			threadCount = 1;
		}

		const std::vector<RcFile>& convertorFiles = (*convertorIt).second;
		assert(convertorFiles.size()>0);

		// implementation note: we insert filenames starting from last, because converting function will take filenames one by one from the end(!) of the array
		for(int i = convertorFiles.size() - 1; i >= 0; --i)
		{
			filesToConvert.m_inputFiles.push_back(convertorFiles[i]);
		}

		CompileFilesMultiThreaded(this, m_threadIdTLSIndex, filesToConvert, threadCount, platform, config, convertor);

		assert(filesToConvert.m_inputFiles.empty());
		assert(filesToConvert.m_outOfMemoryFiles.empty());
	}

	const int numFilesConverted = filesToConvert.m_convertedFiles.size();
	const int numFilesFailed = filesToConvert.m_failedFiles.size();
	assert(numFilesConverted+numFilesFailed == filesToConvert.m_allFiles.size());

	const bool savedTimeLogging = GetTimeLogging();
	SetTimeLogging(false);

	char szTimeMsg[128];
	sprintf_s(szTimeMsg, " in %.1f sec", difftime(time(0), m_startTime));

	RCLog("");

	if (numFilesFailed <= 0)
	{
		RCLog("%d file%s processed%s.", numFilesConverted, (numFilesConverted > 1? "s" : ""), szTimeMsg);
		RCLog("");
	}
	else
	{
		const bool bLogSourceControlInfo = config->HasKey("sourcecontrol");
		string sourceControlClientName;

		if(bLogSourceControlInfo)
		{
			// Lazy registering of source control
			if(!m_bSourceControlCreated)
			{
				CreateSourceControl();
			}
			if(!config->Get("sourcecontrol", sourceControlClientName))
			{
				sourceControlClientName = "";
			}
		}

		RCLog("");
		RCLog(
			"%d of %d file%s were converted%s. Couldn't convert the following file%s:", 
			numFilesConverted, 
			numFilesConverted + numFilesFailed, 
			(numFilesConverted + numFilesFailed > 1 ? "s" : ""), 
			szTimeMsg, 
			(numFilesFailed > 1 ? "s" : ""));
		RCLog("");
		for(size_t i = 0; i < numFilesFailed; ++i)
		{
			const string failedFilename = PathHelpers::Join(filesToConvert.m_failedFiles[i].m_sourceLeftPath, filesToConvert.m_failedFiles[i].m_sourceInnerPathAndName);
			LogFailedFileInfo(config, i, failedFilename, bLogSourceControlInfo, sourceControlClientName.c_str());
		}
		RCLog("");
	}

	delete m_presets;

	SetTimeLogging(savedTimeLogging);

	return numFilesConverted > 0;
}


static void EnableFloatingPointExceptions()
{
	_clearfp();
	unsigned int old;
	_controlfp_s( &old, _EM_INEXACT, _MCW_EM );
}


unsigned int WINAPI ThreadFunc(void* threadDataMemory)
{
//	EnableFloatingPointExceptions();

	ThreadData* data = static_cast<ThreadData*>(threadDataMemory);

	// Initialize the thread local storage, so the log can prepend the thread id to each line.
	TlsSetValue(data->threadIdTLSIndex, &data->threadId);

	// Create a copy of the main configuration.
	Config localConfig;
	localConfig.SetConfigKeyRegistry(data->rc);
	localConfig.Merge(&data->config);	// merge main config into local config

	data->compiler->BeginProcessing();

	for(;;)
	{
		data->pFilesToConvert->lock();

		if (data->pFilesToConvert->m_inputFiles.empty())
		{
			data->pFilesToConvert->unlock();
			break;
		}

		const RcFile fileToConvert = data->pFilesToConvert->m_inputFiles.back();
		data->pFilesToConvert->m_inputFiles.pop_back();

		data->pFilesToConvert->unlock();

		const string sourceInnerPath = PathHelpers::GetDirectory(fileToConvert.m_sourceInnerPathAndName);
		const string sourceFullFileName = PathHelpers::Join(fileToConvert.m_sourceLeftPath, fileToConvert.m_sourceInnerPathAndName);

		int result;

		try
		{
			if (data->rc->CompileFile(data->platform, &localConfig, sourceFullFileName.c_str(), fileToConvert.m_targetLeftPath.c_str(), sourceInnerPath.c_str(), data->compiler, data->convertor ))
			{
				result = 1;
			}
			else
			{
				result = 2;
			}
		}
		catch (std::bad_alloc&)
		{
			result = 3;
		}
		catch (...)
		{
			result = 2;
		}

		data->pFilesToConvert->lock();
		switch (result)
		{
		case 1:
			data->pFilesToConvert->m_convertedFiles.push_back(fileToConvert);
			break;
		case 2:
			data->pFilesToConvert->m_failedFiles.push_back(fileToConvert);
			break;
		case 3:
			data->pFilesToConvert->m_outOfMemoryFiles.push_back(fileToConvert);
			break;
		default:
			assert(0);
			break;
		}
		data->pFilesToConvert->unlock();
	}

	data->compiler->EndProcessing();

	return 0;
}

void ResourceCompiler::EnsureDirectoriesPresent(const char *path)
{
	DWORD dwFileSpecAttr = GetFileAttributes (path);
	if (dwFileSpecAttr == 0xFFFFFFFF && *path)
	{
		EnsureDirectoriesPresent(PathHelpers::GetDirectory(PathHelpers::RemoveSeparator(path)).c_str());
		if(_mkdir(path))
			RCLog("Creating directory failed: %s", path);

//		RCLog("Creating directory %s (%s)", path, _mkdir(path) ? "failed" : "ok");
	}
}
	

// makes the relative path out of any
string NormalizePath(const char* szPath)
{
	char szCurDir[0x800]="";
	GetCurrentDirectory(sizeof(szCurDir),szCurDir);
	strcat(szCurDir, "/");

	char szFullPath[0x800];
	if (!_fullpath(szFullPath, szPath, sizeof(szFullPath)))
		strcpy (szFullPath, szPath);


	char* p, *q;
	string sRes = szPath;
	for (p = szCurDir, q = szFullPath; *p && *q; ++p, ++q)
	{
		if (tolower(*p)==tolower(*q))
			continue;

    if ((*p=='/'||*p=='\\')&&(*q=='/'||*q=='\\'))
			continue;

		return sRes;
	}

	if (*p)
		return szPath;

	return q; // return whatever has left after truncating the leading path
}

//////////////////////////////////////////////////////////////////////////
bool ResourceCompiler::CompileFile(EPlatform platform, IConfig* config, const char* const sourceFullFileName, const char* const targetLeftPath, const char* const sourceInnerPath, ICompiler* compiler, IConvertor* convertor )
{
	CmdLine cmdLine;

//	if (!RCPathFileExists(sourceFullFileName) && strstr(sourceFullFileName,".cba")==0)
//		return false;
// don't check for whether the file exists
//	bool fileExists = (0 != RCPathFileExists(sourceFullFileName));
//	if (!fileExists) // [MichaelS 28/4/2008] If files don't exist, look for the same path with a '.zip' extension.
//		fileExists = (0 != RCPathFileExists((string(sourceFullFileName) + ".zip").GetString()));
//	if (!fileExists)
//		return false;

	const string targetPath = PathHelpers::Join(targetLeftPath, sourceInnerPath);

	// get file extension.
	string ext = PathHelpers::FindExtension(sourceFullFileName);

	{
		string sOverwriteExtension;

		if(config->Get("overwriteextension",sOverwriteExtension))
		{
			ext = sOverwriteExtension;
		}
	}

	ext.MakeLower();

	// get key for special copy/ignore options to certain extensions
	{
		string extkey = "ext_";
		extkey += ext;
		string extcommand;
		if(config->Get(extkey.c_str(), extcommand))
		{
			if(extcommand=="ignore")
			{
				RCLog("Ignoring %s", sourceFullFileName);
				return false;
			}
				
			if(extcommand=="copy")
			{
				string targetFullFileName = targetPath;
				targetFullFileName += PathHelpers::GetFilename(sourceFullFileName);
				if(targetFullFileName != sourceFullFileName)
				{
					// TODO: can compare filestamps of source and destination to avoid copy, but maybe overkill
					RCLog("Copying %s to %s", sourceFullFileName, targetFullFileName.c_str());
					CopyFile(sourceFullFileName, targetFullFileName.c_str(), false); // overwrites any existing file, same as all converters
					FileUtil::SetFileTimes(targetFullFileName.c_str(), FileUtil::GetLastWriteFileTime(sourceFullFileName));
				}
				return true;
			}
		}
	}

	CfgFile CfgFile;				// file specific config file

	Config localConfig;

	localConfig.SetConfigKeyRegistry(this);

	localConfig.Merge(config);	// merge main config into local config

	// Setup conversion context.
	IConvertContext* pCC = convertor->CreateConvertContext();

	pCC->SetConfig(&localConfig);
	pCC->SetPlatform(platform);
	pCC->SetPlattformName(GetSectionName(platform));
	pCC->SetRC(this);
	pCC->SetThreads(m_maxThreads);

	pCC->SetSourceFileFinal(PathHelpers::GetFilename(sourceFullFileName));
	pCC->SetSourceFileFinalExtension(ext);
	pCC->SetSourceFolder(NormalizePath(PathHelpers::GetDirectory(sourceFullFileName)));

	const string outputFolder = NormalizePath(targetPath);
	pCC->SetOutputFolder(outputFolder);
	pCC->SetPresets(m_presets);
	pCC->SetQuiet(m_bQuiet);
	compiler->ConstructAndSetOutputFile(*(ConvertContext*)pCC);

	EnsureDirectoriesPresent(outputFolder.c_str());
	
	const bool bVerbose = config->HasKey("verbose");

	string filenameForUpToDateCheck;
	{
		char buff[MAX_PATH];
		buff[0] = 0;
		compiler->GetFilenameForUpToDateCheck(*(ConvertContext*)pCC, buff, sizeof(buff));
		filenameForUpToDateCheck = buff;
	}

	const FILETIME fileTimeSource = FileUtil::GetLastWriteFileTime(sourceFullFileName);

	if ((!filenameForUpToDateCheck.empty()) && FileUtil::FileTimeIsValid(fileTimeSource) && (!config->HasKey("refresh")))
	{
		const FILETIME fileTime = FileUtil::GetLastWriteFileTime(filenameForUpToDateCheck);

		if (FileUtil::FileTimesAreEqual(fileTimeSource, fileTime))
		{
			AddOutputFile( filenameForUpToDateCheck.c_str(),sourceFullFileName );

			if (bVerbose)
			{
				RCLog("Skipping %s: file %s is up to date", sourceFullFileName, filenameForUpToDateCheck.c_str());
			}
			pCC->Release();
			return true;
		}												
	}

	RCLog("-------------------------------------------------------");
	if (bVerbose)
	{
		RCLog("Path='%s'", PathHelpers::RemoveSeparator(sourceInnerPath).c_str());
		RCLog("File='%s'", PathHelpers::GetFilename(sourceFullFileName).c_str());
	}
	else
	{
		const string sourceInnerPathAndName = PathHelpers::AddSeparator(sourceInnerPath) + PathHelpers::GetFilename(sourceFullFileName);
		RCLog("File='%s'", sourceInnerPathAndName.c_str());
	}

	OutputDebugString("Current file: '");
	OutputDebugString(sourceFullFileName);
	OutputDebugString("' ... ");

	// file name changed - print new header for warnings and errors
	SetHeaderLine(sourceFullFileName);

	// Start Recording Log messages from convertor.
	StartLogRecording();

	// compile file
	bool const bRet = compiler->Process( *(ConvertContext*)pCC );

	if (bRet && FileUtil::FileTimeIsValid(fileTimeSource) && !filenameForUpToDateCheck.empty())
	{
		FileUtil::SetFileTimes(filenameForUpToDateCheck, fileTimeSource);
	}

	// Stop Recording Log messages from convertor.
	StopLogRecording();

	// Only do if no MT
	if (GetMaxNumThreads() == 1)
	{
		m_LogForFileMap[sourceFullFileName] = m_RecordedLogLines;
	}

	OutputDebugString("processed\n");

	if (!bRet)
	{
		LogError("failed to convert file %s",sourceFullFileName);
	}

	// Release cloned config.
//	if (config != m_config)
//		config->Release();

	pCC->Release();

	return bRet;
}

//////////////////////////////////////////////////////////////////////////
void ResourceCompiler::AddOutputFile( const char *sOutputFilename,const char *sInputFilename )
{
	assert(sOutputFilename && sInputFilename);
	assert(sInputFilename[0] && sOutputFilename[0]); 

	m_outputFilesLock.Lock();

	COutputFileList::SFile out;
	out.outputFile = sOutputFilename;
	out.inputFile = sInputFilename;
	m_pOutputFileList->Add(out);

	m_outputFilesLock.Unlock();
}


void ResourceCompiler::SetHeaderLine( const char *inszLine )
{
	m_bWarningHeaderLine=false;
	m_bErrorHeaderLine=false;
	m_sHeaderLine=inszLine;
}

void ResourceCompiler::InitializeThreadIds()
{
	m_threadIdTLSIndex = TlsAlloc();
	if (m_threadIdTLSIndex == TLS_OUT_OF_INDEXES)
	{
		printf("RC Initialization error");
		exit( 100 );
	}
	TlsSetValue(m_threadIdTLSIndex, 0);
}

int ResourceCompiler::GetThreadId() const
{
	const int* pThreadId = (const int*) TlsGetValue(m_threadIdTLSIndex);
	const int threadId = (pThreadId ? (*pThreadId) : -1);
	return threadId;
}

void ResourceCompiler::LogLine( const ELogType ineType, const char* szText )
{
	// Get the index of the thread.
	const int threadIndex = GetThreadId();

	static volatile int g_LogLock;
	WriteLock lock(g_LogLock);

	FILE* fLog = 0;
	if (!m_mainLogFileName.empty())
	{
		fopen_s(&fLog, m_mainLogFileName.c_str(), "a+t");
	}

	if (m_bQuiet)
	{
		if (fLog)
		{
			fprintf(fLog,"%s\n",szText);
			fflush(fLog);
			fclose(fLog);
			fLog = 0;
		}
		return;
	}
				
	char threadString[10];
	threadString[0] = 0;
	if (threadIndex > 0)
	{
		sprintf_s(threadString, "%d> ", threadIndex);
	}

	char timeString[20];
	timeString[0] = 0;
	if (m_bTimeLogging)
	{
		const int seconds = (int)(difftime(time(0), m_startTime) + 0.5);
		const int minutes = seconds / 60;
		sprintf_s(timeString, "%d:%02d ", minutes, seconds - (minutes * 60));
	}

	const char* prefix = "";
	const char* additionalLogFileName = 0;
	bool* pbAdditionalLogHeaderLine = 0;

	switch(ineType)
	{
	case eMessage:
		prefix = "   ";
		break;

	case eWarning:
		prefix = "W: ";
		if (!m_warningLogFileName.empty())
		{
			additionalLogFileName = m_warningLogFileName.c_str();
			pbAdditionalLogHeaderLine = &m_bWarningHeaderLine;
		}
		break;

	case eError:
		prefix = "E: ";
		if (!m_errorLogFileName.empty())
		{
			additionalLogFileName = m_errorLogFileName.c_str();
			pbAdditionalLogHeaderLine = &m_bErrorHeaderLine;
		}
		break;

	default:
		assert(0);
		break;
	}

	if (additionalLogFileName)
	{
		FILE* fAdditionalLog = 0;
		fopen_s(&fAdditionalLog, additionalLogFileName, "a+t");

		if (fAdditionalLog)
		{
			if (*pbAdditionalLogHeaderLine == false)
			{
				fprintf(fAdditionalLog, "-----------------------------------------------------------------\n");
				fprintf(fAdditionalLog, "%s%s%s%s\n", prefix, threadString, timeString, m_sHeaderLine.c_str());
				*pbAdditionalLogHeaderLine = true;
			}
			fprintf(fAdditionalLog, "%s%s%s%s\n", prefix, threadString, timeString, szText);

			fflush(fAdditionalLog);
			fclose(fAdditionalLog);
		}
	}

	if (fLog)
	{
		fprintf(fLog, "%s%s%s%s\n", prefix, threadString, timeString, szText);

		fflush(fLog);
		fclose(fLog);
		fLog = 0;
	}

	if (m_bWarningsAsErrors && (ineType == eWarning || ineType == eError))
	{
		MessageBox( NULL,szText,"RC Compilation Error",MB_OK|MB_ICONERROR );
		exit( EXIT_FAILURE );
	}

	if (m_bLogRecordingEnabled)
	{
		m_RecordedLogLines.push_back( szText );
	}

	printf("%s%s%s%s\n", prefix, threadString, timeString, szText);
}

//////////////////////////////////////////////////////////////////////////
void MessageBoxError( const char *format,... )
{
	va_list	ArgList;
	char		szBuffer[1024];

	va_start(ArgList, format);
	vsprintf(szBuffer, format, ArgList);
	va_end(ArgList);

	string str = "####-ERROR-####: ";
	str += szBuffer;

//	printf( "%s\n",str );
	MessageBox( NULL,szBuffer,_T("Error"),MB_OK|MB_ICONERROR );
}

//////////////////////////////////////////////////////////////////////////
void ResourceCompiler::LogMultiLine( const char *szText )
{
	const char *p=szText;
	char szLine[80],*pLine=szLine;

	for(;;)
	{
		if(*p=='\n' || *p==0 || (pLine-szLine)>=sizeof(szLine)-(5+2+1)) // 5 spaces +2 (W: or E:) +1 to avoid nextline jump
		{
			*pLine=0;                                                     // zero termination
			RCLog("     %s",szLine);                                      // 5 spaces
			pLine=szLine;
			
			if(*p=='\n')
			{
				++p;
				continue;
			}
		}

		if(*p==0)
			return;

		*pLine++=*p++;

	} while(*p);
}

//////////////////////////////////////////////////////////////////////////
void ResourceCompiler::show_help( const bool bDetailed )
{
	RCLog( "" );
	RCLog( "Usage: RC filespec /p=<platform> [/Key1=Value1] [/Key2=Value2] etc..." );

	if(bDetailed)
	{
		RCLog( "" );

		std::map<string,string>::const_iterator it, end=m_KeyHelp.end();

		for(it=m_KeyHelp.begin();it!=end;++it)
		{
			const string &rKey = it->first;
			const string &rHelp = it->second;

			RCLog("/%s",rKey.c_str());
			LogMultiLine(rHelp.c_str());
			RCLog( "" );
			RCLog( "" );
		}
	}
	else
	{
		RCLog( "       RC /help             // will list all usable keys with description" );
		RCLog( "       RC /help >file.txt   // help to file.txt" );
	}

	RCLog( "" );
}

//////////////////////////////////////////////////////////////////////////
static EPlatform GetPlatformFromName( const char *sPlatform )
{
	// Platform name to enum mapping.
	struct {
		const char *name;
		EPlatform platform;
	} platformNames[] =
	{
		{ "PC", ePlatform_PC },
//		{ "XBOX", ePlatform_XBOX },
//		{ "PS2", ePlatform_PS2 },
//		{ "GameCube", ePlatform_GAMECUBE },
//		{ "WII", ePlatform_WII },
		{ "PS3", ePlatform_PS3 },
		{ "X360", ePlatform_X360 }, 
	};
	for (int i = 0; i < sizeof(platformNames)/sizeof(platformNames[0]); i++)
	{
		if (stricmp(platformNames[i].name,sPlatform) == 0)
			return platformNames[i].platform;
	}
	return ePlatform_UNKNOWN;
}




//////////////////////////////////////////////////////////////////////////
void RegisterConvertors( IResourceCompiler *rc )
{
	string strDir;
	{
		char szRCPath[1000];
		if (GetModuleFileName (NULL, szRCPath, sizeof(szRCPath)))
			strDir = PathUtil::GetParentDirectory(szRCPath) + "\\";
	}
	__finddata64_t fd;
	int hSearch = _findfirst64 ((strDir +  "ResourceCompiler*.dll").c_str(), &fd);
	if (hSearch != -1)

	do {
		HMODULE hPlugin = LoadLibrary ((strDir+fd.name).c_str());
		if (!hPlugin)
		{
			RCLog("Error: Couldn't load plug-in module %s", fd.name);
			continue;
		}
		
		FnRegisterConvertors fnRegister = hPlugin?(FnRegisterConvertors)GetProcAddress(hPlugin, "RegisterConvertors"):NULL;
		if (!fnRegister)
		{
			RCLog("Error: plug-in module %s doesn't have RegisterConvertors function", fd.name);
			continue;
		}

		time_t nTime = GetTimestampForLoadedLibrary (hPlugin);
		char* szTime = "unknown";
		if (nTime)
		{
			szTime = asctime(localtime(&nTime));
			szTime[strlen(szTime)-1] = '\0';
		}
//		Info ("timestamp %s", szTime);
		RCLog("");
		RCLog("  Loading \"%s\"", fd.name);

		fnRegister (rc);
	}
	while(_findnext64(hSearch, &fd) != -1);

	_findclose(hSearch);
	RCLog("");
}

void RegisterCompressor( ResourceCompiler *rc )
{
	string strDir;
	{
		char szRCPath[1000];
		if (GetModuleFileName (NULL, szRCPath, sizeof(szRCPath)))
			strDir = PathUtil::GetParentDirectory(szRCPath) + "\\";
	}

	string compressorName = "CryCompressorRC.dll";

	HMODULE hPlugin = LoadLibrary (PathUtil::Make(strDir, compressorName).c_str());
	if (!hPlugin)
	{
		RCLogError("Couldn't load compressor plug-in module %s", compressorName.c_str());
		return;
	}

	FnCreateComressor fnRegister = hPlugin?(FnCreateComressor)GetProcAddress(hPlugin, "RegisterCompressor"):NULL;
	if (!fnRegister)
	{
		RCLogError("Compressor module %s doesn't have RegisterCompressor function", compressorName.c_str());
		return;
	}

	RCLog("CryCompressorRC dll loaded");

	rc->m_pCompressorRoutine = fnRegister (rc);
}

void ResourceCompiler::CreateSourceControl()
{
	assert(!m_bSourceControlCreated);

	m_bSourceControlCreated = true;
	m_pSourceControl = 0;

	string strDir;
	{
		char szRCPath[1000];
		if (GetModuleFileName (NULL, szRCPath, sizeof(szRCPath)))
		{
			strDir = PathUtil::GetParentDirectory(szRCPath) + "\\";
		}
	}

	const string dllName = "CryPerforce.dll";

	HMODULE hPlugin = LoadLibrary(PathUtil::Make(strDir, dllName).c_str());
	if (!hPlugin)
	{
		RCLogError("Couldn't load source control plug-in module %s", dllName.c_str());
		return;
	}

	m_fnCreateSourceControl =
		(hPlugin)
		? (FnCreateSourceControl)GetProcAddress(hPlugin, "CreateSourceControl")
		: NULL;

	m_fnDestroySourceControl =
		(hPlugin)
		? (FnDestroySourceControl)GetProcAddress(hPlugin, "DestroySourceControl")
		: NULL;

	if(!m_fnCreateSourceControl)
	{
		RCLogError("Source control module %s doesn't have CreateSourceControl function", dllName.c_str());
		return;
	}

	if(!m_fnDestroySourceControl)
	{
		RCLogError("Source control module %s doesn't have DestroySourceControl function", dllName.c_str());
		return;
	}

	RCLog("%s loaded", dllName.c_str());

	m_pSourceControl = m_fnCreateSourceControl();

	if(m_pSourceControl == 0)
	{
		RCLogWarning("Source control module %s failed to start (cannot connect to the server?)", dllName.c_str());
	}
}


void ResourceCompiler::LogFailedFileInfo( IConfig* config,int index, const char* fileName, bool bLogSourceControlInfo, const char* sourceControlClientName)
{
	assert(fileName);
	assert(fileName[0]);

	if(m_bSourceControlCreated && m_pSourceControl)
	{
		RCLog("  ConvertError #%i", index);

		RCLog("    FileName:       %s", fileName);

		CSourceControlFileInfo fi;
		CSourceControlUserInfo ui;

		if(!m_pSourceControl->GetFileHeadRevisionInfo(fileName, sourceControlClientName, fi, ui))
		{
			RCLog("    SourceControl:  Failed to obtain file and/or user info!");
		}
		else
		{
			char strTime[100];
			m_pSourceControl->ConvertTimeToText(fi.m_time, strTime, sizeof(strTime));

			char strChangelist[40];
			char strRevision[40];
			sprintf_s(strChangelist, "%d", fi.m_change);
			sprintf_s(strRevision, "%d", fi.m_revision);

			RCLog("    DepotFileName:      %s", fi.m_depotFileName);
			RCLog("    ClientFileName:     %s", fi.m_clientFileName);
			RCLog("    Change Description: %s", fi.m_changeDescription);
			RCLog("    Changelist:         %s", strChangelist);
			RCLog("    Revision:           %s", strRevision);
			RCLog("    CheckInTime:        %s", strTime);
			RCLog("    User:               %s", ui.m_userName);
			RCLog("    Client:             %s", ui.m_clientName);
			RCLog("    UserFullName:       %s", ui.m_userFullName);
			RCLog("    UserMail:           %s", ui.m_email);
			RCLog("    Error:              %s", "UNKNOWN");	// FIXME: use real error message

			const bool bMailErrors = config->GetAs<bool>("MailErrors", false);
			if (bMailErrors)
			{
				string mailBody;

#if defined(_MSC_VER)
				{
					char compName[256];
					DWORD size = ARRAYSIZE(compName);

					typedef BOOL (WINAPI *FP_GetComputerNameExA)(COMPUTER_NAME_FORMAT, LPSTR, LPDWORD);
					FP_GetComputerNameExA pGetComputerNameExA = (FP_GetComputerNameExA) GetProcAddress(LoadLibrary("kernel32.dll"), "GetComputerNameExA");

					if (pGetComputerNameExA)
						pGetComputerNameExA(ComputerNamePhysicalDnsHostname, compName, &size);
					else
						GetComputerName(compName, &size);

					mailBody += string("Report sent from ") + compName + "...\n\n";
				}
#endif

				mailBody += string("FileName: ") + fileName + "\r\n";

				mailBody += string("DepotFileName:      ") + fi.m_depotFileName+ "\r\n";
				mailBody += string("ClientFileName:     ") + fi.m_clientFileName + "\r\n";
				mailBody += string("Change Description: ") + fi.m_changeDescription + "\r\n";
				mailBody += string("Changelist:         ") + strChangelist + "\r\n";
				mailBody += string("Revision:           ") + strRevision + "\r\n";
				mailBody += string("CheckInTime:        ") + strTime+ "\r\n";
				mailBody += string("User:               ") + ui.m_userName+ "\r\n";
				mailBody += string("Client:             ") + ui.m_clientName+ "\r\n";
				mailBody += string("UserFullName:       ") + ui.m_userFullName+ "\r\n";
				mailBody += string("UserMail:           ") + ui.m_email+ "\r\n";
				mailBody += string("Compile Errors:     ") + "\r\n";

				// Add log output from this file.
				LogToFileMap::const_iterator it = m_LogForFileMap.find( fileName );
				if (it != m_LogForFileMap.end())
				{
					const std::vector<string> &logs = it->second;
					for (int i = 0; i < (int)logs.size(); i++)
					{
						mailBody += string("                ") + logs[i] + "\n\n";
					}
				}

				//Attachment.resize(0);

				CSMTPMailer::tstrcol Rcpt;
				CSMTPMailer::tstrcol cc;
				CSMTPMailer::tstrcol bcc;
				CSMTPMailer::tstrcol attachments;

				Rcpt.push_back( ui.m_email );

				string cc_email;
				if (config->Get("cc_email", cc_email))
				{
					int curPos= 0;
					string resToken= cc_email.Tokenize(";,",curPos);
					while (!resToken.empty())
					{
						cc.push_back( resToken );
						resToken= cc_email.Tokenize(";,",curPos);
					}
				}

				string smtp_server = "mail.intern.crytek.de";
				if (config->Get("MailServer", smtp_server))
				{
					CSMTPMailer mail("", "", smtp_server);
					string mailSubject = string("[RC Validator] Asset Compile Failed") + ": " + fi.m_depotFileName;
					bool res = mail.Send("RC-noreply@crytek.de", Rcpt, cc, bcc, mailSubject, mailBody, attachments);
					if (res)
					{
						RCLog("  Mail to %s sent", ui.m_email);
					}
					else
					{
						RCLogWarning("  Failed to send Mail to %s", ui.m_email);
					}
				}
			}
		}

		RCLog("  EndConvertError #%i", index);
	}
	else
	{
		RCLog("  %s", fileName);
	}
}


static string GetTimeAsString(const time_t tm)
{
	char buffer[40] = { 0 };
	ctime_s(buffer, sizeof(buffer), &tm);

	string str = buffer;
	while (StringHelpers::EndsWith(str, "\n"))
	{
		str = str.substr(0, str.length() - 1);
	}

	return str;
}


static CrashHandler s_crashHandler;

//////////////////////////////////////////////////////////////////////////
int __cdecl main(int argc, char **argv, char **envp)
{
	/*
	int tmpDbgFlag;
	tmpDbgFlag = _CrtSetDbgFlag(_CRTDBG_REPORT_FLAG);
	// Clear the upper 16 bits and OR in the desired freqency
	tmpDbgFlag = (tmpDbgFlag & 0x0000FFFF) | (32768 << 16);
	tmpDbgFlag |= _CRTDBG_LEAK_CHECK_DF;
	_CrtSetDbgFlag(tmpDbgFlag);

	// Check heap every 
	//_CrtSetBreakAlloc(2031);
	*/

//	EnableFloatingPointExceptions();

	ResourceCompiler rc;
	Config mainConfig;
	mainConfig.SetConfigKeyRegistry(&rc);
	CmdLine cmdLine;

	rc.QueryVersionInfo();
	SFileVersion fv = rc.GetFileVersion();

	rc.RegisterKey("wait","wait for key after running the application");
	rc.RegisterKey("WX","pause and display message box in case of warning or error");
	rc.RegisterKey("recursive","traverse input directory with sub folders");
	rc.RegisterKey("refresh","force recompilation of resources with up to date timestamp");
	rc.RegisterKey("p","to specify platform (PC,XBOX,PS2,GC,X360,PS3)");
	rc.RegisterKey("statistics","log statistics to rc_stats_* files");
	rc.RegisterKey("clean_targetroot","When targetroot switch specified will clean up this folder after rc runs, to delete all files that where not processed");
	rc.RegisterKey("verbose","to produce detailed printouts");
	rc.RegisterKey("quiet","to suppress all printouts");
	rc.RegisterKey("logfiles","to suppress generating log file rc_log.log");
	rc.RegisterKey("logprefix","prepends this prefix to every log file name used");
	rc.RegisterKey("presetcfg","to define the path to the presets e.g. \"rc_presets_pc.ini\"");
	rc.RegisterKey("sourceroot","to define the source folder");
	rc.RegisterKey("targetroot","to define the destination folder");
	rc.RegisterKey("threads","=N to use N threads,\n"
		"=cores number of threads is same as number of cores,\n"
		"=processors number of threads is same as number of logical processors");
	rc.RegisterKey("failonwarnings","return error code if warnings are encountered");
	rc.RegisterKey("sourcecontrol","output source control information for failed files");
	rc.RegisterKey("help","lists all usable keys of the ResourceCompiler with description");
	rc.RegisterKey("overwriteextension","overwrite the file extension, forces specific convertor type");
	
	rc.RegisterKey("listfile","Specify List file, List file can contain file lists from zip files like: @Levels\\Test\\level.pak|resourcelist.txt");
	rc.RegisterKey("listformat","Specify format of the file name read from the list file. You may use special strings:\n"
		"{0} the file name from the file list,\n"
		"{1} text matching first wildcard from the input file mask,"
		"{2} text matching second wildcard from the input file mask,"\
		"and so on.");
	rc.RegisterKey("copyonly","copy source files to target root without processing");
	rc.RegisterKey("name_as_crc32","When creating Pak File outputs target filename as the CRC32 code without the extension");
	rc.RegisterKey("zip","Compress source files into the zip file specified with this parameter");
	rc.RegisterKey("FolderInZip","Put source files into this specified folder inside of zip file (see 'zip' command");

	rc.RegisterKey("validate","When specified RC is running in a resource validation mode");
	rc.RegisterKey("MailServer","SMTP Mail server used when RC needs to send an e-mail");
	rc.RegisterKey("MailErrors","0=off 1=on When enabled sends an email to the user who checked in asset that failed validation");
	rc.RegisterKey("cc_email","When sending mail this address will be added to CC, semicolon separates multiple addresses");
	rc.RegisterKey("job","Process a job xml file");

	char moduleName[_MAX_PATH];
	GetModuleFileName( NULL, moduleName, _MAX_PATH );//retrieves the PATH for the current module

	// Load main config.
	CfgFile cfgFile;
	if (!cfgFile.Load( string(PathUtil::Make(PathUtil::GetPath(moduleName),RC_INI_FILE))) )
	{
		char str[512];

		sprintf(str,"Resource compiler ini file ('%s') is missing - check current working folder",RC_INI_FILE);

		RCLog("%s",str);

		MessageBox(0,str,"ResourceCompiler Error",MB_OK|MB_ICONERROR);
		return 1;
	}

	cfgFile.SetConfig( eCP_PriorityPlatform,COMMON_SECTION,&mainConfig );

	// Parse command line.
	cmdLine.Parse( argc,argv,&mainConfig );

	bool bWork = true;

	string platformStr;
	if (!((IConfig &)mainConfig).Get( "p", platformStr ))
	{
		// Platform switch not specified.
		RCLog("Platform (/p) not specified, defaulting to PC.");
		RCLog("");
		platformStr = "PC";
		mainConfig.Set(eCP_PriorityCmdline,"p",platformStr.c_str());
	}
	
	// Detect platform.
	EPlatform platform = GetPlatformFromName(platformStr.c_str());
	if (platform == ePlatform_UNKNOWN)
	{
		char str[512];

		sprintf(str,"Unknown platform %s specified",platformStr.c_str());

		RCLog("%s",str);

		MessageBox(0,str,"ResourceCompiler Error",MB_OK|MB_ICONERROR);
		return 1;
	}

	rc.GetHWnd();

	// Load configuration from per platform section. 
//	RCLog("Using platform settings '%s'",rc.GetSectionName(platform));
	cfgFile.SetConfig( eCP_PriorityPlatform, rc.GetSectionName(platform),&mainConfig );

	// initialize rc (also intializes logs)
	rc.Init(&mainConfig);

	s_crashHandler.SetFiles(
		rc.GetMainLogFileName(),
		rc.GetErrorLogFileName(),
		rc.FormLogFileName(RC_FILENAME_CRASH_DUMP));

	RCLog("ResourceCompiler Version %d.%d.%d %s %s",fv.v[2],fv.v[1],fv.v[0], __DATE__, __TIME__);
	RCLog("Copyright (c) 2001-2010 Crytek GmbH. All rights reserved.");
#ifdef _WIN64
	RCLog("64-bit edition");
#endif
	RCLog("");
	RCLog("Started at: %s", GetTimeAsString(rc.GetStartTime()).c_str());

	RCLog("");

	if (argc > 0)
	{
		RCLog("Command line:");
		for (int i = 0; i < argc; ++i)
		{
			RCLog("  \"%s\"", argv[i]);
		}
		RCLog("");
	}

	{ 
		char buff[MAX_PATH];
		const DWORD dwRet = GetCurrentDirectory(sizeof(buff), buff);
		if ((dwRet > 0) || (dwRet < sizeof(buff)))
		{
			RCLog("Current directory:");
			RCLog("  \"%s\"", buff);
			RCLog("");
		}
	}

	rc.SetTimeLogging(true);

	RCLog("Registering sub compilers (ResourceCompiler*.dll)");

	RegisterConvertors( &rc );

	RegisterCompressor( &rc );

	if (!mainConfig.CheckForUnknownKeys())
	{
		RCLogWarning("Unknown command-line option(s) (see above). Use \"RC /help\".");
		if (mainConfig.GetAs<bool>("failonwarnings", false))
		{
			return 1;
		}
	}

	bool bShowUsage = false;
	if (mainConfig.HasKey("job"))
	{
		rc.ProcessJobFile(platform, &mainConfig);
		rc.PostBuild(&mainConfig);      // e.g. writing statistics files
	}
	else if (!cmdLine.m_fileSpec.empty())
	{
		rc.Compile(platform, &mainConfig, cmdLine.m_fileSpec.c_str());
		rc.PostBuild(&mainConfig);      // e.g. writing statistics files
	}
	else
	{
		 bShowUsage = true;
	}

	rc.DeInit();		// to clean up before waiting for user

	rc.SetTimeLogging(false);

	if (bShowUsage && !rc.m_bQuiet)
	{
		rc.show_help(false);
	}

	if(mainConfig.HasKey("help"))
	{
		rc.show_help(true);
	}

	RCLog("");
	RCLog("Finished at: %s", GetTimeAsString(time(0)).c_str());

	if (rc.GetNumErrors() || rc.GetNumWarnings())
	{
		RCLog("");
		RCLog("%d errors, %d warnings.", rc.GetNumErrors(), rc.GetNumWarnings());
	}

	if (mainConfig.HasKey("wait"))
	{
		RCLog("");     
		RCLog("    Press <RETURN> (/wait was specified)");
		getchar();
	}

	const bool bFail = rc.GetNumErrors() || (rc.GetNumWarnings() && mainConfig.GetAs<bool>("failonwarnings", false));
	return (bFail ? 1 : 0);
}

//! Returns the main application window
HWND ResourceCompiler::GetHWnd()
{
	HMODULE hKernel32 = LoadLibrary ("kernel32.dll");
	HWND hResult = GetDesktopWindow();
	if (hKernel32)
	{
		//typedef WINBASEAPI  HWND APIENTRY (*FnGetConsoleWindow )(VOID);
		typedef HWND (APIENTRY*FnGetConsoleWindow )(VOID);
		FnGetConsoleWindow GetConsoleWindow = (FnGetConsoleWindow)GetProcAddress (hKernel32, "GetConsoleWindow");
		if (GetConsoleWindow)
		{
			hResult = GetConsoleWindow();
		}
		FreeLibrary (hKernel32);
	}
	return hResult;
}

HWND ResourceCompiler::GetEmptyWindow()
{
	if (!m_hEmptyWindow)
	{
		const char szClassName[] = "DirectXWnd";
		WNDCLASS wc;
		memset (&wc, 0, sizeof(wc));
		wc.style = CS_OWNDC;
		wc.lpfnWndProc = DefWindowProc;
		wc.cbClsExtra = 0;
		wc.cbWndExtra = 0;
		wc.hInstance  = (HINSTANCE)GetModuleHandle (NULL);
		wc.lpszClassName = szClassName;

		ATOM atomWndClass = RegisterClass (&wc);
		m_hEmptyWindow = CreateWindow (szClassName, "DirectXEmpty", WS_OVERLAPPEDWINDOW, CW_USEDEFAULT,CW_USEDEFAULT,256,256,NULL, NULL, wc.hInstance, 0);
	}
	return m_hEmptyWindow;
}

// ------------------------------------------------------
void ResourceCompiler::AddDependencyMaterial( const char *inszSrcFilename, const char *inszMatName, const char *inszScriptName )
{
	if (!m_bStatistics)
		return;
	CMatDep dep;

	dep.m_sMatName=inszMatName;
	dep.m_sScriptName=inszScriptName;

	m_MaterialDependencies.insert( CMatDepPair(dep,inszSrcFilename) );

//	RCLog("  DepMat: <%s> <%s>",inszMatName,inszScriptName);
}

// ------------------------------------------------------
void ResourceCompiler::AddDependencyFile( const char *inszSrcFilename, const char *inszPathFileName )
{
	if (!m_bStatistics)
		return;
	m_FileDependencies.insert( CFileDepPair(inszPathFileName,inszSrcFilename) );

//	RCLog("  DepFile: <%s>",inszPathFileName);
}

// ------------------------------------------------------
void ResourceCompiler::ShowFileDependencies()
{
	const string filename = FormLogFileName(RC_FILENAME_FILEDEP);

	FILE *out=fopen(filename.c_str(),"wb");

	if(!out)
	{
		LogError("unable to open %s - file it not updated",filename.c_str());
		return;
	}

	CFileDepMap::iterator it;

	string sLastDep="";		// for a nice printout
	bool bFirst=true;

	for(it=m_FileDependencies.begin(); it!=m_FileDependencies.end(); ++it)
	{
		const string &rsDepFile = it->first;
		const string &rsSrcFile = it->second;

		if(bFirst || rsDepFile!=sLastDep)
		{
			fprintf(out,"\r\n");
			fprintf(out,"'%s'\r\n",rsDepFile.c_str());
			sLastDep=rsDepFile;
			bFirst=false;
		}

		fprintf(out,"    used by: '%s'\r\n",rsSrcFile.c_str());
	}

	fprintf(out,"\r\n");
	fprintf(out,"------------------------------------------------------------------------------\r\n");
	fprintf(out,"\r\n");
	fprintf(out,"all used files:\r\n");
	fprintf(out,"\r\n");

	bFirst=true;
	for(it=m_FileDependencies.begin(); it!=m_FileDependencies.end(); ++it)
	{
		const string &rsDepFile = it->first;
		const string &rsSrcFile = it->second;

		if(bFirst || rsDepFile!=sLastDep)
		{
			fprintf(out,"  '%s'\r\n",rsDepFile.c_str());
			sLastDep=rsDepFile;
			bFirst=false;
		}
	}

	fclose(out);
}


// ------------------------------------------------------
void ResourceCompiler::ShowPresetUsage()
{
	const string filename = FormLogFileName(RC_FILENAME_PRESETUSAGE);

	FILE *out=fopen(filename.c_str(),"wb");

	if(!out)
	{
		LogError("unable to open %s - file it not updated",filename.c_str());
		return;
	}

	std::vector<std::pair<string, int> > presetSortedMappingTable(m_StatsFiles.size());
	for (int fileIndex = 0, fileCount = int(m_StatsFiles.size()); fileIndex < fileCount; ++fileIndex)
		presetSortedMappingTable[fileIndex] = std::make_pair(m_StatsFiles[fileIndex]->m_sPreset, fileIndex);
	std::sort(presetSortedMappingTable.begin(), presetSortedMappingTable.end());

	string sLastDep="";		// for a nice printout
	bool bFirst=true;

	fprintf(out,"preset usage:\r\n");
	fprintf(out,"\r\n");

	//for(it=m_PresetUsage.begin(); it!=m_PresetUsage.end(); ++it)
	for (int mappingTableIndex = 0, mappingTableSize = int(presetSortedMappingTable.size()); mappingTableIndex < mappingTableSize; ++mappingTableIndex)
	{
		const uint32 dwFileStatsIndex = presetSortedMappingTable[mappingTableIndex].second;

		CFileStats &stats = *m_StatsFiles[dwFileStatsIndex];

		if(bFirst || stats.m_sPreset!=sLastDep)
		{
			fprintf(out,"\r\n  PRESET %s:\r\n",stats.m_sPreset );
			sLastDep=stats.m_sPreset;
			bFirst=false;
		}

		const __int64 ratio = 
			((stats.m_SrcFileSize > 0) && (stats.m_DstFileSize >= 0)) 
			? (stats.m_DstFileSize * 100) / stats.m_SrcFileSize
			: 0;
		fprintf(out,"    %3I64d%% '%s' %I64d -> %I64d \r\n",
			ratio,stats.m_sSourceFilename,stats.m_SrcFileSize,stats.m_DstFileSize);
	}

	fclose(out);
}

//////////////////////////////////////////////////////////////////////////
void ResourceCompiler::CreateExcelReport( IConfig *config )
{
	const string filename = FormLogFileName(RC_FILENAME_XLS_REPORT);

	RCLog( "Saving Excel Report to %s",filename.c_str() );

	// Read Source control info for failed files.

	string sourceControlClientName;
	const bool bLogSourceControlInfo = config->Get("sourcecontrol",sourceControlClientName);
	if(bLogSourceControlInfo)
	{
		// Lazy registering of source control
		if(!m_bSourceControlCreated)
		{
			CreateSourceControl();
		}
		if (m_bSourceControlCreated && m_pSourceControl)
		{
			RCLog( "Retrieving Source Control information on %d files",m_StatsFiles.size() );
			for (size_t i = 0; i < m_StatsFiles.size(); i++)
			{
				CFileStats &fs = *m_StatsFiles[i];

				// Only query Source control info for failed files and if m_bGetSourceControlInfo is set.
				if (!fs.m_bGetSourceControlInfo)
				{
					continue;
				}

				char szFullPath[MAX_PATH];
				if (!_fullpath(szFullPath, fs.m_sSourceFilename, sizeof(szFullPath)))
				{
					strcpy_s(szFullPath, fs.m_sSourceFilename);
				}

				CSourceControlFileInfo fi;
				CSourceControlUserInfo ui;

				if (m_pSourceControl->GetFileHeadRevisionInfo(szFullPath, sourceControlClientName, fi, ui))
				{
					fs.m_sc.bValid = true;
					fs.SafeStrCopy( fs.m_sc.user,ui.m_userName );
					//fs.SafeStrCopy( fs.m_sc.user_fullname,ui.m_userFullName );
					//fs.SafeStrCopy( fs.m_sc.user_email,ui.m_email );
					fs.SafeStrCopy( fs.m_sc.workspace,ui.m_clientName );
					fs.SafeStrCopy( fs.m_sc.depotFile,fi.m_depotFileName );					
					fs.SafeStrCopy( fs.m_sc.changeDescription,fi.m_changeDescription);
					fs.m_sc.time = fi.m_time;
					fs.m_sc.change = fi.m_change;
					fs.m_sc.revision = fi.m_revision;
				}
			}
			RCLog( "Source Control information retrieved successfully" );
		}
	}

	//////////////////////////////////////////////////////////////////////////
	for (size_t i = 0; i < m_StatsFiles.size(); i++)
	{
		CFileStats &fs = *m_StatsFiles[i];
		if (!fs.m_bSuccess)
		{
			// Add log output from this file.
			LogToFileMap::const_iterator it = m_LogForFileMap.find( fs.m_sSourceFilename.c_str() );
			if (it != m_LogForFileMap.end())
			{
				string sErrors;
				const std::vector<string> &logs = it->second;
				for (size_t log = 0; log < logs.size(); log++)
				{
					sErrors += logs[log] + "\n";
				}
				fs.SafeStrCopy(fs.m_sErrorLog,sErrors);
			}
		}
	}

	CExcelReport report;
	report.Export( this,filename.c_str(),m_StatsFiles );

	RCLog( "Excel Report saved successfully." );
}


// ------------------------------------------------------
void ResourceCompiler::ShowXLSFileSizes()
{
	const string filename = FormLogFileName(RC_FILENAME_XLS_FILESIZES);

	FILE *out=fopen(filename.c_str(),"wb");

	if(!out)
	{
		LogError("unable to open %s - file it not updated",filename.c_str());
		return;
	}

	fprintf(out,
		"DestFileSizeInKB"
		"\tDestFileName"
		"\tPreset"
		"\tWidth"
		"\tHeight"
		"\tAlpha"
		"\tMips"
		"\tMemInKB(in Memory)"
		"\tFormat"
		"\tReduce"
		"\tAttachedMemInKB(e.g. Alpha of 3dC)"
		"\tDestFilePath"
		"\r\n");
	fprintf(out,"----------------------------------------------------------------------------------------------<in Excel friendly format>\r\n");

	// Produce a mapping table into the file stats array that is sorted by file size.
	std::vector<std::pair<__int64, uint32> > fileSizeSortedMappingTable(m_StatsFiles.size());
	for (int fileIndex = 0, fileCount = int(m_StatsFiles.size()); fileIndex < fileCount; ++fileIndex)
		fileSizeSortedMappingTable[fileIndex] = std::make_pair(m_StatsFiles[fileIndex]->m_DstFileSize, fileIndex);
	std::sort(fileSizeSortedMappingTable.begin(), fileSizeSortedMappingTable.end());

	for (int mappingTableIndex = 0, mappingTableSize = fileSizeSortedMappingTable.size(); mappingTableIndex < mappingTableSize; ++mappingTableIndex)
	{
		const uint32 dwFileStatsIndex = fileSizeSortedMappingTable[mappingTableIndex].second;

		CFileStats &stats = *m_StatsFiles[dwFileStatsIndex];

		fprintf(out,
			"%I64d"       // DestFileSize
			"\t%s"        // DestFileName
			"\t%s"        // preset
			"\t%s"        // info
			"\t%s\r\n",   // DestPath
			stats.m_DstFileSize,
			PathHelpers::GetFilename(stats.m_sDestFilename.c_str()).c_str(),
			stats.m_sPreset,
			stats.m_sInfo,
			PathHelpers::GetDirectory(stats.m_sDestFilename.c_str()).c_str());
	}

	fclose(out);
}

/*
// ------------------------------------------------------
void ResourceCompiler::ShowMaterialDependencies()
{
	const char *szFileName=RC_FILENAME_MATDEP;

	FILE *out=fopen(m_exePath+szFileName,"wb");

	if(!out)
	{
		LogError("unable to open %s - file it not updated",szFileName);
		return;
	}

	CMatDepMap::iterator it;

	CMatDep LastDep;		// for a nice printout
	bool bFirst=true;

	// max info
	for(it=m_MaterialDependencies.begin(); it!=m_MaterialDependencies.end(); ++it)
	{
		const CMatDep &rsMatDep = it->first;
		const string &rsSrcFile = it->second;

		if(bFirst || !(rsMatDep==LastDep))
		{
			fprintf(out,"\r\n");
			fprintf(out,"scriptmaterial='%s' materialname='%s'\r\n",rsMatDep.m_sScriptName.c_str(),rsMatDep.m_sMatName.c_str());
			LastDep=rsMatDep; 
			bFirst=false;
		}

		fprintf(out,"    used by: '%s'\r\n",rsSrcFile.c_str());
	}

	fprintf(out,"\r\n");
	fprintf(out,"------------------------------------------------------------------------------\r\n");
	fprintf(out,"\r\n");
	fprintf(out,"all used scriptmaterials:\r\n");
	fprintf(out,"\r\n");

	// only the used scripsmaterials
	bFirst=true;
	for(it=m_MaterialDependencies.begin(); it!=m_MaterialDependencies.end(); ++it)
	{
		const CMatDep &rsMatDep = it->first;
		const string &rsSrcFile = it->second;

		if(bFirst || !(rsMatDep.m_sScriptName==LastDep.m_sScriptName))
		{
			fprintf(out,"  '%s'\r\n",rsMatDep.m_sScriptName.c_str());
			LastDep=rsMatDep; 
			bFirst=false;
		}
	}

	fclose(out);
}
*/

// ------------------------------------------------------
void ResourceCompiler::PostBuild( IConfig *pConfig )
{
	if (m_bStatistics)
	{
		RCLog("writing statistics files (rc_stats_...) ");
		RCLog("");

		ShowFileDependencies();
//		ShowMaterialDependencies();
		ShowPresetUsage();
		ShowXLSFileSizes();

		CreateExcelReport(pConfig);
	}
}

//////////////////////////////////////////////////////////////////////////
void ResourceCompiler::QueryVersionInfo()
{
	char moduleName[_MAX_PATH];
	DWORD dwHandle;
	UINT len;

	char ver[1024*8];

	GetModuleFileName( NULL, moduleName, _MAX_PATH );//retrieves the PATH for the current module
	m_exePath = PathHelpers::AddSeparator( PathHelpers::GetDirectory(moduleName) );

	int verSize = GetFileVersionInfoSize( moduleName,&dwHandle );
	if (verSize > 0)
	{
		GetFileVersionInfo( moduleName,dwHandle,1024*8,ver );
		VS_FIXEDFILEINFO *vinfo;
		VerQueryValue( ver,"\\",(void**)&vinfo,&len );

		m_fileVersion.v[0] = vinfo->dwFileVersionLS & 0xFFFF;
		m_fileVersion.v[1] = vinfo->dwFileVersionLS >> 16;
		m_fileVersion.v[2] = vinfo->dwFileVersionMS & 0xFFFF;
		m_fileVersion.v[3] = vinfo->dwFileVersionMS >> 16;

		m_productVersion.v[0] = vinfo->dwProductVersionLS & 0xFFFF;
		m_productVersion.v[1] = vinfo->dwProductVersionLS >> 16;
		m_productVersion.v[2] = vinfo->dwProductVersionMS & 0xFFFF;
		m_productVersion.v[3] = vinfo->dwProductVersionMS >> 16;
	}
}

//////////////////////////////////////////////////////////////////////////
void ResourceCompiler::DeInit()
{
	m_extensionManager.UnregisterAll();
}

//////////////////////////////////////////////////////////////////////////
void ResourceCompiler::Init( IConfig* config )
{
	m_bCleanTargetFolder = config->HasKey("clean_targetroot");
	m_numFilesProcessed = 0;	// init progress

	m_bQuiet = config->HasKey("quiet");
	m_bWarningsAsErrors = config->HasKey("WX");

	m_startTime = time(0);
	m_bTimeLogging = false;

	InitLogs(config);
	SetRCLog(this);
}

//////////////////////////////////////////////////////////////////////////
static unsigned int CountBitsSetTo1(const DWORD_PTR val)
{
	unsigned int result = 0;
	const size_t bitCount = sizeof(val) * 8;
	for (size_t i = 0; i < bitCount; ++i)
	{
		result += (val & (((DWORD_PTR)1) << i)) ? 1 : 0;
	}
	return result;
}

//////////////////////////////////////////////////////////////////////////
void ResourceCompiler::SetupMaxThreads( IConfig* config )
{
	{
		unsigned int numCoresInSystem = 0;
		unsigned int numCoresAvailableToProcess = 0;
		GetNumCPUCores(numCoresInSystem, numCoresAvailableToProcess);
		if (numCoresAvailableToProcess <= 0)
		{
			numCoresAvailableToProcess = 1;
		}

		unsigned int numLogicalProcessorsAvailableToProcess = 0;
		{
			DWORD_PTR processAffinity = 0;
			DWORD_PTR systemAffinity = 0;
			GetProcessAffinityMask(GetCurrentProcess(), &processAffinity, &systemAffinity);

			numLogicalProcessorsAvailableToProcess = CountBitsSetTo1(processAffinity);

			if (numLogicalProcessorsAvailableToProcess < numCoresAvailableToProcess)
			{
				numLogicalProcessorsAvailableToProcess = numCoresAvailableToProcess;
			}
		}

		unsigned int numLogicalProcessorsInSystem = 0;
		{
			SYSTEM_INFO si;
			GetSystemInfo(&si);
			numLogicalProcessorsInSystem = si.dwNumberOfProcessors;
		}

		m_systemCpuCoreCount = numCoresInSystem;
		m_processCpuCoreCount = numCoresAvailableToProcess;
		m_systemLogicalProcessorCount = numLogicalProcessorsInSystem;
		m_processLogicalProcessorCount = numLogicalProcessorsAvailableToProcess;
	}

	RCLog("");
	RCLog("CPU cores: %d available (%d in system).",
		m_processCpuCoreCount, 
		m_systemCpuCoreCount);				
	RCLog("Logical processors: %d available (%d in system).",
		m_processLogicalProcessorCount, 
		m_systemLogicalProcessorCount);

	m_maxThreads = -1;

	const bool bValidateMode = config->GetAs<bool>("validate", false);
	string threadString;
	string message;

	if (!config->Get("threads", threadString) || bValidateMode)
	{
		m_maxThreads = 1;
		message.Format("/threads was not specified.");
	}
	else if (StringHelpers::Equals(threadString, "cores"))
	{
		m_maxThreads = m_processCpuCoreCount;
		message.Format("/threads specified with value '%s'.", threadString.c_str());
	}
	else if (StringHelpers::Equals(threadString, "processors"))
	{
		m_maxThreads = m_processLogicalProcessorCount;
		message.Format("/threads specified with value '%s'.", threadString.c_str());
	}
	else if (threadString.empty())
	{
		m_maxThreads = m_processCpuCoreCount;
		message.Format("/threads specified without value.");
	}
	else
	{
		const int val = atol(threadString.c_str());
		if (val <= 0)
		{
			m_maxThreads = 1;
			message.Format("/threads specified with bad value '%s'.", threadString.c_str());
		}
		else if (val > m_processLogicalProcessorCount)
		{
			m_maxThreads = m_processLogicalProcessorCount;
			message.Format("/threads specified with too big value '%s'.", threadString.c_str());
		}
		else
		{
			m_maxThreads = val;
			message.Format("/threads specified with value '%s'.", threadString.c_str());
		}
	}

	RCLog("%s Using %d thread%s.", message.c_str(), m_maxThreads, ((m_maxThreads > 1) ? "s" : ""));
	RCLog("");

	assert(m_maxThreads >= 1);
}

//////////////////////////////////////////////////////////////////////////
ICfgFile *ResourceCompiler::CreateCfgFile()
{
	return new CfgFile;
}


//////////////////////////////////////////////////////////////////////////
void ResourceCompiler::VerifyKeyRegistration( const char *szKey )
{
	assert(szKey);
	string sKey = szKey;
	sKey.MakeLower();

	const bool ok = (m_KeyHelp.count(sKey) != 0);

	if(!ok)
	{
		RCLogWarning("Key '%s' was not registered, call RegisterKey() before using the key",szKey);
	}
}

//////////////////////////////////////////////////////////////////////////
bool ResourceCompiler::VerifyKeyRegistration2( const char *szKey )
{
	assert(szKey);
	string sKey = szKey;
	sKey.MakeLower();

	const bool ok = (m_KeyHelp.count(sKey) != 0);

	if(!ok)
	{
		RCLogWarning("Key '%s' is unknown.",szKey);
		return false;
	}

	return true;
}

//////////////////////////////////////////////////////////////////////////
void ResourceCompiler::RegisterKey( const char *key, const char *helptxt )
{
	string sKey = key;

	sKey.MakeLower();

	assert(m_KeyHelp.count(sKey)==0);		// registered twice

	m_KeyHelp[sKey] = helptxt;
}

//////////////////////////////////////////////////////////////////////////
void ResourceCompiler::InitLogs(IConfig *config)
{
	m_logPrefix.clear();
	config->Get("logprefix", m_logPrefix);
	if (m_logPrefix.empty())
	{
		m_logPrefix = m_exePath;
	}

	FileUtil::CreateDirectoryRecursive(PathHelpers::GetDirectory(m_logPrefix + "unused.name"));

	m_mainLogFileName    = FormLogFileName(RC_FILENAME_LOG);
	m_warningLogFileName = FormLogFileName(RC_FILENAME_WARNINGS);
	m_errorLogFileName   = FormLogFileName(RC_FILENAME_ERRORS);

	DeleteFile(m_mainLogFileName);
	DeleteFile(m_warningLogFileName);
	DeleteFile(m_errorLogFileName);

	if (config->HasKey("logfiles"))
	{
		m_mainLogFileName = "";
	}
}

//////////////////////////////////////////////////////////////////////////
string ResourceCompiler::FormLogFileName(const char* suffix) const
{
	return (suffix && suffix[0]) ? m_logPrefix + suffix : string();
}

//////////////////////////////////////////////////////////////////////////
const string& ResourceCompiler::GetMainLogFileName() const
{
	return m_mainLogFileName;
}

//////////////////////////////////////////////////////////////////////////
const string& ResourceCompiler::GetErrorLogFileName() const
{
	return m_errorLogFileName;
}

//////////////////////////////////////////////////////////////////////////
time_t ResourceCompiler::GetStartTime() const
{
	return m_startTime;
}

bool ResourceCompiler::GetTimeLogging() const
{
	return m_bTimeLogging;
}
void ResourceCompiler::SetTimeLogging(bool enable)
{
	m_bTimeLogging = enable;
}

//////////////////////////////////////////////////////////////////////////
void ResourceCompiler::AddFileStats( CFileStats &fs )
{
	CFileStats *pNewFS = new CFileStats(fs);
	m_outputFilesLock.Lock();
	m_StatsFiles.push_back(pNewFS);
	m_outputFilesLock.Unlock();
}

//////////////////////////////////////////////////////////////////////////
int ResourceCompiler::GetMaxNumThreads() const
{
	return m_maxThreads;
}

void ResourceCompiler::Update()
{
}

void ResourceCompiler::LogV( const ELogType ineType, const char* szFormat, va_list args )
{
	if(ineType == eWarning)
	{
		++m_numWarnings;
	}
	else if(ineType == eError)
	{
		++m_numErrors;
	}

	char str[16*1024];

	_vsnprintf_s(str, sizeof(str), sizeof(str) - 1, szFormat, args);

	char* p = str;

	bool bRun=true;

	while(bRun)
	{
		char *start=p;

		// search for end marker
		while(*p!=0)
		{
			// remove nonprintable characters except newlines and tabs
			if( (*p<' ') && (*p!='\n') && (*p!='\t') ) 
				*p=' ';  

			p++;
		}

		if(*p==0)
			bRun=false;

		*p=0;

		LogLine(ineType,start);

		p++;	// jump over end marker
	}
}

//////////////////////////////////////////////////////////////////////////
void ResourceCompiler::StartLogRecording()
{
	if (GetMaxNumThreads() == 1)
	{
		m_bLogRecordingEnabled = true;
		m_RecordedLogLines.clear();
	}
}

//////////////////////////////////////////////////////////////////////////
void ResourceCompiler::StopLogRecording()
{
	m_bLogRecordingEnabled = false;
}

//////////////////////////////////////////////////////////////////////////
void ResourceCompiler::CopyFiles( IConfig *config,const std::vector<RcFile>& files)
{
	const bool bVerbose = config->HasKey("verbose");

	const size_t numFiles = files.size();
	size_t numFilesCopied = 0;
	size_t numFilesUpToDate = 0;
	size_t numFilesMissing = 0;
	size_t numFilesFailed = 0;

	string progressString;
	progressString.Format("Copying %d files", numFiles);

	RCLog("Starting copying %d files", numFiles);

	for (size_t i = 0; i < numFiles; ++i)
	{
		ShowProgress( progressString.c_str(),(i*100)/numFiles );

		const string srcFilename = PathHelpers::Join(files[i].m_sourceLeftPath, files[i].m_sourceInnerPathAndName);
		const string trgFilename = PathHelpers::Join(files[i].m_targetLeftPath, files[i].m_sourceInnerPathAndName);

		if (bVerbose)
		{
			RCLog("Copying %s to %s", srcFilename.c_str(), trgFilename.c_str());
		}

		const bool bSourceExists = FileUtil::FileExists(srcFilename.c_str());

		if (!bSourceExists)
		{
			++numFilesMissing;
			RCLog("Source file %s does not exist", srcFilename.c_str());
		}
		else
		{
			FileUtil::CreateDirectoryRecursive(PathHelpers::GetDirectory(trgFilename));

			//////////////////////////////////////////////////////////////////////////
			// Compare source and target files modify timestamps.
			FILETIME ftSource = FileUtil::GetLastWriteFileTime(srcFilename);
			FILETIME ftTarget = FileUtil::GetLastWriteFileTime(trgFilename);

			if (ftSource.dwHighDateTime == ftTarget.dwHighDateTime &&
				ftSource.dwLowDateTime == ftTarget.dwLowDateTime)
			{
				// Up to date file already exists in target folder
				++numFilesUpToDate;
				AddOutputFile( trgFilename,srcFilename );
				continue;
			}
			//////////////////////////////////////////////////////////////////////////

			SetFileAttributes(trgFilename,FILE_ATTRIBUTE_ARCHIVE);
			const bool bCopied = (::CopyFile( srcFilename,trgFilename,FALSE ) != 0);

			if (bCopied)
			{
				++numFilesCopied;
				SetFileAttributes(trgFilename,FILE_ATTRIBUTE_ARCHIVE);
				FileUtil::SetFileTimes(trgFilename,ftSource);
			}
			else
			{
				++numFilesFailed;
				RCLog("Failed to copy %s to %s", srcFilename.c_str(), trgFilename.c_str());
			}
		}

		AddOutputFile( trgFilename,srcFilename );
	}

	RCLog("Finished copying %d files: %d copied, %d up-to-date, %d missing, %d failed",
		numFiles, numFilesCopied, numFilesUpToDate, numFilesMissing, numFilesFailed);
}

//////////////////////////////////////////////////////////////////////////
bool ResourceCompiler::CreatePakFile(
	IConfig *config,
	const std::vector<RcFile> &sourceFiles,
	const string &folderInPak,
	const string &pakFilename,
	bool bUpdate)
{
	RCLog("Packing %u files to zip file %s", sourceFiles.size(), pakFilename.c_str());

	std::vector<RcFile> files = sourceFiles;

	// Sort all files alphabetically
	{
		struct files_compare : public std::binary_function<RcFile, RcFile, bool> 
		{
			bool operator()(const RcFile& left, const RcFile& right) const
			{
				return stricmp(left.m_sourceInnerPathAndName.c_str(), right.m_sourceInnerPathAndName.c_str()) < 0;
			}
		};
		std::sort(files.begin(), files.end(), files_compare());
	}

	bool bResult = true;

	Crc32Gen crc32generator;
	std::set<unsigned int> crc32set;
	bool name_as_crc32 = false;
	config->Get("name_as_crc32",name_as_crc32);

	if (!bUpdate)
	{
		// Delete old pak file.
		::SetFileAttributes( pakFilename.c_str(),FILE_ATTRIBUTE_ARCHIVE );
		::DeleteFile( pakFilename.c_str() );
	}

	FileUtil::CreateDirectoryRecursive(PathHelpers::GetDirectory(pakFilename));

	std::vector<char> buffer;

	// Add them to pak file.
	PakSystemArchive *pPakFile = GetPakSystem()->OpenArchive( pakFilename.c_str() );
	if (!pPakFile)
	{
		RCLogError( "Failed to create zip file %s",pakFilename.c_str() );
		return false;
	}

	const size_t numFiles = files.size();
	size_t numFilesAdded = 0;
	size_t numFilesUpToDate = 0;
	size_t numFilesSkipped = 0;
	size_t numFilesMissing = 0;
	size_t numFilesFailed = 0;

	const string sProgressOp = "Adding files to Pak file " + pakFilename;

	// Add files to Pak
	for (size_t i = 0; i < numFiles; ++i)
	{
		// Show progress.
		ShowProgress( sProgressOp.c_str(),(i*100)/numFiles );

		string sFileNameInZip = PathHelpers::ToDosPath(PathHelpers::Join(folderInPak, files[i].m_sourceInnerPathAndName));
		const string sRealFilename = PathHelpers::Join(files[i].m_sourceLeftPath, files[i].m_sourceInnerPathAndName);
		
		// Skip files with extensions starting from $ sign or .pak.
		{
			const string ext = PathHelpers::FindExtension(sRealFilename);
			if (!ext.empty() && (ext[0] == '$' || stricmp(ext,"pak")==0))
			{
				++numFilesSkipped;
				continue;
			}
		}

		const FILETIME ft = FileUtil::GetLastWriteFileTime(sRealFilename);
		LARGE_INTEGER lt;
		lt.HighPart = ft.dwHighDateTime;
		lt.LowPart = ft.dwLowDateTime;
		const __int64 modTime = lt.QuadPart;;

		if (name_as_crc32)
		{
			const unsigned int crc32 = crc32generator.GetCRC32Lowercase(sFileNameInZip.c_str());
			if (crc32set.find(crc32) != crc32set.end())
			{
				RCLogError( "Duplicate CRC32 code %X for file %s when creating Pak File: %s",crc32,sFileNameInZip.c_str(),pakFilename.c_str() );
				++numFilesFailed;
				bResult = false;
				break;
			}
			crc32set.insert(crc32);
			sFileNameInZip.Format( "%X",crc32 );
		}

		// Check if file with same name and same timestamp already exist in pak.
		if (GetPakSystem()->CheckIfFileExist( pPakFile,sFileNameInZip.c_str(),modTime ))
		{
			// File with same timestamp is already inside pak file.
			++numFilesUpToDate;
			continue;
		}

		FILE *f = fopen(sRealFilename.c_str(),"rb");
		if (f)
		{
			fseek(f,0,SEEK_END);
			const int nFileSize = ftell(f);
			if (nFileSize > 0)
			{
				buffer.resize(nFileSize);
				fseek(f,0,SEEK_SET);
				if (fread( &buffer[0],1,nFileSize,f ) != nFileSize)
				{
					fclose(f);
					RCLog( "Zip [%s]: error reading source file %s",pakFilename.c_str(),sRealFilename.c_str() );			
					++numFilesFailed;
					bResult = false;
					break;
				}
				GetPakSystem()->AddToArchive( pPakFile,sFileNameInZip.c_str(),&buffer[0],nFileSize,modTime );
				RCLog( "Add file to zip: [%s] %s",pakFilename.c_str(),sFileNameInZip.c_str() );
				++numFilesAdded;
			}
			else
			{
				++numFilesSkipped;
			}
			fclose(f);
		}
		else
		{
			RCLog( "Zip [%s]: source file %s is not found",pakFilename.c_str(),sRealFilename.c_str() );			
			++numFilesMissing;
		}
	}
	GetPakSystem()->CloseArchive(pPakFile);

	// Add this zip to the array.
	m_zipFiles.push_back(pakFilename);

	RCLog("Finished adding %d files to zip file %s:", 
		numFiles, pakFilename.c_str());
	RCLog("    %d added, %d up-to-date, %d skipped, %d missing, %d failed",
		numFilesAdded, numFilesUpToDate, numFilesSkipped, numFilesMissing, numFilesFailed);

	return bResult;
}

//////////////////////////////////////////////////////////////////////////
void ResourceCompiler::CleanTargetFolder( IConfig* config )
{
	std::vector<string> files;
	string targetroot;
	if (!config->Get( "targetroot", targetroot ))
	{
		return;
	}
	RCLog( "Cleaning target folder %s",targetroot.c_str() );

	// Look at the list of processed files.
	m_pOutputFileList->Load( FormLogFileName(RC_FILENAME_OUTPUT_FILE_LIST) );

	// Save output files


	// Collect unique list of source files that where deleted.
	std::vector<string> deletedSourceFiles;
	std::vector<string> deletedTargetFiles;
	string lastInputFile;
	bool bSrcFileExist = false;
	for (size_t i = 0; i < m_pOutputFileList->m_outputFiles.size(); i++)
	{
		// Check if input file exist.
		const COutputFileList::SFile &of = m_pOutputFileList->m_outputFiles[i];
		if (of.inputFile != lastInputFile)
		{
			lastInputFile = of.inputFile;
			if (FileUtil::FileExists(of.inputFile.c_str()))
			{
				bSrcFileExist = true;
			}
			else
			{
				RCLog( "Source file deleted: %s",of.inputFile.c_str() );
				deletedSourceFiles.push_back(of.inputFile);
				bSrcFileExist = false;
			}
		}
		if (!bSrcFileExist)
		{
			deletedTargetFiles.push_back(of.outputFile);
		}
	}
	
	std::sort(deletedSourceFiles.begin(),deletedSourceFiles.end());
	std::sort(deletedTargetFiles.begin(),deletedTargetFiles.end());
	
	//////////////////////////////////////////////////////////////////////////
	// Remove source filename from the output file list
	for (size_t i = 0; i < deletedSourceFiles.size(); i++)
	{
		m_pOutputFileList->RemoveInputFile( deletedSourceFiles[i] );
	}

	//////////////////////////////////////////////////////////////////////////
	// Delete target files from disk
	for (size_t i = 0; i < deletedTargetFiles.size(); i++)
	{
		string filename = deletedTargetFiles[i];
		RCLog( "Delete file %s",filename.c_str() );
		::DeleteFile( filename.c_str() );
	}

	int nDeletedInZip = 0;
	int nScannedFiles = 0;
	int nTotalToScan = m_zipFiles.size()*deletedTargetFiles.size();

	if (nTotalToScan > 0)
	{
		RCLog( "Deleting files from zip archives" );
	}

	// If we created some zip file, check if files need to be deleted from them.
	for (size_t nzip = 0; nzip < m_zipFiles.size() && nTotalToScan > 0; nzip++)
	{
		string zipFilename = m_zipFiles[nzip];

		PakSystemArchive *pPakFile = GetPakSystem()->OpenArchive( zipFilename.c_str() );
		if (pPakFile)
		{
			string progress = string("Deleting files from ") + zipFilename;

			char szRelative[MAX_PATH];
			string zipFileDir = PathHelpers::GetDirectory(zipFilename);

			for (size_t i = 0; i < deletedTargetFiles.size(); i++)
			{
				string filename = deletedTargetFiles[i];
				// Detect path offset of the zip location to source folder
				if (TRUE == PathRelativePathTo( szRelative,zipFileDir.c_str(),FILE_ATTRIBUTE_DIRECTORY,filename.c_str(),FILE_ATTRIBUTE_NORMAL ))
				{
					char szRelative2[MAX_PATH];
					PathCanonicalize(szRelative2,szRelative);
					filename = szRelative2;
				}

				if (GetPakSystem()->DeleteFromArchive( pPakFile,filename.c_str() ))
				{
					RCLog( "Remove file from zip: [%s] %s",zipFilename.c_str(),filename.c_str() );
					nDeletedInZip++;
				}
				nScannedFiles++;
				ShowProgress( progress.c_str(),(nScannedFiles*100)/nTotalToScan );
			}
			GetPakSystem()->CloseArchive( pPakFile );
		}
	}

	RCLog( "Saving rc_deletedfiles.txt" );


	m_pOutputFileList->Sort();
	m_pOutputFileList->Save( FormLogFileName(RC_FILENAME_OUTPUT_FILE_LIST) );

	// store deleted files list.
	FILE *f = fopen( m_exePath+"rc_deletedfiles.txt","wt" );
	if (f)
	{
		for (size_t i = 0; i < deletedTargetFiles.size(); i++)
		{
			fprintf( f,"%s\n",deletedTargetFiles[i].c_str() );
		}
		fclose(f);
	}
}

//////////////////////////////////////////////////////////////////////////
static ICryXML* LoadICryXML()
{
	HMODULE hXMLLibrary = LoadLibrary("CryXML.dll");
	if (NULL == hXMLLibrary)
	{
		RCLogError("Unable to load xml library (CryXML.dll)");
		return 0;
	}
	FnGetICryXML pfnGetICryXML = (FnGetICryXML)GetProcAddress(hXMLLibrary, "GetICryXML");
	if (pfnGetICryXML == 0)
	{
		RCLogError("Unable to load xml library (CryXML.dll) - cannot find exported function GetICryXML().");
		return 0;
	}
	return pfnGetICryXML();
}

//////////////////////////////////////////////////////////////////////////
XmlNodeRef ResourceCompiler::LoadXml( const char *filename )
{
	ICryXML *pCryXML = LoadICryXML();
	if (!pCryXML)
		return false;

	// Get the xml serializer.
	IXMLSerializer* pSerializer = pCryXML->GetXMLSerializer();

	// Read in the input file.
	XmlNodeRef root;
	{
		const bool bRemoveNonessentialSpacesFromContent = false;
		char szErrorBuffer[1024];
		root = pSerializer->Read(FileXmlBufferSource(filename), false, sizeof(szErrorBuffer), szErrorBuffer);
		if (!root)
		{
			RCLogError("Cannot open XML file '%s': %s\n", filename, szErrorBuffer);
			return 0;
		}
	}
	return root;
}

//////////////////////////////////////////////////////////////////////////
bool ResourceCompiler::ProcessJobFile( EPlatform platform,IConfig* config )
{
	// Job file is an XML with multiple jobs for the RC
	string jobFile;
	if (!config->Get( "job", jobFile ))
	{
		RCLogError( "No job file specified" );
		return false;
	}

	CPropertyVars properties(this);
	config->SetToPropertyVars(properties);

	XmlNodeRef root = LoadXml( jobFile.c_str() );
	if (!root)
	{
		RCLogError( "Failed to load job XML file %s",jobFile.c_str() );
		return false;
	}
	for (int i = 0; i < root->getChildCount(); i++)
	{
		XmlNodeRef jobNode = root->getChild(i);
		RunJobXmlNode( properties,platform,config,jobNode );
	}

	return true;
}

//////////////////////////////////////////////////////////////////////////
void ResourceCompiler::RunJobXmlNode( CPropertyVars &properties,EPlatform platform,IConfig* config,XmlNodeRef &jobNode )
{
	if (jobNode->isTag("Properties"))
	{
		// Attributes are config modifiers.
		for (int attr = 0; attr < jobNode->getNumAttributes(); attr++)
		{
			const char *key = "";
			const char *value = "";
			jobNode->getAttributeByIndex(attr,&key,&value);
			string strValue = value;
			properties.ExpandProperties(strValue);
			properties.SetProperty( key,strValue );
		}
		return;
	}

	if (jobNode->isTag("Run"))
	{
		const char *jobListName = jobNode->getAttr("Job");
		if (strlen(jobListName) == 0)
			return;

		// Attributes are config modifiers.
		for (int attr = 0; attr < jobNode->getNumAttributes(); attr++)
		{
			const char *key = "";
			const char *value = "";
			jobNode->getAttributeByIndex(attr,&key,&value);
			string strValue = value;
			properties.ExpandProperties(strValue);
			properties.SetProperty( key,strValue );
		}
		
		XmlNodeRef root = jobNode;
		while (root->getParent()) root = root->getParent();
		// Find JobList.
		XmlNodeRef jobListNode = root->findChild(jobListName);
		if (jobListNode)
		{
			// Execute Job sub nodes.
			for (int i = 0; i < jobListNode->getChildCount(); i++)
			{
				XmlNodeRef subJobNode = jobListNode->getChild(i);
				RunJobXmlNode( properties,platform,config,subJobNode );
			}
		}
		return;
	}

	if (jobNode->isTag("Include"))
	{
		const char *includeFile = jobNode->getAttr("file");
		if (strlen(includeFile) == 0)
			return;
		
		string jobFile;
		config->Get( "job", jobFile );
		string includePath = PathHelpers::AddSeparator(PathHelpers::GetDirectory(jobFile)) + includeFile;

		XmlNodeRef root = LoadXml( includePath );
		if (!root)
		{
			RCLogError( "Cannot open Job include file %s",includePath.c_str() );
			return;
		}

		// Add include sub-nodes
		XmlNodeRef parent = jobNode->getParent();
		for (int i = 0; i < root->getChildCount(); i++)
		{
			XmlNodeRef subJobNode = root->getChild(i);
			parent->addChild(subJobNode);
		}
		return;
	}

	// Condition node.
	if (jobNode->isTag("if") || jobNode->isTag("ifnot"))
	{
		bool bIf = false;
		for (int attr = 0; attr < jobNode->getNumAttributes(); attr++)
		{
			const char *key = "";
			const char *value = "";
			jobNode->getAttributeByIndex(attr,&key,&value);

			string propValue;
			properties.GetProperty(key,propValue);
			if (stricmp(value,propValue.c_str()) == 0)
			{
				// match.
				bIf = true;
			}
		}
		if (jobNode->isTag("ifnot"))
		{
			bIf = !bIf; // Invert
		}
		if (bIf)
		{
			// Exec sub-nodes
			for (int i = 0; i < jobNode->getChildCount(); i++)
			{
				XmlNodeRef subJobNode = jobNode->getChild(i);
				RunJobXmlNode( properties,platform,config,subJobNode );
			}
		}
	}

	if (jobNode->isTag("Job"))
	{
		RCLog( "-------------------------------------------------------------------" );
		string jobLog = "Job: ";
		// Delete all config entries from previous job.
		config->ClearPriorityUsage(eCP_PriorityJob);

		bool bCleanJob = false;;

		// Attributes are config modifiers.
		for (int attr = 0; attr < jobNode->getNumAttributes(); attr++)
		{
			const char *key = "";
			const char *value = "";
			jobNode->getAttributeByIndex(attr,&key,&value);

			if (stricmp(key,"input") == 0)
			{
				jobLog += string("/") + key + "=" + value + " ";
				continue;
			}
			if (stricmp(key,"clean_targetroot") == 0)
			{
				bCleanJob = true;
			}
			string valueStr = value;
			properties.ExpandProperties(valueStr);
			config->Set( eCP_PriorityJob,key,valueStr );
			jobLog += string("/") + key + "=" + valueStr + " ";
		}

		string fileSpec = jobNode->getAttr( "input" );
		properties.ExpandProperties(fileSpec);
		if (!fileSpec.empty())
		{
			RCLog(jobLog);
			Compile( platform,config,fileSpec );
		}
		else if (bCleanJob)
		{
			CleanTargetFolder(config);

		}
		// Delete all config entries from our job.
		config->ClearPriorityUsage(eCP_PriorityJob);
	}
}

//////////////////////////////////////////////////////////////////////////
void ResourceCompiler::ShowProgress( const char *sOperation,int nPercent )
{
	// Show progress
	static int nLastPercent = -1;
	if (nPercent != nLastPercent)
	{
		char str[1024];
		_snprintf_s(str, sizeof(str), sizeof(str) - 1,"Progress: %d%% %s", nPercent, sOperation);
		SetConsoleTitle(str);
	}
}
