/* -----------------------------------------------------------------------------

	Parts:
	Copyright (c) 2006 Simon Brown                 si@sjbrown.co.uk
	Copyright (c) 2006 Ignacio Castano             icastano@nvidia.com

	Permission is hereby granted, free of charge, to any person obtaining
	a copy of this software and associated documentation files (the 
	"Software"), to	deal in the Software without restriction, including
	without limitation the rights to use, copy, modify, merge, publish,
	distribute, sublicense, and/or sell copies of the Software, and to 
	permit persons to whom the Software is furnished to do so, subject to 
	the following conditions:

	The above copyright notice and this permission notice shall be included
	in all copies or substantial portions of the Software.

	THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
	OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
	MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
	IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY 
	CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, 
	TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE 
	SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
	
   -------------------------------------------------------------------------- */

#include "stdafx.h"
#include "OptimalCompressor.h"


// Data structures and tables below were created by opt.cpp.
// Create a .bat file with following three rows and launch it.
//   call "%VS80COMNTOOLS%vsvars32.bat"
//   cl opt.cpp
//   opt > tables.txt

#if 0
// opt.cpp
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <float.h>


class OptimalSingleColorEncodingBuilder
{
	struct TableElem
	{
		int val0;
		int val1;
	    float err;
	};


	static int FloatToClosestNonNegativeInt( float f )
	{
    	return (f <= 0) ? 0 : ((int) (f + 0.5f));
    }

	static void ComputeTable(TableElem* const Table, const int* const expand, int const size, int const mul, int const div)
	{
		for (int i = 0; i < 256; i++)
		{
			float bestErr = FLT_MAX;

			for (int min = 0; min < size; min++)
			{
				for (int max = 0; max < size; max++)
				{
					int mine = expand[min];
					int maxe = expand[max];

    	            float computedVal = ((div-mul)*mine + mul*maxe) / (float)div;
        	        float realErr = (computedVal - i);
            	    if(realErr < 0 ) realErr = -realErr;
                    float directx10_MaxAllowedError = 0.03f * abs(max - min); // 3% is max allowed error in DirectX10
					float err = realErr + directx10_MaxAllowedError;

					if (err < bestErr)
					{
						Table[i].val0 = min;
						Table[i].val1 = max;
						Table[i].err = realErr;
						bestErr = err;
					}
				}
			}
		}
	}


	static void ComputeTable256(unsigned char* const Table, const int* const expand, int targetSize)
	{
    	const int sourceSize = 256;

		for (int i = 0; i < sourceSize; i++)
		{
			float bestErr = FLT_MAX;

			for (int j = 0; j < targetSize; j++)
			{
                float err = i - expand[j];
                if(err < 0) err = -err;

				if (err < bestErr)
				{
					Table[i] = j;
					bestErr = err;
				}
			}
		}
	}


	static void PrintHeader()
	{
   		printf(
	    	"struct OptimalEncodingPair\n"
    	    "{\n"
        	"\tunsigned char val0;\n"
	        "\tunsigned char val1;\n"
	        "\tunsigned char err;\n"
	        "\tunsigned char _pad_;\n"
	        "};\n"
	        "\n");
	}


	static void PrintTable256(const char* msg, const unsigned char* const Table)
	{
    	const int size = 256;
	   	printf("static const unsigned char %s[%i] =\n", msg, size);
   		printf("{\n");

	 	for(int i=0; i<size; ++i)
    	{
    		printf( "\t%2i%s     // %3i",
	        	Table[i],
	            ((i==size-1)?" ":","),
    	        i );
		   	printf("\n");
    	}
	
   		printf("};\n");
	   	printf("\n");
	}


	static void PrintTable(const char* msg, const TableElem* const Table, int const size, bool verbose)
	{
	   	printf("static const OptimalEncodingPair %s[256] =\n", msg);
   		printf("{\n");

	 	for(int i=0; i<size; ++i)
    	{
    		printf( "\t{ 0x%02x, 0x%02x, %2i, 0 }%s     // %3i",
	        	Table[i].val0,
    	        Table[i].val1,
        	    FloatToClosestNonNegativeInt(6 * Table[i].err),
	            ((i==size-1)?" ":","),
    	        i );
            if(verbose)
            {
				printf( "(err:%5.2f, err*6:%5.2f)",
	        	    Table[i].err,
		            6 * Table[i].err);
            }
		   	printf("\n");
    	}

   		printf("};\n");
	   	printf("\n");
	}

	static void PrintComparision(const char* msg, TableElem* const Table0, TableElem* const Table1, int const size)
	{
	   	printf("comparision %s\n", msg);
	   	printf("{\n");
	
 		for(int i=0; i<size; ++i)
	    {
    		float errDelta = Table1[i].err - Table0[i].err;
    		printf( "\t%f   // %3i %s\n", errDelta, i, (errDelta<0)?"/* second table wins! */":"");
	    }

   		printf("};\n");
	}

public:
	static void MakeTables(bool verbose)
	{
		int Expand4to8[16];
		int Expand5to8[32];
		int Expand6to8[64];

		TableElem OMatch5_1_3[256];
		TableElem OMatch6_1_3[256];

		TableElem OMatch5_1_2[256];
		TableElem OMatch6_1_2[256];

		unsigned char Cnv8to4[256];

		for(int i=0;i<16;i++)
        {
			Expand4to8[i] = (i<<4) | i;
        }

		for(int i=0;i<32;i++)
        {
			Expand5to8[i] = (i<<3) | (i>>2);
        }

		for(int i=0;i<64;i++)
        {
			Expand6to8[i] = (i<<2) | (i>>4);
        }

		ComputeTable(OMatch5_1_3, Expand5to8, 32, 1,3);
		ComputeTable(OMatch6_1_3, Expand6to8, 64, 1,3);

		ComputeTable(OMatch5_1_2, Expand5to8, 32, 1,2);
		ComputeTable(OMatch6_1_2, Expand6to8, 64, 1,2);

        ComputeTable256(Cnv8to4, Expand4to8, 16);

        PrintHeader();

		PrintTable("OptimalEncoding5bit_1_3", OMatch5_1_3,256, verbose);
		PrintTable("OptimalEncoding6bit_1_3", OMatch6_1_3,256, verbose);

		PrintTable("OptimalEncoding5bit_1_2", OMatch5_1_2,256, verbose);
		PrintTable("OptimalEncoding6bit_1_2", OMatch6_1_2,256, verbose);

		PrintTable256("Convert8bitTo4bit", Cnv8to4);

        if(verbose)
		{
			PrintComparision("5bit 1/3 vs 1/2", OMatch5_1_3,OMatch5_1_2,256);
			PrintComparision("6bit 1/3 vs 1/2", OMatch6_1_3,OMatch6_1_2,256);
		}
	}
 };



int main(int argc, char **argv)
{
	OptimalSingleColorEncodingBuilder::MakeTables( (argc>1) );
	return 0;
}
#endif


struct OptimalEncodingPair
{
	unsigned char val0;
	unsigned char val1;
	unsigned char err;
	unsigned char _pad_;
};

static const OptimalEncodingPair OptimalEncoding5bit_1_3[256] =
{
	{ 0x00, 0x00,  0, 0 },     //   0
	{ 0x00, 0x00,  6, 0 },     //   1
	{ 0x00, 0x01,  4, 0 },     //   2
	{ 0x00, 0x01,  2, 0 },     //   3
	{ 0x00, 0x01,  8, 0 },     //   4
	{ 0x01, 0x00,  2, 0 },     //   5
	{ 0x01, 0x00,  4, 0 },     //   6
	{ 0x01, 0x01,  6, 0 },     //   7
	{ 0x01, 0x01,  0, 0 },     //   8
	{ 0x01, 0x01,  6, 0 },     //   9
	{ 0x01, 0x02,  4, 0 },     //  10
	{ 0x00, 0x04,  0, 0 },     //  11
	{ 0x00, 0x04,  6, 0 },     //  12
	{ 0x02, 0x01,  2, 0 },     //  13
	{ 0x00, 0x05,  2, 0 },     //  14
	{ 0x02, 0x02,  6, 0 },     //  15
	{ 0x02, 0x02,  0, 0 },     //  16
	{ 0x01, 0x04,  4, 0 },     //  17
	{ 0x02, 0x03,  4, 0 },     //  18
	{ 0x01, 0x05,  0, 0 },     //  19
	{ 0x01, 0x05,  6, 0 },     //  20
	{ 0x03, 0x02,  2, 0 },     //  21
	{ 0x04, 0x00,  0, 0 },     //  22
	{ 0x03, 0x03,  6, 0 },     //  23
	{ 0x03, 0x03,  0, 0 },     //  24
	{ 0x04, 0x01,  2, 0 },     //  25
	{ 0x03, 0x04,  6, 0 },     //  26
	{ 0x03, 0x04,  0, 0 },     //  27
	{ 0x04, 0x02,  4, 0 },     //  28
	{ 0x03, 0x05,  4, 0 },     //  29
	{ 0x04, 0x03,  0, 0 },     //  30
	{ 0x04, 0x03,  6, 0 },     //  31
	{ 0x03, 0x06,  2, 0 },     //  32
	{ 0x04, 0x04,  0, 0 },     //  33
	{ 0x04, 0x04,  6, 0 },     //  34
	{ 0x03, 0x07,  0, 0 },     //  35
	{ 0x04, 0x05,  2, 0 },     //  36
	{ 0x06, 0x02,  6, 0 },     //  37
	{ 0x06, 0x02,  0, 0 },     //  38
	{ 0x05, 0x04,  4, 0 },     //  39
	{ 0x06, 0x03,  4, 0 },     //  40
	{ 0x05, 0x05,  0, 0 },     //  41
	{ 0x05, 0x05,  6, 0 },     //  42
	{ 0x07, 0x02,  2, 0 },     //  43
	{ 0x04, 0x08,  0, 0 },     //  44
	{ 0x04, 0x08,  6, 0 },     //  45
	{ 0x07, 0x03,  0, 0 },     //  46
	{ 0x04, 0x09,  2, 0 },     //  47
	{ 0x06, 0x06,  6, 0 },     //  48
	{ 0x06, 0x06,  0, 0 },     //  49
	{ 0x05, 0x08,  4, 0 },     //  50
	{ 0x06, 0x07,  4, 0 },     //  51
	{ 0x05, 0x09,  0, 0 },     //  52
	{ 0x05, 0x09,  6, 0 },     //  53
	{ 0x07, 0x06,  2, 0 },     //  54
	{ 0x08, 0x04,  0, 0 },     //  55
	{ 0x07, 0x07,  6, 0 },     //  56
	{ 0x07, 0x07,  0, 0 },     //  57
	{ 0x08, 0x05,  2, 0 },     //  58
	{ 0x07, 0x08,  6, 0 },     //  59
	{ 0x07, 0x08,  0, 0 },     //  60
	{ 0x08, 0x06,  4, 0 },     //  61
	{ 0x07, 0x09,  4, 0 },     //  62
	{ 0x08, 0x07,  0, 0 },     //  63
	{ 0x08, 0x07,  6, 0 },     //  64
	{ 0x07, 0x0a,  2, 0 },     //  65
	{ 0x08, 0x08,  0, 0 },     //  66
	{ 0x08, 0x08,  6, 0 },     //  67
	{ 0x07, 0x0b,  0, 0 },     //  68
	{ 0x08, 0x09,  2, 0 },     //  69
	{ 0x0a, 0x06,  6, 0 },     //  70
	{ 0x0a, 0x06,  0, 0 },     //  71
	{ 0x09, 0x08,  4, 0 },     //  72
	{ 0x0a, 0x07,  4, 0 },     //  73
	{ 0x09, 0x09,  0, 0 },     //  74
	{ 0x09, 0x09,  6, 0 },     //  75
	{ 0x0b, 0x06,  2, 0 },     //  76
	{ 0x08, 0x0c,  0, 0 },     //  77
	{ 0x08, 0x0c,  6, 0 },     //  78
	{ 0x0b, 0x07,  0, 0 },     //  79
	{ 0x08, 0x0d,  2, 0 },     //  80
	{ 0x0a, 0x0a,  6, 0 },     //  81
	{ 0x0a, 0x0a,  0, 0 },     //  82
	{ 0x09, 0x0c,  4, 0 },     //  83
	{ 0x0a, 0x0b,  4, 0 },     //  84
	{ 0x09, 0x0d,  0, 0 },     //  85
	{ 0x09, 0x0d,  6, 0 },     //  86
	{ 0x0b, 0x0a,  2, 0 },     //  87
	{ 0x0c, 0x08,  0, 0 },     //  88
	{ 0x0b, 0x0b,  6, 0 },     //  89
	{ 0x0b, 0x0b,  0, 0 },     //  90
	{ 0x0c, 0x09,  2, 0 },     //  91
	{ 0x0b, 0x0c,  6, 0 },     //  92
	{ 0x0b, 0x0c,  0, 0 },     //  93
	{ 0x0c, 0x0a,  4, 0 },     //  94
	{ 0x0b, 0x0d,  4, 0 },     //  95
	{ 0x0c, 0x0b,  0, 0 },     //  96
	{ 0x0c, 0x0b,  6, 0 },     //  97
	{ 0x0b, 0x0e,  2, 0 },     //  98
	{ 0x0c, 0x0c,  0, 0 },     //  99
	{ 0x0c, 0x0c,  6, 0 },     // 100
	{ 0x0b, 0x0f,  0, 0 },     // 101
	{ 0x0c, 0x0d,  2, 0 },     // 102
	{ 0x0e, 0x0a,  6, 0 },     // 103
	{ 0x0e, 0x0a,  0, 0 },     // 104
	{ 0x0d, 0x0c,  4, 0 },     // 105
	{ 0x0e, 0x0b,  4, 0 },     // 106
	{ 0x0d, 0x0d,  0, 0 },     // 107
	{ 0x0d, 0x0d,  6, 0 },     // 108
	{ 0x0f, 0x0a,  2, 0 },     // 109
	{ 0x0c, 0x10,  0, 0 },     // 110
	{ 0x0c, 0x10,  6, 0 },     // 111
	{ 0x0f, 0x0b,  0, 0 },     // 112
	{ 0x0c, 0x11,  2, 0 },     // 113
	{ 0x0e, 0x0e,  6, 0 },     // 114
	{ 0x0e, 0x0e,  0, 0 },     // 115
	{ 0x0d, 0x10,  4, 0 },     // 116
	{ 0x0e, 0x0f,  4, 0 },     // 117
	{ 0x0d, 0x11,  0, 0 },     // 118
	{ 0x0d, 0x11,  6, 0 },     // 119
	{ 0x0f, 0x0e,  2, 0 },     // 120
	{ 0x10, 0x0c,  0, 0 },     // 121
	{ 0x0f, 0x0f,  6, 0 },     // 122
	{ 0x0f, 0x0f,  0, 0 },     // 123
	{ 0x10, 0x0d,  2, 0 },     // 124
	{ 0x0f, 0x10,  6, 0 },     // 125
	{ 0x0f, 0x10,  0, 0 },     // 126
	{ 0x10, 0x0e,  4, 0 },     // 127
	{ 0x0f, 0x11,  4, 0 },     // 128
	{ 0x10, 0x0f,  0, 0 },     // 129
	{ 0x10, 0x0f,  6, 0 },     // 130
	{ 0x0f, 0x12,  2, 0 },     // 131
	{ 0x10, 0x10,  0, 0 },     // 132
	{ 0x10, 0x10,  6, 0 },     // 133
	{ 0x0f, 0x13,  0, 0 },     // 134
	{ 0x10, 0x11,  2, 0 },     // 135
	{ 0x12, 0x0e,  6, 0 },     // 136
	{ 0x12, 0x0e,  0, 0 },     // 137
	{ 0x11, 0x10,  4, 0 },     // 138
	{ 0x12, 0x0f,  4, 0 },     // 139
	{ 0x11, 0x11,  0, 0 },     // 140
	{ 0x11, 0x11,  6, 0 },     // 141
	{ 0x13, 0x0e,  2, 0 },     // 142
	{ 0x10, 0x14,  0, 0 },     // 143
	{ 0x10, 0x14,  6, 0 },     // 144
	{ 0x13, 0x0f,  0, 0 },     // 145
	{ 0x10, 0x15,  2, 0 },     // 146
	{ 0x12, 0x12,  6, 0 },     // 147
	{ 0x12, 0x12,  0, 0 },     // 148
	{ 0x11, 0x14,  4, 0 },     // 149
	{ 0x12, 0x13,  4, 0 },     // 150
	{ 0x11, 0x15,  0, 0 },     // 151
	{ 0x11, 0x15,  6, 0 },     // 152
	{ 0x13, 0x12,  2, 0 },     // 153
	{ 0x14, 0x10,  0, 0 },     // 154
	{ 0x13, 0x13,  6, 0 },     // 155
	{ 0x13, 0x13,  0, 0 },     // 156
	{ 0x14, 0x11,  2, 0 },     // 157
	{ 0x13, 0x14,  6, 0 },     // 158
	{ 0x13, 0x14,  0, 0 },     // 159
	{ 0x14, 0x12,  4, 0 },     // 160
	{ 0x13, 0x15,  4, 0 },     // 161
	{ 0x14, 0x13,  0, 0 },     // 162
	{ 0x14, 0x13,  6, 0 },     // 163
	{ 0x13, 0x16,  2, 0 },     // 164
	{ 0x14, 0x14,  0, 0 },     // 165
	{ 0x14, 0x14,  6, 0 },     // 166
	{ 0x13, 0x17,  0, 0 },     // 167
	{ 0x14, 0x15,  2, 0 },     // 168
	{ 0x16, 0x12,  6, 0 },     // 169
	{ 0x16, 0x12,  0, 0 },     // 170
	{ 0x15, 0x14,  4, 0 },     // 171
	{ 0x16, 0x13,  4, 0 },     // 172
	{ 0x15, 0x15,  0, 0 },     // 173
	{ 0x15, 0x15,  6, 0 },     // 174
	{ 0x17, 0x12,  2, 0 },     // 175
	{ 0x14, 0x18,  0, 0 },     // 176
	{ 0x14, 0x18,  6, 0 },     // 177
	{ 0x17, 0x13,  0, 0 },     // 178
	{ 0x14, 0x19,  2, 0 },     // 179
	{ 0x16, 0x16,  6, 0 },     // 180
	{ 0x16, 0x16,  0, 0 },     // 181
	{ 0x15, 0x18,  4, 0 },     // 182
	{ 0x16, 0x17,  4, 0 },     // 183
	{ 0x15, 0x19,  0, 0 },     // 184
	{ 0x15, 0x19,  6, 0 },     // 185
	{ 0x17, 0x16,  2, 0 },     // 186
	{ 0x18, 0x14,  0, 0 },     // 187
	{ 0x17, 0x17,  6, 0 },     // 188
	{ 0x17, 0x17,  0, 0 },     // 189
	{ 0x18, 0x15,  2, 0 },     // 190
	{ 0x17, 0x18,  6, 0 },     // 191
	{ 0x17, 0x18,  0, 0 },     // 192
	{ 0x18, 0x16,  4, 0 },     // 193
	{ 0x17, 0x19,  4, 0 },     // 194
	{ 0x18, 0x17,  0, 0 },     // 195
	{ 0x18, 0x17,  6, 0 },     // 196
	{ 0x17, 0x1a,  2, 0 },     // 197
	{ 0x18, 0x18,  0, 0 },     // 198
	{ 0x18, 0x18,  6, 0 },     // 199
	{ 0x17, 0x1b,  0, 0 },     // 200
	{ 0x18, 0x19,  2, 0 },     // 201
	{ 0x1a, 0x16,  6, 0 },     // 202
	{ 0x1a, 0x16,  0, 0 },     // 203
	{ 0x19, 0x18,  4, 0 },     // 204
	{ 0x1a, 0x17,  4, 0 },     // 205
	{ 0x19, 0x19,  0, 0 },     // 206
	{ 0x19, 0x19,  6, 0 },     // 207
	{ 0x1b, 0x16,  2, 0 },     // 208
	{ 0x18, 0x1c,  0, 0 },     // 209
	{ 0x18, 0x1c,  6, 0 },     // 210
	{ 0x1b, 0x17,  0, 0 },     // 211
	{ 0x18, 0x1d,  2, 0 },     // 212
	{ 0x1a, 0x1a,  6, 0 },     // 213
	{ 0x1a, 0x1a,  0, 0 },     // 214
	{ 0x19, 0x1c,  4, 0 },     // 215
	{ 0x1a, 0x1b,  4, 0 },     // 216
	{ 0x19, 0x1d,  0, 0 },     // 217
	{ 0x19, 0x1d,  6, 0 },     // 218
	{ 0x1b, 0x1a,  2, 0 },     // 219
	{ 0x1c, 0x18,  0, 0 },     // 220
	{ 0x1b, 0x1b,  6, 0 },     // 221
	{ 0x1b, 0x1b,  0, 0 },     // 222
	{ 0x1c, 0x19,  2, 0 },     // 223
	{ 0x1b, 0x1c,  6, 0 },     // 224
	{ 0x1b, 0x1c,  0, 0 },     // 225
	{ 0x1c, 0x1a,  4, 0 },     // 226
	{ 0x1b, 0x1d,  4, 0 },     // 227
	{ 0x1c, 0x1b,  0, 0 },     // 228
	{ 0x1c, 0x1b,  6, 0 },     // 229
	{ 0x1b, 0x1e,  2, 0 },     // 230
	{ 0x1c, 0x1c,  0, 0 },     // 231
	{ 0x1c, 0x1c,  6, 0 },     // 232
	{ 0x1b, 0x1f,  0, 0 },     // 233
	{ 0x1c, 0x1d,  2, 0 },     // 234
	{ 0x1e, 0x1a,  6, 0 },     // 235
	{ 0x1e, 0x1a,  0, 0 },     // 236
	{ 0x1d, 0x1c,  4, 0 },     // 237
	{ 0x1e, 0x1b,  4, 0 },     // 238
	{ 0x1d, 0x1d,  0, 0 },     // 239
	{ 0x1d, 0x1d,  6, 0 },     // 240
	{ 0x1f, 0x1a,  2, 0 },     // 241
	{ 0x1d, 0x1e,  2, 0 },     // 242
	{ 0x1f, 0x1b,  6, 0 },     // 243
	{ 0x1f, 0x1b,  0, 0 },     // 244
	{ 0x1e, 0x1d,  4, 0 },     // 245
	{ 0x1e, 0x1e,  6, 0 },     // 246
	{ 0x1e, 0x1e,  0, 0 },     // 247
	{ 0x1e, 0x1e,  6, 0 },     // 248
	{ 0x1e, 0x1f,  4, 0 },     // 249
	{ 0x1e, 0x1f,  2, 0 },     // 250
	{ 0x1e, 0x1f,  8, 0 },     // 251
	{ 0x1f, 0x1e,  2, 0 },     // 252
	{ 0x1f, 0x1e,  4, 0 },     // 253
	{ 0x1f, 0x1f,  6, 0 },     // 254
	{ 0x1f, 0x1f,  0, 0 }      // 255
};

static const OptimalEncodingPair OptimalEncoding6bit_1_3[256] =
{
	{ 0x00, 0x00,  0, 0 },     //   0
	{ 0x00, 0x01,  2, 0 },     //   1
	{ 0x00, 0x01,  4, 0 },     //   2
	{ 0x01, 0x00,  2, 0 },     //   3
	{ 0x01, 0x01,  0, 0 },     //   4
	{ 0x01, 0x02,  2, 0 },     //   5
	{ 0x01, 0x02,  4, 0 },     //   6
	{ 0x02, 0x01,  2, 0 },     //   7
	{ 0x02, 0x02,  0, 0 },     //   8
	{ 0x02, 0x03,  2, 0 },     //   9
	{ 0x02, 0x03,  4, 0 },     //  10
	{ 0x03, 0x02,  2, 0 },     //  11
	{ 0x03, 0x03,  0, 0 },     //  12
	{ 0x03, 0x04,  2, 0 },     //  13
	{ 0x03, 0x04,  4, 0 },     //  14
	{ 0x04, 0x03,  2, 0 },     //  15
	{ 0x04, 0x04,  0, 0 },     //  16
	{ 0x04, 0x05,  2, 0 },     //  17
	{ 0x04, 0x05,  4, 0 },     //  18
	{ 0x05, 0x04,  2, 0 },     //  19
	{ 0x05, 0x05,  0, 0 },     //  20
	{ 0x05, 0x06,  2, 0 },     //  21
	{ 0x05, 0x06,  4, 0 },     //  22
	{ 0x06, 0x05,  2, 0 },     //  23
	{ 0x06, 0x06,  0, 0 },     //  24
	{ 0x06, 0x07,  2, 0 },     //  25
	{ 0x06, 0x07,  4, 0 },     //  26
	{ 0x07, 0x06,  2, 0 },     //  27
	{ 0x07, 0x07,  0, 0 },     //  28
	{ 0x07, 0x08,  2, 0 },     //  29
	{ 0x07, 0x08,  4, 0 },     //  30
	{ 0x08, 0x07,  2, 0 },     //  31
	{ 0x08, 0x08,  0, 0 },     //  32
	{ 0x08, 0x09,  2, 0 },     //  33
	{ 0x08, 0x09,  4, 0 },     //  34
	{ 0x05, 0x10,  0, 0 },     //  35
	{ 0x09, 0x09,  0, 0 },     //  36
	{ 0x09, 0x0a,  2, 0 },     //  37
	{ 0x06, 0x10,  2, 0 },     //  38
	{ 0x06, 0x11,  0, 0 },     //  39
	{ 0x0a, 0x0a,  0, 0 },     //  40
	{ 0x0a, 0x0b,  2, 0 },     //  41
	{ 0x07, 0x11,  2, 0 },     //  42
	{ 0x08, 0x10,  0, 0 },     //  43
	{ 0x0b, 0x0b,  0, 0 },     //  44
	{ 0x0b, 0x0c,  2, 0 },     //  45
	{ 0x10, 0x02,  0, 0 },     //  46
	{ 0x09, 0x11,  0, 0 },     //  47
	{ 0x0c, 0x0c,  0, 0 },     //  48
	{ 0x0c, 0x0d,  2, 0 },     //  49
	{ 0x10, 0x05,  0, 0 },     //  50
	{ 0x0b, 0x10,  0, 0 },     //  51
	{ 0x0d, 0x0d,  0, 0 },     //  52
	{ 0x0d, 0x0e,  2, 0 },     //  53
	{ 0x10, 0x08,  0, 0 },     //  54
	{ 0x0c, 0x11,  0, 0 },     //  55
	{ 0x0e, 0x0e,  0, 0 },     //  56
	{ 0x0e, 0x0f,  2, 0 },     //  57
	{ 0x10, 0x0b,  0, 0 },     //  58
	{ 0x0e, 0x10,  0, 0 },     //  59
	{ 0x0f, 0x0f,  0, 0 },     //  60
	{ 0x10, 0x0d,  2, 0 },     //  61
	{ 0x10, 0x0e,  0, 0 },     //  62
	{ 0x0f, 0x11,  0, 0 },     //  63
	{ 0x0f, 0x12,  2, 0 },     //  64
	{ 0x10, 0x10,  0, 0 },     //  65
	{ 0x11, 0x0f,  0, 0 },     //  66
	{ 0x0f, 0x14,  0, 0 },     //  67
	{ 0x11, 0x10,  2, 0 },     //  68
	{ 0x11, 0x11,  0, 0 },     //  69
	{ 0x13, 0x0e,  0, 0 },     //  70
	{ 0x0f, 0x17,  0, 0 },     //  71
	{ 0x12, 0x11,  2, 0 },     //  72
	{ 0x12, 0x12,  0, 0 },     //  73
	{ 0x14, 0x0f,  0, 0 },     //  74
	{ 0x0f, 0x1a,  0, 0 },     //  75
	{ 0x13, 0x12,  2, 0 },     //  76
	{ 0x13, 0x13,  0, 0 },     //  77
	{ 0x16, 0x0e,  0, 0 },     //  78
	{ 0x0f, 0x1d,  0, 0 },     //  79
	{ 0x14, 0x13,  2, 0 },     //  80
	{ 0x14, 0x14,  0, 0 },     //  81
	{ 0x17, 0x0f,  0, 0 },     //  82
	{ 0x18, 0x0e,  2, 0 },     //  83
	{ 0x15, 0x14,  2, 0 },     //  84
	{ 0x15, 0x15,  0, 0 },     //  85
	{ 0x19, 0x0e,  0, 0 },     //  86
	{ 0x19, 0x0f,  2, 0 },     //  87
	{ 0x16, 0x15,  2, 0 },     //  88
	{ 0x16, 0x16,  0, 0 },     //  89
	{ 0x1a, 0x0f,  0, 0 },     //  90
	{ 0x16, 0x17,  4, 0 },     //  91
	{ 0x17, 0x16,  2, 0 },     //  92
	{ 0x17, 0x17,  0, 0 },     //  93
	{ 0x17, 0x18,  2, 0 },     //  94
	{ 0x17, 0x18,  4, 0 },     //  95
	{ 0x18, 0x17,  2, 0 },     //  96
	{ 0x18, 0x18,  0, 0 },     //  97
	{ 0x18, 0x19,  2, 0 },     //  98
	{ 0x18, 0x19,  4, 0 },     //  99
	{ 0x15, 0x20,  0, 0 },     // 100
	{ 0x19, 0x19,  0, 0 },     // 101
	{ 0x19, 0x1a,  2, 0 },     // 102
	{ 0x16, 0x20,  2, 0 },     // 103
	{ 0x16, 0x21,  0, 0 },     // 104
	{ 0x1a, 0x1a,  0, 0 },     // 105
	{ 0x1a, 0x1b,  2, 0 },     // 106
	{ 0x17, 0x21,  2, 0 },     // 107
	{ 0x18, 0x20,  0, 0 },     // 108
	{ 0x1b, 0x1b,  0, 0 },     // 109
	{ 0x1b, 0x1c,  2, 0 },     // 110
	{ 0x20, 0x12,  0, 0 },     // 111
	{ 0x19, 0x21,  0, 0 },     // 112
	{ 0x1c, 0x1c,  0, 0 },     // 113
	{ 0x1c, 0x1d,  2, 0 },     // 114
	{ 0x20, 0x15,  0, 0 },     // 115
	{ 0x1b, 0x20,  0, 0 },     // 116
	{ 0x1d, 0x1d,  0, 0 },     // 117
	{ 0x1d, 0x1e,  2, 0 },     // 118
	{ 0x20, 0x18,  0, 0 },     // 119
	{ 0x1c, 0x21,  0, 0 },     // 120
	{ 0x1e, 0x1e,  0, 0 },     // 121
	{ 0x1e, 0x1f,  2, 0 },     // 122
	{ 0x20, 0x1b,  0, 0 },     // 123
	{ 0x1e, 0x20,  0, 0 },     // 124
	{ 0x1f, 0x1f,  0, 0 },     // 125
	{ 0x20, 0x1d,  2, 0 },     // 126
	{ 0x20, 0x1e,  0, 0 },     // 127
	{ 0x1f, 0x21,  0, 0 },     // 128
	{ 0x1f, 0x22,  2, 0 },     // 129
	{ 0x20, 0x20,  0, 0 },     // 130
	{ 0x21, 0x1f,  0, 0 },     // 131
	{ 0x1f, 0x24,  0, 0 },     // 132
	{ 0x21, 0x20,  2, 0 },     // 133
	{ 0x21, 0x21,  0, 0 },     // 134
	{ 0x23, 0x1e,  0, 0 },     // 135
	{ 0x1f, 0x27,  0, 0 },     // 136
	{ 0x22, 0x21,  2, 0 },     // 137
	{ 0x22, 0x22,  0, 0 },     // 138
	{ 0x24, 0x1f,  0, 0 },     // 139
	{ 0x1f, 0x2a,  0, 0 },     // 140
	{ 0x23, 0x22,  2, 0 },     // 141
	{ 0x23, 0x23,  0, 0 },     // 142
	{ 0x26, 0x1e,  0, 0 },     // 143
	{ 0x1f, 0x2d,  0, 0 },     // 144
	{ 0x24, 0x23,  2, 0 },     // 145
	{ 0x24, 0x24,  0, 0 },     // 146
	{ 0x27, 0x1f,  0, 0 },     // 147
	{ 0x28, 0x1e,  2, 0 },     // 148
	{ 0x25, 0x24,  2, 0 },     // 149
	{ 0x25, 0x25,  0, 0 },     // 150
	{ 0x29, 0x1e,  0, 0 },     // 151
	{ 0x29, 0x1f,  2, 0 },     // 152
	{ 0x26, 0x25,  2, 0 },     // 153
	{ 0x26, 0x26,  0, 0 },     // 154
	{ 0x2a, 0x1f,  0, 0 },     // 155
	{ 0x26, 0x27,  4, 0 },     // 156
	{ 0x27, 0x26,  2, 0 },     // 157
	{ 0x27, 0x27,  0, 0 },     // 158
	{ 0x27, 0x28,  2, 0 },     // 159
	{ 0x27, 0x28,  4, 0 },     // 160
	{ 0x28, 0x27,  2, 0 },     // 161
	{ 0x28, 0x28,  0, 0 },     // 162
	{ 0x28, 0x29,  2, 0 },     // 163
	{ 0x28, 0x29,  4, 0 },     // 164
	{ 0x25, 0x30,  0, 0 },     // 165
	{ 0x29, 0x29,  0, 0 },     // 166
	{ 0x29, 0x2a,  2, 0 },     // 167
	{ 0x26, 0x30,  2, 0 },     // 168
	{ 0x26, 0x31,  0, 0 },     // 169
	{ 0x2a, 0x2a,  0, 0 },     // 170
	{ 0x2a, 0x2b,  2, 0 },     // 171
	{ 0x27, 0x31,  2, 0 },     // 172
	{ 0x28, 0x30,  0, 0 },     // 173
	{ 0x2b, 0x2b,  0, 0 },     // 174
	{ 0x2b, 0x2c,  2, 0 },     // 175
	{ 0x30, 0x22,  0, 0 },     // 176
	{ 0x29, 0x31,  0, 0 },     // 177
	{ 0x2c, 0x2c,  0, 0 },     // 178
	{ 0x2c, 0x2d,  2, 0 },     // 179
	{ 0x30, 0x25,  0, 0 },     // 180
	{ 0x2b, 0x30,  0, 0 },     // 181
	{ 0x2d, 0x2d,  0, 0 },     // 182
	{ 0x2d, 0x2e,  2, 0 },     // 183
	{ 0x30, 0x28,  0, 0 },     // 184
	{ 0x2c, 0x31,  0, 0 },     // 185
	{ 0x2e, 0x2e,  0, 0 },     // 186
	{ 0x2e, 0x2f,  2, 0 },     // 187
	{ 0x30, 0x2b,  0, 0 },     // 188
	{ 0x2e, 0x30,  0, 0 },     // 189
	{ 0x2f, 0x2f,  0, 0 },     // 190
	{ 0x30, 0x2d,  2, 0 },     // 191
	{ 0x30, 0x2e,  0, 0 },     // 192
	{ 0x2f, 0x31,  0, 0 },     // 193
	{ 0x2f, 0x32,  2, 0 },     // 194
	{ 0x30, 0x30,  0, 0 },     // 195
	{ 0x31, 0x2f,  0, 0 },     // 196
	{ 0x2f, 0x34,  0, 0 },     // 197
	{ 0x31, 0x30,  2, 0 },     // 198
	{ 0x31, 0x31,  0, 0 },     // 199
	{ 0x33, 0x2e,  0, 0 },     // 200
	{ 0x2f, 0x37,  0, 0 },     // 201
	{ 0x32, 0x31,  2, 0 },     // 202
	{ 0x32, 0x32,  0, 0 },     // 203
	{ 0x34, 0x2f,  0, 0 },     // 204
	{ 0x2f, 0x3a,  0, 0 },     // 205
	{ 0x33, 0x32,  2, 0 },     // 206
	{ 0x33, 0x33,  0, 0 },     // 207
	{ 0x36, 0x2e,  0, 0 },     // 208
	{ 0x2f, 0x3d,  0, 0 },     // 209
	{ 0x34, 0x33,  2, 0 },     // 210
	{ 0x34, 0x34,  0, 0 },     // 211
	{ 0x37, 0x2f,  0, 0 },     // 212
	{ 0x38, 0x2e,  2, 0 },     // 213
	{ 0x35, 0x34,  2, 0 },     // 214
	{ 0x35, 0x35,  0, 0 },     // 215
	{ 0x39, 0x2e,  0, 0 },     // 216
	{ 0x39, 0x2f,  2, 0 },     // 217
	{ 0x36, 0x35,  2, 0 },     // 218
	{ 0x36, 0x36,  0, 0 },     // 219
	{ 0x3a, 0x2f,  0, 0 },     // 220
	{ 0x36, 0x37,  4, 0 },     // 221
	{ 0x37, 0x36,  2, 0 },     // 222
	{ 0x37, 0x37,  0, 0 },     // 223
	{ 0x37, 0x38,  2, 0 },     // 224
	{ 0x37, 0x38,  4, 0 },     // 225
	{ 0x38, 0x37,  2, 0 },     // 226
	{ 0x38, 0x38,  0, 0 },     // 227
	{ 0x38, 0x39,  2, 0 },     // 228
	{ 0x38, 0x39,  4, 0 },     // 229
	{ 0x39, 0x38,  2, 0 },     // 230
	{ 0x39, 0x39,  0, 0 },     // 231
	{ 0x39, 0x3a,  2, 0 },     // 232
	{ 0x39, 0x3a,  4, 0 },     // 233
	{ 0x3a, 0x39,  2, 0 },     // 234
	{ 0x3a, 0x3a,  0, 0 },     // 235
	{ 0x3a, 0x3b,  2, 0 },     // 236
	{ 0x3a, 0x3b,  4, 0 },     // 237
	{ 0x3b, 0x3a,  2, 0 },     // 238
	{ 0x3b, 0x3b,  0, 0 },     // 239
	{ 0x3b, 0x3c,  2, 0 },     // 240
	{ 0x3b, 0x3c,  4, 0 },     // 241
	{ 0x3c, 0x3b,  2, 0 },     // 242
	{ 0x3c, 0x3c,  0, 0 },     // 243
	{ 0x3c, 0x3d,  2, 0 },     // 244
	{ 0x3c, 0x3d,  4, 0 },     // 245
	{ 0x3d, 0x3c,  2, 0 },     // 246
	{ 0x3d, 0x3d,  0, 0 },     // 247
	{ 0x3d, 0x3e,  2, 0 },     // 248
	{ 0x3d, 0x3e,  4, 0 },     // 249
	{ 0x3e, 0x3d,  2, 0 },     // 250
	{ 0x3e, 0x3e,  0, 0 },     // 251
	{ 0x3e, 0x3f,  2, 0 },     // 252
	{ 0x3e, 0x3f,  4, 0 },     // 253
	{ 0x3f, 0x3e,  2, 0 },     // 254
	{ 0x3f, 0x3f,  0, 0 }      // 255
};

static const OptimalEncodingPair OptimalEncoding5bit_1_2[256] =
{
	{ 0x00, 0x00,  0, 0 },     //   0
	{ 0x00, 0x00,  6, 0 },     //   1
	{ 0x00, 0x00, 12, 0 },     //   2
	{ 0x00, 0x01,  6, 0 },     //   3
	{ 0x00, 0x01,  0, 0 },     //   4
	{ 0x00, 0x01,  6, 0 },     //   5
	{ 0x01, 0x01, 12, 0 },     //   6
	{ 0x01, 0x01,  6, 0 },     //   7
	{ 0x01, 0x01,  0, 0 },     //   8
	{ 0x01, 0x01,  6, 0 },     //   9
	{ 0x01, 0x01, 12, 0 },     //  10
	{ 0x01, 0x02,  6, 0 },     //  11
	{ 0x01, 0x02,  0, 0 },     //  12
	{ 0x01, 0x02,  6, 0 },     //  13
	{ 0x02, 0x02, 12, 0 },     //  14
	{ 0x02, 0x02,  6, 0 },     //  15
	{ 0x02, 0x02,  0, 0 },     //  16
	{ 0x00, 0x04,  3, 0 },     //  17
	{ 0x00, 0x04,  9, 0 },     //  18
	{ 0x02, 0x03,  6, 0 },     //  19
	{ 0x02, 0x03,  0, 0 },     //  20
	{ 0x01, 0x04,  3, 0 },     //  21
	{ 0x01, 0x04,  9, 0 },     //  22
	{ 0x03, 0x03,  6, 0 },     //  23
	{ 0x03, 0x03,  0, 0 },     //  24
	{ 0x02, 0x04,  3, 0 },     //  25
	{ 0x02, 0x04,  9, 0 },     //  26
	{ 0x03, 0x04,  9, 0 },     //  27
	{ 0x03, 0x04,  3, 0 },     //  28
	{ 0x03, 0x04,  3, 0 },     //  29
	{ 0x03, 0x04,  9, 0 },     //  30
	{ 0x03, 0x05,  9, 0 },     //  31
	{ 0x03, 0x05,  3, 0 },     //  32
	{ 0x04, 0x04,  0, 0 },     //  33
	{ 0x04, 0x04,  6, 0 },     //  34
	{ 0x03, 0x06,  9, 0 },     //  35
	{ 0x03, 0x06,  3, 0 },     //  36
	{ 0x04, 0x05,  0, 0 },     //  37
	{ 0x04, 0x05,  6, 0 },     //  38
	{ 0x03, 0x07,  9, 0 },     //  39
	{ 0x03, 0x07,  3, 0 },     //  40
	{ 0x05, 0x05,  0, 0 },     //  41
	{ 0x05, 0x05,  6, 0 },     //  42
	{ 0x05, 0x05, 12, 0 },     //  43
	{ 0x05, 0x06,  6, 0 },     //  44
	{ 0x05, 0x06,  0, 0 },     //  45
	{ 0x05, 0x06,  6, 0 },     //  46
	{ 0x06, 0x06, 12, 0 },     //  47
	{ 0x06, 0x06,  6, 0 },     //  48
	{ 0x06, 0x06,  0, 0 },     //  49
	{ 0x04, 0x08,  3, 0 },     //  50
	{ 0x04, 0x08,  9, 0 },     //  51
	{ 0x06, 0x07,  6, 0 },     //  52
	{ 0x06, 0x07,  0, 0 },     //  53
	{ 0x05, 0x08,  3, 0 },     //  54
	{ 0x05, 0x08,  9, 0 },     //  55
	{ 0x07, 0x07,  6, 0 },     //  56
	{ 0x07, 0x07,  0, 0 },     //  57
	{ 0x06, 0x08,  3, 0 },     //  58
	{ 0x06, 0x08,  9, 0 },     //  59
	{ 0x07, 0x08,  9, 0 },     //  60
	{ 0x07, 0x08,  3, 0 },     //  61
	{ 0x07, 0x08,  3, 0 },     //  62
	{ 0x07, 0x08,  9, 0 },     //  63
	{ 0x07, 0x09,  9, 0 },     //  64
	{ 0x07, 0x09,  3, 0 },     //  65
	{ 0x08, 0x08,  0, 0 },     //  66
	{ 0x08, 0x08,  6, 0 },     //  67
	{ 0x07, 0x0a,  9, 0 },     //  68
	{ 0x07, 0x0a,  3, 0 },     //  69
	{ 0x08, 0x09,  0, 0 },     //  70
	{ 0x08, 0x09,  6, 0 },     //  71
	{ 0x07, 0x0b,  9, 0 },     //  72
	{ 0x07, 0x0b,  3, 0 },     //  73
	{ 0x09, 0x09,  0, 0 },     //  74
	{ 0x09, 0x09,  6, 0 },     //  75
	{ 0x09, 0x09, 12, 0 },     //  76
	{ 0x09, 0x0a,  6, 0 },     //  77
	{ 0x09, 0x0a,  0, 0 },     //  78
	{ 0x09, 0x0a,  6, 0 },     //  79
	{ 0x0a, 0x0a, 12, 0 },     //  80
	{ 0x0a, 0x0a,  6, 0 },     //  81
	{ 0x0a, 0x0a,  0, 0 },     //  82
	{ 0x08, 0x0c,  3, 0 },     //  83
	{ 0x08, 0x0c,  9, 0 },     //  84
	{ 0x0a, 0x0b,  6, 0 },     //  85
	{ 0x0a, 0x0b,  0, 0 },     //  86
	{ 0x09, 0x0c,  3, 0 },     //  87
	{ 0x09, 0x0c,  9, 0 },     //  88
	{ 0x0b, 0x0b,  6, 0 },     //  89
	{ 0x0b, 0x0b,  0, 0 },     //  90
	{ 0x0a, 0x0c,  3, 0 },     //  91
	{ 0x0a, 0x0c,  9, 0 },     //  92
	{ 0x0b, 0x0c,  9, 0 },     //  93
	{ 0x0b, 0x0c,  3, 0 },     //  94
	{ 0x0b, 0x0c,  3, 0 },     //  95
	{ 0x0b, 0x0c,  9, 0 },     //  96
	{ 0x0b, 0x0d,  9, 0 },     //  97
	{ 0x0b, 0x0d,  3, 0 },     //  98
	{ 0x0c, 0x0c,  0, 0 },     //  99
	{ 0x0c, 0x0c,  6, 0 },     // 100
	{ 0x0b, 0x0e,  9, 0 },     // 101
	{ 0x0b, 0x0e,  3, 0 },     // 102
	{ 0x0c, 0x0d,  0, 0 },     // 103
	{ 0x0c, 0x0d,  6, 0 },     // 104
	{ 0x0b, 0x0f,  9, 0 },     // 105
	{ 0x0b, 0x0f,  3, 0 },     // 106
	{ 0x0d, 0x0d,  0, 0 },     // 107
	{ 0x0d, 0x0d,  6, 0 },     // 108
	{ 0x0d, 0x0d, 12, 0 },     // 109
	{ 0x0d, 0x0e,  6, 0 },     // 110
	{ 0x0d, 0x0e,  0, 0 },     // 111
	{ 0x0d, 0x0e,  6, 0 },     // 112
	{ 0x0e, 0x0e, 12, 0 },     // 113
	{ 0x0e, 0x0e,  6, 0 },     // 114
	{ 0x0e, 0x0e,  0, 0 },     // 115
	{ 0x0c, 0x10,  3, 0 },     // 116
	{ 0x0c, 0x10,  9, 0 },     // 117
	{ 0x0e, 0x0f,  6, 0 },     // 118
	{ 0x0e, 0x0f,  0, 0 },     // 119
	{ 0x0d, 0x10,  3, 0 },     // 120
	{ 0x0d, 0x10,  9, 0 },     // 121
	{ 0x0f, 0x0f,  6, 0 },     // 122
	{ 0x0f, 0x0f,  0, 0 },     // 123
	{ 0x0e, 0x10,  3, 0 },     // 124
	{ 0x0e, 0x10,  9, 0 },     // 125
	{ 0x0f, 0x10,  9, 0 },     // 126
	{ 0x0f, 0x10,  3, 0 },     // 127
	{ 0x0f, 0x10,  3, 0 },     // 128
	{ 0x0f, 0x10,  9, 0 },     // 129
	{ 0x0f, 0x11,  9, 0 },     // 130
	{ 0x0f, 0x11,  3, 0 },     // 131
	{ 0x10, 0x10,  0, 0 },     // 132
	{ 0x10, 0x10,  6, 0 },     // 133
	{ 0x0f, 0x12,  9, 0 },     // 134
	{ 0x0f, 0x12,  3, 0 },     // 135
	{ 0x10, 0x11,  0, 0 },     // 136
	{ 0x10, 0x11,  6, 0 },     // 137
	{ 0x0f, 0x13,  9, 0 },     // 138
	{ 0x0f, 0x13,  3, 0 },     // 139
	{ 0x11, 0x11,  0, 0 },     // 140
	{ 0x11, 0x11,  6, 0 },     // 141
	{ 0x11, 0x11, 12, 0 },     // 142
	{ 0x11, 0x12,  6, 0 },     // 143
	{ 0x11, 0x12,  0, 0 },     // 144
	{ 0x11, 0x12,  6, 0 },     // 145
	{ 0x12, 0x12, 12, 0 },     // 146
	{ 0x12, 0x12,  6, 0 },     // 147
	{ 0x12, 0x12,  0, 0 },     // 148
	{ 0x10, 0x14,  3, 0 },     // 149
	{ 0x10, 0x14,  9, 0 },     // 150
	{ 0x12, 0x13,  6, 0 },     // 151
	{ 0x12, 0x13,  0, 0 },     // 152
	{ 0x11, 0x14,  3, 0 },     // 153
	{ 0x11, 0x14,  9, 0 },     // 154
	{ 0x13, 0x13,  6, 0 },     // 155
	{ 0x13, 0x13,  0, 0 },     // 156
	{ 0x12, 0x14,  3, 0 },     // 157
	{ 0x12, 0x14,  9, 0 },     // 158
	{ 0x13, 0x14,  9, 0 },     // 159
	{ 0x13, 0x14,  3, 0 },     // 160
	{ 0x13, 0x14,  3, 0 },     // 161
	{ 0x13, 0x14,  9, 0 },     // 162
	{ 0x13, 0x15,  9, 0 },     // 163
	{ 0x13, 0x15,  3, 0 },     // 164
	{ 0x14, 0x14,  0, 0 },     // 165
	{ 0x14, 0x14,  6, 0 },     // 166
	{ 0x13, 0x16,  9, 0 },     // 167
	{ 0x13, 0x16,  3, 0 },     // 168
	{ 0x14, 0x15,  0, 0 },     // 169
	{ 0x14, 0x15,  6, 0 },     // 170
	{ 0x13, 0x17,  9, 0 },     // 171
	{ 0x13, 0x17,  3, 0 },     // 172
	{ 0x15, 0x15,  0, 0 },     // 173
	{ 0x15, 0x15,  6, 0 },     // 174
	{ 0x15, 0x15, 12, 0 },     // 175
	{ 0x15, 0x16,  6, 0 },     // 176
	{ 0x15, 0x16,  0, 0 },     // 177
	{ 0x15, 0x16,  6, 0 },     // 178
	{ 0x16, 0x16, 12, 0 },     // 179
	{ 0x16, 0x16,  6, 0 },     // 180
	{ 0x16, 0x16,  0, 0 },     // 181
	{ 0x14, 0x18,  3, 0 },     // 182
	{ 0x14, 0x18,  9, 0 },     // 183
	{ 0x16, 0x17,  6, 0 },     // 184
	{ 0x16, 0x17,  0, 0 },     // 185
	{ 0x15, 0x18,  3, 0 },     // 186
	{ 0x15, 0x18,  9, 0 },     // 187
	{ 0x17, 0x17,  6, 0 },     // 188
	{ 0x17, 0x17,  0, 0 },     // 189
	{ 0x16, 0x18,  3, 0 },     // 190
	{ 0x16, 0x18,  9, 0 },     // 191
	{ 0x17, 0x18,  9, 0 },     // 192
	{ 0x17, 0x18,  3, 0 },     // 193
	{ 0x17, 0x18,  3, 0 },     // 194
	{ 0x17, 0x18,  9, 0 },     // 195
	{ 0x17, 0x19,  9, 0 },     // 196
	{ 0x17, 0x19,  3, 0 },     // 197
	{ 0x18, 0x18,  0, 0 },     // 198
	{ 0x18, 0x18,  6, 0 },     // 199
	{ 0x17, 0x1a,  9, 0 },     // 200
	{ 0x17, 0x1a,  3, 0 },     // 201
	{ 0x18, 0x19,  0, 0 },     // 202
	{ 0x18, 0x19,  6, 0 },     // 203
	{ 0x17, 0x1b,  9, 0 },     // 204
	{ 0x17, 0x1b,  3, 0 },     // 205
	{ 0x19, 0x19,  0, 0 },     // 206
	{ 0x19, 0x19,  6, 0 },     // 207
	{ 0x19, 0x19, 12, 0 },     // 208
	{ 0x19, 0x1a,  6, 0 },     // 209
	{ 0x19, 0x1a,  0, 0 },     // 210
	{ 0x19, 0x1a,  6, 0 },     // 211
	{ 0x1a, 0x1a, 12, 0 },     // 212
	{ 0x1a, 0x1a,  6, 0 },     // 213
	{ 0x1a, 0x1a,  0, 0 },     // 214
	{ 0x18, 0x1c,  3, 0 },     // 215
	{ 0x18, 0x1c,  9, 0 },     // 216
	{ 0x1a, 0x1b,  6, 0 },     // 217
	{ 0x1a, 0x1b,  0, 0 },     // 218
	{ 0x19, 0x1c,  3, 0 },     // 219
	{ 0x19, 0x1c,  9, 0 },     // 220
	{ 0x1b, 0x1b,  6, 0 },     // 221
	{ 0x1b, 0x1b,  0, 0 },     // 222
	{ 0x1a, 0x1c,  3, 0 },     // 223
	{ 0x1a, 0x1c,  9, 0 },     // 224
	{ 0x1b, 0x1c,  9, 0 },     // 225
	{ 0x1b, 0x1c,  3, 0 },     // 226
	{ 0x1b, 0x1c,  3, 0 },     // 227
	{ 0x1b, 0x1c,  9, 0 },     // 228
	{ 0x1b, 0x1d,  9, 0 },     // 229
	{ 0x1b, 0x1d,  3, 0 },     // 230
	{ 0x1c, 0x1c,  0, 0 },     // 231
	{ 0x1c, 0x1c,  6, 0 },     // 232
	{ 0x1b, 0x1e,  9, 0 },     // 233
	{ 0x1b, 0x1e,  3, 0 },     // 234
	{ 0x1c, 0x1d,  0, 0 },     // 235
	{ 0x1c, 0x1d,  6, 0 },     // 236
	{ 0x1b, 0x1f,  9, 0 },     // 237
	{ 0x1b, 0x1f,  3, 0 },     // 238
	{ 0x1d, 0x1d,  0, 0 },     // 239
	{ 0x1d, 0x1d,  6, 0 },     // 240
	{ 0x1d, 0x1d, 12, 0 },     // 241
	{ 0x1d, 0x1e,  6, 0 },     // 242
	{ 0x1d, 0x1e,  0, 0 },     // 243
	{ 0x1d, 0x1e,  6, 0 },     // 244
	{ 0x1e, 0x1e, 12, 0 },     // 245
	{ 0x1e, 0x1e,  6, 0 },     // 246
	{ 0x1e, 0x1e,  0, 0 },     // 247
	{ 0x1e, 0x1e,  6, 0 },     // 248
	{ 0x1e, 0x1e, 12, 0 },     // 249
	{ 0x1e, 0x1f,  6, 0 },     // 250
	{ 0x1e, 0x1f,  0, 0 },     // 251
	{ 0x1e, 0x1f,  6, 0 },     // 252
	{ 0x1f, 0x1f, 12, 0 },     // 253
	{ 0x1f, 0x1f,  6, 0 },     // 254
	{ 0x1f, 0x1f,  0, 0 }      // 255
};

static const OptimalEncodingPair OptimalEncoding6bit_1_2[256] =
{
	{ 0x00, 0x00,  0, 0 },     //   0
	{ 0x00, 0x00,  6, 0 },     //   1
	{ 0x00, 0x01,  0, 0 },     //   2
	{ 0x01, 0x01,  6, 0 },     //   3
	{ 0x01, 0x01,  0, 0 },     //   4
	{ 0x01, 0x01,  6, 0 },     //   5
	{ 0x01, 0x02,  0, 0 },     //   6
	{ 0x02, 0x02,  6, 0 },     //   7
	{ 0x02, 0x02,  0, 0 },     //   8
	{ 0x02, 0x02,  6, 0 },     //   9
	{ 0x02, 0x03,  0, 0 },     //  10
	{ 0x03, 0x03,  6, 0 },     //  11
	{ 0x03, 0x03,  0, 0 },     //  12
	{ 0x03, 0x03,  6, 0 },     //  13
	{ 0x03, 0x04,  0, 0 },     //  14
	{ 0x04, 0x04,  6, 0 },     //  15
	{ 0x04, 0x04,  0, 0 },     //  16
	{ 0x04, 0x04,  6, 0 },     //  17
	{ 0x04, 0x05,  0, 0 },     //  18
	{ 0x05, 0x05,  6, 0 },     //  19
	{ 0x05, 0x05,  0, 0 },     //  20
	{ 0x05, 0x05,  6, 0 },     //  21
	{ 0x05, 0x06,  0, 0 },     //  22
	{ 0x06, 0x06,  6, 0 },     //  23
	{ 0x06, 0x06,  0, 0 },     //  24
	{ 0x06, 0x06,  6, 0 },     //  25
	{ 0x06, 0x07,  0, 0 },     //  26
	{ 0x07, 0x07,  6, 0 },     //  27
	{ 0x07, 0x07,  0, 0 },     //  28
	{ 0x07, 0x07,  6, 0 },     //  29
	{ 0x07, 0x08,  0, 0 },     //  30
	{ 0x08, 0x08,  6, 0 },     //  31
	{ 0x08, 0x08,  0, 0 },     //  32
	{ 0x00, 0x10,  3, 0 },     //  33
	{ 0x08, 0x09,  0, 0 },     //  34
	{ 0x01, 0x10,  3, 0 },     //  35
	{ 0x09, 0x09,  0, 0 },     //  36
	{ 0x02, 0x10,  3, 0 },     //  37
	{ 0x09, 0x0a,  0, 0 },     //  38
	{ 0x03, 0x10,  3, 0 },     //  39
	{ 0x0a, 0x0a,  0, 0 },     //  40
	{ 0x04, 0x10,  3, 0 },     //  41
	{ 0x0a, 0x0b,  0, 0 },     //  42
	{ 0x05, 0x10,  3, 0 },     //  43
	{ 0x0b, 0x0b,  0, 0 },     //  44
	{ 0x06, 0x10,  3, 0 },     //  45
	{ 0x0b, 0x0c,  0, 0 },     //  46
	{ 0x07, 0x10,  3, 0 },     //  47
	{ 0x0c, 0x0c,  0, 0 },     //  48
	{ 0x08, 0x10,  3, 0 },     //  49
	{ 0x0c, 0x0d,  0, 0 },     //  50
	{ 0x09, 0x10,  3, 0 },     //  51
	{ 0x0d, 0x0d,  0, 0 },     //  52
	{ 0x0a, 0x10,  3, 0 },     //  53
	{ 0x0d, 0x0e,  0, 0 },     //  54
	{ 0x0b, 0x10,  3, 0 },     //  55
	{ 0x0e, 0x0e,  0, 0 },     //  56
	{ 0x0c, 0x10,  3, 0 },     //  57
	{ 0x0e, 0x0f,  0, 0 },     //  58
	{ 0x0d, 0x10,  3, 0 },     //  59
	{ 0x0f, 0x0f,  0, 0 },     //  60
	{ 0x0e, 0x10,  3, 0 },     //  61
	{ 0x0f, 0x10,  3, 0 },     //  62
	{ 0x0f, 0x10,  3, 0 },     //  63
	{ 0x0f, 0x11,  3, 0 },     //  64
	{ 0x10, 0x10,  0, 0 },     //  65
	{ 0x0f, 0x12,  3, 0 },     //  66
	{ 0x10, 0x11,  0, 0 },     //  67
	{ 0x0f, 0x13,  3, 0 },     //  68
	{ 0x11, 0x11,  0, 0 },     //  69
	{ 0x0f, 0x14,  3, 0 },     //  70
	{ 0x11, 0x12,  0, 0 },     //  71
	{ 0x0f, 0x15,  3, 0 },     //  72
	{ 0x12, 0x12,  0, 0 },     //  73
	{ 0x0f, 0x16,  3, 0 },     //  74
	{ 0x12, 0x13,  0, 0 },     //  75
	{ 0x0f, 0x17,  3, 0 },     //  76
	{ 0x13, 0x13,  0, 0 },     //  77
	{ 0x0f, 0x18,  3, 0 },     //  78
	{ 0x13, 0x14,  0, 0 },     //  79
	{ 0x0f, 0x19,  3, 0 },     //  80
	{ 0x14, 0x14,  0, 0 },     //  81
	{ 0x0f, 0x1a,  3, 0 },     //  82
	{ 0x14, 0x15,  0, 0 },     //  83
	{ 0x0f, 0x1b,  3, 0 },     //  84
	{ 0x15, 0x15,  0, 0 },     //  85
	{ 0x0f, 0x1c,  3, 0 },     //  86
	{ 0x15, 0x16,  0, 0 },     //  87
	{ 0x0f, 0x1d,  3, 0 },     //  88
	{ 0x16, 0x16,  0, 0 },     //  89
	{ 0x0f, 0x1e,  3, 0 },     //  90
	{ 0x16, 0x17,  0, 0 },     //  91
	{ 0x0f, 0x1f,  3, 0 },     //  92
	{ 0x17, 0x17,  0, 0 },     //  93
	{ 0x17, 0x17,  6, 0 },     //  94
	{ 0x17, 0x18,  0, 0 },     //  95
	{ 0x18, 0x18,  6, 0 },     //  96
	{ 0x18, 0x18,  0, 0 },     //  97
	{ 0x10, 0x20,  3, 0 },     //  98
	{ 0x18, 0x19,  0, 0 },     //  99
	{ 0x11, 0x20,  3, 0 },     // 100
	{ 0x19, 0x19,  0, 0 },     // 101
	{ 0x12, 0x20,  3, 0 },     // 102
	{ 0x19, 0x1a,  0, 0 },     // 103
	{ 0x13, 0x20,  3, 0 },     // 104
	{ 0x1a, 0x1a,  0, 0 },     // 105
	{ 0x14, 0x20,  3, 0 },     // 106
	{ 0x1a, 0x1b,  0, 0 },     // 107
	{ 0x15, 0x20,  3, 0 },     // 108
	{ 0x1b, 0x1b,  0, 0 },     // 109
	{ 0x16, 0x20,  3, 0 },     // 110
	{ 0x1b, 0x1c,  0, 0 },     // 111
	{ 0x17, 0x20,  3, 0 },     // 112
	{ 0x1c, 0x1c,  0, 0 },     // 113
	{ 0x18, 0x20,  3, 0 },     // 114
	{ 0x1c, 0x1d,  0, 0 },     // 115
	{ 0x19, 0x20,  3, 0 },     // 116
	{ 0x1d, 0x1d,  0, 0 },     // 117
	{ 0x1a, 0x20,  3, 0 },     // 118
	{ 0x1d, 0x1e,  0, 0 },     // 119
	{ 0x1b, 0x20,  3, 0 },     // 120
	{ 0x1e, 0x1e,  0, 0 },     // 121
	{ 0x1c, 0x20,  3, 0 },     // 122
	{ 0x1e, 0x1f,  0, 0 },     // 123
	{ 0x1d, 0x20,  3, 0 },     // 124
	{ 0x1f, 0x1f,  0, 0 },     // 125
	{ 0x1e, 0x20,  3, 0 },     // 126
	{ 0x1f, 0x20,  3, 0 },     // 127
	{ 0x1f, 0x20,  3, 0 },     // 128
	{ 0x1f, 0x21,  3, 0 },     // 129
	{ 0x20, 0x20,  0, 0 },     // 130
	{ 0x1f, 0x22,  3, 0 },     // 131
	{ 0x20, 0x21,  0, 0 },     // 132
	{ 0x1f, 0x23,  3, 0 },     // 133
	{ 0x21, 0x21,  0, 0 },     // 134
	{ 0x1f, 0x24,  3, 0 },     // 135
	{ 0x21, 0x22,  0, 0 },     // 136
	{ 0x1f, 0x25,  3, 0 },     // 137
	{ 0x22, 0x22,  0, 0 },     // 138
	{ 0x1f, 0x26,  3, 0 },     // 139
	{ 0x22, 0x23,  0, 0 },     // 140
	{ 0x1f, 0x27,  3, 0 },     // 141
	{ 0x23, 0x23,  0, 0 },     // 142
	{ 0x1f, 0x28,  3, 0 },     // 143
	{ 0x23, 0x24,  0, 0 },     // 144
	{ 0x1f, 0x29,  3, 0 },     // 145
	{ 0x24, 0x24,  0, 0 },     // 146
	{ 0x1f, 0x2a,  3, 0 },     // 147
	{ 0x24, 0x25,  0, 0 },     // 148
	{ 0x1f, 0x2b,  3, 0 },     // 149
	{ 0x25, 0x25,  0, 0 },     // 150
	{ 0x1f, 0x2c,  3, 0 },     // 151
	{ 0x25, 0x26,  0, 0 },     // 152
	{ 0x1f, 0x2d,  3, 0 },     // 153
	{ 0x26, 0x26,  0, 0 },     // 154
	{ 0x1f, 0x2e,  3, 0 },     // 155
	{ 0x26, 0x27,  0, 0 },     // 156
	{ 0x1f, 0x2f,  3, 0 },     // 157
	{ 0x27, 0x27,  0, 0 },     // 158
	{ 0x27, 0x27,  6, 0 },     // 159
	{ 0x27, 0x28,  0, 0 },     // 160
	{ 0x28, 0x28,  6, 0 },     // 161
	{ 0x28, 0x28,  0, 0 },     // 162
	{ 0x20, 0x30,  3, 0 },     // 163
	{ 0x28, 0x29,  0, 0 },     // 164
	{ 0x21, 0x30,  3, 0 },     // 165
	{ 0x29, 0x29,  0, 0 },     // 166
	{ 0x22, 0x30,  3, 0 },     // 167
	{ 0x29, 0x2a,  0, 0 },     // 168
	{ 0x23, 0x30,  3, 0 },     // 169
	{ 0x2a, 0x2a,  0, 0 },     // 170
	{ 0x24, 0x30,  3, 0 },     // 171
	{ 0x2a, 0x2b,  0, 0 },     // 172
	{ 0x25, 0x30,  3, 0 },     // 173
	{ 0x2b, 0x2b,  0, 0 },     // 174
	{ 0x26, 0x30,  3, 0 },     // 175
	{ 0x2b, 0x2c,  0, 0 },     // 176
	{ 0x27, 0x30,  3, 0 },     // 177
	{ 0x2c, 0x2c,  0, 0 },     // 178
	{ 0x28, 0x30,  3, 0 },     // 179
	{ 0x2c, 0x2d,  0, 0 },     // 180
	{ 0x29, 0x30,  3, 0 },     // 181
	{ 0x2d, 0x2d,  0, 0 },     // 182
	{ 0x2a, 0x30,  3, 0 },     // 183
	{ 0x2d, 0x2e,  0, 0 },     // 184
	{ 0x2b, 0x30,  3, 0 },     // 185
	{ 0x2e, 0x2e,  0, 0 },     // 186
	{ 0x2c, 0x30,  3, 0 },     // 187
	{ 0x2e, 0x2f,  0, 0 },     // 188
	{ 0x2d, 0x30,  3, 0 },     // 189
	{ 0x2f, 0x2f,  0, 0 },     // 190
	{ 0x2e, 0x30,  3, 0 },     // 191
	{ 0x2f, 0x30,  3, 0 },     // 192
	{ 0x2f, 0x30,  3, 0 },     // 193
	{ 0x2f, 0x31,  3, 0 },     // 194
	{ 0x30, 0x30,  0, 0 },     // 195
	{ 0x2f, 0x32,  3, 0 },     // 196
	{ 0x30, 0x31,  0, 0 },     // 197
	{ 0x2f, 0x33,  3, 0 },     // 198
	{ 0x31, 0x31,  0, 0 },     // 199
	{ 0x2f, 0x34,  3, 0 },     // 200
	{ 0x31, 0x32,  0, 0 },     // 201
	{ 0x2f, 0x35,  3, 0 },     // 202
	{ 0x32, 0x32,  0, 0 },     // 203
	{ 0x2f, 0x36,  3, 0 },     // 204
	{ 0x32, 0x33,  0, 0 },     // 205
	{ 0x2f, 0x37,  3, 0 },     // 206
	{ 0x33, 0x33,  0, 0 },     // 207
	{ 0x2f, 0x38,  3, 0 },     // 208
	{ 0x33, 0x34,  0, 0 },     // 209
	{ 0x2f, 0x39,  3, 0 },     // 210
	{ 0x34, 0x34,  0, 0 },     // 211
	{ 0x2f, 0x3a,  3, 0 },     // 212
	{ 0x34, 0x35,  0, 0 },     // 213
	{ 0x2f, 0x3b,  3, 0 },     // 214
	{ 0x35, 0x35,  0, 0 },     // 215
	{ 0x2f, 0x3c,  3, 0 },     // 216
	{ 0x35, 0x36,  0, 0 },     // 217
	{ 0x2f, 0x3d,  3, 0 },     // 218
	{ 0x36, 0x36,  0, 0 },     // 219
	{ 0x2f, 0x3e,  3, 0 },     // 220
	{ 0x36, 0x37,  0, 0 },     // 221
	{ 0x2f, 0x3f,  3, 0 },     // 222
	{ 0x37, 0x37,  0, 0 },     // 223
	{ 0x37, 0x37,  6, 0 },     // 224
	{ 0x37, 0x38,  0, 0 },     // 225
	{ 0x38, 0x38,  6, 0 },     // 226
	{ 0x38, 0x38,  0, 0 },     // 227
	{ 0x38, 0x38,  6, 0 },     // 228
	{ 0x38, 0x39,  0, 0 },     // 229
	{ 0x39, 0x39,  6, 0 },     // 230
	{ 0x39, 0x39,  0, 0 },     // 231
	{ 0x39, 0x39,  6, 0 },     // 232
	{ 0x39, 0x3a,  0, 0 },     // 233
	{ 0x3a, 0x3a,  6, 0 },     // 234
	{ 0x3a, 0x3a,  0, 0 },     // 235
	{ 0x3a, 0x3a,  6, 0 },     // 236
	{ 0x3a, 0x3b,  0, 0 },     // 237
	{ 0x3b, 0x3b,  6, 0 },     // 238
	{ 0x3b, 0x3b,  0, 0 },     // 239
	{ 0x3b, 0x3b,  6, 0 },     // 240
	{ 0x3b, 0x3c,  0, 0 },     // 241
	{ 0x3c, 0x3c,  6, 0 },     // 242
	{ 0x3c, 0x3c,  0, 0 },     // 243
	{ 0x3c, 0x3c,  6, 0 },     // 244
	{ 0x3c, 0x3d,  0, 0 },     // 245
	{ 0x3d, 0x3d,  6, 0 },     // 246
	{ 0x3d, 0x3d,  0, 0 },     // 247
	{ 0x3d, 0x3d,  6, 0 },     // 248
	{ 0x3d, 0x3e,  0, 0 },     // 249
	{ 0x3e, 0x3e,  6, 0 },     // 250
	{ 0x3e, 0x3e,  0, 0 },     // 251
	{ 0x3e, 0x3e,  6, 0 },     // 252
	{ 0x3e, 0x3f,  0, 0 },     // 253
	{ 0x3f, 0x3f,  6, 0 },     // 254
	{ 0x3f, 0x3f,  0, 0 }      // 255
};

static const unsigned char Convert8bitTo4bit[256] =
{
	 0,     //   0
	 0,     //   1
	 0,     //   2
	 0,     //   3
	 0,     //   4
	 0,     //   5
	 0,     //   6
	 0,     //   7
	 0,     //   8
	 1,     //   9
	 1,     //  10
	 1,     //  11
	 1,     //  12
	 1,     //  13
	 1,     //  14
	 1,     //  15
	 1,     //  16
	 1,     //  17
	 1,     //  18
	 1,     //  19
	 1,     //  20
	 1,     //  21
	 1,     //  22
	 1,     //  23
	 1,     //  24
	 1,     //  25
	 2,     //  26
	 2,     //  27
	 2,     //  28
	 2,     //  29
	 2,     //  30
	 2,     //  31
	 2,     //  32
	 2,     //  33
	 2,     //  34
	 2,     //  35
	 2,     //  36
	 2,     //  37
	 2,     //  38
	 2,     //  39
	 2,     //  40
	 2,     //  41
	 2,     //  42
	 3,     //  43
	 3,     //  44
	 3,     //  45
	 3,     //  46
	 3,     //  47
	 3,     //  48
	 3,     //  49
	 3,     //  50
	 3,     //  51
	 3,     //  52
	 3,     //  53
	 3,     //  54
	 3,     //  55
	 3,     //  56
	 3,     //  57
	 3,     //  58
	 3,     //  59
	 4,     //  60
	 4,     //  61
	 4,     //  62
	 4,     //  63
	 4,     //  64
	 4,     //  65
	 4,     //  66
	 4,     //  67
	 4,     //  68
	 4,     //  69
	 4,     //  70
	 4,     //  71
	 4,     //  72
	 4,     //  73
	 4,     //  74
	 4,     //  75
	 4,     //  76
	 5,     //  77
	 5,     //  78
	 5,     //  79
	 5,     //  80
	 5,     //  81
	 5,     //  82
	 5,     //  83
	 5,     //  84
	 5,     //  85
	 5,     //  86
	 5,     //  87
	 5,     //  88
	 5,     //  89
	 5,     //  90
	 5,     //  91
	 5,     //  92
	 5,     //  93
	 6,     //  94
	 6,     //  95
	 6,     //  96
	 6,     //  97
	 6,     //  98
	 6,     //  99
	 6,     // 100
	 6,     // 101
	 6,     // 102
	 6,     // 103
	 6,     // 104
	 6,     // 105
	 6,     // 106
	 6,     // 107
	 6,     // 108
	 6,     // 109
	 6,     // 110
	 7,     // 111
	 7,     // 112
	 7,     // 113
	 7,     // 114
	 7,     // 115
	 7,     // 116
	 7,     // 117
	 7,     // 118
	 7,     // 119
	 7,     // 120
	 7,     // 121
	 7,     // 122
	 7,     // 123
	 7,     // 124
	 7,     // 125
	 7,     // 126
	 7,     // 127
	 8,     // 128
	 8,     // 129
	 8,     // 130
	 8,     // 131
	 8,     // 132
	 8,     // 133
	 8,     // 134
	 8,     // 135
	 8,     // 136
	 8,     // 137
	 8,     // 138
	 8,     // 139
	 8,     // 140
	 8,     // 141
	 8,     // 142
	 8,     // 143
	 8,     // 144
	 9,     // 145
	 9,     // 146
	 9,     // 147
	 9,     // 148
	 9,     // 149
	 9,     // 150
	 9,     // 151
	 9,     // 152
	 9,     // 153
	 9,     // 154
	 9,     // 155
	 9,     // 156
	 9,     // 157
	 9,     // 158
	 9,     // 159
	 9,     // 160
	 9,     // 161
	10,     // 162
	10,     // 163
	10,     // 164
	10,     // 165
	10,     // 166
	10,     // 167
	10,     // 168
	10,     // 169
	10,     // 170
	10,     // 171
	10,     // 172
	10,     // 173
	10,     // 174
	10,     // 175
	10,     // 176
	10,     // 177
	10,     // 178
	11,     // 179
	11,     // 180
	11,     // 181
	11,     // 182
	11,     // 183
	11,     // 184
	11,     // 185
	11,     // 186
	11,     // 187
	11,     // 188
	11,     // 189
	11,     // 190
	11,     // 191
	11,     // 192
	11,     // 193
	11,     // 194
	11,     // 195
	12,     // 196
	12,     // 197
	12,     // 198
	12,     // 199
	12,     // 200
	12,     // 201
	12,     // 202
	12,     // 203
	12,     // 204
	12,     // 205
	12,     // 206
	12,     // 207
	12,     // 208
	12,     // 209
	12,     // 210
	12,     // 211
	12,     // 212
	13,     // 213
	13,     // 214
	13,     // 215
	13,     // 216
	13,     // 217
	13,     // 218
	13,     // 219
	13,     // 220
	13,     // 221
	13,     // 222
	13,     // 223
	13,     // 224
	13,     // 225
	13,     // 226
	13,     // 227
	13,     // 228
	13,     // 229
	14,     // 230
	14,     // 231
	14,     // 232
	14,     // 233
	14,     // 234
	14,     // 235
	14,     // 236
	14,     // 237
	14,     // 238
	14,     // 239
	14,     // 240
	14,     // 241
	14,     // 242
	14,     // 243
	14,     // 244
	14,     // 245
	14,     // 246
	15,     // 247
	15,     // 248
	15,     // 249
	15,     // 250
	15,     // 251
	15,     // 252
	15,     // 253
	15,     // 254
	15      // 255
};





static int ComputeGreenError(const ColorBlock4x4& colors, const ColorBlockDXT1& block)
{
	int palette[4];
	palette[0] = (block.col0.g << 2) | (block.col0.g >> 4);
	palette[1] = (block.col1.g << 2) | (block.col1.g >> 4);
	palette[2] = (2 * palette[0] + palette[1] + 1) / 3;			// Sokov: '+1' to have better rounding after division by 3
	palette[3] = (2 * palette[1] + palette[0] + 1) / 3;			// Sokov: '+1' to have better rounding after division by 3

	int totalError = 0;

	for( unsigned int i = 0; i < 16; ++i )
	{
		const int green = colors.color(i).g;
		
		int error = abs(green - palette[0]);
		error = min(error, abs(green - palette[1]));
		error = min(error, abs(green - palette[2]));
		error = min(error, abs(green - palette[3]));
		
		totalError += error;
	}

	return totalError;
}

static uint32 ComputeGreenIndices(const ColorBlock4x4& colors, const ColorBGRA8 palette[4])
{
	const int color0 = palette[0].g;
	const int color1 = palette[1].g;
	const int color2 = palette[2].g;
	const int color3 = palette[3].g;
	
	uint32 indices = 0;
	for(unsigned int i = 0; i < 16; ++i)
	{
		const int color = colors.color(i).g;
		
		unsigned int d0 = abs(color0 - color);
		unsigned int d1 = abs(color1 - color);
		unsigned int d2 = abs(color2 - color);
		unsigned int d3 = abs(color3 - color);
		
		unsigned int b0 = d0 > d3;
		unsigned int b1 = d1 > d2;
		unsigned int b2 = d0 > d2;
		unsigned int b3 = d1 > d3;
		unsigned int b4 = d2 > d3;
		
		unsigned int x0 = b1 & b2;
		unsigned int x1 = b0 & b3;
		unsigned int x2 = b0 & b4;
		
		indices |= (x2 | ((x0 | x1) << 1)) << (2 * i);
	}

	return indices;
}

static int ComputeAlphaError(const ColorBlock4x4& colors, const AlphaBlockDXT5& block)
{
	uint8 alphas[8];
	block.evaluatePalette(alphas);

	int totalError = 0;

	for(unsigned int i = 0; i < 16; ++i)
	{
		const int alpha = colors.color(i).a;

		int besterror = 256*256;
		for(unsigned int p = 0; p < 8; ++p)
		{
			const int delta = (int)alphas[p] - alpha;
			int error = delta * delta;

			if (error < besterror)
			{
				besterror = error;
			}
		}

		totalError += besterror;
	}

	return totalError;
}


static void ComputeAlphaIndices(const ColorBlock4x4& colors, AlphaBlockDXT5& block)
{
	uint8 alphas[8];
	block.evaluatePalette(alphas);

	for(unsigned int i = 0; i < 16; ++i)
	{
		const int alpha = colors.color(i).a;

		int besterror = 256*256;
		unsigned int best;
		for(unsigned int p = 0; p < 8; ++p)
		{
			const int delta = (int)alphas[p] - alpha;
			int error = delta * delta;

			if (error < besterror)
			{
				besterror = error;
				best = p;
			}
		}

		block.setByIndex(i, best);		// FIXME: form whole 48-bit value here, then set it once. no need to call setByIndex() 16 times
	}
}





void OptimalCompressor::CompressDXT1FullyTransparent(ColorBlockDXT1& outBlock)
{
	outBlock.col0.u = 0;
	outBlock.col1.u = 0;
	outBlock.indices = 0xFFffFFff;
}


void OptimalCompressor::CompressDXT1SingleColor3(ColorBGRA8 const color, ColorBlockDXT1& outBlock)
{
	outBlock.col0.b = OptimalEncoding5bit_1_2[color.b].val0;
	outBlock.col0.g = OptimalEncoding6bit_1_2[color.g].val0;
	outBlock.col0.r = OptimalEncoding5bit_1_2[color.r].val0;

	outBlock.col1.b = OptimalEncoding5bit_1_2[color.b].val1;
	outBlock.col1.g = OptimalEncoding6bit_1_2[color.g].val1;
	outBlock.col1.r = OptimalEncoding5bit_1_2[color.r].val1;

	outBlock.indices = 0xAAaaAAaa;

	if( outBlock.col0.u > outBlock.col1.u )
	{
		std::swap( outBlock.col0.u, outBlock.col1.u );
	}
}


void OptimalCompressor::CompressDXT1SingleColor4(ColorBGRA8 const color, ColorBlockDXT1& outBlock)
{
	outBlock.col0.b = OptimalEncoding5bit_1_3[color.b].val0;
	outBlock.col0.g = OptimalEncoding6bit_1_3[color.g].val0;
	outBlock.col0.r = OptimalEncoding5bit_1_3[color.r].val0;

	outBlock.col1.b = OptimalEncoding5bit_1_3[color.b].val1;
	outBlock.col1.g = OptimalEncoding6bit_1_3[color.g].val1;
	outBlock.col1.r = OptimalEncoding5bit_1_3[color.r].val1;

	outBlock.indices = 0xAAaaAAaa;

	if( outBlock.col0.u < outBlock.col1.u )
	{
		std::swap( outBlock.col0.u, outBlock.col1.u );
		outBlock.indices = 0xFFffFFff;
	}
}


void OptimalCompressor::CompressDXT1SingleColor3or4(ColorBGRA8 const color, ColorBlockDXT1& outBlock)
{
	const unsigned int error_1_3 =
		  OptimalEncoding5bit_1_3[color.b].err
		+ OptimalEncoding6bit_1_3[color.g].err
		+ OptimalEncoding5bit_1_3[color.r].err;

	const unsigned int error_1_2 = 
		  OptimalEncoding5bit_1_2[color.b].err 
		+ OptimalEncoding6bit_1_2[color.g].err 
		+ OptimalEncoding5bit_1_2[color.r].err;

	if( error_1_3 <= error_1_2 )
    {
    	CompressDXT1SingleColor4( color, outBlock );
    }
    else
    {
    	CompressDXT1SingleColor3( color, outBlock );
    }
}


void OptimalCompressor::CompressDXT1Green(const ColorBlock4x4& colors, ColorBlockDXT1& outBlock)
{
	// Get min/max green.
	int ming = 256;
	int maxg = -1;
	for( unsigned int i = 0; i < 16; ++i )
	{
		const int value = colors.color(i).g;
		if(value < ming)
		{
			ming = value;
		}
		if(value > maxg)
		{
			maxg = value;
		}
	}

	if(ming==maxg)
	{
		OptimalCompressor::CompressDXT1SingleColor4( ColorBGRA8(0,ming,0,0), outBlock );
		return;
	}

	// find best green color palette
	{
		outBlock.col0.r = 0;
		outBlock.col0.b = 0;
		outBlock.col1.r = 0;
		outBlock.col1.b = 0;

		// round 8-bit min/max values to 6-bit values (with small expanding, to avoid missing good encoding pair)
		// FIXME: Sokov: we need to try expand ranges even more (it's possible to have lower error sometimes)
		{
			ming = (ming >> 2) - 1;
			if(ming < 0)
			{
				ming = 0;
			}

			maxg = (maxg >> 2) + 1;
			if(maxg > 63)
			{
				maxg = 63;
			}
		}

		int besterror = INT_MAX;			
		int bestmin = 0;
		int bestmax = 0;
		for( int maxvalue = ming; maxvalue <= maxg; ++maxvalue )
		{
			for( int minvalue = ming; minvalue <= maxvalue; ++minvalue )
			{
				outBlock.col0.g = maxvalue;
				outBlock.col1.g = minvalue;
				const int error = ComputeGreenError(colors, outBlock);

				if(error < besterror)
				{
					besterror = error;
					bestmin = minvalue;
					bestmax = maxvalue;
				}
			}
		}

		// Some bad decoders use 3-color palette instead of 4-color palette when color0<=color1,
		// so below we enforcing color0>color1
		if(bestmin==bestmax)
		{
			if(bestmin > 0)
			{
				--bestmin;
			}
			else
			{
				++bestmax;
			}
		}

		outBlock.col0.g = bestmax;
		outBlock.col1.g = bestmin;

		ColorBGRA8 palette[4];

		outBlock.evaluatePalette4KeepAlpha(palette);
		outBlock.indices = ComputeGreenIndices(colors, palette);
	}
}


void OptimalCompressor::CompressDXT3Alpha(const ColorBlock4x4& colors, AlphaBlockDXT3& outBlock)
{
	const ColorBGRA8* const c = colors.colors();

	outBlock.alpha0 = Convert8bitTo4bit[ c[ 0].a ];
	outBlock.alpha1 = Convert8bitTo4bit[ c[ 1].a ];
	outBlock.alpha2 = Convert8bitTo4bit[ c[ 2].a ];
	outBlock.alpha3 = Convert8bitTo4bit[ c[ 3].a ];
	outBlock.alpha4 = Convert8bitTo4bit[ c[ 4].a ];
	outBlock.alpha5 = Convert8bitTo4bit[ c[ 5].a ];
	outBlock.alpha6 = Convert8bitTo4bit[ c[ 6].a ];
	outBlock.alpha7 = Convert8bitTo4bit[ c[ 7].a ];
	outBlock.alpha8 = Convert8bitTo4bit[ c[ 8].a ];
	outBlock.alpha9 = Convert8bitTo4bit[ c[ 9].a ];
	outBlock.alphaA = Convert8bitTo4bit[ c[10].a ];
	outBlock.alphaB = Convert8bitTo4bit[ c[11].a ];
	outBlock.alphaC = Convert8bitTo4bit[ c[12].a ];
	outBlock.alphaD = Convert8bitTo4bit[ c[13].a ];
	outBlock.alphaE = Convert8bitTo4bit[ c[14].a ];
	outBlock.alphaF = Convert8bitTo4bit[ c[15].a ];
}


void OptimalCompressor::CompressDXT5Alpha(const ColorBlock4x4& colors, AlphaBlockDXT5& outBlock)
{
	// get the ranges for 6-alpha and 8-alpha interpolation
	int min6 = 256;
	int max6 = -1;
	int min8 = 256;
	int max8 = -1;
	
	for( unsigned int i = 0; i < 16; ++i )
	{
		const int value = colors.color(i).a;

		if( (value != 0) && (value < min6) )
		{
			min6 = value;
		}
		if( (value != 255) && (value > max6) )
		{
			max6 = value;
		}
		if( value < min8 )
		{
			min8 = value;
		}
		if( value > max8 )
		{
			max8 = value;
		}
	}
	
	// try to use 8-alpha lossless compression
	if( max8 - min8 < 8 )
	{
		if(max8 == min8)
		{
			if(max8 == 255)
			{
				outBlock.alpha0 = 255;
				outBlock.alpha1 = 254;
			}
			else
			{
				outBlock.alpha0 = max8+1;
				outBlock.alpha1 = max8;
			}
		}
		else
		{
			outBlock.alpha0 = max8;
			outBlock.alpha1 = min8;
		}

		ComputeAlphaIndices(colors, outBlock);

		return;
	}

	// try to use 6-alpha lossless compression
	if( max6 - min6 < 6 )
	{
		if(max6 < 0)
		{
			outBlock.alpha0 = 0;
			outBlock.alpha1 = 255;
		}
		else
		{
			outBlock.alpha0 = min6;
			outBlock.alpha1 = max6;
		}

		ComputeAlphaIndices(colors, outBlock);

		return;
	}
	
	// FIXME: Sokov: we need to try expand ranges a bit (it's possible to have lower error sometimes)

	// find best 8-alpha palette
	int besterror8 = INT_MAX;
	int bestmin8 = 0;
	int bestmax8 = 0;
	{
		for (int maxa = min8+7; maxa <= max8; ++maxa)
		{
			for (int mina = min8; mina <= maxa-7; ++mina)
			{
				outBlock.alpha0 = maxa;
				outBlock.alpha1 = mina;
				const int error = ComputeAlphaError(colors, outBlock);

				if (error < besterror8)
				{
					besterror8 = error;
					bestmin8 = mina;
					bestmax8 = maxa;
				}
			}
		}
	}

	// find best 6-alpha palette
	int besterror6 = INT_MAX;			
	int bestmin6 = 0;
	int bestmax6 = 0;
	{
		for (int maxa = min6+5; maxa <= max6; ++maxa)
		{
			for (int mina = min6; mina <= maxa-5; ++mina)
			{
				outBlock.alpha0 = mina;
				outBlock.alpha1 = maxa;
				const int error = ComputeAlphaError(colors, outBlock);

				if (error < besterror6)
				{
					besterror6 = error;
					bestmin6 = mina;
					bestmax6 = maxa;
				}
			}
		}
	}

	if(besterror6 < besterror8)
	{
		outBlock.alpha0 = bestmin6;
		outBlock.alpha1 = bestmax6;
	}
	else
	{
		outBlock.alpha0 = bestmax8;
		outBlock.alpha1 = bestmin8;
	}

	ComputeAlphaIndices(colors, outBlock);
}
