////////////////////////////////////////////////////////////////////////////
//
//  Crytek Engine Source File.
//  Copyright (C) Crytek GmbH
// -------------------------------------------------------------------------
//  File name:   ChunkCompiler.cpp
//  Version:     v1.00
//  Created:     5/04/2010 by Sergey Sokov
//  Compilers:   Visual Studio.NET
//  Description: 
// -------------------------------------------------------------------------
//  History:
//
////////////////////////////////////////////////////////////////////////////

#include "StdAfx.h"
#include "ConvertContext.h"
#include "iconfig.h"
#include "ChunkCompiler.h"
#include "ChunkFileHelpers.h"
#include "CryHeaders.h"
#include "ResourceCompiler.h"
										  

typedef CryStackStringT<char, 16> String;

typedef CHUNK_TABLE_ENTRY_0745 ChunkTableEntry;

struct ChunkTypeInfo
{
	uint32 type;
	const char* name;
	const char* errorText;
};

static ChunkTypeInfo s_chunkTypeInfo[] = 
{
#define CHUNK_TYPE_ENTRY(v, e) { v, #v, e }

	// Based on enum ChunkTypes from CryHeaders.h

	CHUNK_TYPE_ENTRY(ChunkType_ANY,              "unused generic type 0"),

	CHUNK_TYPE_ENTRY(ChunkType_Mesh,             ""),
	CHUNK_TYPE_ENTRY(ChunkType_Helper,           ""),
	CHUNK_TYPE_ENTRY(ChunkType_VertAnim,         ""),
	CHUNK_TYPE_ENTRY(ChunkType_BoneAnim,         ""),
	CHUNK_TYPE_ENTRY(ChunkType_GeomNameList,     "obsolete"),
	CHUNK_TYPE_ENTRY(ChunkType_BoneNameList,     ""),
	CHUNK_TYPE_ENTRY(ChunkType_MtlList,          "obsolete"),
	CHUNK_TYPE_ENTRY(ChunkType_MRM,              "obsolete"),
	CHUNK_TYPE_ENTRY(ChunkType_SceneProps,       ""),
	CHUNK_TYPE_ENTRY(ChunkType_Light,            ""),
	CHUNK_TYPE_ENTRY(ChunkType_PatchMesh,        "not implemented"),
	CHUNK_TYPE_ENTRY(ChunkType_Node,             ""),
	CHUNK_TYPE_ENTRY(ChunkType_Mtl,              ""),
	CHUNK_TYPE_ENTRY(ChunkType_Controller,       ""),
	CHUNK_TYPE_ENTRY(ChunkType_Timing,           ""),
	CHUNK_TYPE_ENTRY(ChunkType_BoneMesh,         ""),
	CHUNK_TYPE_ENTRY(ChunkType_BoneLightBinding, ""),
	CHUNK_TYPE_ENTRY(ChunkType_MeshMorphTarget,  ""),
	CHUNK_TYPE_ENTRY(ChunkType_BoneInitialPos,   ""),
	CHUNK_TYPE_ENTRY(ChunkType_SourceInfo,       ""),
	CHUNK_TYPE_ENTRY(ChunkType_MtlName,          ""),
	CHUNK_TYPE_ENTRY(ChunkType_ExportFlags,      ""),
	CHUNK_TYPE_ENTRY(ChunkType_DataStream,       ""),
	CHUNK_TYPE_ENTRY(ChunkType_MeshSubsets,      ""),
	CHUNK_TYPE_ENTRY(ChunkType_MeshPhysicsData,  ""),

	CHUNK_TYPE_ENTRY(ChunkType_CompiledBones,            ""),
	CHUNK_TYPE_ENTRY(ChunkType_CompiledPhysicalBones,    ""),
	CHUNK_TYPE_ENTRY(ChunkType_CompiledMorphTargets,     ""),
	CHUNK_TYPE_ENTRY(ChunkType_CompiledPhysicalProxies,  ""),
	CHUNK_TYPE_ENTRY(ChunkType_CompiledIntFaces,         ""),
	CHUNK_TYPE_ENTRY(ChunkType_CompiledIntSkinVertices,  ""),
	CHUNK_TYPE_ENTRY(ChunkType_CompiledExt2IntMap,       ""),

	CHUNK_TYPE_ENTRY(ChunkType_BreakablePhysics, ""),
	CHUNK_TYPE_ENTRY(ChunkType_FaceMap,          ""),
	CHUNK_TYPE_ENTRY(ChunkType_SpeedInfo,        ""),
	CHUNK_TYPE_ENTRY(ChunkType_FootPlantInfo,    ""),
	CHUNK_TYPE_ENTRY(ChunkType_BonesBoxes,       ""),
	CHUNK_TYPE_ENTRY(ChunkType_FoliageInfo,      ""),
	CHUNK_TYPE_ENTRY(ChunkType_Timestamp,        ""),
	CHUNK_TYPE_ENTRY(ChunkType_GlobalAnimationHeaderCAF, ""),
	CHUNK_TYPE_ENTRY(ChunkType_GlobalAnimationHeaderAIM, "")

#undef CHUNK_TYPE_ENTRY
};

static std::map<uint32, const ChunkTypeInfo*> s_chunkTypeToInfoMap;
static CryCriticalSectionRC s_chunkTypeToInfoMapLock;


//////////////////////////////////////////////////////////////////////////
CChunkCompiler::CChunkCompiler()
{
	m_refCount = 1;
}

//////////////////////////////////////////////////////////////////////////
CChunkCompiler::~CChunkCompiler()
{
}

//////////////////////////////////////////////////////////////////////////
// ICompiler + IConvertor methods.
//////////////////////////////////////////////////////////////////////////
void CChunkCompiler::Release()
{
	if (--m_refCount <= 0)
	{
		delete this;
	}
}

//////////////////////////////////////////////////////////////////////////
// ICompiler methods.
//////////////////////////////////////////////////////////////////////////
void CChunkCompiler::ConstructAndSetOutputFile(ConvertContext &cc)
{
	cc.SetOutputFile(cc.sourceFileFinal);
}

//////////////////////////////////////////////////////////////////////////
void CChunkCompiler::GetFilenameForUpToDateCheck(ConvertContext &cc, char* filenameBuffer, size_t bufferSize) const
{
	string const filename(cc.getOutputPath());

	if (filenameBuffer && (filename.length() < bufferSize))
	{
		strcpy(filenameBuffer, filename.c_str());		
	}
}

//////////////////////////////////////////////////////////////////////////
// IConvertor methods.
//////////////////////////////////////////////////////////////////////////
ICompiler* CChunkCompiler::CreateCompiler()
{
	// Only ever return one compiler, since we don't support multithreading. Since
	// the compiler is just this object, we can tell whether we have already returned
	// a compiler by checking the ref count.
	if (m_refCount >= 2)
		return 0;

	// Until we support multithreading for this convertor, the compiler and the
	// convertor may as well just be the same object.
	++m_refCount;
	return this;
}

//////////////////////////////////////////////////////////////////////////
bool CChunkCompiler::SupportsMultithreading() const
{
	return false;
}

//////////////////////////////////////////////////////////////////////////
int CChunkCompiler::GetNumPlatforms() const
{
	return 3;
}

//////////////////////////////////////////////////////////////////////////
EPlatform CChunkCompiler::GetPlatform(int index) const
{
	switch (index)
	{
	case 0:	return ePlatform_PC;
	case 1:	return ePlatform_X360;
	case 2:	return ePlatform_PS3;
	default: return ePlatform_UNKNOWN;
	}
}

//////////////////////////////////////////////////////////////////////////
int CChunkCompiler::GetNumExt() const 
{
	return 1; 
}

//////////////////////////////////////////////////////////////////////////
const char* CChunkCompiler::GetExt(int index) const 
{ 
	return "chunk"; 
}

//////////////////////////////////////////////////////////////////////////
IConvertContext* CChunkCompiler::CreateConvertContext() const 
{ 
	return new ConvertContext; 
}

//////////////////////////////////////////////////////////////////////////


//////////////////////////////////////////////////////////////////////////
// Auxiliary functions
//////////////////////////////////////////////////////////////////////////

// Implementation note:
// 
// vsnprintf() can produce weird results (on different systems) if the buffer is
// shorter than formatted output. See page http://perfec.to/vsnprintf/ for details.
// Below is a quote from the page describing the "buffer is short" behaviour:
// 
// A) vsnprintf() returns -1, but provides no clue as to how big the buffer should be. 
// B) vsnprintf() returns a value equal to the buffer size passed, but provides no other
//                clue about how big the buffer should be. 
// C) vsnprintf() returns a value equal to the buffer size passed, minus 1. 
//                This should indicate the formatted string, plus terminal \0 fit exactly.
//                But on some systems, the string was silently truncated to fit the
//                given buffer size. 
// D) vsnprintf() returns a value larger than the given buffer size.
//                In this case, the formatted string was truncated, but vsnprintf()
//                is telling us how big we need to make the buffer. 
// 
// So in the following function we try to handle all known strange cases.
//
template <typename StringType>
void appendVsprintf(StringType& s, const char* const format, va_list parg)
{
	if ((format == 0) || (format[0] == 0))
	{
		return;
	}

	char* buff = 0;
	size_t capacity = 0;

	size_t wantedCapacity = _vscprintf(format, parg);  // note: use strlen(format) if _vscprintf() is unavailable
	wantedCapacity += 2;  // to avoid getting case 'C)' in first run (see comments above the function)

	for (;;)
	{
		if (wantedCapacity > capacity)
		{
			delete [] buff;
			capacity = wantedCapacity;
			buff = new char[capacity + 1];
		}

		const int countWritten = _vsnprintf(buff, capacity, format, parg);

		if ((countWritten >= 0) && (capacity > (size_t)countWritten + 1))
		{
			// output was not truncated
			buff[countWritten] = 0;
			break;
		}

		wantedCapacity = capacity * 2;
	}

	s.append(buff);
	delete [] buff;
}


class Writer
{
public:
	enum EInfoType
	{
		eInfoType_Info    = BIT(0),
		eInfoType_Warning = BIT(1),
		eInfoType_Error   = BIT(2)
	};

	Writer()
		: m_f(0)
	{
	}

	~Writer()
	{
		finish();
	}

	bool start(int logInfoTypeMask, int fileInfoTypeMask, const char* filename)
	{
		finish();

		m_filename = (filename && filename[0]) ? filename : "";
		m_logInfoTypeMask = logInfoTypeMask;
		m_fileInfoTypeMask = fileInfoTypeMask;

		if (!m_filename.empty())
		{
			m_f = fopen(m_filename.c_str(), "wt");
			if (m_f == 0)
			{
				RCLogError("Cannot create file %s.", m_filename.c_str());
				return false;
			}
		}

		return true;
	}

	void finish()
	{
		if (m_f)
		{
			fclose(m_f);
			m_f = 0;
		}
	}

	void write(EInfoType const infoType, const char* const szFormat, ...)
	{
		if (((m_logInfoTypeMask | m_fileInfoTypeMask) & (int)infoType) == 0)
		{
			return;
		}

		String s;

		{
			va_list args;
			va_start(args, szFormat);
			appendVsprintf(s, szFormat, args);
			va_end(args);
		}

		if ((m_fileInfoTypeMask & (int)infoType) && m_f)
		{
			fprintf(m_f, "%s\n", s.c_str());
		}

		if (m_logInfoTypeMask & (int)infoType)
		{
			switch (infoType)
			{
			case eInfoType_Info:
				RCLog("%s", s.c_str());
				break;
			case eInfoType_Warning:
				RCLogWarning("%s", s.c_str());
				break;
			case eInfoType_Error:
			default:
				RCLogError("%s", s.c_str());
				break;
			}
		}
	}

private:
	int m_logInfoTypeMask;
	int m_fileInfoTypeMask;
	String m_filename;
	FILE* m_f;
};


static int qsortCompare_ChunkTableEntry_FileOffset(const void* p0, const void* p1)
{
	const ChunkTableEntry* const e1 = (const ChunkTableEntry*) p0;
	const ChunkTableEntry* const e2 = (const ChunkTableEntry*) p1;
	return e1->chdr.FileOffset - e2->chdr.FileOffset;
}


static void fillChunkTypeToInfoMap()
{
	CryAutoLockRC lock(s_chunkTypeToInfoMapLock);

	if (!s_chunkTypeToInfoMap.empty())
	{
		return;
	}

	const size_t chunkTypeCount = sizeof(s_chunkTypeInfo) / sizeof(s_chunkTypeInfo[0]);

	for (size_t i = 0; i < chunkTypeCount; ++i)
	{
		s_chunkTypeToInfoMap[s_chunkTypeInfo[i].type] = &s_chunkTypeInfo[i];
	}

	if (s_chunkTypeToInfoMap.size() != chunkTypeCount)
	{
		RCLogError("Some chunk types are duplicated - contact RC programmer");
		exit(-1);
	}
}


static String getFileHeaderError(const FILE_HEADER& fileHeader, const char* const filename)
{
	String err;

	if (memcmp(&fileHeader.Signature[0], FILE_SIGNATURE, sizeof(fileHeader.Signature)))
	{
		err.Format(
			"Signature in file header of file %s is not '%s'.",
			filename, FILE_SIGNATURE);
		return err;
	}

	if ((fileHeader.FileType != FileType_Geom) && (fileHeader.FileType != FileType_Anim))
	{
		err.Format(
			"File type specified in file header of file %s is neither FileType_Geom (0x%08x), nor FileType_Anim (0x%08x): it's an unknown value (0x%08x).",
			filename, FileType_Geom, FileType_Anim, fileHeader.FileType);
		return err;
	}

	return err;
}


const bool isInside(size_t offset0, size_t size0, size_t offset1, size_t size1)
{
	// check overflow in 'parent' segment sizes
	if (offset0 + size0 < offset0)
	{
		RCLogError("Unexpected data in isInside() - Contact RC programmer");
		return false;
	}

	// check overflow in 'child' segment sizes
	if (offset1 + size1 < offset1)
	{
		return false;
	}

	if ((offset1 < offset0) || (offset1 + size1 > offset0 + size0))
	{
		return false;
	}

	return true;
}


static bool scanChunkFile(const char* filename, Writer& w)
{
	fillChunkTypeToInfoMap();

	// This function assumes that the system is little-endian, so let's check it.
	// (note: changing this function to support big-endian systems is doable)
	bool bSystemIsLittleEndian;
	{
		const int32 a = 1;
		bSystemIsLittleEndian = (((const char*)&a)[0] == 1);
	}
	if (!bSystemIsLittleEndian)
	{
		w.write(Writer::eInfoType_Error, "Unexpected big-endian system. Contact RC programmer.");
		return false;
	}

	if ((filename == 0) || (filename[0] == 0))
	{
		w.write(Writer::eInfoType_Error, "Name of chunk file for dumping is damaged. Contact RC programmer.");
		return false;
	}

	CCryFile file;

	if (!file.Open(filename, "rb"))
	{
		w.write(Writer::eInfoType_Error, "File %s failed to open for reading", filename);
		return false;
	}

	w.write(Writer::eInfoType_Info, "<<< Dump of chunk file >>>");

	w.write(Writer::eInfoType_Info, "");
	w.write(Writer::eInfoType_Info, "FileName: %s", file.GetFilename());
	w.write(Writer::eInfoType_Info, "FileSize: %lu", (unsigned long)file.GetLength());

	//
	// The header
	//
	FILE_HEADER fileHeader;	
	bool bHeaderIsLittleEndian;

	w.write(Writer::eInfoType_Info, "");
	w.write(Writer::eInfoType_Info, "File header");
	{
		const size_t nRead = file.ReadRaw(&fileHeader, sizeof(fileHeader));
		if (nRead != sizeof(fileHeader))
		{
			w.write(Writer::eInfoType_Error,
				"*** Failed to read file header (%lu bytes) from file %s (file size is %lu bytes)",
				(unsigned long)sizeof(fileHeader), file.GetFilename(), (unsigned long)file.GetLength());
			return false;
		}

		const String errOriginalEndianness = getFileHeaderError(fileHeader, file.GetFilename());
		SwapEndianBase(&fileHeader, 1);		// temporarily change endianness of the header
		const String errChangedEndianness = getFileHeaderError(fileHeader, file.GetFilename());
		SwapEndianBase(&fileHeader, 1);		// restore endianness of the header

		if (errOriginalEndianness.empty() && errChangedEndianness.empty())
		{
			w.write(Writer::eInfoType_Error, "*** Unexpected file header parsing error - contact RC programmer");
			return false;
		}

		if ((!errOriginalEndianness.empty()) && (!errChangedEndianness.empty()))
		{
			w.write(Writer::eInfoType_Error, "*** %s", errOriginalEndianness.c_str());
			return false;
		}

		bHeaderIsLittleEndian = (errOriginalEndianness.empty() ? bSystemIsLittleEndian : !bSystemIsLittleEndian);

		if (bHeaderIsLittleEndian != bSystemIsLittleEndian)
		{
			SwapEndianBase(&fileHeader, 1);
		}

		w.write(Writer::eInfoType_Info, "  Endianness: %s", (bHeaderIsLittleEndian ? "little-endian" : "big-endian"));
		w.write(Writer::eInfoType_Info, "  FileType: %s", ((fileHeader.FileType == FileType_Geom) ? "FileType_Geom" : "FileType_Anim"));

		if ((fileHeader.Version != ChunkFileVersion) && (fileHeader.Version != ChunkFileVersion_Align))
		{
			w.write(Writer::eInfoType_Error, 
				"*** Chunk file version specified in file header of file %s is neither ChunkFileVersion (0x%08x), nor ChunkFileVersion_Align (0x%08x): it's an unknown value (0x%08x) '%s'.",
				file.GetFilename(), ChunkFileVersion, ChunkFileVersion_Align, fileHeader.Version);
			return false;
		}

		w.write(Writer::eInfoType_Info, "  Aligned: %s", ((fileHeader.Version == ChunkFileVersion_Align) ? "Yes" : "No"));

		w.write(Writer::eInfoType_Info, "  ChunkTableOffset: %d (0x%08x)", fileHeader.ChunkTableOffset, fileHeader.ChunkTableOffset);
	}

	//
	// The chunk table (count, then entries)
	//
	if (file.Seek(fileHeader.ChunkTableOffset, SEEK_SET) != 0)
	{
		w.write(Writer::eInfoType_Error, 
			"*** Chunk table offset (%d (0x%08x)) is bad in header of file %s", 
			fileHeader.ChunkTableOffset, fileHeader.ChunkTableOffset, file.GetFilename());
		return false;
	}

	uint32 chunkCount;
	{
		const size_t nRead = file.ReadRaw(&chunkCount, sizeof(chunkCount));
		if (nRead != sizeof(chunkCount))
		{
			w.write(Writer::eInfoType_Error, 
				"*** Failed to read chunk count from file %s (file size is %lu bytes)",
				file.GetFilename(), (unsigned long)file.GetLength());
			return false;
		}

		if (bHeaderIsLittleEndian != bSystemIsLittleEndian)
		{
			SwapEndianBase(&chunkCount, 1);
		}

		static const int MAX_CHUNKS_NUM = 10000;

		if ((chunkCount <= 0) || (chunkCount > MAX_CHUNKS_NUM))
		{
			w.write(Writer::eInfoType_Error, 
				"*** Chunk count (%d (0x%08x)) is bad in file %s", 
				chunkCount, chunkCount, file.GetFilename());
			return false;
		}
	}
	w.write(Writer::eInfoType_Info, "");
	w.write(Writer::eInfoType_Info, "Chunk count: %d", chunkCount);

	ChunkTableEntry* pChunks = new ChunkTableEntry[chunkCount];
	assert(pChunks);

	if (fileHeader.Version == ChunkFileVersion_Align)
	{
		const size_t nRead = file.ReadRaw(pChunks, chunkCount * sizeof(*pChunks));
		if (nRead != chunkCount * sizeof(*pChunks))
		{
			w.write(Writer::eInfoType_Error,  
				"*** Failed to read chunk table (%d entries, %d bytes each) from file %s (file size is %lu bytes)",
				(int)chunkCount, (int)sizeof(*pChunks), file.GetFilename(), (unsigned long)file.GetLength());
			delete [] pChunks;
			return false;
		}

		if (bHeaderIsLittleEndian != bSystemIsLittleEndian)
		{
			SwapEndianBase(pChunks, chunkCount);
		}
	}
	else
	{
		CHUNK_HEADER *pChunkHeaders = new CHUNK_HEADER[chunkCount];
		assert(pChunkHeaders);

		const size_t nRead = file.ReadRaw(pChunkHeaders, chunkCount * sizeof(*pChunkHeaders));
		if (nRead != chunkCount * sizeof(*pChunkHeaders))
		{
			w.write(Writer::eInfoType_Error, 
				"*** Failed to read chunk table (%d entries, %d bytes each) from file %s (file size is %lu bytes)",
				(int)chunkCount, (int)sizeof(*pChunkHeaders), file.GetFilename(), (unsigned long)file.GetLength());
			delete [] pChunkHeaders;
			delete [] pChunks;
			return false;
		}

		if (bHeaderIsLittleEndian != bSystemIsLittleEndian)
		{
			SwapEndianBase(pChunkHeaders, chunkCount);
		}

		for (size_t i = 0; i < chunkCount; ++i)
		{
			pChunks[i].chdr = pChunkHeaders[i];
			pChunks[i].ChunkSize = -1;
		}

		delete [] pChunkHeaders;

		qsort(pChunks, chunkCount, sizeof(*pChunks), qsortCompare_ChunkTableEntry_FileOffset);

		const size_t nEndOfChunks = (fileHeader.ChunkTableOffset == sizeof(fileHeader))
			? file.GetLength()
			: fileHeader.ChunkTableOffset;

		for (size_t i = 0; i < chunkCount; ++i)
		{
			// calculate the chunk size, based on the very next chunk with greater offset
			// or the end of the raw data portion of the file

			const size_t nextFileOffset = (i + 1 < chunkCount)
				? pChunks[i + 1].chdr.FileOffset
				: nEndOfChunks;

			pChunks[i].ChunkSize = nextFileOffset - pChunks[i].chdr.FileOffset;
		}
	}

	//
	// Check chunks' types
	//
	for (size_t i = 0; i < chunkCount; ++i)
	{
		const ChunkTableEntry& c = pChunks[i];

		const ChunkTypeInfo* pChunkTypeInfo = 0;
		{
			std::map<uint32, const ChunkTypeInfo*>::iterator it = s_chunkTypeToInfoMap.find(c.chdr.ChunkType);
			if (it != s_chunkTypeToInfoMap.end())
			{
				pChunkTypeInfo = it->second;
			}
		}

		const char* chunkTypeName = (pChunkTypeInfo == 0) ? "UNKNOWN_TYPE" : pChunkTypeInfo->name;

		String err;
		if (pChunkTypeInfo == 0)
		{
			err = "Unknown chunk type";
		}
		else if (pChunkTypeInfo->errorText[0])
		{
			err.Format("Bad chunk type (%s)", pChunkTypeInfo->errorText);
		}

		const bool bConsole = ((c.chdr.ChunkVersion & CONSOLE_VERSION_MASK) != 0);
		const uint32 chunkVersion = c.chdr.ChunkVersion & ~CONSOLE_VERSION_MASK;

		w.write(Writer::eInfoType_Info, 
			"%s  [%3d]: type: 0x%08x %-35s, version: 0x%08x, size: %7d, fileOffset: %9d, chunkID: %3d, console: %s",
			(err.empty() ? "" : "***"), (int)i, (unsigned int)c.chdr.ChunkType, chunkTypeName, chunkVersion, c.ChunkSize, c.chdr.FileOffset, c.chdr.ChunkID, (bConsole ? "Yes": "No"));

		if (!err.empty())
		{
			w.write(Writer::eInfoType_Error, 
				"*** %s", err.c_str());
			return false;
		}
	}

	//
	// Chunks
	//
	{
		const size_t fileSize = file.GetLength();
		std::vector<uchar> data;

		for (size_t i = 0; i < chunkCount; ++i)
		{
			const ChunkTableEntry& c = pChunks[i];

			if ((c.ChunkSize <= 0) || 
				(c.chdr.FileOffset <= 0) || 
				(!isInside(0, fileSize, (size_t)c.chdr.FileOffset, (size_t)c.ChunkSize)))
			{
				w.write(Writer::eInfoType_Error, 
					"Chunk [%d] has bad size and/or offset: size: %d, fileOffset: %d, fileSize: %lu",
					(int)i, c.ChunkSize, c.chdr.FileOffset, c.chdr.ChunkID, (unsigned long)fileSize);
				return false;
			}

			data.resize(c.ChunkSize);

			file.Seek(c.chdr.FileOffset, SEEK_SET);
			const size_t pos = file.GetPosition();
			if (pos != (size_t)c.chdr.FileOffset)
			{
				w.write(Writer::eInfoType_Error, 
					"Chunk [%d] seek error", (int)i);
				return false;
			}

			const size_t readen = file.ReadRaw(&data[0], c.ChunkSize);
			if (readen != (size_t)c.ChunkSize)
			{
				w.write(Writer::eInfoType_Error, 
					"Chunk [%d] reading error", (int)i);
				return false;
			}

			// Check chunk's data
			// TODO: write code here
		}
	}

	//
	// Check chunk IDs for uniqueness
	//
	{
		typedef std::set<int> ChunkIdSet;
		ChunkIdSet chunkIdSet;

		for (size_t i = 0; i < chunkCount; ++i)
		{
			chunkIdSet.insert(pChunks[i].chdr.ChunkID);
		}

		if (chunkCount != chunkIdSet.size())
		{
			const int duplicateCount = (int)(chunkCount - chunkIdSet.size());
			w.write(Writer::eInfoType_Error, 
				"*** %d duplicate chunk ID%s found in file %s", 
				duplicateCount, ((duplicateCount > 1) ? "s" : ""), file.GetFilename());
			delete [] pChunks;
			return false;
		}
	}

	delete [] pChunks;

	w.write(Writer::eInfoType_Info, "");
	w.write(Writer::eInfoType_Info, "<<< End of the dump >>>");
	w.write(Writer::eInfoType_Info, "");

	return true;
}


static bool debugDumpChunkFile(const char* filename)
{
	if ((filename == 0) || (filename[0] == 0))
	{
		RCLogError("Name of chunk file is empty. Contact RC programmer.");
		return false;
	}

	const String dumpFilename = String(filename) + ".dumpChunk";
	RCLog("Dumping chunk file %s -> %s...", filename, dumpFilename.c_str());

	Writer w;
	const int logInfoTypeMask = Writer::eInfoType_Warning | Writer::eInfoType_Error;
	const int fileInfoTypeMask = Writer::eInfoType_Info | Writer::eInfoType_Warning | Writer::eInfoType_Error;
	w.start(logInfoTypeMask, fileInfoTypeMask, dumpFilename);

	const bool ok = scanChunkFile(filename, w);

	w.finish();

	if (!ok)
	{
		return false;
	}

	RCLog("Dumping chunk file %s finished.", filename);

	return true;
}


static bool debugValidateChunkFile(const char* filename)
{
	if ((filename == 0) || (filename[0] == 0))
	{
		RCLogError("Name of chunk file is empty. Contact RC programmer.");
		return false;
	}

	Writer w;
	const int logInfoTypeMask = Writer::eInfoType_Warning | Writer::eInfoType_Error;
	const int fileInfoTypeMask = 0;
	w.start(logInfoTypeMask, fileInfoTypeMask, "");

	const bool ok = scanChunkFile(filename, w);

	w.finish();

	return ok;
}


//////////////////////////////////////////////////////////////////////////
// IConvertor methods.
//////////////////////////////////////////////////////////////////////////
bool CChunkCompiler::Process(ConvertContext &cc)
{
	const string sourceFile = cc.getSourcePath();
	const string outputFile = cc.getOutputPath();

	cc.pRC->AddOutputFile( cc.getOutputPath(),cc.getSourcePath() );

	try
	{
		if (cc.config->HasKey("SkipMissing"))
		{
			// Skip missing source files.
			const DWORD dwFileSpecAttr = GetFileAttributes(sourceFile.c_str());
			if (dwFileSpecAttr == 0xFFFFFFFF)
			{
				// Skip missing file instead of reporting it as an error.
				return true;
			}
		}

		{
			bool dump = false;
			if (cc.config->HasKey("debugdump"))
			{
				if (!cc.config->Get("debugdump",dump))
				{
					// No any value specified. Assume 'true'.
					dump = true;
				}
			}

			if (dump)
			{
				const bool ok = debugDumpChunkFile(sourceFile.c_str());
				return ok;
			}
		}

		{
			bool validate = false;
			if (cc.config->HasKey("debugvalidate"))
			{
				if (!cc.config->Get("debugvalidate",validate))
				{
					// No any value specified. Assume 'true'.
					validate = true;
				}
			}

			if (validate)
			{
				const bool ok = debugValidateChunkFile(sourceFile.c_str());
				return ok;
			}
		}

		if (cc.config->HasKey("RealignChunks"))
		{
			const bool ok = ChunkFileHelpers::RealignChunks(sourceFile, outputFile);
			return ok;
		}

		RCLogError("Chunk file %s: no any processing option was specified (available options are: 'debugdump', 'debugvalidate', 'RealignChunks')", sourceFile.c_str());
		return false;
	}
	catch (char*)
	{
		Beep(1000, 1000);
		RCLogError("Unexpected failure in processing %s - contact an RC programmer.", sourceFile.c_str());
		return false;
	}

	return true;
}
