#include "stdafx.h"
#include "XMLConverter.h"
#include "IRCLog.h"
#include "IXmlSerializer.h"
#include "XmlBinaryHeaders.h"
#include "XMLBinaryReader.h"
#include "XMLBinaryWriter.h"
#include "DbgHelp.h"
#include "IConfig.h"
#include "IResCompiler.h"
#include "StringHelpers.h"


static const char* s_extensions[] = 
{
	"xml"
};

XMLConverter::XMLConverter(ICryXML* pCryXML)
	: m_pCryXML(pCryXML)
	, m_refCount(1)
{
	m_pCryXML->AddRef();
}

XMLConverter::~XMLConverter()
{
	m_pCryXML->Release();
}

void XMLConverter::Release()
{
	if (--m_refCount <= 0)
		delete this;
}

void XMLConverter::Init(IConfig* config, const char* exePath)
{
	m_filter.clear();

	string xmlFilterFile;
	config->Get("xmlFilterFile", xmlFilterFile);
	if (xmlFilterFile.empty())
	{
		return;
	}

	FILE* const f = fopen(xmlFilterFile.c_str(),"rt");
	if (!f)
	{
		return;
	}

	char line[1024];
	while (fgets(line, sizeof(line), f) != NULL)
	{
		if (line[0])
		{
			string sLine = line;
			sLine.Trim();
			if (!sLine.empty() && (::tolower(sLine[0]) == 'a' || ::tolower(sLine[0]) == 'e'))
			{
				FilterElement fe;
				fe.type = (::tolower(sLine[0]) == 'a') 
					? XMLBinary::IFilter::eType_AttributeName 
					: XMLBinary::IFilter::eType_ElementName;
				sLine = sLine.substr(1, sLine.npos);
				sLine.TrimLeft();
				if (!sLine.empty() && (sLine[0] == '+' || sLine[0] == '-'))
				{
					fe.bAccept = (sLine[0] == '+');
					sLine = sLine.substr(1, sLine.npos);
					sLine.TrimLeft();
					if (!sLine.empty())
					{
						fe.wildcards = sLine;
						m_filter.push_back(fe);
					}
				}
			}
		}
	}

	fclose(f);
}

//////////////////////////////////////////////////////////////////////////
class CXmlBinaryDataWriterFile : public XMLBinary::IDataWriter
{
public:
	CXmlBinaryDataWriterFile( const char *file ) { m_file = fopen( file,"wb" ); }
	~CXmlBinaryDataWriterFile() { if (m_file) fclose( m_file ); }
	virtual bool IsOk() { return m_file != 0; }
	virtual void Write(const void* pData, size_t size) { if (m_file) fwrite( pData,size,1,m_file ); }
private:
	FILE *m_file;
};

//////////////////////////////////////////////////////////////////////////
class CXmlFilter : public XMLBinary::IFilter
{
public:
	CXmlFilter(std::vector<XMLConverter::FilterElement>* pFilterArray)
		: m_pFilterArray((pFilterArray && !pFilterArray->empty()) ? pFilterArray : 0)
	{
	}

	virtual bool IsAccepted(EType type, const char* pName) const
	{
		if (!m_pFilterArray)
		{
			return true;
		}

		const string name(pName);
		for (size_t i = 0; i < m_pFilterArray->size(); ++i)
		{
			if (((*m_pFilterArray)[i].type == type) &&
				StringHelpers::MatchesWildcardsIgnoreCase(name, (*m_pFilterArray)[i].wildcards))
			{
				return (*m_pFilterArray)[i].bAccept;
			}
		}

		return true;
	}

private:
	std::vector<XMLConverter::FilterElement>* m_pFilterArray;
};

//////////////////////////////////////////////////////////////////////////
static bool xmlsAreEqual(XmlNodeRef node0, XmlNodeRef node1, XMLBinary::IFilter* pFilter)
{
	{
		const char* const tag0 = node0->getTag();
		const char* const tag1 = node1->getTag();

		if (strcmp(tag0, tag1) != 0)
		{
			return false;
		}
	}

	{
		const char* const content0 = node0->getContent();
		const char* const content1 = node1->getContent();

		if (strcmp(content0, content1) != 0)
		{
			return false;
		}
	}

	{
		const int count0 = node0->getNumAttributes();
		const int count1 = node1->getNumAttributes();
		int index1 = 0;
		int filteredCount1 = 0;

		for (int index0 = 0; index0 < count0; ++index0)
		{
			const char* key0;
			const char* value0;
			node0->getAttributeByIndex(index0, &key0, &value0);

			const char* key1;
			const char* value1;
			for (;;)
			{
				if (index1 >= count1)
				{
					return false;
				}
				node1->getAttributeByIndex(index1++, &key1, &value1);
				if (!pFilter || pFilter->IsAccepted(XMLBinary::IFilter::eType_AttributeName, key1))
				{
					++filteredCount1;
					break;
				}
			}

			if ((strcmp(key0, key1) != 0) ||
				(strcmp(value0, value1) != 0))
			{
				return false;
			}
		}

		if (count0 != filteredCount1)
		{
			return false;
		}
	}

	{
		const int count0 = node0->getChildCount();
		const int count1 = node1->getChildCount();
		int index1 = 0;
		int filteredCount1 = 0;

		for (int index0 = 0; index0 < count0; ++index0)
		{
			XmlNodeRef child0 = node0->getChild(index0);

			XmlNodeRef child1;
			for (;;)
			{
				if (index1 >= count1)
				{
					return false;
				}
				child1 = node1->getChild(index1++);
				if (!pFilter || pFilter->IsAccepted(XMLBinary::IFilter::eType_ElementName, child1->getTag()))
				{
					++filteredCount1;
					break;
				}
			}

			const bool bEqual = xmlsAreEqual(child0, child1, pFilter);
			if (!bEqual)
			{
				return false;
			}
		}

		if (count0 != filteredCount1)
		{
			return false;
		}
	}

	return true;
}


bool XMLConverter::Process( ConvertContext &cc )
{
	const bool bNeedSwapEndian = (cc.platform == ePlatform_X360 || cc.platform == ePlatform_PS3);
	if (bNeedSwapEndian)
	{
		RCLog("XML: Endian conversion specified");
	}

	// Get the files to process.
	const string sOutputFile = cc.getOutputPath();
	const string sInputFile = cc.getSourcePath();

	cc.pRC->AddOutputFile( cc.getOutputPath(),cc.getSourcePath() );

	// Get the xml serializer.
	IXMLSerializer* pSerializer = m_pCryXML->GetXMLSerializer();

	// Read in the input file.
	XmlNodeRef root;
	{
		FILE* pSourceFile = fopen(sInputFile.c_str(), "rb");
		if (pSourceFile == 0)
		{
			RCLogError("XML: Cannot open file \"%s\": %s\n", sInputFile.c_str(), strerror(errno));
			return false;
		}
		fclose(pSourceFile);
		const bool bRemoveNonessentialSpacesFromContent = true;
		char szErrorBuffer[1024];
		szErrorBuffer[0] = 0;
		root = pSerializer->Read(FileXmlBufferSource(sInputFile.c_str()), bRemoveNonessentialSpacesFromContent, sizeof(szErrorBuffer), szErrorBuffer);
		if (!root)
		{
			const char* const pErrorStr = 
				(szErrorBuffer[0])
				? &szErrorBuffer[0] 
				: "Probably this file has bad XML syntax or it's not XML file at all.";
			RCLogError("XML: Cannot read file \"%s\": %s.\n", sInputFile.c_str(), pErrorStr);
			return false;
		}
	}

	// Create filter to get rid of unneded elements and attributes
	CXmlFilter filter(&m_filter);

	// Write out the destination file.
	{
		FILE* pDestinationFile = fopen(sOutputFile.c_str(), "wb");
		if (pDestinationFile == 0)
		{
			RCLogError("XML: Cannot open file \"%s\": %s\n", sInputFile.c_str(), strerror(errno));
			return false;
		}
		fclose(pDestinationFile);

		CXmlBinaryDataWriterFile outputFile(sOutputFile.c_str());
		XMLBinary::CXMLBinaryWriter xmlBinaryWriter;
		string error;
		const bool ok = xmlBinaryWriter.WriteNode(&outputFile, root, bNeedSwapEndian, &filter, error);
		if (!ok)
		{
			remove(sOutputFile.c_str());
			RCLogError("XML: Failed to write binary xml file \"%s\": %s\n", sOutputFile.c_str(), error.c_str());
			return false;
		}
	}

	// Verify that the output file was written
	{
		FILE* pSourceFile = fopen(sOutputFile.c_str(), "rb");
		if (pSourceFile == 0)
		{
			RCLogError("XML: Failed to write file \"%s\": %s\n", sOutputFile.c_str(), strerror(errno));
			return false;
		}
		fclose(pSourceFile);
	}

	// Check that the output binary XML file has same content as the input text XML file
	if (!bNeedSwapEndian) 
	{
		// Read in the input file.
		XmlNodeRef rootTxt;
		{
			const bool bRemoveNonessentialSpacesFromContent = true;
			char szErrorBuffer[1024];
			szErrorBuffer[0] = 0;
			rootTxt = pSerializer->Read(FileXmlBufferSource(sInputFile.c_str()), bRemoveNonessentialSpacesFromContent, sizeof(szErrorBuffer), szErrorBuffer);
			if (!rootTxt)
			{
				RCLogError("XML: Cannot read file \"%s\" in second pass: %s.\n", sInputFile.c_str(), &szErrorBuffer[0]);
				return false;
			}
		}

		XMLBinary::XMLBinaryReader binReader;
		XmlNodeRef rootBin = binReader.Parse(sOutputFile.c_str());
		if (!rootBin)
		{
			RCLogError("XML: Cannot read binary XML file \"%s\".\n", sOutputFile.c_str());
			return false;
		}

		if (!xmlsAreEqual(rootBin, rootTxt, &filter))
		{
			RCLogError("XML: Source XML file \"%s\" and result binary XML file \"%s\" are different.", sInputFile.c_str(), sOutputFile.c_str());
			return false;
		}
	}

	return true;
}

void XMLConverter::ConstructAndSetOutputFile(ConvertContext &cc)
{
	string sExtension = PathHelpers::FindExtension(cc.sourceFileFinal);
	sExtension = "bin" + sExtension;
	cc.SetOutputFile(PathHelpers::ReplaceExtension(cc.sourceFileFinal, sExtension));
}

void XMLConverter::GetFilenameForUpToDateCheck(ConvertContext &cc, char* filenameBuffer, size_t bufferSize) const
{
	string const filename(cc.getOutputPath());

	if (filenameBuffer && (filename.length() < bufferSize))
	{
		strcpy(filenameBuffer, filename.c_str());		
	}
}

int XMLConverter::GetNumPlatforms() const
{
	return 3;
}

EPlatform XMLConverter::GetPlatform( int index ) const
{
	switch (index)
	{
	case 0:
		return ePlatform_PC;
	case 1:
		return ePlatform_X360;
	case 2:
		return ePlatform_PS3;
	default:
		return ePlatform_UNKNOWN;
	}
}

int XMLConverter::GetNumExt() const
{
	return sizeof(s_extensions) / sizeof(s_extensions[0]);
}

const char* XMLConverter::GetExt(int index) const
{
	return s_extensions[index];
}

//////////////////////////////////////////////////////////////////////////
ICompiler* XMLConverter::CreateCompiler()
{
	// Only ever return one compiler, since we don't support multithreading. Since
	// the compiler is just this object, we can tell whether we have already returned
	// a compiler by checking the ref count.
	if (m_refCount >= 2)
		return 0;

	// Until we support multithreading for this convertor, the compiler and the
	// convertor may as well just be the same object.
	++m_refCount;
	return this;
}

bool XMLConverter::SupportsMultithreading() const
{
	return false;
}
