#ifndef _M3dView
#define _M3dView
//-
// ==========================================================================
// Copyright (C) 1995 - 2006 Autodesk, Inc., and/or its licensors.  All
// rights reserved.
//
// The coded instructions, statements, computer programs, and/or related
// material (collectively the "Data") in these files contain unpublished
// information proprietary to Autodesk, Inc. ("Autodesk") and/or its
// licensors,  which is protected by U.S. and Canadian federal copyright law
// and by international treaties.
//
// The Data may not be disclosed or distributed to third parties or be
// copied or duplicated, in whole or in part, without the prior written
// consent of Autodesk.
//
// The copyright notices in the Software and this entire statement,
// including the above license grant, this restriction and the following
// disclaimer, must be included in all copies of the Software, in whole
// or in part, and all derivative works of the Software, unless such copies
// or derivative works are solely in the form of machine-executable object
// code generated by a source language processor.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND.
// AUTODESK DOES NOT MAKE AND HEREBY DISCLAIMS ANY EXPRESS OR IMPLIED
// WARRANTIES INCLUDING, BUT NOT LIMITED TO, THE WARRANTIES OF
// NON-INFRINGEMENT, MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
// OR ARISING FROM A COURSE OF DEALING, USAGE, OR TRADE PRACTICE. IN NO
// EVENT WILL AUTODESK AND/OR ITS LICENSORS BE LIABLE FOR ANY LOST
// REVENUES, DATA, OR PROFITS, OR SPECIAL, DIRECT, INDIRECT, OR
// CONSEQUENTIAL DAMAGES, EVEN IF AUTODESK AND/OR ITS LICENSORS HAS
// BEEN ADVISED OF THE POSSIBILITY OR PROBABILITY OF SUCH DAMAGES.
// ==========================================================================
//+
//
// CLASS:    M3dView
//
// ****************************************************************************

#if defined __cplusplus

// ****************************************************************************
// INCLUDED HEADER FILES


#include <maya/MStatus.h>
#include <maya/MObject.h>

#if defined(OSMac_)
#	if defined(OSMac_MachO_)
#		include <AGL/agl.h>
#	else
#		include <agl.h>
#	endif
typedef WindowRef MWindow;

#elif defined (SGI) || defined (LINUX)

#include <GL/glx.h>
#include <X11/Intrinsic.h>
typedef  Widget MWindow;

#elif defined (_WIN32)

#include "windows.h"
#include <gl/Gl.h>
typedef  HWND MWindow;
#else
#error Unknown OS
#endif //OSMac_

// ****************************************************************************
// DECLARATIONS

class MString;
class MDagPath;
class MPoint;
class MVector;
class MMatrix;
class MColor;
class MPxGlBuffer;
class MImage;

// ****************************************************************************
// CLASS DECLARATION (M3dView)

#if (defined (SGI) || defined (LINUX) || defined (OSMac_))
	#if (defined (SGI) || defined(LINUX))
		typedef Window M3dWindow;
	#elif defined(OSMac_)
		typedef WindowPtr		M3dWindow;
	#endif // defined (SGI)  || defined(LINUX)
#elif defined(_WIN32)
typedef HWND M3dWindow;
#else
#error Unknown OS
#endif // OSUnix_

//! \ingroup OpenMayaUI
//! \brief A 3-D view
/*!
  M3dView provides methods for working with 3D model views.  3D views
  are based on OpenGL drawing areas.

  Maya can operate in two different color modes, RGBA and color index.
  Color index mode is used to increase performance when shading is not
  required.  Drawing in color index mode is more complicated, but this
  class provides methods to simplify color selection.

  Maya has four color tables that can be used in RGBA, and that must
  be used in color index mode.  These four color tables represent four
  sets of bit planes that are independent of each other.  So, for
  example, it is possible to clear all active objects from the display
  and redraw them without redrawing the dormant and templated objects.
  The active and dormant color tables contain the same colors, but use
  different bitplanes.

  The extra performance of color index mode comes at the cost of a
  limited number of colors.  If this restriction causes difficulty,
  then it is possible for the user to force all displays into RGBA
  mode where any color may be used.

  When an object is affected by another in the scene, it is drawn in a
  magenta colour by default.  This is denoted in the DisplayStatus
  enum by kActiveAffected.  These objects are drawn in the active
  planes even though they are dormant for performance reasons.
*/
class OPENMAYAUI_EXPORT M3dView {

public:

	//! Display styles for a 3D view.
    enum DisplayStyle {
        kBoundingBox,			//!< Bounding box display.
        kFlatShaded,			//!< Flat shaded display.
        kGouraudShaded,			//!< Gouraud shaded display.
        kWireFrame,				//!< Wire frame display.
        kPoints					//!< Points only display.
    };

	//! Drawing modes for individual objects
    enum DisplayStatus {
        kActive,				//!< Object is active (selected).
        kLive,					//!< Object is live (construction surface).
        kDormant,				//!< Object is domant.
        kInvisible,				//!< Object is invisible (not drawn).
        kHilite,				//!< Object is hilited (has selectable components).
        kTemplate,				//!< Object is templated (Not renderable).
        kActiveTemplate,		//!< Object is active and templated.
        kActiveComponent,		//!< Object has active components.
		kLead,					//!< Last selected object.
		kIntermediateObject,	//!< Construction object (not drawn).
		kActiveAffected,		//!< Affected by active object(s).
        kNoStatus				//!< Object does not have a valid display status.
    };

	//! Reference to color palettes.
    enum ColorTable {
        kActiveColors = kActive,			//!< Colors for active objects
        kDormantColors = kDormant,			//!< Colors for dormant objects
        kTemplateColor = kTemplate,			//!< Colors for templated objects
        kBackgroundColor					//!< Colors for background color
    };

	//! Alignemnt values when drawing text.
    enum TextPosition {
        kLeft,					//!< Draw text to the left of the point
        kCenter,				//!< Draw text centered around the point
        kRight					//!< Draw text to the right of the point
    };

	//! Display modes
	enum DisplayObjects {
		kDisplayEverything	= 0,			//!< Show everything
		kDisplayNurbsCurves,				//!< Show nurbs curves
		kDisplayNurbsSurfaces,				//!< Show nurbs surfaces
		kDisplayMeshes,						//!< Show meshes
		kDisplayPlanes,						//!< Show planes
		kDisplayLights,						//!< Show lights
		kDisplayCameras,					//!< Show camera
		kDisplayJoints,						//!< Show joints
		kDisplayIkHandles,					//!< Show IK handles
		kDisplayDeformers,					//!< Show deformers
		kDisplayDynamics,					//!< Show dynamics
		kDisplayLocators,					//!< Show locators
		kDisplayDimensions,					//!< Show dimensions
		kDisplaySelectHandles,				//!< Show selection handles
		kDisplayPivots,						//!< Show pivots
		kDisplayTextures,					//!< Show textures
		kDisplayGrid,						//!< Show the grid
		kDisplayCVs,						//!< Show NURBS CVs
		kDisplayHulls,						//!< Show NURBS hulls
		kDisplayStrokes,					//!< Show strokes
		kDisplaySubdivSurfaces				//!< Show subdivision surfaces
	};

	//! Lighting mode used in this 3D view.
	enum LightingMode {
		kLightAll,							//!< All lights ON mode
		kLightSelected,						//!< Selected lights ON mode
		kLightActive,						//!< Active lights ON mode
		kLightDefault						//!< Default light ON mode
	};

	//! Current hardware rendergn engine used in this view.
	enum RendererName {
		//! Equivalent to when the renderer name is "base_OpenGL_Renderer" when queried from the "modelEditor" command
		kDefaultQualityRenderer,
		//! Equivalent to when the renderer name is "hwRender_OpenGL_Renderer" when queried from the "modelEditor" command
		kHighQualityRenderer,
		//! An externally defined renderer name has been set.
		kExternalRenderer
	};

	//! Possible depth buffer formats to read into.
	enum DepthBufferFormat
	{
		kDepth_8 = 0,						//!< 8 bits.
		kDepth_Float						//!< Floating point.
	};

    M3dView();
    virtual ~M3dView();

	// global methods
    static M3dView		active3dView( MStatus * ReturnStatus = NULL );
	static unsigned int     numberOf3dViews();
	static MStatus      get3dView( const unsigned int index,
							                M3dView & view );

#if defined(OSMac_)
    //! Mac OS X
	AGLContext			display ( MStatus * ReturnStatus = NULL );
	// deviceContext() method not needed on the Mac
#elif defined (SGI) || defined (LINUX)
	//! Linux
    Display *			display( MStatus * ReturnStatus = NULL );
	//! Linux
	GLXContext	        glxContext( MStatus * ReturnStatus = NULL );
#elif defined (_WIN32)
	//! Windows
	HGLRC  				display( MStatus * ReturnStatus = NULL );
	//! Windows
	HDC     			deviceContext( MStatus * ReturnStatus = NULL );
#else
#error Unknown OS
#endif

	static MWindow		applicationShell( MStatus * ReturnStatus = NULL );
	void		getScreenPosition( int &x, int &y, MStatus *ReturnStatus = NULL ) const; 
	M3dWindow  	window( MStatus * ReturnStatus = NULL );
    int         portWidth( MStatus * ReturnStatus = NULL );
    int         portHeight( MStatus * ReturnStatus = NULL );

	// Viewport sizing methods
    MStatus		pushViewport (unsigned int x, unsigned int y, unsigned int width, unsigned int height);
    MStatus		popViewport  ();
    MStatus		viewport (unsigned int &x, unsigned int &y, unsigned int &width, unsigned int &height) const;


	// OpenGL wrapper methods
    MStatus     beginGL();
    MStatus     endGL();

    //  OpenGL Selection
    //
    void            beginSelect (GLuint *buffer = NULL, GLsizei size = 0);
    GLint           endSelect   ();
    bool            selectMode  () const;
	bool 			textureMode () const;
    void            loadName    (GLuint name);
    void            pushName    (GLuint name);
    void            popName     ();
    void            initNames   ();


    // Overlay plane methods

    MStatus     beginOverlayDrawing();
    MStatus     endOverlayDrawing();
    MStatus     clearOverlayPlane();

    // Color methods

    MStatus     setDrawColor( unsigned int index,
							  ColorTable table = kActiveColors );
    MStatus     setDrawColor( const MColor & color );

    bool        isColorIndexMode( MStatus * ReturnStatus = NULL );
    unsigned int    numDormantColors( MStatus * ReturnStatus = NULL );
    unsigned int    numActiveColors( MStatus * ReturnStatus = NULL );
    unsigned int    numUserDefinedColors( MStatus * ReturnStatus = NULL );

    MStatus     setUserDefinedColor( unsigned int index, const MColor & color );
    unsigned int    userDefinedColorIndex( unsigned int index,
                                       MStatus * ReturnStatus = NULL );

    MColor      templateColor( MStatus * ReturnStatus = NULL );
    MColor      backgroundColor( MStatus * ReturnStatus = NULL );

    MColor      colorAtIndex( unsigned int index, ColorTable table = kActiveColors,
                              MStatus * ReturnStatus = NULL );
	MStatus		getColorIndexAndTable( unsigned int glindex, unsigned int &index,
									   ColorTable &table ) const;
	MStatus		colorMask(bool &r, bool &g, bool &b, bool &a);
	MStatus		setColorMask(bool r, bool g, bool b, bool a);


    // Text drawing methods

    MStatus     drawText( const MString & text, const MPoint position,
                          TextPosition textPosition = kLeft );

    // Camera methods

    MStatus     getCamera( MDagPath & camera );
    MStatus     setCamera( MDagPath & camera );

    MStatus     refresh( bool all = false, bool force = false );
    MStatus     refresh( bool all, bool force, bool offscreen );
    MStatus     refresh( MPxGlBuffer &buffer );
    MStatus     refresh( MPxGlBuffer &buffer, bool offscreen );
    MStatus     refresh( MPxGlBuffer &buffer, bool offscreen, const MMatrix& projectionMatrix );

	// Light methods
	MStatus		getLightCount( unsigned int &count, bool visible = TRUE );
	MStatus		getLightingMode(LightingMode &mode );
	MStatus		getLightPath( unsigned int lightNumber, MDagPath &light );
	MStatus		isLightVisible( unsigned int lightNumber, bool &visible );
	MStatus		getLightIndex( unsigned int lightNumber, unsigned int &lightIndex );

    // Transformation methods

    MStatus     viewToWorld( short x_pos, short y_pos,
                             MPoint & worldPt, MVector & worldVector ) const;
    MStatus     viewToWorld( short x_pos, short y_pos,
                             MPoint & nearClipPt, MPoint & farClipPt ) const;
    MStatus     viewToObjectSpace( short x_pos, short y_pos,
                                   const MMatrix & localMatrixInverse,
                                   MPoint & oPt, MVector & oVector ) const;
    bool        worldToView( const MPoint& worldPt,
                             short& x_pos, short& y_pos,
                             MStatus * ReturnStatus = NULL ) const;
	MStatus     projectionMatrix( MMatrix& projectionMatrix ) const;

	MStatus     modelViewMatrix( MMatrix& modelViewMatrix ) const;

	//	View selected prefix
	//
	MString			viewSelectedPrefix(MStatus *ReturnStatus) const;
	MStatus			setViewSelectedPrefix( const MString &prefix);

	//	isolateSelect behavior
	//
	bool			showViewSelectedChildren(MStatus *ReturnStatus) const;
	MStatus			setShowViewSelectedChildren(bool);

	//	Method to get a M3dView from a modelPanel
	//
	static MStatus 	getM3dViewFromModelPanel(const MString &modelPaneName,
											 M3dView &view);

	//	Method to get a M3dView from a modelEditor
	//
	static MStatus 	getM3dViewFromModelEditor(const MString &modelPaneName,
											  M3dView &view);

    // Display style methods
	//
    DisplayStyle    displayStyle( MStatus * ReturnStatus = NULL ) const;
    bool        isShadeActiveOnly( MStatus * ReturnStatus = NULL ) const;
    MStatus     setDisplayStyle ( DisplayStyle style, bool activeOnly = false);


	// Renderer methods
	RendererName	getRendererName(MStatus *ReturnStatus /* = NULL */) const;
	MString			rendererString( MStatus * ReturnStatus = NULL ) const;
	bool			wireframeOnlyInShadedMode(MStatus *ReturnStatus /* = NULL */) const;

	// Viewport buffer methods
	MStatus			readColorBuffer( MImage &image, bool readRGBA = false );
	MStatus			writeColorBuffer( const MImage &image, signed short x = 0, signed short y = 0) const;
	MStatus			readDepthMap(unsigned short x, unsigned short y,
								unsigned int width, unsigned int height,
								unsigned char* bufferPtr,
								DepthBufferFormat depthMapPrecision);
	MStatus			readBufferTo2dTexture(unsigned short x, unsigned short y,
								unsigned int width, unsigned int height);

	// Texture mipmap methods
	bool			usingMipmappedTextures() const;

BEGIN_NO_SCRIPT_SUPPORT:
	//!     NO SCRIPT SUPPORT
	MStatus			getRendererString( MString &stringName ) const;

END_NO_SCRIPT_SUPPORT:

protected:
// No protected members

private:
    static const char* className();
    M3dView( const void * );
    const void * fPtr;
};

#endif /* __cplusplus */
#endif /* _M3dView */
