#ifndef _MDagMessage
#define _MDagMessage
//-
// ==========================================================================
// Copyright (C) 1995 - 2006 Autodesk, Inc., and/or its licensors.  All
// rights reserved.
//
// The coded instructions, statements, computer programs, and/or related
// material (collectively the "Data") in these files contain unpublished
// information proprietary to Autodesk, Inc. ("Autodesk") and/or its
// licensors,  which is protected by U.S. and Canadian federal copyright law
// and by international treaties.
//
// The Data may not be disclosed or distributed to third parties or be
// copied or duplicated, in whole or in part, without the prior written
// consent of Autodesk.
//
// The copyright notices in the Software and this entire statement,
// including the above license grant, this restriction and the following
// disclaimer, must be included in all copies of the Software, in whole
// or in part, and all derivative works of the Software, unless such copies
// or derivative works are solely in the form of machine-executable object
// code generated by a source language processor.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND.
// AUTODESK DOES NOT MAKE AND HEREBY DISCLAIMS ANY EXPRESS OR IMPLIED
// WARRANTIES INCLUDING, BUT NOT LIMITED TO, THE WARRANTIES OF
// NON-INFRINGEMENT, MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
// OR ARISING FROM A COURSE OF DEALING, USAGE, OR TRADE PRACTICE. IN NO
// EVENT WILL AUTODESK AND/OR ITS LICENSORS BE LIABLE FOR ANY LOST
// REVENUES, DATA, OR PROFITS, OR SPECIAL, DIRECT, INDIRECT, OR
// CONSEQUENTIAL DAMAGES, EVEN IF AUTODESK AND/OR ITS LICENSORS HAS
// BEEN ADVISED OF THE POSSIBILITY OR PROBABILITY OF SUCH DAMAGES.
// ==========================================================================
//+
//
// CLASS:    MDagMessage
//
// ****************************************************************************

#if defined __cplusplus

// ****************************************************************************
// INCLUDED HEADER FILES


#include <maya/MMessage.h>
#include <maya/MString.h>

class MDagPath;

// ****************************************************************************
// CLASS DECLARATION (MDagMessage)

//! \ingroup OpenMaya
//! \brief Dependency graph messages. 
/*!
	This class is used to register callbacks for dependency graph messages.

	There are 6 types of add callback methods which will add callbacks for the
	following messages

		\li <b>Parent Added</b>
		\li <b>Parent Removed</b>
		\li <b>Child Added</b>
		\li <b>Child Removed</b>
		\li <b>Child Reordered</b>
		\li <b>Instance Added</b>
		\li <b>Instance Removed</b>


	Methods exist to register callbacks for every time any DAG node is
	affected as well as methods that work on specific nodes.
	Each method returns an id which is used to remove the callback.

    To remove a callback use MMessage::removeCallback.
	All callbacks that are registered by a plug-in must be removed by that
	plug-in when it is unloaded. Failure to do so will result in a fatal error.

	NOTE: It is possible to get Parent Added and Child Added messages before
	the node name has been set.  This can happen if the node is newly created.
	Additionally, the MDagPath string path names passed to the callback may
	not be set as yet if the node has not been added to the model.
*/
class OPENMAYA_EXPORT MDagMessage : public MMessage
{
public:
	//! The type of DAG changed messages that have occurred
	enum DagMessage {
		//! an invalid message was used.
		kInvalidMsg = -1,
		//! a dummy enum used for looping through the message types
		kParentAdded,
		//!	a parent was removed from a DAG node
		kParentRemoved,
		//!	a child was added to a DAG node
		kChildAdded,
		//!	a child was removed from a DAG node
		kChildRemoved,
		//!	a child of a DAG node was reordered
		kChildReordered,
		//!	a DAG node was instanced
		kInstanceAdded,
		//!	a DAG node instance was removed
		kInstanceRemoved,
		//!	last value of the enum
		kLast
	};

public:

	//! \brief Pointer to a callback function which takes an MDagMessage, two MDagPaths and a clientData pointer.
	/*!
	\param[in] msgType Type of message which caused the function to be called.
	\param[in,out] child First DAG path, usually used to pass the child of a parent/child pair.
	\param[in,out] parent Second DAG path, usually use to pass the parent of a parent/child pair.
	\param[in,out] clientData User-defined data which was supplied when the callback was added.
	*/
	typedef void (*MMessageParentChildFunction)( MDagMessage::DagMessage msgType,MDagPath &child, MDagPath &parent, void * clientData );


	// Parent added callback for all nodes
	static MCallbackId	addParentAddedCallback(
								MMessage::MParentChildFunction func,
								void * clientData = NULL,
								MStatus * ReturnStatus = NULL );

	// Parent added callback for a specified node
	static MCallbackId	addParentAddedDagPathCallback(
								MDagPath &node,
								MMessage::MParentChildFunction func,
								void * clientData = NULL,
								MStatus * ReturnStatus = NULL );

	// Parent removed callback for all nodes
	static MCallbackId  addParentRemovedCallback(
								MMessage::MParentChildFunction func,
								void * clientData = NULL,
								MStatus * ReturnStatus = NULL );

	// Parent removed callback for a specified node
	static MCallbackId  addParentRemovedDagPathCallback(
								MDagPath &node,
								MMessage::MParentChildFunction func,
								void * clientData = NULL,
								MStatus * ReturnStatus = NULL );

	// Child added callback for all nodes
	static MCallbackId	addChildAddedCallback(
								MMessage::MParentChildFunction func,
								void * clientData = NULL,
								MStatus * ReturnStatus = NULL );

	// Child added callback for a specified node
	static MCallbackId	addChildAddedDagPathCallback(
								MDagPath &node,
								MMessage::MParentChildFunction func,
								void * clientData = NULL,
								MStatus * ReturnStatus = NULL );

	// Child removed callback for all nodes
	static MCallbackId	addChildRemovedCallback(
								MMessage::MParentChildFunction func,
								void * clientData = NULL,
								MStatus * ReturnStatus = NULL );

	// Child removed callback for a specified node
	static MCallbackId	addChildRemovedDagPathCallback(
								MDagPath &node,
								MMessage::MParentChildFunction func,
								void * clientData = NULL,
								MStatus * ReturnStatus = NULL );

	// Child reordered callback for all nodes
	static MCallbackId	addChildReorderedCallback(
								MMessage::MParentChildFunction func,
								void * clientData = NULL,
								MStatus * ReturnStatus = NULL );

	// Child reordered callback for a specified node
	static MCallbackId	addChildReorderedDagPathCallback(
								MDagPath &node,
								MMessage::MParentChildFunction func,
								void * clientData = NULL,
								MStatus * ReturnStatus = NULL );



	//	Any DAG change callback with the DagMessage for all nodes
	static MCallbackId	addDagCallback(
								DagMessage msgType,
								MDagMessage::MMessageParentChildFunction func,
								void * clientData = NULL,
								MStatus * ReturnStatus = NULL );

	//	Any Dag change callback with the DagMessage for a specified node
	static MCallbackId	addDagDagPathCallback(
								MDagPath &node,
								DagMessage msgType,
								MDagMessage::MMessageParentChildFunction func,
								void * clientData = NULL,
								MStatus * ReturnStatus = NULL );

	//	Adds a Dag change callback for all known Dag change messages for
	//	all nodes.
	static MCallbackId	addAllDagChangesCallback(
								MDagMessage::MMessageParentChildFunction func,
								void * clientData = NULL,
								MStatus * ReturnStatus = NULL );

	//	Adds a Dag change callback for all known Dag change messages for
	//	the specified node.
	static MCallbackId	addAllDagChangesDagPathCallback(
								MDagPath &node,
								MDagMessage::MMessageParentChildFunction func,
								void * clientData = NULL,
								MStatus * ReturnStatus = NULL );



	//	Adds a Dag change callback for the instancing of
	//	all the nodes in the DAG
	static MCallbackId	addInstanceAddedCallback(
								MMessage::MParentChildFunction func,
								void * clientData = NULL,
								MStatus * ReturnStatus = NULL );


	//	Adds a Dag change callback for the instancing of
	//	the specified node.
	static MCallbackId	addInstanceAddedDagPathCallback(
								MDagPath &node,
								MMessage::MParentChildFunction func,
								void * clientData = NULL,
								MStatus * ReturnStatus = NULL );

	//	Adds a Dag change callback for the instance removed/deleted of
	//	all the nodes in the DAG
	static MCallbackId	addInstanceRemovedCallback(
								MMessage::MParentChildFunction func,
								void * clientData = NULL,
								MStatus * ReturnStatus = NULL );

	//	Adds a Dag change callback for the instance removed of
	//	the specified node.
	static MCallbackId	addInstanceRemovedDagPathCallback(
								MDagPath &node,
								MMessage::MParentChildFunction func,
								void * clientData = NULL,
								MStatus * ReturnStatus = NULL );


BEGIN_NO_SCRIPT_SUPPORT:

	//! Obsolete and no script support
	static MCallbackId	addParentAddedCallback(
								MDagPath &node,
								MMessage::MParentChildFunction func,
								void * clientData = NULL,
								MStatus * ReturnStatus = NULL );

	//! Obsolete and no script support
	static MCallbackId  addParentRemovedCallback(
								MDagPath &node,
								MMessage::MParentChildFunction func,
								void * clientData = NULL,
								MStatus * ReturnStatus = NULL );

	//! obsolete and no script support
	static MCallbackId	addChildAddedCallback(
								MDagPath &node,
								MMessage::MParentChildFunction func,
								void * clientData = NULL,
								MStatus * ReturnStatus = NULL );

	//! Obsolete and no script support
	static MCallbackId	addChildRemovedCallback(
								MDagPath &node,
								MMessage::MParentChildFunction func,
								void * clientData = NULL,
								MStatus * ReturnStatus = NULL );

	//! Obsolete and no script support
	static MCallbackId	addChildReorderedCallback(
								MDagPath &node,
								MMessage::MParentChildFunction func,
								void * clientData = NULL,
								MStatus * ReturnStatus = NULL );

	//! Obsolete and no script support
	static MCallbackId	addInstanceAddedCallback(
								MDagPath &node,
								MMessage::MParentChildFunction func,
								void * clientData = NULL,
								MStatus * ReturnStatus = NULL );

	//! Obsolete and no script support
	static MCallbackId	addInstanceRemovedCallback(
								MDagPath &node,
								MMessage::MParentChildFunction func,
								void * clientData = NULL,
								MStatus * ReturnStatus = NULL );

	//! Obsolete and no script support
	static MCallbackId	addDagCallback(
								DagMessage msgType,
								MMessage::MParentChildFunction func,
								void * clientData = NULL,
								MStatus * ReturnStatus = NULL );

	//! Obsolete and no script support
	static MCallbackId	addDagCallback(
								MDagPath &node,
								DagMessage msgType,
								MMessage::MParentChildFunction func,
								void * clientData = NULL,
								MStatus * ReturnStatus = NULL );

	//! obsolete and no script support
	static MCallbackId	addDagCallback(
								MDagPath &node,
								DagMessage msgType,
								MDagMessage::MMessageParentChildFunction func,
								void * clientData = NULL,
								MStatus * ReturnStatus = NULL );

	//! Obsolete and no script support
	static MCallbackId	addAllDagChangesCallback(
								MDagPath &node,
								MDagMessage::MMessageParentChildFunction func,
								void * clientData = NULL,
								MStatus * ReturnStatus = NULL );

END_NO_SCRIPT_SUPPORT:

private:
	static MCallbackId	addDagCallback(
								MDagPath *dagPath,
								DagMessage msgType,
								MMessage::MParentChildFunction func,
								void * clientData = NULL,
								MStatus * ReturnStatus = NULL );


	static MCallbackId	addDagCallback(
								MDagPath *dagPath,
								DagMessage msgType,
								MDagMessage::MMessageParentChildFunction func,
								void * clientData = NULL,
								MStatus * ReturnStatus = NULL );

	static MCallbackId	addAllDagChangesCallback(
								MDagPath *dagPath,
								MDagMessage::MMessageParentChildFunction func,
								void * clientData = NULL,
								MStatus * ReturnStatus = NULL );

	static const char* className();
};

#endif /* __cplusplus */
#endif /* _MDagMessage */
