#ifndef _MHardwareRenderer
#define _MHardwareRenderer
//-
// ==========================================================================
// Copyright (C) 1995 - 2006 Autodesk, Inc., and/or its licensors.  All
// rights reserved.
//
// The coded instructions, statements, computer programs, and/or related
// material (collectively the "Data") in these files contain unpublished
// information proprietary to Autodesk, Inc. ("Autodesk") and/or its
// licensors,  which is protected by U.S. and Canadian federal copyright law
// and by international treaties.
//
// The Data may not be disclosed or distributed to third parties or be
// copied or duplicated, in whole or in part, without the prior written
// consent of Autodesk.
//
// The copyright notices in the Software and this entire statement,
// including the above license grant, this restriction and the following
// disclaimer, must be included in all copies of the Software, in whole
// or in part, and all derivative works of the Software, unless such copies
// or derivative works are solely in the form of machine-executable object
// code generated by a source language processor.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND.
// AUTODESK DOES NOT MAKE AND HEREBY DISCLAIMS ANY EXPRESS OR IMPLIED
// WARRANTIES INCLUDING, BUT NOT LIMITED TO, THE WARRANTIES OF
// NON-INFRINGEMENT, MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
// OR ARISING FROM A COURSE OF DEALING, USAGE, OR TRADE PRACTICE. IN NO
// EVENT WILL AUTODESK AND/OR ITS LICENSORS BE LIABLE FOR ANY LOST
// REVENUES, DATA, OR PROFITS, OR SPECIAL, DIRECT, INDIRECT, OR
// CONSEQUENTIAL DAMAGES, EVEN IF AUTODESK AND/OR ITS LICENSORS HAS
// BEEN ADVISED OF THE POSSIBILITY OR PROBABILITY OF SUCH DAMAGES.
// ==========================================================================
//+
//
// CLASS:    MHardwareRenderer
//
// ****************************************************************************

#if defined __cplusplus

// ****************************************************************************
// INCLUDED HEADER FILES


#include <maya/MString.h>
#include <maya/MDrawProcedureBase.h>
#include <maya/MColor.h>
#include <maya/MGLFunctionTable.h>

class MStringArray;
class MGeometryData;
class MImage;

#define kDefaultQuadColor  MColor(0.5f, 0.5f, 0.5f, 1.0f)

// ****************************************************************************
// CLASS DECLARATION (MHardwareRenderer)

//! \ingroup OpenMayaRender
//! \brief Static hardware renderer interface class
/*!
 MHardwareRenderer is an interface class which wraps the hardware
 renderer.
*/
class OPENMAYARENDER_EXPORT MHardwareRenderer
{
public:
	//! Draw Procedure status codes
	enum DrawProcedureStatusCode {
		//! Success
		kSuccess = 0,
		//! Failure
		kFailure,
		//! Item already exists
		kItemExists,
		//! Item is not found
		kItemNotFound,
		//! Location not found
		kLocationNotFound
	};

	//! Specify the pixel format of the current buffer
	enum BufferPixelFormat {
		//! 8 bit Red, Green, Blue, and Alpha channel
		kRGBA_Fix8,
		//! 16 bit Red, Green, Blue, and Alpha channel
		kRGBA_Float16,
		//! 32 bit floating point depth buffer
		kDepth_Float32
	};

	//! Draw Procedure call locations
	enum CallLocation {
		//! Before rendering one frame. No model or view matrices
		// can be assumed to have been set up at this time,
		// nor an assumption on the background.
		kPreRendering = 0,
		//! Before rendering one "exposure". If multiple exposures are
		//! are required to render one frame. After the frame buffer is cleared
		//! and model and view matrices are set up for the current exposure.
		kPreExposure = 1,
		//! After rendering one "exposure".
		kPostExposure = 2,
		//! After rendering one frame. Before a possible fame buffer swap.
		kPostRendering = 3
	};

	//! Default geometry shapes
	enum GeometricShape {
		//! Sphere with radius 1, centered at 0,0,0.
		kDefaultSphere = 0,
		//! Plane with width and height of 1, centered at 0,0,0.
		//! Assuming "Y-Up" orientation: width = x-axis, and height = y-axis.
		kDefaultPlane,
		//! Cube with width, height and depth of 1, centered at 0,0,0.
		kDefaultCube
	};

	static MHardwareRenderer*	theRenderer();

	const MString &				backEndString() const;


	// Draw procedure methods
	DrawProcedureStatusCode		addDrawProcedure( const MString & backEndString,
									MDrawProcedureBase *drawProcedureptr,
									CallLocation location);
	DrawProcedureStatusCode		removeDrawProcedure( const MString & backEndString,
									MDrawProcedureBase *drawProcedureptr,
									CallLocation location );
	DrawProcedureStatusCode		insertDrawProcedure( const MString & backEndString,
									   MDrawProcedureBase *drawProcedurePtr,
									   CallLocation location,
									   unsigned int listIndex,
									   bool moveExistingItem = false);
	DrawProcedureStatusCode		getDrawProcedureCount( const MString & backEndString,
									  CallLocation location,
									  unsigned int &count ) const;
	DrawProcedureStatusCode		getDrawProcedureListNames( const MString & backEndString,
												CallLocation location,
												MStringArray &namesOfProcs ) const;
	MDrawProcedureBase * findDrawProcedure( const MString & backEndString,
									CallLocation location,
									const MString &procName ) const;

	// Context handling methods
	MStatus						makeResourceContextCurrent(const MString &backEndString);

	MStatus						getBufferSize( const MString& backEndString,
												unsigned int &width,
												unsigned int &height ) const;

	MStatus						getColorBufferPixelFormat( const MString& backEndString,
														BufferPixelFormat &fmt ) const;

	MStatus						getDepthBufferPixelFormat( const MString& backEndString,
														BufferPixelFormat &fmt ) const;

	MStatus						getCurrentExposureNumber( const MString& backEndString,
														  unsigned int &number ) const;
	MStatus						getTotalExposureCount( const MString& backEndString,
														  unsigned int &number ) const;

	MStatus restoreCurrent(const MString& backEndString) const;


	// Swatch rendering methods
	MStatus						makeSwatchContextCurrent( const MString& backEndString,
														  unsigned int & width,
														  unsigned int & height );

	MStatus						readSwatchContextPixels( const MString& backEndString,
												   MImage & image );

	MGeometryData *				referenceDefaultGeometry(
									MHardwareRenderer::GeometricShape geomShape,
									unsigned int & numberOfData,
									unsigned int* & pIndexing,
									unsigned int & indexLength);
	MStatus						dereferenceGeometry(MGeometryData * pGeomData,
											unsigned int numberOfData);

	void getSwatchOrthoCameraSetting( double& l, double& r,
									  double& b, double& t,
									  double& n, double& f ) const;

	void getSwatchPerspectiveCameraSetting( double& l, double& r,
									  double& b, double& t,
									  double& n, double& f ) const;
	void getSwatchPerspectiveCameraTranslation( float& x,  float& y, float& z, float& w ) const;

	void getSwatchLightDirection( float& x,  float& y, float& z, float& w ) const;

	void drawSwatchBackGroundQuads( const MColor & quadColor = kDefaultQuadColor,
									bool textured = false,
									unsigned int numberOfRepeats = 8) const;

	MGLFunctionTable *			glFunctionTable() const;

protected:
	// Destructor and constructor are protected
	MHardwareRenderer();
	~MHardwareRenderer();

	// Static global instance of wrapper class
	//
	static	MHardwareRenderer *	fsHardwareRenderer;
	static  MGLFunctionTable *	fsGLFunctionTable;

private:
// No private members

};

#endif /* __cplusplus */
#endif /* _MHardwareRenderer */


