#ifndef _MPxCacheFormat
#define _MPxCacheFormat
//-
// ==========================================================================
// Copyright (C) 1995 - 2006 Autodesk, Inc., and/or its licensors.  All
// rights reserved.
//
// The coded instructions, statements, computer programs, and/or related
// material (collectively the "Data") in these files contain unpublished
// information proprietary to Autodesk, Inc. ("Autodesk") and/or its
// licensors,  which is protected by U.S. and Canadian federal copyright law
// and by international treaties.
//
// The Data may not be disclosed or distributed to third parties or be
// copied or duplicated, in whole or in part, without the prior written
// consent of Autodesk.
//
// The copyright notices in the Software and this entire statement,
// including the above license grant, this restriction and the following
// disclaimer, must be included in all copies of the Software, in whole
// or in part, and all derivative works of the Software, unless such copies
// or derivative works are solely in the form of machine-executable object
// code generated by a source language processor.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND.
// AUTODESK DOES NOT MAKE AND HEREBY DISCLAIMS ANY EXPRESS OR IMPLIED
// WARRANTIES INCLUDING, BUT NOT LIMITED TO, THE WARRANTIES OF
// NON-INFRINGEMENT, MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
// OR ARISING FROM A COURSE OF DEALING, USAGE, OR TRADE PRACTICE. IN NO
// EVENT WILL AUTODESK AND/OR ITS LICENSORS BE LIABLE FOR ANY LOST
// REVENUES, DATA, OR PROFITS, OR SPECIAL, DIRECT, INDIRECT, OR
// CONSEQUENTIAL DAMAGES, EVEN IF AUTODESK AND/OR ITS LICENSORS HAS
// BEEN ADVISED OF THE POSSIBILITY OR PROBABILITY OF SUCH DAMAGES.
// ==========================================================================
//+
//
// CLASS:    MPxCacheFormat
//
// ****************************************************************************

#if defined __cplusplus

// ****************************************************************************
// INCLUDED HEADER FILES


#include <maya/MObject.h>
#include <maya/MString.h>
#include <maya/MTypes.h>

class MTime;
class MFloatVectorArray;
class MDoubleArray;
class MFloatArray;
class MIntArray;
class MVectorArray;

// ****************************************************************************
// CLASS DECLARATION (MPxCacheFormat)

//! \ingroup OpenMaya MPx
//! \brief CacheFormat definition. 
/*!
  The MPxCacheFormat class can be used to implement support for new cache
  file formats in Maya.
*/
class OPENMAYA_EXPORT MPxCacheFormat
{
public:
	//! Cache file access modes.
    typedef enum {
        kRead,		//!< read only
        kWrite,		//!< write only
	kReadWrite 	//!< read and write (e.g. append, edit, etc.)
    } FileAccessMode;

	MPxCacheFormat();
	virtual ~MPxCacheFormat();

	/*!
	  Attempt to open the specified cache format. It is important
	  that this function only return success if the cache file is
	  definitely supported by this implementation.

	  \param[in] fileName Name of the cache file to open
	  \param[in] mode Access mode for the cache file

	  \return success if the cache file is supported.
	*/
	virtual MStatus		open( const MString& fileName, FileAccessMode mode);

	//! Close the current current cache file.
	virtual void		close();

	//! Confirm wether the current cache file is valid.
	virtual MStatus		isValid();

	//! Rewind to the beginning of the cache
	virtual MStatus		rewind();

	/*!
	  Provide the disk file extension for this format. This is not
	  the same as the format's key, which is used by the plugin to
	  identify itself.
	*/
	virtual MString		extension();

	/*!
	  Read the header from the current cache data file, and store
	  any data that may be required.
	*/
	virtual MStatus		readHeader();

	/*! Write the header for the current cache.
	 */
	virtual MStatus		writeHeader(const MString& version, MTime& startTime, MTime& endTime);

	/*! 
	  Perform any actions required prior to writing a chunk's
	  information. A chunk contains the cache information for a
	  specfic time, and may contain multiple channels.
	*/
	virtual void		beginWriteChunk();

	//! Perform any actions required after writing a chunk's information. A chunk contains the cache
	//! information for a specfic time, and may contain multiple channels.
	virtual void		endWriteChunk();

	//! Perform any actions required prior to reading a chunk's information. A chunk contains the cache
	//! information for a specfic time, and may contain multiple channels.
	virtual MStatus		beginReadChunk();

	/*!
	  Perform any actions required after reading a chunk's
	  information. A chunk contains the cache information for a
	  specfic time, and may contain multiple channels.
	*/
	virtual void		endReadChunk();

	/*! Write the current time to the cache.
	  \param[in] time Time to write.
	 */
	virtual MStatus		writeTime(MTime& time);

	/*! Read the current time from the cache.
	  \param[out] time Variable to read the time value into.
	 */
	virtual MStatus		readTime(MTime& time);

	//! Find a specific time in the cache.
	virtual	MStatus		findTime(MTime& time, MTime& foundTime);

	//! Find the next time in the cache.
	virtual MStatus		readNextTime(MTime& foundTime);

	//! Extract size information from the cache.
	virtual unsigned	readArraySize();

	//! Write data to the cache.
	virtual MStatus		writeDoubleArray(const MDoubleArray&);
	virtual MStatus		writeFloatArray(const MFloatArray&);
	virtual MStatus		writeIntArray(const MIntArray&);
	virtual MStatus		writeDoubleVectorArray(const MVectorArray& array);
	virtual MStatus     writeFloatVectorArray(const MFloatVectorArray& array);
	virtual MStatus		writeInt32(int);

	//! Read data from the cache.
	virtual MStatus		readDoubleArray(MDoubleArray&, unsigned size);
	virtual MStatus		readFloatArray(MFloatArray&, unsigned size);
	virtual MStatus		readIntArray(MIntArray&, unsigned size);
	virtual MStatus		readDoubleVectorArray(MVectorArray&, unsigned arraySize);
	virtual MStatus     readFloatVectorArray(MFloatVectorArray& array, unsigned arraySize);
	virtual int		readInt32();

	//! Write a channel's name to the cache. Each cached object is contained within a separate channel.
	virtual MStatus        	writeChannelName(const MString & name);

	//! Seek to a specific channel name in the cache.
	virtual MStatus        	findChannelName(const MString & name);

	//! Read the next channel name in the cache.
	virtual MStatus        	readChannelName(MString& name);

protected:

private:
};

#endif /* __cplusplus */
#endif /* _MPxCacheFormat */
