#ifndef __MTemplateCommand_h
#define __MTemplateCommand_h
//-
// ==========================================================================
// Copyright (C) 1995 - 2006 Autodesk, Inc., and/or its licensors.  All
// rights reserved.
//
// The coded instructions, statements, computer programs, and/or related
// material (collectively the "Data") in these files contain unpublished
// information proprietary to Autodesk, Inc. ("Autodesk") and/or its
// licensors,  which is protected by U.S. and Canadian federal copyright law
// and by international treaties.
//
// The Data may not be disclosed or distributed to third parties or be
// copied or duplicated, in whole or in part, without the prior written
// consent of Autodesk.
//
// The copyright notices in the Software and this entire statement,
// including the above license grant, this restriction and the following
// disclaimer, must be included in all copies of the Software, in whole
// or in part, and all derivative works of the Software, unless such copies
// or derivative works are solely in the form of machine-executable object
// code generated by a source language processor.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND.
// AUTODESK DOES NOT MAKE AND HEREBY DISCLAIMS ANY EXPRESS OR IMPLIED
// WARRANTIES INCLUDING, BUT NOT LIMITED TO, THE WARRANTIES OF
// NON-INFRINGEMENT, MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
// OR ARISING FROM A COURSE OF DEALING, USAGE, OR TRADE PRACTICE. IN NO
// EVENT WILL AUTODESK AND/OR ITS LICENSORS BE LIABLE FOR ANY LOST
// REVENUES, DATA, OR PROFITS, OR SPECIAL, DIRECT, INDIRECT, OR
// CONSEQUENTIAL DAMAGES, EVEN IF AUTODESK AND/OR ITS LICENSORS HAS
// BEEN ADVISED OF THE POSSIBILITY OR PROBABILITY OF SUCH DAMAGES.
// ==========================================================================
//+
//
// CLASS:    MTemplateCommand
//
// ****************************************************************************

#if defined __cplusplus

// ****************************************************************************
// INCLUDED HEADER FILES

#include <maya/MTypes.h>
#include <maya/MPxCommand.h>
#include <maya/MObject.h>
#include <maya/MFnPlugin.h>
#include <maya/MSyntax.h>
#include <maya/MDagModifier.h>
#include <maya/MObjectHandle.h>
#include <maya/MArgDatabase.h>

// ****************************************************************************
// DECLARATIONS

// ****************************************************************************
// CLASS DECLARATION (MTemplateCommand)

//! \ingroup OpenMaya
//! \brief Template class for non undoable actions.
/*!
  Helper template class which make writing not undoable actions easier.

  \par Parameters:
  \li ActionClass : derived from MPxCommand and used in the creator() method
  \li CommandName : name of the command
  \li CommandSyntax : function pointer to syntax creator
*/
template <class ActionClass, const char* CommandName, MCreateSyntaxFunction CommandSyntax >
class MTemplateAction : public MPxCommand
{
public:
	//! Default constructor
	MTemplateAction()
	{
	}

	//! Execution of the action. Default behaviour is failure.
	virtual MStatus	doIt ( const MArgList& )
	{
		return MStatus::kFailure;
	}

	//! Method called by Maya to create an instance of this class.
	static void* creator()
	{
		return new ActionClass;
	}

	/*! Plug-in registration method.

	\param[in] obj Plug-in object taken from the initializePlugin()
	entry point.
 */
	static MStatus registerCommand( MObject obj )
	{
		MFnPlugin plugin( obj );
		return plugin.registerCommand( CommandName, creator, CommandSyntax );
	}

	/*! Plug-in dregistration method.

	\param[in] obj Plug-in object taken from the initializePlugin()
	entry point.
	*/
	static MStatus deregisterCommand( MObject obj )
	{
		MFnPlugin plugin( obj );
		return plugin.deregisterCommand( CommandName );
	}
};
//! \ingroup OpenMaya
//! \brief Template class for undoable commands.
/*!
  Helper template class which make writing undoable commands easier.

  \par Parameters:
  \li CommandClass : derived from MPxCommand and used in the creator() method
  \li CommandName : name of the command
  \li CommandSyntax : function pointer to syntax creator
*/
template <class CommandClass, const char* CommandName, MCreateSyntaxFunction CommandSyntax>
class MTemplateCommand : public MTemplateAction<CommandClass,CommandName, CommandSyntax>
{
public:
	//! Define the command as undoable
	virtual bool isUndoable() const
	{
		return true;
	}
};


//
//	Utility strings + functions that template classes
//	may need.
//

// Strings we need for the command
const char *CreateCommandFlag = "-c";
const char *CreateCommandFlagLong = "-create";
const char *DeleteCommandFlag = "-d";
const char *DeleteCommandFlagLong = "-delete";

// Syntax for the command
MSyntax MTemplateCreateNodeCommand_newSyntax()
{
	MSyntax syntax;
	syntax.addFlag( CreateCommandFlag, CreateCommandFlagLong );
	syntax.addFlag( DeleteCommandFlag, DeleteCommandFlagLong );
	return syntax;
}

// Null syntax
MSyntax MTemplateCommand_nullSyntax()
{
	MSyntax syntax;
	return syntax;
}

// Should get duplicate error if used more than once in
// a plug-in binary
MObjectHandle MTemplateCreateNodeCommand_nodeHandle;

//! \ingroup OpenMaya
//! \brief Template class for undoable commands to create/delete a node.
/*!
  Helper template class which make it easier to write commands to
  create/delete a node.

  The command is tied into an MDGModifier to create a node with undo support.

  \note Only one such command per plug-in binary is allowed because
  MTemplateCreateNodeCommand_nodeHandle is a global variable.

  \par Parameters:
  \li CommandClass : derived from MPxCommand and used in the creator() method
  \li CommandName : name of the command
  \li NodeName : node that should be created
*/
template <class CommandClass, const char *CommandName, const char* NodeName>
class MTemplateCreateNodeCommand : public MTemplateCommand<CommandClass,CommandName,MTemplateCreateNodeCommand_newSyntax/*specialization*/>
{
public:

	/*! Execution of the command.

	\param[in] args command arguments
	*/
	virtual MStatus doIt( const MArgList& args )
	{
		MArgDatabase argData( this->syntax(), args );

		creating = true;
		if ( argData.isFlagSet( CreateCommandFlag ) )
			creating = true;
		else if ( argData.isFlagSet( DeleteCommandFlag ) )
			creating = false;
		else
			return MStatus::kFailure;

		MObject newNode;
		MStatus status;
		if ( creating )
		{
			if ( MTemplateCreateNodeCommand_nodeHandle.isValid() )
			{
				this->displayError("Must match -create/-delete calls");
				return MStatus::kFailure;
			}
			newNode = modifier.createNode( NodeName, MObject::kNullObj, &status );
			MTemplateCreateNodeCommand_nodeHandle = newNode;
		}
		else
		{
			if ( MTemplateCreateNodeCommand_nodeHandle.isValid() )
				status = modifier.deleteNode( MTemplateCreateNodeCommand_nodeHandle.object() );
			 MTemplateCreateNodeCommand_nodeHandle = MObject::kNullObj;
		}

		return redoIt();
	}

	//! Redo the create/delete
	virtual MStatus redoIt()
	{
		MStatus status;
		status = modifier.doIt();
		if ( MStatus::kSuccess == status )
			return MStatus::kSuccess;

		return MStatus::kFailure;
	}

	//! Undo the create/delete
	virtual MStatus undoIt()
	{
		return modifier.undoIt();
	}

private:
	bool creating;
	MDagModifier modifier;
};

#endif /* _cplusplus */
#endif /* __MTemplateCommand_h */

