#ifndef __FBCONTROLS_H__
#define __FBCONTROLS_H__
/**************************************************************************
 Copyright (c) 1994 - 2006 Autodesk, Inc. and/or its licensors.
 All Rights Reserved.
 
 The coded instructions, statements, computer programs, and/or related 
 material (collectively the "Data") in these files contain unpublished 
 information proprietary to Autodesk, Inc. and/or its licensors, which is 
 protected by Canada and United States of America federal copyright law 
 and by international treaties.
 
 The Data may not be disclosed or distributed to third parties, in whole 
 or in part, without the prior written consent of Autodesk, Inc. 
 ("Autodesk").
 
 THE DATA IS PROVIDED "AS IS" AND WITHOUT WARRANTY.
 ALL WARRANTIES ARE EXPRESSLY EXCLUDED AND DISCLAIMED. AUTODESK MAKES NO 
 WARRANTY OF ANY KIND WITH RESPECT TO THE DATA, EXPRESS, IMPLIED OR 
 ARISING BY CUSTOM OR TRADE USAGE, AND DISCLAIMS ANY IMPLIED WARRANTIES 
 OF TITLE, NON-INFRINGEMENT, MERCHANTABILITY OR FITNESS FOR A PARTICULAR 
 PURPOSE OR USE. WITHOUT LIMITING THE FOREGOING, AUTODESK DOES NOT 
 WARRANT THAT THE OPERATION OF THE DATA WILL BE UNINTERRUPTED OR ERROR 
 FREE.
 
 IN NO EVENT SHALL AUTODESK, ITS AFFILIATES, PARENT COMPANIES, LICENSORS 
 OR SUPPLIERS ("AUTODESK GROUP") BE LIABLE FOR ANY LOSSES, DAMAGES OR 
 EXPENSES OF ANY KIND (INCLUDING WITHOUT LIMITATION PUNITIVE OR MULTIPLE 
 DAMAGES OR OTHER SPECIAL, DIRECT, INDIRECT, EXEMPLARY, INCIDENTAL, LOSS 
 OF PROFITS, REVENUE OR DATA, COST OF COVER OR CONSEQUENTIAL LOSSES OR 
 DAMAGES OF ANY KIND), HOWEVER CAUSED, AND REGARDLESS OF THE THEORY OF 
 LIABILITY, WHETHER DERIVED FROM CONTRACT, TORT (INCLUDING, BUT NOT 
 LIMITED TO, NEGLIGENCE), OR OTHERWISE, ARISING OUT OF OR RELATING TO THE 
 DATA OR ITS USE OR ANY OTHER PERFORMANCE, WHETHER OR NOT AUTODESK HAS 
 BEEN ADVISED OF THE POSSIBILITY OF SUCH LOSS OR DAMAGE.
 
**************************************************************************/

/**	\file fbcontrols.h
*	Contains the user interface components for the SDK.
*	All of the user interface elements available for use with the SDK are 
*	defined in this file. These elements (for the most part) derive from 
*	the FBVisualComponent base class and provide the functionality 
*	required to develop an interface for a plugin.
*/

#include <kaydaradef.h>
#ifndef FBSDK_DLL
	#define FBSDK_DLL K_DLLEXPORT
#endif

#include <fbsdk/fbcomponent.h>

K_FORWARD( IKtLayoutRegion );
K_FORWARD( IKtObject );
K_FORWARD( IKtView );

#ifdef FBSDKUseNamespace
	namespace FBSDKNamespace {
#endif

FB_DEFINE_COMPONENT( FBSDK_DLL, Layout				);
FB_DEFINE_COMPONENT( FBSDK_DLL, VisualComponent	);
FB_DEFINE_COMPONENT( FBSDK_DLL, View				);

#ifdef K_DISABLE_UI //{

#else // }{

FB_DEFINE_COMPONENT( FBSDK_DLL, TreeNode			);


#define SPLIT_WIDTH				10

/** Register a control
*	\param	ClassName		Control class name to register.
*	\param	Path			Internal path to store control.
*	\param	Label			Short description of control.
*	\param	Description		Long description of control.
*	\param	IconFilename	Filename of associated icon.
*	\warning The application path to specify is not a directory structure path.
*/
#define FBRegisterControl( ClassName, Path, Label, Description, IconFilename ) \
	HIObject RegisterControl##ClassName( HIObject /*pOwner*/,char * /*pName*/,void * /*pData*/) \
	{\
		ClassName *Class = new ClassName; \
		return IQ(Class,IObject); \
	} \
	FBLibraryModule( ClassName )\
	{\
		FBRegisterObject( ClassName,Path,Label,Description,RegisterControl##ClassName, true, IconFilename );\
		FBRegisterObject( ClassName##1,"FBSDK",Label,Description,RegisterControl##ClassName, true, IconFilename );\
	}

////////////////////////////////////////////////////////////////////////////////////
// General
////////////////////////////////////////////////////////////////////////////////////

//! General directions for UI components.
enum FBOrientation { 
	kFBHorizontal,		//!< Horizontal 
	kFBVertical			//!< Vertical	
};

FB_DEFINE_ENUM( FBSDK_DLL, Orientation ); 

#endif //K_DISABLE_UI


////////////////////////////////////////////////////////////////////////////////////
// Input type, modifiers, and keys
////////////////////////////////////////////////////////////////////////////////////
//! Types of input events.
enum FBInputType {
        kFBKeyPress,            //!< A keyboard key was pressed.
        kFBKeyRelease,          //!< A keyboard key was released.
        kFBButtonPress,         //!< A mouse button was pressed.
        kFBButtonRelease,       //!< A mouse button was released.
        kFBMotionNotify,        //!< The mouse has been moved.
        kFBButtonDoubleClick,   //!< A mouse button was double clicked.
        kFBMouseEnter,          //!< The mouse pointer is entering the window.
        kFBMouseLeave,          //!< The mouse pointer is leaving the window.
        kFBMouseWheelNotify,    //!< The mouse wheel has moved.
        kFBDragging,            //!< The mouse is dragging items.
        kFBDropping,            //!< The mouse is dropping items.
        kFBKeyPressRaw,         //!< A keyboard key was pressed.
        kFBKeyReleaseRaw,       //!< A keyboard key was released.
        kFBUnknownInput         //!< The internal event could not be translated.
};

//! Input Modifiers (Ctrl, Alt, Shift).
enum FBInputModifier {	
		kFBKeyNone	= 0,		//!< No modifier.
		kFBKeyShift	= 1 << 0,	//!< Shift was pressed.
		kFBKeyCtrl	= 1 << 1,	//!< Control was pressed.
		kFBKeyAlt	= 1 << 2	//!< Alt was pressed.
};	

//! Keyboard inputs.
enum FBInputKey 	{	
		kFBKeyReturn	= 0x0D,			//!< Return.
		kFBKeyBackSpace	= 0x08,			//!< Backspace.
		kFBKeyTab		= 0x09,			//!< Tab.
		kFBKeyEscape	= 0x1B,			//!< Escape.

		kFBKeyPageUp	= 0x121,		//!< Page Up.
		kFBKeyPageDown	= 0x122,		//!< Page Down.
		kFBKeyEnd		= 0x123,		//!< End.
		kFBKeyHome		= 0x124,		//!< Home.
		kFBKeyLeft		= 0x125,		//!< Left.
		kFBKeyUp		= 0x126,		//!< Up.
		kFBKeyRight		= 0x127,		//!< Right.
		kFBKeyDown		= 0x128,		//!< Down.

		kFBKeyIns		= 0x12D,		//!< Insert.
		kFBKeyDel		= 0x12E,		//!< Delete.

		kFBKeyF1		= 0x170,		//!< F1.
		kFBKeyF2		= 0x171,		//!< F2.
		kFBKeyF3		= 0x172,		//!< F3.
		kFBKeyF4		= 0x173,		//!< F4.
		kFBKeyF5		= 0x174,		//!< F5.
		kFBKeyF6		= 0x175,		//!< F6.
		kFBKeyF7		= 0x176,		//!< F7.
		kFBKeyF8		= 0x177,		//!< F8.
		kFBKeyF9		= 0x178,		//!< F9.
		kFBKeyF10		= 0x179,		//!< F10.
		kFBKeyF11		= 0x17A,		//!< F11.
		kFBKeyF12		= 0x17B			//!< F12.
};

FB_DEFINE_ENUM( FBSDK_DLL, InputType		);
FB_DEFINE_ENUM( FBSDK_DLL, InputModifier	);
FB_DEFINE_ENUM( FBSDK_DLL, InputKey			);

enum FBColorIndex
{
	// "Traditional" colors
	kFBColorIndexBackground,
	kFBColorIndexBackground1,
	kFBColorIndexForeground,
	kFBColorIndexForeground1,
	kFBColorIndexForeground2,
	kFBColorIndexAlternate1,
	kFBColorIndexAlternate2,
	kFBColorIndexHighlight,
	kFBColorIndexShadow,
	kFBColorIndexWhite,
	kFBColorIndexBlack,
	kFBColorIndexLtGray,
	kFBColorIndexGray,
	kFBColorIndexDkGray,
	kFBColorIndexRed,
	kFBColorIndexGreen,
	kFBColorIndexBlue,
	// Colors to be defined and used by skins
	kFBColorIndexStdButton1,
	kFBColorIndexStdButton2,
	kFBColorIndexFlatButtonActive1,
	kFBColorIndexFlatButtonActive2,
	kFBColorIndexFlatButtonPassive1,
	kFBColorIndexFlatButtonPassive2,
	kFBColorIndexStdCheckboxClear1,
	kFBColorIndexStdCheckboxClear2,
	kFBColorIndexStdCheckboxAmbig1,
	kFBColorIndexStdCheckboxAmbig2,
	kFBColorIndexStdCheckboxDisabled1,
	kFBColorIndexStdCheckboxDisabled2,
	kFBColorIndexStdField1,
	kFBColorIndexStdField2,
	kFBColorIndexStdScrollActive1,
	kFBColorIndexStdScrollActive2,
	kFBColorIndexStdScrollPassive1,
	kFBColorIndexStdScrollPassive2,
	kFBColorIndexStdScrollCursor,
	kFBColorIndexStdListBg1,
	kFBColorIndexStdListBgSelected1,
    kFBColorIndexStdListBgSelected2,
    kFBColorIndexStdListBgFocusSelected1,
	kFBColorIndexStdListText,
	kFBColorIndexStdListTextSelected,
	kFBColorIndexStdListTextDragover,
	kFBColorIndexStdListLine,
	kFBColorIndexStdTabActive1,
	kFBColorIndexStdTabActive2,
	kFBColorIndexStdTabPassive1,
	kFBColorIndexStdTabPassive2,
	kFBColorIndexFlatTabBg1,
	kFBColorIndexFlatTabTextSelected,
	kFBColorIndexDisabled1,
	kFBColorIndexDisabled2,
	kFBColorIndexSelected,
	kFBColorIndexTimelineOutside,
	kFBColorIndexTimelineOutside1,
	kFBColorIndexTimelineCenter,
	kFBColorIndexTimelineCenter1,
	kFBColorIndexTimelineCenter2,
	kFBColorIndexTimelineShadow,
	kFBColorIndexTimelineHighlight,
	kFBColorIndexTimelineManipulator,
    kFBColorIndexStdListBgFocusSelected2,
    kFBColorIndexTreeBgTitle,
	kFBColorIndexTreeBgSubtitle,
	kFBColorIndexTreeBgSubtitleLine,
    kFBColorIndexTreeBgNode,
    kFBColorIndexTreeEditboxBg,
	kFBColorIndexTreeSplitter,
    kFBColorIndexTreeKeyingGroup,
    kFBColorIndexTreeOddRow,
    kFBColorIndexTreeSelect,
	kFBColorIndexTreeKeyingSelect,
	kFBColorIndexDkGreen,
	// Total number of colors
	kFBColorIndexSize
};


#ifndef K_DISABLE_UI

////////////////////////////////////////////////////////////////////////////////////
// FBVisualComponent
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBLayout );
__FB_FORWARD( FBVisualComponent );

FB_FORWARD( FBStringList );

FB_DEFINE_COMPONENT( FBSDK_DLL, FBVisualComponent );

//! Types of attachments between UI regions.
enum FBAttachType {	
	kFBAttachLeft,		//!< Attach to left		[min(x1,x2)]	
	kFBAttachRight,		//!< Attach to right	[max(x1,x2)]	
	kFBAttachTop,		//!< Attach to top		[min(y1,y2)]	
	kFBAttachBottom,	//!< Attach to bottom	[max(y1,y2)]	
	kFBAttachWidth,		//!< Attach to width	[abs(x2-x1)]	
	kFBAttachHeight,	//!< Attach to height	[abs(y2-y1)]	
	kFBAttachCenter,	//!< Attach to center	[center(x1,y1,x2,y2)]	
	kFBAttachNone		//!< No attachment.						
};
FB_DEFINE_ENUM( FBSDK_DLL, AttachType ); 

//! Different border types available.
enum FBBorderStyle {	
	kFBNoBorder,					//!< No border.						
	kFBStandardBorder,				//!< Standard border.				
	kFBEmbossBorder,				//!< Embossed border.				
	kFBEmbossSmoothBorder,			//!< Smooth border.					
	kFBEmbossEdgeSmoothBorder,		//!< Edged smooth border.			
	kFBEmbossSmoothEdgeBorder,		//!< Smoothed edges border.			
	kFBStandardSmoothBorder,		//!< Standard smooth border.			
	kFBStandardEdgeSmoothBorder,	//!< Standard edged smooth border.	
	kFBStandardSmoothEdgeBorder,	//!< Standard smoothed edges border.
	kFBHighlightBorder,				//!< Highlight border.				
	kFBPickingBorder				//!< Picking border.				
};
FB_DEFINE_ENUM( FBSDK_DLL, BorderStyle ); // FBPropertyBorderStyle 

/**	Visual Component base class.
*	All of the user interface elements available in the SDK derive from this class.
*/
class FBSDK_DLL FBVisualComponent : public FBComponent {
	__FBClassDeclare( FBVisualComponent,FBComponent );
protected:
	friend class FBLayout;

	/**	Get internal toolkit object.
	*	\internal 
	*	\return Handle to internal toolkit object.
	*/
	virtual HIKtObject GetHIKtObject();

public:
	//! Constructor.
	FBVisualComponent();

    IObject_Declare(Implementation);		// Interface to IObject
    ICallback_Declare(Implementation);		// Interface to ICallback

	/** Add a child component.
	*	\param	pChild	Visual component to add as a child.
	*	\param	pId		User reference number to associate with \e pChild(default=0).
	*	\return	Operation was successful (\b true or \b false).
	*/
	virtual bool AddChild( HFBVisualComponent pChild,int pId=0);

	/** Get a child component.
	*	\param	pId User reference number to look for child with(default=0).
	*	\return Handle to child (NULL if not found).
	*/
	virtual HFBVisualComponent GetChild( int pId=0 );

	/** Refresh component. 
	*	\param pNow Refresh immediately if \b true (default = \b false).
	*/
    virtual void Refresh(bool pNow=false);

	//! Exposed view callback function.
	virtual void ViewExpose();

	/** Input callback function.
	*	\param pMouseX		Mouse X position.
	*	\param pMouseY		Mouse Y position.
	*	\param pAction		Mouse action.
	*	\param pButtonKey	Keyboard input.
	*	\param pModifier	Keyboard intput modifier.
	*/
	virtual void ViewInput(int pMouseX,int pMouseY,int pAction,int pButtonKey,int pModifier);

	/**	Is component a view?
	*	\return \b true if component is a view.
	*/
	virtual bool IsView();

	/** Region information structure.
	*	Contains region attachment and relative value information.
	*/
	struct FBSDK_DLL __Region
	{
		FBPropertyInt				X;			//!< <b>Property:</b> Region X offset.
		FBPropertyInt				Y;			//!< <b>Property:</b> Region Y offset.
		FBPropertyInt				Width;		//!< <b>Property:</b> Region width offset.
		FBPropertyInt				Height;		//!< <b>Property:</b> Region height offset.

		//! Structure for attachment ratio.
		struct FBSDK_DLL __Ratio
		{
			FBPropertyFloat			X;		//!< <b>Property:</b> Ratio for X attachment.
			FBPropertyFloat			Y;		//!< <b>Property:</b> Ratio for Y attachment.
			FBPropertyFloat			Width;	//!< <b>Property:</b> Ratio for Width attachment.
			FBPropertyFloat			Height;	//!< <b>Property:</b> Ratio for Height attachment.
		} Ratio;

		//! Structure for attachment types.
		struct FBSDK_DLL __AttachType
		{
			FBPropertyAttachType	X;		//!< <b>Property:</b> X Attachment type.
			FBPropertyAttachType	Y;		//!< <b>Property:</b> Y Attachment type.
			FBPropertyAttachType	Width;	//!< <b>Property:</b> Width Attachment type.
			FBPropertyAttachType	Height;	//!< <b>Property:</b> Height Attachment type.
		} AttachType;

		//! Structure for attachment components.
		struct FBSDK_DLL __AttachTo
		{
			FBPropertyVisualComponent	X;		//!< <b>Property:</b> X Attachment source.
			FBPropertyVisualComponent	Y;		//!< <b>Property:</b> Y Attachment source.
			FBPropertyVisualComponent	Width;	//!< <b>Property:</b> Width Attachment source.
			FBPropertyVisualComponent	Height;	//!< <b>Property:</b> Height Attachment source.
		} AttachTo;

		//! Structure for absolute positions
		struct FBSDK_DLL __Pos
		{
			FBPropertyInt	X[2];			//!< <b>Property:</b> Region X position (0=min,1=max).
			FBPropertyInt	Y[2];			//!< <b>Property:</b> Region Y position (0=min,1=max).
		} Position;

	} Region;

	//! Border information structure.
	struct FBSDK_DLL __Border
	{
		FBPropertyString		Caption;			//!< <b>Property:</b> Caption to display in border.
		FBPropertyBool			ShowCaption;		//!< <b>Property:</b> Show caption?
		FBPropertyBorderStyle	Style;				//!< <b>Property:</b> Style of border.
		FBPropertyBool			InSet;				//!< <b>Property:</b> Is border inset?
		FBPropertyInt			Width;				//!< <b>Property:</b> Width of border.
		FBPropertyInt			Spacing;			//!< <b>Property:</b> Spacing of border.
		FBPropertyFloat			MaxAngle;			//!< <b>Property:</b> Max angle for rounding.
		FBPropertyInt			CornerRadius;		//!< <b>Property:</b> Corner radius (rounded).
	} Border;
};


////////////////////////////////////////////////////////////////////////////////////
// FBEventInput
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBEventInput );

//! Input event class.
class FBSDK_DLL FBEventInput : public FBEvent {
	__FBClassDeclare( FBEventInput,FBEvent );
public:
	/**	Constructor.
	*	\param pEvent	 Base event (internal) to obtain information from.
	*/
	FBEventInput( HKEventBase pEvent );		

	FBPropertyInputType InputType;		//!< <b>Read Only Property:</b> Input type.
	FBPropertyInt		X;				//!< <b>Read Only Property:</b> Mouse X Position.
	FBPropertyInt		Y;				//!< <b>Read Only Property:</b> Mouse Y Position.
	FBPropertyInt		MouseButton;	//!< <b>Read Only Property:</b> Mouse Button.
	FBPropertyInt		Key;			//!< <b>Read Only Property:</b> Input key.
	FBPropertyInt		KeyState;		//!< <b>Read Only Property:</b> State of key.
};

////////////////////////////////////////////////////////////////////////////////////
// FBEventDragAndDrop
////////////////////////////////////////////////////////////////////////////////////
//__FB_FORWARD( FBDragAndDrop );
__FB_FORWARD( FBEventDragAndDrop );

//! State of Drag and Drop.
enum FBDragAndDropState {	
	kFBDragAndDropBegin,	//!< Begin a drag and drop sequence.
	kFBDragAndDropDrag,		//!< Dragging.
	kFBDragAndDropDrop,		//!< Dropping.
	kFBDragAndDropEnd,		//!< End of drag and drop.
	kFBDragOnEmpty,			//!< Empty the drag and drop stack.
	kFBDragOnEmptyDrop		//!< Dropping empty stack.
};

FB_DEFINE_ENUM( FBSDK_DLL, DragAndDropState ); 

//! Drag and drop interface.
class FBSDK_DLL FBEventDragAndDrop : public FBEvent 
{
	__FBClassDeclare( FBEventDragAndDrop, FBEvent );

public:
	/**	Constructor.
	*	\param pEvent Base event (internal) to obtain information from.
	*/
	FBEventDragAndDrop ( HKEventBase pEvent );

	/** Accept a drag and drop sequence.
	*	This will cause the region in question to accept a drag and drop action when this
	*	event occurs.
	*/
	virtual void Accept();

	/**	Add an item to the drag and drop list.
	*	\param pComponent Item to add to the list.
	*	\param pId User-defined reference for the item (default = 0 ).
	*/
	virtual void Add( HFBComponent pComponent, int pId=0);

	/** Get the FBComponent specified by \e pIndex from the Drag and Drop list.
	*	\param	pIndex	Index in list where to get FBComponent.
	*	\return	Handle to FBComponent in list at \e pIndex.
	*/
	virtual HFBComponent Get( int pIndex );

	//! Clear drag and drop list.
	virtual void Clear();			

	/**	Get the number of items in the DragAndDrop list.
	*	\return Number of items in DragAndDrop list.
	*/
	virtual int GetCount();

	//-- Properties
	FBPropertyDragAndDropState	State;		//!< <b>Property:</b> Drag and drop sub-event.
	FBPropertyInt				PosX;		//!< <b>Property:</b> X position of mouse.
	FBPropertyInt				PosY;		//!< <b>Property:</b> Y position of mouse.
	FBPropertykReference		Data[2];	//!< <b>Property:</b> User specified reference. (for example, FBSpread:row)
};

////////////////////////////////////////////////////////////////////////////////////
// FBEventShow
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBEventShow );

//! Show event class.
class FBSDK_DLL FBEventShow : public FBEvent {
	__FBClassDeclare( FBEventShow,FBEvent );
public:
	/**	Constructor.
	*	\param pEvent Base event (internal) to obtain information from.
	*/
	FBEventShow( HKEventBase pEvent );		

	FBPropertyBool	Shown;		//!< <b>Read Only Property:</b> Was layer just shown?
};

////////////////////////////////////////////////////////////////////////////////////
// FBEventActivate
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBEventActivate );

//! Activation event.
class FBSDK_DLL FBEventActivate : public FBEvent {
	__FBClassDeclare( FBEventActivate,FBEvent );
public:
	/**	Constructor.
	*	\param pEvent Base event (internal) to obtain information from.
	*/
	FBEventActivate( HKEventBase pEvent );

	FBPropertykReference Data;				//!< <b>Read Write Property:</b> Generic data of event.
};

////////////////////////////////////////////////////////////////////////////////////
// FBEventMenu
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD(FBEventMenu);
__FB_FORWARD(FBMenuItem);

/**	Menu Item class declaration.
*	\param	ClassName	Name of menu item class to declare.
*	\param	Parent		Name of menu item class's parent class.
*/
#define FBMenuItemDeclare( ClassName,Parent ) \
	FBClassDeclare( ClassName,Parent ); \
	public: \
		ClassName():Parent() { FBClassInit; FBCreate(); } \
	private:

/**	Menu item class implementation.
*	\param	ThisComponent	Menu item class to implement.
*/
#define FBMenuItemImplementation( ThisComponent ) \
	FBClassImplementation( ThisComponent )

/**	Menu item registration.
*	\param	ClassName	Menu item class to register.
*/
#define FBRegisterMenuItem( ClassName ) \
	HIObject RegisterMenuItem##ClassName( HIObject /*pOwner*/,char * /*pName*/,void * /*pData*/) \
	{\
		return NULL;\
	}\
	FBLibraryModule( ClassName )	\
	{	\
	}	\
	ClassName* g##ClassName;

/**	Menu item activation.
*	\param	ClassName	Menu item class to activate.
*/
#define FBMenuItemActivation( ClassName )	\
	extern ClassName* g##ClassName;			\
	g##ClassName = new ClassName;			\
	g##ClassName->Activate();

/**	Get the handle of a menu item.
*	\param	ClassName	Menu item class to get a handle for.
*	\retval	Handle		Handle to the menu item.	
*/
#define FBMenuItemHandle( ClassName, Handle )	\
	extern ClassName* g##ClassName;				\
	Handle = g##ClassName;

//! Types of menu items available.
enum FBMenuItemType {	
	kFBMenuItemMotionImport,	//!< Motion Files->Import.
	kFBMenuItemSceneImport, 	//!< Scenes->Import.
	kFBMenuItemMotionExport,	//!< Motion Files->Export.
	kFBMenuItemSceneExport		//!< Scenes->Export.
};

FB_DEFINE_ENUM( FBSDK_DLL, MenuItemType );

//! Menu event
class FBSDK_DLL FBEventMenu : public FBEvent
{
	__FBClassDeclare(FBEventMenu, FBEvent);

public:
	FBPropertyInt		Id;			//!< <b>Read Write Property:</b> Id number for menu item.
	FBPropertyString	Name;		//!< <b>Read Write Property:</b> Name of menu item.

	/** Constructor.
	*	\param	pEvent	Base event object.
	*/
	FBEventMenu( HKEventBase pEvent);
};

//! Menu Item.
class FBSDK_DLL FBMenuItem : public FBVisualComponent
{
	__FBClassDeclare( FBMenuItem, FBVisualComponent );

public:
	//! Constructor.
	FBMenuItem();

	/**	Initialization function.
	*	\param	pType			Type of menu item to be set.
	*	\param	pExtension		File extension to use as filter for popup.
	*	\param	pLabel			Short description of file format (usually one word identifier)
	*	\param	pDescription	Long description of file format
	*/
	void Set(FBMenuItemType	pType, char* pExtension, char* pLabel, char* pDescription);

	/** Is menu item active?
	*	\return \b true if menu item has been registered (activated).
	*/
	bool IsActive();

	/** Activate menu item.
	*	This will register the callback functions for the menu item, as well as make the item
	*	itself appear in the menu. Once activated, you have no more control over the 
	*	menu item, seeing as it will be registered and cannot be activated twice.
	*	This function should be the last one you call when activating the menu item.
	*	\param	pState	Activateif \b true (default = \b true).
	*/
	void Activate( bool pState = true );

	/** Internal callback function for activated menu item.
	*	Registers internally the menu item.
	*	\param	pSender	Sending register.
	*	\param	pEvent	Event being sent.
	*/
	virtual void OnItemSelectEvent(HIRegister pSender, HKEvent pEvent);

public:
	FBPropertyComponent			Component;			//!< <b>Read Write Property:</b> Component handle.
	FBPropertyString			Extension;			//!< <b>Read Write Property:</b> Default extension.
	FBPropertyString			Label;				//!< <b>Read Write Property:</b> Label of the menu item.
	FBPropertyString			Description;		//!< <b>Read Write Property:</b> Description of the menu item.
	FBPropertyMenuItemType		Type;				//!< <b>Read Write Property:</b> Type of menu item.
	
public:
	/**	The callback function for when the menu item is selected.
	*	In order to use this class correctly, you must create a class that inherits directly
	*	from this class, overloading this virtual function. This function will then be called 
	*	when the menu item is selected.
	*	\param	pFilename	Filename from popup (default = NULL).
	*	\return	Operation was successful (\b true or \b false).
	*/
	virtual	bool Execute(char *pFilename=NULL)	= 0;
};

////////////////////////////////////////////////////////////////////////////////////
// FBLayout
////////////////////////////////////////////////////////////////////////////////////

/** User interface layout class.
*	The user interface in MOTIONBUILDER is managed using layout
*	objects (class FBLayout). Layouts are responsible for
*	managing areas of the screen called regions. UI
*	components such as buttons, viewers, and edit boxes
*	are bound to regions. In other words, when a UI
*	component is bound to a region, the region defines
*	how big it is and how it behaves when the layout is
*	resized.
*
*	<b>Types of Layouts</b>
*	With the different types of plugins available, there are
*	different kinds of layouts, depending on where the
*	plugin fits into the MOTIONBUILDER architecture. The following
*	types of layouts exist:
*	<ul>
*	<li>Device layouts.
*	<li>Constraint layouts.</li>
*	<li>Manipulator layouts.</li>
*	<li>Shader layouts.</li>
*	</ul>
*/
class FBSDK_DLL FBLayout : public FBVisualComponent {
	__FBClassDeclare( FBLayout,FBVisualComponent );
public:

	//! Type of split style (sub-division) for layout.
	enum FBSplitStyle 
	{	
		kFBNoSplit	= 0,	//!< No split.
		kFBHSplit	= 1,	//!< Horizontal split.
		kFBVSplit	= 2,	//!< Vertical split.
		kFBHVSplit	= 3		//!< Horizontal and Vertical split.
	};

	//! Constructor.
	FBLayout(HIObject pObject=NULL);

	/**	Add a region to the layout.
	*	\param	pName		Name of region.
	*	\param	pTitle		Title to display.
	*
	*	\param	pX			X: Position.
	*	\param	pXType		X: Type of Attachment.
	*	\param	pXRelative	X: Item to attach to.
	*	\param	pMultX		X: Multiplier of relative value.
	*
	*	\param	pY			Y: Position.
	*	\param	pYType		Y: Type of Attachment.
	*	\param	pYRelative	Y: Item to attach to.
	*	\param	pMultY		Y: Multiplier of relative value.
	*
	*	\param	pW			W: Width of region.
	*	\param	pWType		W: Type of Attachment.
	*	\param	pWRelative	W: Item to attach to.
	*	\param	pMultW		W: Multiplier of relative value.
	*
	*	\param	pH			H: Height of region.
	*	\param	pHType		H: Type of Attachment.
	*	\param	pHRelative	H: Item to attach to.
	*	\param	pMultH		H: Multiplier of relative value.
	*
	*	\return	Operation was successful (\b true or \b false).
	*/
	virtual bool AddRegion(	char *pName,	char *pTitle,
							int pX,	FBAttachType pXType,	char *pXRelative,	float pMultX,
							int pY,	FBAttachType pYType,	char *pYRelative,	float pMultY,
							int pW,	FBAttachType pWType,	char *pWRelative,	float pMultW,
							int pH,	FBAttachType pHType,	char *pHRelative,	float pMultH);

	/**	Move a region.
	*	\param	pName		Name of region to move.
	*	\param	pX			New X position.
	*	\param	pY			New Y position.
	*	\return	Operation was successful (\b true or \b false).
	*/
	virtual bool MoveRegion( char *pName, int pX, int pY );

	/**	Change a region's size.
	*	\param	pName		Name of region to resize.
	*	\param	pW			New region width.
	*	\param	pH			New region height.
	*	\return	Operation was successful (\b true or \b false).
	*/
	virtual bool SizeRegion( char *pName, int pW, int pH );

	/**	Remove a region.
	*	\param	pName		Name of region to remove.
	*	\return	Operation was successful (\b true or \b false).
	*/
	virtual bool RemoveRegion( char *pName );

	/** Rename a region.
	*	\param	pOldName	Region's old name.
	*	\param	pNewName	Region's new name.
	*	\return	Operation was successful (\b true or \b false).
	*/
	virtual bool RenameRegion( char *pOldName, char *pNewName );

	/**	Verify if a region with \e pName exists.
	*	\param	pName		Name of region to check.
	*	\return	Operation was successful (\b true or \b false).
	*/
	virtual bool GetRegion( char *pName );

	/**	Get region \e pName information (position and size)
	*	\param	pName		Name of region.
	*	\param	pComputed	Is the information retrieved relative or absolute?
	*	\retval	pX			Position in X of the region.
	*	\retval	pY			Position in Y of the region.
	*	\retval pW			Width of the region.
	*	\retval pH			Height of the region.
	*	\return	Operation was successful (\b true or \b false).
	*/
	virtual bool GetRegionPositions	(	char *pName,	bool pComputed,
										int *pX,		int *pY, 
										int *pW=NULL,	int *pH=NULL	);

	//@{
	/**	Set control of a region to a visual component.
	*	\param	pName		Name of region to affect.
	*	\param	pComponent	Component to control region.
	*	\return	Operation was successful (\b true or \b false).
	*/
	virtual bool SetControl( char *pName, HFBVisualComponent pComponent );
	virtual bool SetControl( char *pName, FBVisualComponent &pComponent );
	//@}

	/** Set or Get internal toolkit: object.
	*	\internal 
	*	\param	pName		Name of region.
	*	\return	Handle to internal toolkit object.
	*/
	virtual HIKtObject GetHIKtObject( char *pName );

	/** Set internal toolkit view.
	*	\internal 
	*	\param	pName		Name of region.
	*	\param	pView		View to set as view for region.
	*	\return	Operation was successful (\b true or \b false).
	*/
	virtual bool SetHIKtView( char *pName, HIKtView pView );

	/** Get internal toolkit view.
	*	\internal
	*	\param	pName		Name of region.
	*	\return	Handle to internal toolkit view object.
	*/
	virtual HIKtView GetHIKtView( char *pName );

	//@{
	/**	Set view.
	*	\param	pName		Name of Region.
	*	\param	pComponent	Component to set as view.
	*	\return	Operation was successful (\b true or \b false).
	*/
	virtual bool SetView( char *pName, HFBVisualComponent pComponent );
	virtual bool SetView( char *pName, FBVisualComponent &pComponent );
	//@}

	/** Set border properties for a region.
	*	\param	pName			Name of Region to change border properties.
	*	\param	pType			Border style to use.
	*	\param	pShowTitle		Show region title?
	*	\param	pInSet			Is region inset?
	*	\param	pWidth			Border width.
	*	\param	pSpacing		Border spacing.
	*	\param	pMaxAngle		Max angle for rounding.
	*	\param	pCornerRadius	Corner radius for rounding.
	*	\return	Operation was successful (\b true or \b false).
	*/
	virtual bool SetBorder	(	char *pName,		FBBorderStyle pType,	bool pShowTitle,
								bool pInSet,		int pWidth,				int pSpacing,
								float pMaxAngle,	int pCornerRadius );
	
	/**	Set a region's splitstyle.
	*	\param	pName		Name of Region to set splitstyle.
	*	\param	pRegionType	Split style give to region.
	*	\return	Operation was successful (\b true or \b false).
	*/
	virtual bool SetSplitStyle ( char *pName, FBSplitStyle pRegionType );

	/**	Get a region's splitstyle.
	*	\param	pName		Name of Region to get splitstyle from.
	*	\return Split style of specified region.
	*/
	virtual FBSplitStyle GetSplitStyle ( char *pName );

	/** Set a region's title.
	*	\param	pName		Name of region to change title.
	*	\param	pTitle		New title for region.
	*	\return	Operation was successful (\b true or \b false).
	*/
	virtual bool SetRegionTitle ( char *pName, char* pTitle );

    IObject_Declare(Implementation);		// Interface to IObject.

	//--- Events
	FBPropertyEvent		OnInput;		//!< <b>Event:</b> Input.
	FBPropertyEvent		OnPaint;		//!< <b>Event:</b> Paint layout.
	FBPropertyEvent		OnIdle;			//!< <b>Event:</b> Idle.
	FBPropertyEvent		OnShow;			//!< <b>Event:</b> Show layout.
	FBPropertyEvent		OnResize;		//!< <b>Event:</b> Resize layout.
};

////////////////////////////////////////////////////////////////////////////////////
// FBTabPanel
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBTabPanel );

//! Tab panel.
class FBSDK_DLL FBTabPanel : public FBVisualComponent {
	__FBClassDeclare( FBTabPanel,FBVisualComponent );
public:
	//! Constructor.
	FBTabPanel();

	FBPropertyStringList	Items;			//!< <b>List:</b> Names for tab panels.
	FBPropertyBool			Enabled;		//!< <b>Read Write Property:</b> Is tab panel enabled?
	FBPropertyInt			ItemIndex;		//!< <b>Read Write Property:</b> Current tab panel.
	FBPropertyLayout		Layout;			//!< <b>Read Write Property:</b> Layout for current tab panel.
	FBPropertyEvent			OnChange;		//!< <b>Event:</b> Tab panel change.
};

////////////////////////////////////////////////////////////////////////////////////
// FBButton
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBButton );

/** Style of buttons.
*	\warning Not all button styles are completely functional.
*/
enum FBButtonStyle {	
	kFBPushButton,		//!< Normal button.
	kFBBitmapButton,	//!< Button with bitmap on it.
	kFBCheckbox,		//!< Check box.
	kFBRadioButton,		//!< Radio button.
	kFB2States,			//!< 2 state button (2 colors).
	kFBBitmap2States	//!< 2 state button with 2 bitmaps.
};

/** Possible button states.
*   Currently, only two button states are possible.
*/
enum FBButtonState {
    kFBButtonState0,    //!< State is 0, usually meaning not active
    kFBButtonState1     //!< State is 1, usually meaning active
};

FB_DEFINE_ENUM( FBSDK_DLL, ButtonStyle ); 

//! Button.
class FBSDK_DLL FBButton : public FBVisualComponent {
	__FBClassDeclare( FBButton,FBVisualComponent );
public:
	//! Constructor.
	FBButton();
    
    /** Queries the color associated with a button state.
    *   This method is only useful for buttons of style kFB2States.
	*	\param	pState	The state to be queried.
	*	\return The color vector.
	*/
    FBColor GetStateColor( FBButtonState pState );

    /** Returns whether or not the item \e pIndex is currently selected.
	*	\param	pState	The state to be set.
	*	\param	pColor	The desired color vector.
	*/
    void    SetStateColor( FBButtonState pState, const FBColor& pColor );

	FBPropertyString		Caption;		//!< <b>Read Write Property:</b> Caption to appear on button.
	FBPropertyBool			Enabled;		//!< <b>Read Write Property:</b> Is button enabled?
	FBPropertyInt			State;			//!< <b>Read Write Property:</b> Current state of button.
	FBPropertyButtonStyle	Style;			//!< <b>Read Write Property:</b> Button style.
	FBPropertyBool			ReadOnly;		//!< <b>Read Write Property:</b> Is button read only?

	FBPropertyEvent			OnClick;		//!< <b>Event:</b> Button clicked.
};

////////////////////////////////////////////////////////////////////////////////////
// FBLabel
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBLabel );

//! Text justification styles.
enum FBTextJustify {	
	kFBTextJustifyLeft,		//!< Left justify.
	kFBTextJustifyRight,	//!< Right justify.
	kFBTextJustifyCenter	//!< Center alignment.
};

FB_DEFINE_ENUM( FBSDK_DLL, TextJustify ); 

//! Text label.
class FBSDK_DLL FBLabel : public FBVisualComponent {
	__FBClassDeclare( FBLabel,FBVisualComponent );
public:
	//! Constructor.
	FBLabel();

	FBPropertyString		Caption;		//!< <b>Read Write Property:</b> Caption to appear in label.
	FBPropertyBool			Enabled;		//!< <b>Read Write Property:</b> Is label enabled?
	FBPropertyTextJustify	Justify;		//!< <b>Read Write Property:</b> Text justification for label.
};

////////////////////////////////////////////////////////////////////////////////////
// FBImage
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBImage );

//! Image.
class FBSDK_DLL FBImage : public FBVisualComponent {
	__FBClassDeclare( FBImage,FBVisualComponent );
public:
	//! Constructor.
	FBImage();

	FBPropertyString		Filename;		//!< <b>Read Write Property:</b> Filename for image.
	FBPropertyInt			Width;			//!< <b>Read Write Property:</b> Image width.
	FBPropertyInt			Height;			//!< <b>Read Write Property:</b> Image height.

	FBPropertyEvent			OnDragAndDrop;	//!< <b>Event:</b> Drag and drop.
};

////////////////////////////////////////////////////////////////////////////////////
// FBEdit
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBEdit );

//! Text edit box.
class FBSDK_DLL FBEdit : public FBVisualComponent {
	__FBClassDeclare( FBEdit,FBVisualComponent );
public:
	//! Constructor.
	FBEdit();

	FBPropertyString		Text;			//!< <b>Read Write Property:</b> Text displayed.
	FBPropertyBool			Enabled;		//!< <b>Read Write Property:</b> Is text edit box enabled?
	FBPropertyBool			PasswordMode;	//!< <b>Read Write Property:</b> Set password mode for this edit box.

	FBPropertyEvent			OnChange;		//!< <b>Event:</b> Text changed.
};

////////////////////////////////////////////////////////////////////////////////////
// FBEditNumber
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBEditNumber );

//! Number edit box.
class FBSDK_DLL FBEditNumber : public FBVisualComponent {
	__FBClassDeclare( FBEditNumber,FBVisualComponent );
public:
	//! Constructor.
	FBEditNumber();

	FBPropertyBool			Enabled;		//!< <b>Read Write Property:</b> Is number edit box enabled?
	FBPropertyDouble		Value;			//!< <b>Read Write Property:</b> Current value.
	FBPropertyDouble		Min;			//!< <b>Read Write Property:</b> Minimum value.
	FBPropertyDouble		Max;			//!< <b>Read Write Property:</b> Maximum value.
	FBPropertyDouble		Precision;		//!< <b>Read Write Property:</b> Precision of value.
	FBPropertyDouble		LargeStep;		//!< <b>Read Write Property:</b> Large step value.
	FBPropertyDouble		SmallStep;		//!< <b>Read Write Property:</b> Small step value.

	FBPropertyEvent			OnChange;		//!< <b>Event:</b> Number changed.
};

////////////////////////////////////////////////////////////////////////////////////
// FBEditColor
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBEditColor );

//! Color edit widget.
class FBSDK_DLL FBEditColor : public FBVisualComponent {
	__FBClassDeclare( FBEditColor,FBVisualComponent );
public:
	//! Constructor.
	FBEditColor();

	FBPropertyBool			Enabled;		//!< <b>Read Write Property:</b> Is color edit widget enabled?
	FBPropertyColor			Value;			//!< <b>Read Write Property:</b> Current value of color.
	FBPropertyInt			ColorMode;		//!< <b>Read Write Property:</b> 3 for RGB, 4 for RGBA (Default = 3)

	FBPropertyEvent			OnChange;		//!< <b>Event:</b> Color changed.
};

////////////////////////////////////////////////////////////////////////////////////
// FBEditVector
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBEditVector );

//! Vector edit widget.
class FBSDK_DLL FBEditVector : public FBVisualComponent {
	__FBClassDeclare( FBEditVector,FBVisualComponent );
public:
	//! Constructor.
	FBEditVector();

	FBPropertyBool			Enabled;		//!< <b>Read Write Property:</b> Is color edit widget enabled?
	FBPropertyVector3d		Value;			//!< <b>Read Write Property:</b> Current value of vector.

	FBPropertyEvent			OnChange;		//!< <b>Event:</b> Vector value changed.
};

////////////////////////////////////////////////////////////////////////////////////
// FBList
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBList );

//! List style or direction.
enum FBListStyle { 
	kFBDropDownList,	//!< Drop down list.
	kFBVerticalList		//!< Vertical list.
};

FB_DEFINE_ENUM( FBSDK_DLL, ListStyle );	// FBPropertyListStyle

//! List of items.
class FBSDK_DLL FBList : public FBVisualComponent {
	__FBClassDeclare( FBList,FBVisualComponent );
public:
	//! Constructor.
	FBList();

	FBPropertyStringList	Items;			//!< <b>List:</b> Names of items in list.

	FBPropertyInt			ItemIndex;		//!< <b>Read Write Property:</b> Current item index.
	FBPropertyBool			Enabled;		//!< <b>Read Write Property:</b> Is list enabled?
	FBPropertyBool			MultiSelect;	//!< <b>Read Write Property:</b> Can multiple items be selected?
	FBPropertyBool			ExtendedSelect;	//!< <b>Read Write Property:</b> Extended selection state?
	FBPropertyListStyle		Style;			//!< <b>Read Write Property:</b> Style or direction of list.

	FBPropertyEvent			OnChange;		//!< <b>Event:</b> List changed.
	FBPropertyEvent			OnDragAndDrop;	//!< <b>Event:</b> Drag and drop event.

	/** Returns whether or not the item \e pIndex is currently selected.
	*	\param	pIndex	Index to see if select or not.
	*	\return \b true if item at \e pIndex is selected.
	*/
	virtual bool IsSelected( int pIndex );

	/** Set the current selected state of item at \e pIndex to \e pSelected.
	*	\param	pIndex		Index to affect item at.
	*	\param	pSelected	State to set item at \e pIndex to.
	*/
	virtual void Selected( int pIndex, bool pSelected );
};

////////////////////////////////////////////////////////////////////////////////////
// FBSlider
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBSlider );

//! Slider.
class FBSDK_DLL FBSlider : public FBVisualComponent {
	__FBClassDeclare( FBSlider,FBVisualComponent );
public:
	//! Constructor.
	FBSlider();

	FBPropertyBool			Enabled;			//!< <b>Read Write Property:</b> Is slider enabled?
	FBPropertyDouble		Min;				//!< <b>Read Write Property:</b> Minimum value.
	FBPropertyDouble		Max;				//!< <b>Read Write Property:</b> Maximum value.
	FBPropertyDouble		Value;				//!< <b>Read Write Property:</b> Current value.
	FBPropertyOrientation	Orientation;		//!< <b>Read Write Property:</b> Slider orientation.

	FBPropertyEvent			OnChange;			//!< <b>Event:</b> Slider value changed.
};

////////////////////////////////////////////////////////////////////////////////////
// FBLayoutRegion
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBLayoutRegion );

//! Layout region.
class FBSDK_DLL FBLayoutRegion : public FBVisualComponent {
	__FBClassDeclare( FBLayoutRegion,FBVisualComponent );
public:
	//! Constructor.
	FBLayoutRegion();
};

////////////////////////////////////////////////////////////////////////////////////
// FBThermometer
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBThermometer );

//! Thermometer.
class FBSDK_DLL FBThermometer : public FBVisualComponent {
	__FBClassDeclare( FBThermometer,FBVisualComponent );
public:
	//! Constructor.
	FBThermometer();

	FBPropertyString	Caption;		//!< <b>Read Write Property:</b> Caption for thermometer.
	FBPropertyBool		Enabled;		//!< <b>Read Write Property:</b> Is thermometer enabled?
	FBPropertyFloat		Min;			//!< <b>Read Write Property:</b> Minimum value.
	FBPropertyFloat		Max;			//!< <b>Read Write Property:</b> Maximum value.
	FBPropertyFloat		Value;			//!< <b>Read Write Property:</b> Current value.
};

////////////////////////////////////////////////////////////////////////////////////
// FBSpread
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBSpread );

//! Different styles of spreadsheet cell styles.
enum FBCellStyle  {	
	kFBCellStyleDefault,		//!< Default cell style.
	kFBCellStyleString,			//!< String.
	kFBCellStyleDouble,			//!< Double.
	kFBCellStyleInteger,		//!< Integer.
	kFBCellStyleButton,			//!< Button.
	kFBCellStyle2StatesButton,	//!< 2 state button.
	kFBCellStyle3StatesButton,	//!< 3 state button.
	kFBCellStyleMenu,			//!< Menu.
	kFBCellStyleVoid,			//!< Void (no value).
	kFBCellStyleView,			//!< View (user definable).
    kFBCellStyleTime            //!< Time.
};

FB_DEFINE_ENUM( FBSDK_DLL, CellStyle ); // FBPropertyCellStyle

__FB_FORWARD( FBSpreadPart );

/** Spreadsheet part.
*	\remarks Due to protected constructor, this can only be created by a child object.
*/
class FBSDK_DLL FBSpreadPart : public FBComponent {
	__FBClassDeclare( FBSpreadPart,FBComponent );
protected:
	/**	Constructor.
	*	\param		pParent	Parent Spreadsheet.
	*	\remarks	This constructor is protected, therefore only children of this object may be created.
	*/
	FBSpreadPart( HFBSpread pParent );

public:
	FBPropertyInt			Row;			//!< <b>Read Only Property:</b> Row number.
	FBPropertyInt			Column;			//!< <b>Read Only Property:</b> Column number.
	FBPropertyBool			ReadOnly;		//!< <b>Read Write Property:</b> Is SpreadPart read-only?
	FBPropertyBool			Enabled;		//!< <b>Read Write Property:</b> Is SpreadPart enabled?
	FBPropertyCellStyle		Style;			//!< <b>Read Write Property:</b> Style of cell
	FBPropertyTextJustify	Justify;		//!< <b>Read Write Property:</b> Text justification for SpreadPart
};

__FB_FORWARD( FBSpreadCell );

//! Spreadsheet cell.
class FBSDK_DLL FBSpreadCell : public FBSpreadPart {
	__FBClassDeclare( FBSpreadCell, FBSpreadPart );
public:

	/**	Constructor.
	*	\param	pParent	Parent spreadsheet.
	*	\param	pRow	Row to which cell belongs.
	*	\param	pCol	Column to which cell belongs.
	*/
	FBSpreadCell( HFBSpread pParent, kReference pRow, int pCol );
};

__FB_FORWARD( FBSpreadRow );

//! Spreadsheet row.
class FBSDK_DLL FBSpreadRow : public FBSpreadPart {
	__FBClassDeclare( FBSpreadRow,FBSpreadPart );
public:
	/**	Constructor.
	*	\param	pParent	Parent spreadsheet.
	*	\param	pRow	User-defined reference to assign to row.
	*/
	FBSpreadRow( HFBSpread pParent, kReference pRow );

	//! Remove (destroy) row.
	virtual void Remove();

	/**	Edit the row caption.
	*	This will initiate the UI edit of a row caption.
	*	\return	Operation was successful (\b true or \b false).
	*/
	virtual bool EditCaption();

	FBPropertyString		Caption;		//!< <b>Read Write Property:</b> Caption to display with row.
	FBPropertykReference	Parent;		//!< <b>Read Write Property:</b> Parent of row (reference).
	FBPropertyBool			Selected;		//!< <b>Read Write Property:</b> Is row selected? 
};

__FB_FORWARD( FBSpreadColumn );

//! Spreadsheet column.
class FBSDK_DLL FBSpreadColumn : public FBSpreadPart {
	__FBClassDeclare( FBSpreadColumn,FBSpreadPart );
public:
	/**	Constructor.
	*	\param	pParent	Parent spreadsheet.
	*	\param	pCol	Column number.
	*/
	FBSpreadColumn( HFBSpread pParent, int pCol );

	FBPropertyString		Caption;		//!< <b>Read Write Property:</b> Caption of the column.
	FBPropertyInt			Width;			//!< <b>Read Write Property:</b> Column width.
	FBPropertyTextJustify	Justify;		//!< <b>Read Write Property:</b> Text justification.
};

//! Base spreadsheet class.
class FBSDK_DLL FBSpread : public FBVisualComponent {
	__FBClassDeclare( FBSpread,FBVisualComponent );
public:
	/** Constructor.
	*/
	FBSpread();

	FBPropertyString		Caption;		//!< <b>Read Write Property:</b> Caption to display for spreadsheet.
	FBPropertykReference	Row;			//!< <b>Read Write Property:</b> Current row.
	FBPropertyInt			Column;			//!< <b>Read Write Property:</b> Current column.
	FBPropertyBool			MultiSelect;	//!< <b>Read Write Property:</b> Can there be multiple selections?

	FBPropertyEvent			OnCellChange;	//!< <b>Event:</b> Cell value changed.
	FBPropertyEvent			OnRowClick;		//!< <b>Event:</b> Row clicked.
	FBPropertyEvent			OnColumnClick;	//!< <b>Event:</b> Column clicked.
	FBPropertyEvent			OnDragAndDrop;	//!< <b>Event:</b> Drag and drop event.

	/** Add a row.
	*	\param	pString		Text to display with row.
	*	\param	pRef		User-defined reference for row(default=0).
	*/
	virtual void RowAdd( char * pString, kReference pRef=0 );

	/** Sort rows.
	*	\param	pAscending	If \b true, sort ascending.
	*/
	virtual void RowSort( bool pAscending=true );

	/**	Add a column.
	*	\param	pString		Text to display with column.
	*	\param	pRef		User-define column reference number(default=0).
	*/
	virtual void ColumnAdd( char * pString, kReference pRef=0 );

	/** Get the current cell.
	*	\return	A copy of the the current cell.
	*/
	virtual FBSpreadCell GetCurrentCell();
	
	/**	Get a cell from row and column numbers.
	*	\param	pRef		Row reference.
	*	\param	pColumn		Column number.
	*	\return	A copy of the cell.
	*/
	virtual FBSpreadCell GetCell( kReference pRef,int pColumn );

	/**	Get a column from a column number.
	*	\param	pColumn		Column number.
	*	\return A copy of column.
	*/
	virtual FBSpreadColumn GetColumn( int pColumn );

	/**	Get a row from a row reference.
	*	\param	pRef	Reference to a row.
	*	\return	A copy of the row.
	*/
	virtual FBSpreadRow GetRow( kReference pRef );
	
	/** Set a cell's value.
	*	\param	pRef		Row of cell.
	*	\param	pColumn		Column of cell.
	*	\param	pString		Cell's new value.
	*/
	virtual void SetCell( kReference pRef,int pColumn, char*        pString );

	//@{
	/** Set a cell's value.
	*	\param	pRef		Row of cell.
	*	\param	pColumn		Column of cell.
	*	\param	pValue		Cell's new value.
	*/
	virtual void SetCell( kReference pRef,int pColumn, int		    pValue );
    virtual void SetCell( kReference pRef,int pColumn, kLongLong    pValue );
	virtual void SetCell( kReference pRef,int pColumn, double	    pValue );
	//@}

	/**	Get a cell's value.
	*	\param	pRef		Row of cell.
	*	\param	pColumn		Column of cell.
	*	\retval	pString		String value of cell.
	*/
	virtual void GetCell( kReference pRef,int pColumn, char*	    &pString );

	//@{
	/**	Get a cell's value.
	*	\param	pRef		Row of cell.
	*	\param	pColumn		Column of cell.
	*	\retval	pValue		Numeric value of cell.
	*/
	virtual void GetCell( kReference pRef,int pColumn, int		    &pValue );
    virtual void GetCell( kReference pRef,int pColumn, kLongLong    &pValue );
	virtual void GetCell( kReference pRef,int pColumn, double	    &pValue );
	//@}

	/** Set a cell's internal toolkit view.
	*	\internal 
	*	\param	pRef		Row of cell.
	*	\param	pColumn		Column of cell.
	*	\param	pView		View to use to set cell's view.
	*/
	virtual void SetCellView( kReference pRef,int pColumn, HIKtView pView );

	/**	Get a cell's internal toolkit view.
	*	\internal 
	*	\param	pRef		Row of cell.
	*	\param	pColumn		Column of cell.
	*	\retval pView		Handle of view.
	*/
	virtual void GetCellView( kReference pRef,int pColumn, HIKtView &pView );

	/** Clear spreadsheet
	*	This function will empty spreadsheet of all its rows, columns and cells.
	*/
	virtual void Clear();
};

__FB_FORWARD( FBEventSpread );

//! Spreadsheet event.
class FBSDK_DLL FBEventSpread : public FBEvent {
	__FBClassDeclare( FBEventSpread,FBEvent );
public:
	/**	Constructor.
	*	\param pEvent Base event (internal) to obtain information from.
	*/
	FBEventSpread( HKEventBase pEvent );

	FBPropertyInt Row;				//!< <b>Read Only Property:</b> Row of event.
	FBPropertyInt Column;			//!< <b>Read Only Property:</b> Column of event.

	/** <b>Read Only Property:</b> Action associated to the spread event. 
	*	This value is not always set and depends on the action being performed on the spreadsheet.
	*/
	FBPropertyInt Action;
};

////////////////////////////////////////////////////////////////////////////////////
// FBContainer
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBContainer );

//! Different icon positions possible.
enum FBIconPosition {	
	kFBIconLeft,		//!< Icon on left of text.
	kFBIconTop,			//!< Icon on top of text.
};

FB_DEFINE_ENUM( FBSDK_DLL, IconPosition ); // FBPropertyIconPosition

//! Container
class FBSDK_DLL FBContainer : public FBVisualComponent {
	__FBClassDeclare( FBContainer,FBVisualComponent );
public:
	/** Constructor.
	*/
	FBContainer();

	/**	Set an item's icon.
	*	\param	pRef					Reference to item in container.
	*	\param	pImage					Handle to image to use.
	*	\param	pUseACopyOfTheImage		Create a copy of the image?(default=true)
	*	\return	Operation was successful (\b true or \b false).
	*/
	virtual bool ItemIconSet( kReference pRef, HFBImage pImage, bool pUseACopyOfTheImage=true );

	/**	Set an item's icon.
	*	\param	pRef		Reference to item in container.
	*	\param	pFilename	Name of file where image is located.
	*	\return	Operation was successful (\b true or \b false).
	*/
	virtual bool ItemIconSet( kReference pRef, char *pFilename );

	/**	Edit a container item.
	*	\param	pRef	 Reference of container to edit.
	*	\return	Operation was successful (\b true or \b false).
	*/
	virtual bool ItemNameEdit( kReference pRef );

	/** Get the selected item.
	*	\return Index of current selection.
	*/
	virtual int GetSelection();

	FBPropertyStringList	Items;					//!< <b>List:</b> Names of items in container.

	FBPropertyBool			Enabled;				//!< <b>Read Write Property:</b> Is container enabled?
	FBPropertyInt			ItemIndex;				//!< <b>Read Write Property:</b> Current item selected.
	FBPropertyOrientation	Orientation;			//!< <b>Read Write Property:</b> Orientation of container.
	FBPropertyBool			ItemWrap;				//!< <b>Read Write Property:</b> Are items wrapped when enough space is available?
	FBPropertyIconPosition	IconPosition;			//!< <b>Read Write Property:</b> Where the icon is positioned for the items.
	FBPropertyInt			ItemWidth;				//!< <b>Read Write Property:</b> Item width.
	FBPropertyInt			ItemHeight;				//!< <b>Read Write Property:</b> Item height.
													
	FBPropertyEvent			OnChange;				//!< <b>Event:</b> Container contents changed.
	FBPropertyEvent			OnDblClick;				//!< <b>Event:</b> Double click.
	FBPropertyEvent			OnDragAndDrop;			//!< <b>Event:</b> Drag and Drop event.
};

////////////////////////////////////////////////////////////////////////////////////
// FBTimeCode
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBTimeCode );

class FBSDK_DLL FBTimeCode : public FBVisualComponent {
	__FBClassDeclare( FBTimeCode,FBVisualComponent );
public:
	/** Constructor.
	*/
	FBTimeCode();

	FBPropertyBool			Enabled;		//!< <b>Read Write Property:</b> Is the object enabled?
	FBPropertyTime			Value;			//!< <b>Read Write Property:</b> Current timecode value.

	FBPropertyEvent			OnChange;		//!< <b>Event:</b> Timecode changed.
};

////////////////////////////////////////////////////////////////////////////////////
// FBTree
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBTreeNode		);
__FB_FORWARD( FBTree			);

////////////////////////////////////////////////////////////////////////////////////
// FBEventTreeClick
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBEventTreeSelect );

//! Activation event.
class FBSDK_DLL FBEventTreeSelect : public FBEvent {
	__FBClassDeclare( FBEventTreeSelect,FBEvent );
public:
	/**	Constructor.
	*	\param pEvent Base event (internal) to obtain information from.
	*/
	FBEventTreeSelect( HKEventBase pEvent );
	FBPropertykReference Data;		    //!< <b>Read Write Property:</b> Generic data of event.
};

//! A node in the tree view.
class FBSDK_DLL FBTreeNode : public FBComponent 
{
	__FBClassDeclare( FBTreeNode,FBComponent );
public:
    /** Constructor
    *   \param  pTree   Parent tree.
    *   \param  pObject Internal object.
    */
	FBTreeNode(HFBTree pTree, HIObject pObject);

    FBPropertykReference	Reference;      //!< <b>Read Write Property:</b> Data to be associated to this node.
};

//! Tree list view.
class FBSDK_DLL FBTree : public FBVisualComponent 
{
	__FBClassDeclare( FBTree,FBVisualComponent );
public:
	//! Constructor.
	FBTree();

	/**	Get the root node.
    *   \return the root node of the tree.
	*/
	HFBTreeNode GetRoot();

	/**	Clear the tree (remove all nodes).
	*/
	void Clear();

	/**	Insert node at the end.
    *   \param  pNode   Node under which the new node will appear.
    *   \param  pName   Text to display for this node.
    *   \return the newly created node.
	*/
	HFBTreeNode InsertLast( HFBTreeNode pNode, char* pName );

    FBPropertyEvent		OnSelect;           //!< <b>Event:</b> A node was selected.

///////////////////////////////////////////////////////////////////////////////
//  WARNING!
//	Anything beyond these lines is under development and subject to change.
///////////////////////////////////////////////////////////////////////////////
#ifndef DOXYGEN_SHOULD_SKIP_THIS
	FBPropertyEvent		OnClick;
	FBPropertyEvent		OnDblClick;
	FBPropertyEvent		OnDeselect;
	FBPropertyEvent		OnExpanded;
	FBPropertyEvent		OnExpanding;
	FBPropertyEvent		OnCollapsed;
	FBPropertyEvent		OnCollapsing;
	FBPropertyEvent		OnChange;
	FBPropertyEvent		OnDragAndDrop;
	FBPropertyEvent		OnEditNode;

	FBPropertyInt			SelectedCount;     
	FBPropertyInt			VisibleItemCount;
	//FBPropertyTreeNode		Root;                 
	//FBPropertyTreeOptions	Options;              
	FBPropertyBool			AllowExpansion;       
	FBPropertyBool			AllowCollapse;        
	FBPropertyBool			MultiDrag;            
	FBPropertyBool			AutoScroll;           
	FBPropertyInt			Indent;               
	FBPropertyBool			CheckBoxes;           
	FBPropertyBool			ShowLines;				
	//FBProperty?				DefaultOpen;          
	FBPropertyInt			ItemHeight;           
	FBPropertyBool			MultiSelect;          
	//FBProperty?				ScrollBarMode;        
	FBPropertyBool			EditNodeOn2Select;    
	FBPropertyInt			TreeHeight;           
	FBPropertyInt			TreeWidth;            
	FBPropertyBool			SelectionActive;      
	//FBPropertyTreeNode		NodeActive;           
	FBPropertyBool			DeselectOnCollapse;   
	FBPropertyBool			NoSelectOnDrag;       
	FBPropertyBool			NoSelectOnRightClick; 
	//FBProperty?				ParentSelection;      
	FBPropertyBool			HighlightOnRightClick;
	FBPropertyBool			AutoScrollOnExpand;   
	FBPropertyBool			AutoExpandOnDragOver; 
	FBPropertyBool			AutoExpandOnDblClick; 
#endif // #ifndef DOXYGEN_SHOULD_SKIP_THIS 

};

////////////////////////////////////////////////////////////////////////////////////
// FBView
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBView );

//! Generic view.
class FBSDK_DLL FBView : public FBVisualComponent {
	__FBClassDeclare( FBView,FBVisualComponent );
public:
    /** Constructor
    */
	FBView();

    IObject_Declare(Implementation);	// Interface to IObject

	/**	Refresh view.
	*	\param	pNow		If \b true, refresh immediately (default = \b false).
	*/
    virtual void Refresh(bool pNow=false);

	/** Checks if object is a view.
	*	\return	Is object a view? (\b true or \b false)
	*/
	virtual bool IsView();

	/**	Draw a string in the view.
	*	\param	pText		Text to draw.
	*	\param	pX			X position of string.
	*	\param	pY			Y position of string.
	*	\param	pEnable		Is string enabled? (default =-1)
	*/
	virtual void DrawString(char *pText,float pX,float pY, int pEnable=-1);

	/**	Set view's viewport
	*	\param	pX	Viewport X value.
	*	\param	pY	Viewport Y value.
	*	\param	pW	Viewport W (width) value.
	*	\param	pH	Viewport H (height) value.
	*	\return	Operation was successful (\b true or \b false).
	*/
	virtual bool SetViewport(int pX,int pY,int pW,int pH);

	FBPropertyBool		GraphicOGL;   	//!< <b>Read Only Property:</b> Indicates if the view is OpenGL.
	FBPropertyBool		DoubleBuffer;   //!< <b>Read Only Property:</b> Indicates if the view is double buffered.
};

////////////////////////////////////////////////////////////////////////////////////
// FBProgress
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBProgress );

//! Progress bar.
class FBSDK_DLL FBProgress : public FBVisualComponent {
	__FBClassDeclare( FBProgress,FBVisualComponent );
public:
	/** Constructor.
	*/
	FBProgress();

	FBPropertyString	Caption;	//!< <b>Read Write Property:</b> Caption to be displayed for progress bar.
	FBPropertyString	Text;		//!< <b>Read Write Property:</b> Text to display on progress bar.
	FBPropertyInt		Percent;	//!< <b>Read Write Property:</b> Percent completed for the operation.
};

////////////////////////////////////////////////////////////////////////////////////
// FBPopup
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBPopup );

/** Popup window.
*	This class lets a window (inheriting from FBLayout) be created for another interface.
*/
class FBSDK_DLL FBPopup : public FBLayout {
	__FBClassDeclare( FBPopup,FBLayout );
public:
	/** Constructor.
	*/
	FBPopup();

	/**	Show popup.
	*	\param	pParent		Parent object for popup
	*	\return	Operation was successful (\b true or \b false).
	*/
	virtual bool Show( HFBVisualComponent pParent=NULL );

	/**	Close popup.
	*	\param	pOk		Equivalent of \b OK button clicked if \b true (default = \b false).
	*/
	virtual void Close(bool pOk=false);

	FBPropertyString		Caption;	//!< <b>Read Write Property:</b> Caption to display in popup.
	FBPropertyBool			Modal;		//!< <b>Read Write Property:</b> Modal?
};

////////////////////////////////////////////////////////////////////////////////////
// FBFilePopup
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBFilePopup );

//! Different types of file popup windows.
enum FBFilePopupStyle { 
	kFBFilePopupOpen,		//!< Open file popup (Shows 'Open Directory').
	kFBFilePopupSave		//!< Save file popup (Shows 'Save Directory').
};

FB_DEFINE_ENUM( FBSDK_DLL, FilePopupStyle ); // FBPropertyFilePopupStyle

//! File Popup (for open/save).
class FBSDK_DLL FBFilePopup : public FBVisualComponent {
	__FBClassDeclare( FBFilePopup,FBVisualComponent );
public:
	/** Constructor.
	*/
	FBFilePopup();

	/**	Execute file popup.
	*	\return	\b true if \b OK is clicked by user.
	*/
	virtual bool Execute();

	FBPropertyString			Caption;		//!< <b>Read Write Property:</b> Caption to put in popup window.
	FBPropertyString			FileName;		//!< <b>Read Write Property:</b> File selected.
	FBPropertyString			Path;			//!< <b>Read Write Property:</b> Path of file selected.
	FBPropertyString			FullFilename;	//!< <b>Read Only Property:</b> Full filename (path and file).
	FBPropertyString			Filter;			//!< <b>Read Write Property:</b> Filter to use for popup window file selection.
	FBPropertyFilePopupStyle	Style;			//!< <b>Read Write Property:</b> Style of file popup.
};

////////////////////////////////////////////////////////////////////////////////////
// FBFolderPopup
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBFolderPopup );

//! Folder Popup (for selecting a directory).
class FBSDK_DLL FBFolderPopup : public FBVisualComponent {
	__FBClassDeclare( FBFolderPopup,FBVisualComponent );
public:
	/** Constructor.
	*/
	FBFolderPopup();

	/**	Execute folder popup.
	*	\return	\b true if \b OK is clicked by user.
	*/
	virtual bool Execute();

	FBPropertyString			Caption;		//!< <b>Read Write Property:</b> Caption to put in popup window.
	FBPropertyString			Path;			//!< <b>Read Write Property:</b> Path of folder selected.
};

/**	Dialog popup box.
*	Opens a message box containing a message and up to three buttons. Waits for the user to click a button.
*	\param	pBoxTitle			Title of message box.
*
*	\param	pMessage			Message to place in box.
*	\param	pButton1Str			String for first button (Cannot be NULL).
*	\param	pButton2Str			String for second button (NULL will not create a button).
*	\param	pButton3Str			String for third button (NULL will not create a button).
*	\param	pDefaultButton		Indicates the default (pre-selected) button (default is 0).
*	\param	pScrolledMessage	Scroll message (default is 0).
*	\return	The number of the button selected.
*/
FBSDK_DLL int FBMessageBox( char* pBoxTitle, char* pMessage, char* pButton1Str, char* pButton2Str=0, char* pButton3Str=0, int pDefaultButton=0, int pScrolledMessage=0);

/**	Dialog popup box with 'don't show again' option.
*	Opens a message box containing a message and up to three buttons. Waits for the user to click a button.
*	This dialog also gives the user the option of never showing the dialog again.
*
*	\param	pBoxTitle			Title of message box.
*	\param	pMessage			Message to place in box.
*	\param	pButton1Str			String for first button (Cannot be NULL).
*	\param	pButton2Str			String for second button (NULL will not create a button).
*	\param	pButton3Str			String for third button (NULL will not create a button).
*	\param	pCheckBoxStr		Check box string (Cannot be NULL).
*	\retval	pCheckBoxValue		Check box value.
*	\param	pDefaultButton		Indicates the default (pre-selected) button (default is 0).
*	\param	pScrolledMessage	Scroll message (default is 0).
*	\return	The number of the button selected.
*/
FBSDK_DLL int FBMessageBoxWithCheck( char* pBoxTitle, char* pMessage, char* pButton1Str, char* pButton2Str, char* pButton3Str, char* pCheckBoxStr, bool& pCheckBoxValue, int pDefaultButton=0, int pScrolledMessage=0);

//! User input types for a popup
enum FBPopupInputType{
	kFBPopupBool,		//!< Boolean input.
	kFBPopupChar,		//!< Character input.
	kFBPopupString,		//!< String input.
	kFBPopupInt,		//!< Integer input.
	kFBPopupFloat,		//!< Float input.
	kFBPopupDouble,		//!< Double input.
	kFBPopupPassword	//!< Password input (String with '*'s).
};

/**	Dialog popup box to get user input.
*	Opens a message box, with up to three buttons, asking the user to enter data. 
*	The type of data to be entered is specified by the \e pValue and \e pValueType parameters.
*
*	\param	pBoxTitle			Title of message box.
*	\param	pMessage			Message to place in box.
*	\retval	pValue				Value entered by user (must correspond with pValueType).
*	\param	pValueType			Type of pointer specified in pValue.
*	\param	pButton1Str			String for first button (Cannot be NULL).
*	\param	pButton2Str			String for second button (NULL will not create a button).
*	\param	pButton3Str			String for third button (NULL will not create a button).
*	\param	pDefaultButton		Indicates the default (pre-selected) button(default=0).
*	\return	The number of the button selected.
*/
FBSDK_DLL int FBMessageBoxGetUserValue( char* pBoxTitle, char* pMessage, void* pValue, FBPopupInputType pValueType, char* pButton1Str, char* pButton2Str=0, char* pButton3Str=0, int pDefaultButton=0 );

////////////////////////////////////////////////////////////////////////////////////
// FBMemo
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBMemo );

//! Tab panel.
class FBSDK_DLL FBMemo : public FBEdit {
	__FBClassDeclare( FBMemo,FBEdit );
public:
	//! Constructor.
	FBMemo();

	void SetStrings(HFBStringList pLines);
	void GetStrings(HFBStringList pLines);
};


////////////////////////////////////////////////////////////////////////////////////
// FBEditProperty
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBEditProperty );

/** Property editor widget.
*   This widget allows users to edit the values of a property without having to
*   manually customize the GUI depending on the type of the property being edited.
*   
*   SDK objects can have three types of properties:
*       - An internal property which maps to an actual property that can be seen
*         in the property editor tool of the application. This type of property
*         is usually obtained from the PropertyList data member.
*       - SDK-only property which does not maps onto an existing property of the
*         encapsulated object. The existence of these types of property is
*         often to make the object interface simpler. All the FBPropertyList-types
*         will fall into this category, except for FBPropertyListObjects.
*       - SDK property which maps onto an existing object property, but does so
*         indirectly using function calls instead of direct property access. This
*         is usually for historical reason. In this case the property will usually
*         be present twice in the PropertyList: once as an SDK-Only property and
*         another time as an internal property.
*
*   Another limitation of this widget is that it can only display non hidden
*   internal properties. To get around this issue, the property flag can be
*   changed to unhide it. Doing so will also cause the property to be visible
*   via the property tool.
*
*   @code
*       // In a tool header file...
*       FBEditProperty mEditProperty;
*
*       // In a tool source file...
*       HFBModel lModel = FBFindModelByName( "ModelName" );
*       if( lModel )
*       {
*           HFBProperty lProperty = lModel->PropertyList.Find( "RotationOrder" );
*           if( lProperty &&
*               lProperty->IsInternal() &&
*               !lProperty->GetPropertyFlag( kFBPropertyFlagHideProperty ))
*           {
*               mEditProperty.Property = lProperty;
*           }
*       }
*   @encode
*   
*/
class FBSDK_DLL FBEditProperty : public FBVisualComponent {
	__FBClassDeclare( FBEditProperty, FBVisualComponent );
public:
	//! Constructor.
	FBEditProperty();

    FBPropertyReference Property;   //!< <b>Read Write Property:</b> Property to edit. Set to NULL to disable.

    FBPropertyDouble    SliderMin;  //!< <b>Read Write Property:</b> Should the property be editable using a slider, set the minimum value atainable with the slider.
    FBPropertyDouble    SliderMax;  //!< <b>Read Write Property:</b> Should the property be editable using a slider, set the maximum value atainable with the slider.
    FBPropertyDouble    SmallInc;   //!< <b>Read Write Property:</b> Indicate the small increment applied when click-draging on the property value (usually right-click-dragging)
    FBPropertyDouble    LargeInc;   //!< <b>Read Write Property:</b> Indicate the large increment applied when click-draging on the property value (usually left-click-dragging)
    FBPropertyDouble    Precision;  //!< <b>Read Write Property:</b> Used to specify the width and precision of the value shown. A value of 7.2 indicates to show at minimum 7 numbers, with 2 decimals.
};


////////////////////////////////////////////////////////////////////////////////////
// FBEditPropertyModern
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBEditPropertyModern );

/** Property editor widget.
*   This is a more modern version of the widget FBEditProperty which is used in the
*   property editor tool of the application.
*
*   See class FBEditProperty for more details.
*/
class FBSDK_DLL FBEditPropertyModern : public FBVisualComponent {
	__FBClassDeclare( FBEditPropertyModern, FBVisualComponent );
public:
	//! Constructor.
	FBEditPropertyModern();

    /** Set the background color index.
    *   Use the system-defined color palette to set the backgound color.
    *   By default the color used is kFBColorIndexStdListBg1
    */
    void SetBackgroundColorIndex( FBColorIndex pIndex );

    FBPropertyReference Property;   //!< <b>Read Write Property:</b> Property to edit. Set to NULL to disable.

    FBPropertyDouble    SliderMin;  //!< <b>Read Write Property:</b> Should the property be editable using a slider, set the minimum value atainable with the slider.
    FBPropertyDouble    SliderMax;  //!< <b>Read Write Property:</b> Should the property be editable using a slider, set the maximum value atainable with the slider.
    FBPropertyDouble    SmallInc;   //!< <b>Read Write Property:</b> Indicate the small increment applied when click-draging on the property value (usually right-click-dragging)
    FBPropertyDouble    LargeInc;   //!< <b>Read Write Property:</b> Indicate the large increment applied when click-draging on the property value (usually left-click-dragging)
    FBPropertyDouble    Precision;  //!< <b>Read Write Property:</b> Used to specify the width and precision of the value shown. A value of 7.2 indicates to show at minimum 7 numbers, with 2 decimals.
};

#endif //}


#ifdef FBSDKUseNamespace
	}
#endif

#endif
