#ifndef __FBDATA_H__
#define __FBDATA_H__
/**************************************************************************
 Copyright (c) 1994 - 2006 Autodesk, Inc. and/or its licensors.
 All Rights Reserved.
 
 The coded instructions, statements, computer programs, and/or related 
 material (collectively the "Data") in these files contain unpublished 
 information proprietary to Autodesk, Inc. and/or its licensors, which is 
 protected by Canada and United States of America federal copyright law 
 and by international treaties.
 
 The Data may not be disclosed or distributed to third parties, in whole 
 or in part, without the prior written consent of Autodesk, Inc. 
 ("Autodesk").
 
 THE DATA IS PROVIDED "AS IS" AND WITHOUT WARRANTY.
 ALL WARRANTIES ARE EXPRESSLY EXCLUDED AND DISCLAIMED. AUTODESK MAKES NO 
 WARRANTY OF ANY KIND WITH RESPECT TO THE DATA, EXPRESS, IMPLIED OR 
 ARISING BY CUSTOM OR TRADE USAGE, AND DISCLAIMS ANY IMPLIED WARRANTIES 
 OF TITLE, NON-INFRINGEMENT, MERCHANTABILITY OR FITNESS FOR A PARTICULAR 
 PURPOSE OR USE. WITHOUT LIMITING THE FOREGOING, AUTODESK DOES NOT 
 WARRANT THAT THE OPERATION OF THE DATA WILL BE UNINTERRUPTED OR ERROR 
 FREE.
 
 IN NO EVENT SHALL AUTODESK, ITS AFFILIATES, PARENT COMPANIES, LICENSORS 
 OR SUPPLIERS ("AUTODESK GROUP") BE LIABLE FOR ANY LOSSES, DAMAGES OR 
 EXPENSES OF ANY KIND (INCLUDING WITHOUT LIMITATION PUNITIVE OR MULTIPLE 
 DAMAGES OR OTHER SPECIAL, DIRECT, INDIRECT, EXEMPLARY, INCIDENTAL, LOSS 
 OF PROFITS, REVENUE OR DATA, COST OF COVER OR CONSEQUENTIAL LOSSES OR 
 DAMAGES OF ANY KIND), HOWEVER CAUSED, AND REGARDLESS OF THE THEORY OF 
 LIABILITY, WHETHER DERIVED FROM CONTRACT, TORT (INCLUDING, BUT NOT 
 LIMITED TO, NEGLIGENCE), OR OTHERWISE, ARISING OUT OF OR RELATING TO THE 
 DATA OR ITS USE OR ANY OTHER PERFORMANCE, WHETHER OR NOT AUTODESK HAS 
 BEEN ADVISED OF THE POSSIBILITY OF SUCH LOSS OR DAMAGE.
 
**************************************************************************/

/**	\file fbdata.h
*	Animation data types.
*	This file contains the definitions for various animation data
*	types in the Open Reality SDK.
*/

#include <kaydaradef.h>
#ifndef FBSDK_DLL 
	#define FBSDK_DLL K_DLLIMPORT
#endif

#include <fbsdk/fbcomponent.h>
#include <fbsdk/fbcore.h>	// CNT_STATUS_LIVE

//@{
//! Animation node data types.
// Basic types
#define ANIMATIONNODE_TYPE_NUMBER               "Number"
#define ANIMATIONNODE_TYPE_TIME                 "Time"
#define ANIMATIONNODE_TYPE_VECTOR               "Vector"
#define ANIMATIONNODE_TYPE_VECTOR_4             "Vector4"

// Types which are numbers
#define ANIMATIONNODE_TYPE_ACTION               "Action"
#define ANIMATIONNODE_TYPE_BOOL                 "Bool"
#define ANIMATIONNODE_TYPE_ENUM                 "Enum"
#define ANIMATIONNODE_TYPE_INTEGER              "Integer"
#define ANIMATIONNODE_TYPE_OCCLUSION            "Occlusion"
#define ANIMATIONNODE_TYPE_FIELDOFVIEWX         "FieldOfViewX"
#define ANIMATIONNODE_TYPE_FIELDOFVIEWY         "FieldOfViewY"
#define ANIMATIONNODE_TYPE_OPTICALCENTERX       "OpticalCenterX"
#define ANIMATIONNODE_TYPE_OPTICALCENTERY       "OpticalCenterY"

// Types which are vectors (3 members)
#define ANIMATIONNODE_TYPE_COLOR                "Color"
#define ANIMATIONNODE_TYPE_LOCAL_TRANSLATION    "Lcl Translation"
#define ANIMATIONNODE_TYPE_LOCAL_ROTATION       "Lcl Rotation"
#define ANIMATIONNODE_TYPE_LOCAL_SCALING        "Lcl Scaling"
#define ANIMATIONNODE_TYPE_TRANSLATION          "Translation"
#define ANIMATIONNODE_TYPE_ROTATION             "Rotation"
#define ANIMATIONNODE_TYPE_SCALING              "Scaling"

// Types which are vectors (4 members)
#define ANIMATIONNODE_TYPE_COLOR_RGBA           "ColorAndAlpha"

//@}

K_FORWARD( IConnector );
K_FORWARD( KFCurve );

#ifdef FBSDKUseNamespace
	namespace FBSDKNamespace {
#endif

FB_FORWARD( FBEvaluateInfo );

typedef double DTNumber;
typedef double DTVector[3];
		
//! Types of interpolation for an FCurve.
enum FBInterpolation	{	
	kFBInterpolationConstant,	//!< Constant interpolation.
	kFBInterpolationLinear,		//!< Linear interpolation.
	kFBInterpolationCubic		//!< Cubic interpolation.
};
FB_DEFINE_ENUM( FBSDK_DLL, Interpolation ); 

/** Methods of tangent calculation.
*	This is only relevant when interpolation is CUBIC.
*/
enum FBTangentMode	{
	kFBTangentModeAuto,		//!< This is the equivalent to a cardinal spline with no parametrization
 	kFBTangentModeTCB,		//!< TCB spline (3 parameters: TENSION, CONTINUITY, BIAS)
 	kFBTangentModeUser,		//!< Used to represent all splines with no lost data (HERMITE, BEZIER, CATMUL, etc.) 
 	kFBTangentModeBreak		//!< Like USER but left slope may differ from right.
};
FB_DEFINE_ENUM( FBSDK_DLL, TangentMode );	
 
////////////////////////////////////////////////////////////////////////////////////
// FBFCurveKey
////////////////////////////////////////////////////////////////////////////////////
FB_FORWARD( FBFCurve );
__FB_FORWARD( FBFCurveKey );
FB_FORWARD( FBPropertyListFCurveKey );

//! KeyFrame for an FCurve.
class FBSDK_DLL FBFCurveKey
{
private:
	HDataFBFCurveKey mLocalPtr;
	void FBFCurveKeyInitProperties();
public:
	void InitProperties();

	/**	Constructor
	*	\param	pFCurve		Parent FCurve (default is NULL).
	*	\param	pKeyIndex	Key frame index (default is 1).
	*/
	FBFCurveKey( HKFCurve pFCurve=NULL, int pKeyIndex=1 );

	/**	Constructor
	*	\param	pFCurveKey	FCurveKey to copy information from.
	*/
	FBFCurveKey( const FBFCurveKey &pFCurveKey );

	/**	Destructor.
	*/
	~FBFCurveKey();

	/**	Overloaded == operator.
	*	\param	pValue	FCurveKey to compare with.
	*	\return	\b true if equal.
	*	\warning.	The overload of the == operator will always return \b false.
	*/
	bool operator==(const FBFCurveKey &pValue) const;

    // To convert the derivative values to degrees (as shown in Exact mode):
    //    degrees( atan( RightDerivative ))
    //    -1 * degrees( atan( LeftDerivative )) 
    //
    // To convert from the exact degree in Exact mode in the GUI to the corresponding
    // derivative:
    //    tan( radians( RightDerivative ))
    //    tan( radians( -1 * LeftDerivative ))
	FBPropertyFloat	Value;				//!< <b>Read Write Property:</b> Value of Key
	FBPropertyFloat	LeftDerivative;		//!< <b>Read Write Property:</b> Left derivative, in units/seconds.
	FBPropertyFloat	RightDerivative;	//!< <b>Read Write Property:</b> Right derivative, in units/seconds.

	// TCB
	FBPropertyFloat	Tension;			//!< <b>Read Write Property:</b> Tension (TCB).
	FBPropertyFloat	Continuity;			//!< <b>Read Write Property:</b> Continuity (TCB).
	FBPropertyFloat	Bias;				//!< <b>Read Write Property:</b> Bias (TCB).
	
	FBPropertyTime			Time;			//!< <b>Read Write Property:</b> Time of key.
	FBPropertyInterpolation Interpolation;	//!< <b>Read Write Property:</b> Type of interpolation.
	FBPropertyTangentMode	TangentMode;	//!< <b>Read Write Property:</b> Tangeant calculation method.
	
	friend class FBFCurve;
	friend class FBPropertyListFCurveKey;
};

////////////////////////////////////////////////////////////////////////////////////
// FBPropertyListFCurveKey
////////////////////////////////////////////////////////////////////////////////////
FB_FORWARD( FBFCurveKey );
typedef class FBSDK_DLL FBPropertyBaseList< FBFCurveKey > FBPropertyBaseListFCurveKey;
 
//! List of FCurveKey
class FBSDK_DLL FBPropertyListFCurveKey : public FBPropertyBaseList< FBFCurveKey >
{
private:
	/**	Add an FCurveKey.
	*	\param		pItem	Key to add.
	*	\return		Number of keys in list.
	*	\warning	This function is private.
	*/
	virtual int	 Add	( FBFCurveKey pItem );

	/**	Remove an FCurveKey.
	*	\param		pItem	Key to remove.
	*	\return		Number of keys in list.
	*	\warning	This function is private.
	*/
    virtual int  Remove	( FBFCurveKey pItem );

public:

	/**	Remove an FCurveKey.
	*	\param		pIndex	Index of key to remove.
	*/
    virtual void RemoveAt( int pIndex );

	/**	Overloaded [] operator.
	*	\param	pIndex	Index of key to access.
	*	\return Key at \e pIndex.
	*/
	virtual FBFCurveKey operator[](int pIndex);

	/**	Get a count of the number of keys in list.
	*	\return	Number of keys in list.
	*/
	virtual int  GetCount();
};

////////////////////////////////////////////////////////////////////////////////////
// FBFCurve
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBFCurve);

//! FCurve class.
class FBSDK_DLL FBFCurve : public FBComponent {
	__FBClassDeclare( FBFCurve,FBComponent );
public:
	/**	Constructor.
	*	\param	pName	Name of FCurve (default is NULL).
	*	\param	pObject	For internal use only (default is NULL).
	*/
	FBFCurve(char *pName = NULL, HIObject pObject=NULL);

	/**	Evaluate FCurve at \e pTime.
	*	\param	pTime	Time at which FCurve is to be evaluated.
	*	\return	Value of FCurve at \e pTime.
	*/
	float	Evaluate(FBTime &pTime);

	/**	Setup function to begin adding keys.
	*	\param	pKeyCount	Key to begin adding at(default is -1).
	*/
	void	EditBegin(int pKeyCount=-1);

	/**	End key adding sequence.
	*	\param	pKeyCount	Key to finish adding at (default is -1).
	*/
	void	EditEnd(int pKeyCount=-1);

	//!	Empty FCurve of all keys.
	void	EditClear();

    /** Add a key at the specified time.
    *   @param pTime    Time at which to insert the key.
    *   @param pValue   Value of the key.
    *   @return     The position of the new key in the list of FCurve keys.
    *   @warning    Since there are no parameter to indicate the interpolation and tangent mode,
    *               the neighouring keys may be affected by the newly inserted key.
    */
    int KeyAdd( FBTime &pTime, double pValue );

    /** Remove key at index.
    *   @param  pIndex  Index of the key to remove.
    *   @return         If the removal was succesful or not.
    */
    bool KeyRemove( int pIndex );

    /**	Insert a key without affecting the curve shape
	*	\param pTime Time at which the key is to be inserted.
	*   \param pInterpolation Interpolation type of the inserted key.
	*   \param pTangentMode Tangent calculation method of the inserted key.
	*/
	void	KeyInsert( FBTime &pTime, FBInterpolation pInterpolation = kFBInterpolationCubic, FBTangentMode pTangentMode = kFBTangentModeAuto );


	/**	Replace keys within a range in current function curve with keys found in a source function curve.
	*   \param pSource Source function curve.
	*   \param pStart Start of time range.
	*   \param pStop End of time range.
	*   \param pUseExactGivenSpan When \c false, the time of the first and last key in the range will be used.
	*   \param pKeyStartEndOnNoKey When \c true, inserts a key at the beginning and at the end of the range if there is no key to insert.
	*/
	void Replace( FBFCurve& pSource, FBTime pStart = FBTime::MinusInfinity, FBTime pStop = FBTime::Infinity, bool pUseExactGivenSpan = false, bool pKeyStartEndOnNoKey = true );

	FBPropertyListFCurveKey	Keys;		//!< <b>List:</b> Keys.
};

FB_DEFINE_COMPONENT( FBSDK_DLL, FCurve );

////////////////////////////////////////////////////////////////////////////////////
// FBPropertyListAnimationNode
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBAnimationNode);
__FB_FORWARD( FBPropertyListAnimationNode);
FB_DEFINE_LIST( FBSDK_DLL, AnimationNode );

//! List of animation nodes
class FBSDK_DLL FBPropertyListAnimationNode : public FBPropertyBaseList< HFBAnimationNode >
{
public:
	/**	Add an animation node to the list.
	*	\param		pItem	Animation node to add.
	*	\return		Number of animation nodes in list after operation.
	*/
	virtual int	 Add	( HFBAnimationNode pItem );
	/**	Remove an animation node from the list.
	*	\param		pItem	Animation Node to remove.
	*	\return		Number of animation nodes in list after operation.
	*/
    virtual int  Remove	( HFBAnimationNode pItem );
	/**	Remove an animation node from \e pIndex.
	*	\param	pIndex	Index of animation node to remove.
	*/
    virtual void RemoveAt( int pIndex );
	/**	Overloaded [] operator.
	*	\param	pIndex	Index of animation node to access.
	*	\return Animation node at \e pIndex.
	*/
	virtual HFBAnimationNode operator[](int pIndex);
	/**	Get a count of the number of animation nodes in list.
	*	\return	Number of animation nodes in list.
	*/
	virtual int  GetCount();
	/**	Find an animation node.
	*	\param	pNodeName	String name of animation node to find.
	*	\return	Handle to the animation node by the name of \e pNodeName.
	*/
	HFBAnimationNode Find( char *pNodeName );
};

FB_DEFINE_COMPONENT( FBSDK_DLL, AnimationNode	);
////////////////////////////////////////////////////////////////////////////////////
// FBEventAnimationNode
////////////////////////////////////////////////////////////////////////////////////
//! Event based on animation node 

//! Types of transformation.
enum FBEventAnimationNodeType {
	kFBEventAnimationNodeDataChange,
	kFBEventAnimationNodeConstraintChange,
	kFBEventAnimationNodeNone
};

FB_DEFINE_ENUM( FBSDK_DLL, EventAnimationNodeType );

__FB_FORWARD( FBEventAnimationNode );

/** The transform manipulator has begun doing modifications
*	This event occurs every time a manipulator:
*	- receives mouse input
*	- begins/ends a transaction
*	- manipulates/moves and object
*/
class FBSDK_DLL FBEventAnimationNode: public FBEvent {
	__FBClassDeclare( FBEventAnimationNode,FBEvent );
public:
	/**	Constructor.
	*	\param pEvent Base event (internal) to obtain information from.
	*/
	FBEventAnimationNode( HKEventBase pEvent );		

	FBPropertyEventAnimationNodeType		Type;	//!< <b>Read Only Property:</b> Type of selection event.
};

////////////////////////////////////////////////////////////////////////////////////
// FBPropertyEventAnimationNode
////////////////////////////////////////////////////////////////////////////////////
//! \b PropertyEvent: UI idle event.
class FBSDK_DLL FBPropertyEventAnimationNode : public FBPropertyEvent
{
  public:
    /** Add/Remove a callback.
	*	\param	pOwner		Callback owner.
	*	\param	pHandler	Callback handler.
	*/
	virtual void Add	( HICallback pOwner, kICallbackHandler pHandler );
	virtual void Remove	( HICallback pOwner, kICallbackHandler pHandler );
};

typedef bool (* kFBDataHandler )(void* pBuffer,HFBEvaluateInfo pEvaluateInfo,HFBAnimationNode pAnimationNode,void* p1,void* p2);

////////////////////////////////////////////////////////////////////////////////////
// FBAnimationNode
////////////////////////////////////////////////////////////////////////////////////
//! Animation node class.
class FBSDK_DLL FBAnimationNode : public FBComponent {
	__FBClassDeclare( FBAnimationNode,FBComponent );
public:
	/**	Constructor.
	*	\param	pName		Name of animation node (default is NULL).
	*	\param	pObject		For internal use only (default is NULL).
	*/
	FBAnimationNode(char *pName = NULL, HIObject pObject=NULL);

	/**	Add a key to the animation node.
	*	\param	pTime	Time to add key at.
	*	\param	pData	Value of data to add at \e pTime.
	*/
	void KeyAdd( FBTime &pTime, double *pData );

	/**	Add a key to the animation node at current time.
	*	\param	pData	Value of data to add.
	*/
	void KeyAdd( double *pData );

	/**	Set the current candidate values for current time.
	*	\return \b true if successful.
	*/
    bool SetCandidate( void *Data );

	/**	Keys the current candidate values for current time.
	*/
    void KeyCandidate();

	/**	Get sizeof void Data Ptr
	*/
	int GetSizeOfData();

	/**	If the DataPtr is of numeric value type ... get the size of the array ex: Light Intensity:1, Translation 3
	*	\return	Size of DataPtr array.
	*/
	int GetDataDoubleArrayCount();

	/**	Write data to animation node.
	*	\param	Data			Data to write to animation node.
	*	\param	pEvaluateInfo	Node evaluation information (access to system and local time).
	*	\param	pInfoType		Type of information being written (default is live).
	*	\return	Status of animation node.
	*/
    int WriteData( void *Data,HFBEvaluateInfo pEvaluateInfo,int pInfoType=CNT_STATUS_LIVE );

	/**	Read data from animation node.
	*	\param	Data	Data to read from animation node.
	*	\param	pEvaluateInfo	Node evaluation information (access to system and local time).
	*	\return	Status of animation node.
	*/
    int ReadData( void *Data,HFBEvaluateInfo pEvaluateInfo );

	/**	Read the last data evaluated for this animation node ... 
	*	this call doesn't generate a pull on the connection attached to this
	*   AnimationNode. No validation is done on the pointer size.
	*	You must provide a buffer that is at least GetSizerOfData() size.
	*	\param	Data	Data to read from animation node.
	*	\return	\b true if successful.
	*/
    bool ReadData( void *Data );

	/**	Verifies if there is a key at the current.
	*	\return	\b true if there is a key at the current time.
	*/
    bool IsKey();

	/**	Remove key at current time
	*/
    void KeyRemove();

	/**	Register/Unregister Data Handler that take control of the handler
	*/
	void RegisterDataHandler(kFBDataHandler pDataHandler,void* p1,void* p2 );
	void UnregisterDataHandler(kFBDataHandler pDataHandler );

	/**	Get animation to record from animation node.
	*	\return	Animation node with data to record.
	*/
	HFBAnimationNode GetAnimationToRecord();

	/**	Convert global time to node time. (NOTE: Only used in the context of a story clip)
	*	\param	pKeyTime	Time of the key to convert.
	*/
	FBTime ConvertGlobalToNodeTime(FBTime& pKeyTime);
	
	/**	Convert node time to global time. (NOTE: Only used in the context of a story clip)
	*	\param	pKeyTime	Time of the key to convert.
	*/
	FBTime ConvertNodeToGlobalTime(FBTime& pKeyTime);

	FBPropertyString			UserName;		//!< <b>Read Only Property:</b> Name of animation node.
	FBPropertykReference		Reference;		//!< <b>Read Write Property:</b> User-defined reference.
	FBPropertyInt				KeyCount;		//!< <b>Read Only Property:</b> Number of keys.
	FBPropertyFCurve			FCurve;			//!< <b>Read Write Property:</b> FCurve for animation.
	FBPropertyBool				Live;			//!< <b>Read Write Property:</b> Is animation live?
	FBPropertyBool				RecordMode;		//!< <b>Read Write Property:</b> Is the node in recording mode (device connectors)?

	FBPropertyListAnimationNode	Nodes;			//!< <b>List:</b> List of animation nodes.

	FBPropertyEventAnimationNode	OnChange;		//!< <b>Event:</b> Called when the value of this property is modified

	friend class FBBox;
};

////////////////////////////////////////////////////////////////////////////////////
// FBPropertyListTake
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBTake);
__FB_FORWARD( FBPropertyListTake );
FB_DEFINE_LIST( FBSDK_DLL, Take );

//! List of takes.
class FBSDK_DLL FBPropertyListTake : public FBPropertyBaseList< HFBTake >
{
public:
	/**	Add a take to the list.
	*	\param	pItem	Take to add.
	*	\return	Number of takes in list after operation.
	*/
	virtual int	 Add	( HFBTake pItem );
	/**	Remove a take from the list.
	*	\param	pItem	Take to remove.
	*	\return	Number of takes in list after operation.
	*/
    virtual int  Remove	( HFBTake pItem );
	/**	Remove a take from \e pIndex.
	*	\param	pIndex	Index of take to remove.
	*/
    virtual void RemoveAt( int pIndex );
	/**	Overloaded [] operator.
	*	\param	pIndex	Index of take to access.
	*	\return Take at \e pIndex.
	*/
	virtual HFBTake operator[](int pIndex);
	/**	Get a count of the number of takes in list.
	*	\return	Number of takes in list.
	*/
	virtual int  GetCount();
};

////////////////////////////////////////////////////////////////////////////////////
// FBTake
////////////////////////////////////////////////////////////////////////////////////

//! Take class.
class FBSDK_DLL FBTake : public FBComponent {
	__FBClassDeclare( FBTake,FBComponent );
public:
	/**	Constructor.
	*	\param	pName		Name of take (default is NULL).
	*	\param	pObject		For internal use only (default is NULL).
	*/
	FBTake(char *pName = NULL, HIObject pObject=NULL);

	/** Deletion method.
    *   Using this method to delete the take insure that the destruction process
    *   follows the same path as if the GUI had been used.
    */
	virtual void FBDelete();

    /**	Copy the take.
	*	Will create a copy of the current take, with the current take data.
	*	This is analogous to creating a new take, and copying the current take data into it.
	*	\param	pNewTakeName	The name for the new take.
	*	\return	Handle to the newly created take.
	*/
	HFBTake	CopyTake( char *pNewTakeName );

	/**	Plot the animation on selected models.
	*	Will plot the animation of the take in question on the selected models in the scene.
	*	\param	pPlotPeriod		Period for the plot.
	*/
	void PlotTakeOnSelected(FBTime pPlotPeriod);

	/** Plot the animation on selected models for all takes.
	*	Will plot the animation for all takes on the selected models in the scene.
	*	\param	pPlotPeriod		Period for the plot.
	*/
	void PlotAllTakesOnSelected(FBTime pPlotPeriod);

	FBPropertyTimeSpan	ReferenceTimeSpan;		//!< <b>Read Write Property:</b> Reference time span.
	FBPropertyTimeSpan	LocalTimeSpan;			//!< <b>Read Write Property:</b> Local time span.
	FBPropertyString	Comments;				//!< <b>Read Write Property:</b> Take comments.
};

FB_DEFINE_COMPONENT( FBSDK_DLL, Take );

#ifdef FBSDKUseNamespace
	}
#endif

#endif

