#ifndef __FBMATH_H__
#define __FBMATH_H__
/**************************************************************************
 Copyright (c) 1994 - 2006 Autodesk, Inc. and/or its licensors.
 All Rights Reserved.
 
 The coded instructions, statements, computer programs, and/or related 
 material (collectively the "Data") in these files contain unpublished 
 information proprietary to Autodesk, Inc. and/or its licensors, which is 
 protected by Canada and United States of America federal copyright law 
 and by international treaties.
 
 The Data may not be disclosed or distributed to third parties, in whole 
 or in part, without the prior written consent of Autodesk, Inc. 
 ("Autodesk").
 
 THE DATA IS PROVIDED "AS IS" AND WITHOUT WARRANTY.
 ALL WARRANTIES ARE EXPRESSLY EXCLUDED AND DISCLAIMED. AUTODESK MAKES NO 
 WARRANTY OF ANY KIND WITH RESPECT TO THE DATA, EXPRESS, IMPLIED OR 
 ARISING BY CUSTOM OR TRADE USAGE, AND DISCLAIMS ANY IMPLIED WARRANTIES 
 OF TITLE, NON-INFRINGEMENT, MERCHANTABILITY OR FITNESS FOR A PARTICULAR 
 PURPOSE OR USE. WITHOUT LIMITING THE FOREGOING, AUTODESK DOES NOT 
 WARRANT THAT THE OPERATION OF THE DATA WILL BE UNINTERRUPTED OR ERROR 
 FREE.
 
 IN NO EVENT SHALL AUTODESK, ITS AFFILIATES, PARENT COMPANIES, LICENSORS 
 OR SUPPLIERS ("AUTODESK GROUP") BE LIABLE FOR ANY LOSSES, DAMAGES OR 
 EXPENSES OF ANY KIND (INCLUDING WITHOUT LIMITATION PUNITIVE OR MULTIPLE 
 DAMAGES OR OTHER SPECIAL, DIRECT, INDIRECT, EXEMPLARY, INCIDENTAL, LOSS 
 OF PROFITS, REVENUE OR DATA, COST OF COVER OR CONSEQUENTIAL LOSSES OR 
 DAMAGES OF ANY KIND), HOWEVER CAUSED, AND REGARDLESS OF THE THEORY OF 
 LIABILITY, WHETHER DERIVED FROM CONTRACT, TORT (INCLUDING, BUT NOT 
 LIMITED TO, NEGLIGENCE), OR OTHERWISE, ARISING OUT OF OR RELATING TO THE 
 DATA OR ITS USE OR ANY OTHER PERFORMANCE, WHETHER OR NOT AUTODESK HAS 
 BEEN ADVISED OF THE POSSIBILITY OF SUCH LOSS OR DAMAGE.
 
**************************************************************************/

/**	\file fbmath.h
*	Contains routines for vector and matrix manipulation.
*/

#include <kaydaradef.h>
#ifndef FBSDK_DLL 
	#define FBSDK_DLL K_DLLIMPORT
#endif

#include <fbsdk/fbtypes.h>

#ifdef FBSDKUseNamespace
	namespace FBSDKNamespace {
#endif

////////////////////////////////////////////////////////////////////////////////////
// Matrix
////////////////////////////////////////////////////////////////////////////////////
/**	Invert a matrix.
*	\retval	pMatrix	Calculated inverse matrix.
*	\param	pSrc	Source matrix to invert.
*/
FBSDK_DLL void FBMatrixInverse (FBMatrix  &pMatrix,FBMatrix  &pSrc);

/**	Multiply two matrices.
*	\retval	pMatrix	Calculated resulting matrix.
*	\param	pA		1st matrix.
*	\param	pB		2nd matrix.
*/
FBSDK_DLL void FBMatrixMult	(FBMatrix  &pMatrix,FBMatrix  &pA,FBMatrix  &pB);

/** Multiply a vertex by a matrix.
*	\retval	pOutVertex	Resulting vertex.
*	\param	pMatrix		Matrix to affect the vertex with.
*	\param	pVertex		Source vertex.
*/
FBSDK_DLL void FBVertexMatrixMult( FBVertex &pOutVertex, FBMatrix &pMatrix,	FBVertex &pVertex );

/** Multiply a vector by a matrix.
*	\retval	pOutVector	Resulting vector.
*	\param	pMatrix		Matrix to affect the vector with.
*	\param	pVector		Source vector.
*/
FBSDK_DLL void FBVectorMatrixMult( FBVector4d &pOutVector, FBMatrix &pMatrix, FBVector4d &pVector );

////////////////////////////////////////////////////////////////////////////////////
// T,R,S to Matrix
////////////////////////////////////////////////////////////////////////////////////
// N.B. Rotation is in EulerXYZ
/**	Convert a translation vector to a matrix.
*	\retval	pMatrix	Calculated resulting matrix.
*	\param	pVector	Translation vector.
*/
FBSDK_DLL void FBTranslationToMatrix	(FBMatrix  &pMatrix,		FBTVector pVector);

/**	Convert a rotation vector to a matrix.
*	\retval	pMatrix	Calculated resulting matrix.
*	\param	pVector	Rotation vector.
*	\warning Rotation is in EulerXYZ
*/
FBSDK_DLL void FBRotationToMatrix		(FBMatrix  &pMatrix,		FBRVector pVector);

/**	Convert a scaling vector to a matrix.
*	\retval	pMatrix	Calculated resulting matrix.
*	\param	pVector	Scaling vector.
*/
FBSDK_DLL void FBScalingToMatrix		(FBMatrix  &pMatrix,		FBSVector pVector);

/**	Convert translation, rotation, and scaling vectors to a matrix.
*	\retval	pMatrix	Calculated resulting matrix.
*	\param	pTVector	Translation vector.
*	\param	pRVector	Rotation vector.
*	\param	pSVector	Scaling vector.
*	\warning Rotation is in EulerXYZ
*/
FBSDK_DLL void FBTRSToMatrix			(FBMatrix  &pMatrix,		FBTVector pTVector, FBRVector pRVector, FBSVector pSVector);

////////////////////////////////////////////////////////////////////////////////////
// Matrix to T,R,S
////////////////////////////////////////////////////////////////////////////////////
// Rotation is in EulerXYZ
/**	Obtain translation vector from a matrix.
*	\retval	pVector	Extracted translation vector.
*	\param	pMatrix	Input matrix.
*/
FBSDK_DLL void FBMatrixToTranslation	(FBTVector &pVector,		FBMatrix pMatrix);

/**	Obtain rotation vector from a matrix.
*	\retval	pVector	Extracted rotation vector.
*	\param	pMatrix	Input matrix.
*	\warning Rotation is in EulerXYZ
*/
FBSDK_DLL void FBMatrixToRotation		(FBRVector &pVector,		FBMatrix pMatrix);

/**	Obtain scaling vector from a matrix.
*	\retval	pVector	Extracted scaling vector.
*	\param	pMatrix	Input matrix.
*/
FBSDK_DLL void FBMatrixToScaling		(FBSVector &pVector,		FBMatrix pMatrix);

/**	Obtain translation, rotation, and scaling vectors from a matrix.
*	\retval	pTVector	Extracted translation vector.
*	\retval	pRVector	Extracted rotation vector.
*	\retval	pSVector	Extracted scaling vector.
*	\param	pMatrix		Input matrix.
*	\warning Rotation is in EulerXYZ
*/
FBSDK_DLL void FBMatrixToTRS			(FBTVector &pTVector, FBRVector &pRVector, FBSVector &pSVector,FBMatrix pMatrix);

////////////////////////////////////////////////////////////////////////////////////
// Quaternion
////////////////////////////////////////////////////////////////////////////////////
/**	Get a quaternion from a rotation vector.
*	\retval	pQuaternion	Calculated quaternion.
*	\param	pVector		Input rotation vector.
*	\warning Rotation is in EulerXYZ
*/
FBSDK_DLL void FBRotationToQuaternion	(FBQuaternion &pQuaternion,	FBRVector pVector);

/**	Get a rotation vector from a quaternion vector.
*	\retval pVector		Calculated rotation vector.
*	\param	pQuaternion	Input quaternion.
*	\warning Rotation is in EulerXYZ
*/
FBSDK_DLL void FBQuaternionToRotation	(FBRVector &pVector,		FBQuaternion pQuaternion);


////////////////////////////////////////////////////////////////////////////////////
// Local/Global conversions
////////////////////////////////////////////////////////////////////////////////////
/**	Get local matrix from parent and child matrices.
*	Will calculate the local matrix from two global matrices. The resulting matrix will be a local matrix
*	containing the local transformations to go from the parent referentialto the child referential.
*	\retval	pMatrix			Calculated local matrix.
*	\param	pMatrixParent	Parent matrix (new base referential).
*	\param	pMatrixChild	Child matrix.
*/
FBSDK_DLL void FBGetLocalMatrix			(FBMatrix &pMatrix, FBMatrix pMatrixParent, FBMatrix pMatrixChild);

/**	Get global matrix from parent and child matrices.
*	From an input referential, this function will calculate the global matrix corresponding to 
*	the input local matrix (which is with respect to the parent matrix).
*	\retval	pMatrix			Calculated local matrix.
*	\param	pMatrixParent	Parent matrix.
*	\param	pLocalMatrix	Local matrix.
*/
FBSDK_DLL void FBGetGlobalMatrix		(FBMatrix &pMatrix, FBMatrix pMatrixParent, FBMatrix pLocalMatrix);

////////////////////////////////////////////////////////////////////////////////////
// Vector operations
////////////////////////////////////////////////////////////////////////////////////
/**	Add two vectors together (\e pResult = \e pV1 \b + \e pV2)
*	\retval	pResult	Resulting vector.
*	\param	pV1		1st vector.
*	\param	pV2		2nd vector.
*/
FBSDK_DLL void		FBAdd				(FBTVector &pResult, FBTVector &pV1, FBTVector &pV2);

/**	Subtract \e pV2 from \e pV1 (\e pResult = \e pV1 \b - \e pV2)
*	\retval	pResult	Resulting vector.
*	\param	pV1		1st vector.
*	\param	pV2		2nd vector.
*/
FBSDK_DLL void		FBSub				(FBTVector &pResult, FBTVector &pV1, FBTVector &pV2);

/**	Multiply \e pV2 from \e pV1 (\e pResult = \e pV1 \b * \e pV2)
*	\retval	pResult	Resulting vector.
*	\param	pV1		1st vector.
*	\param	pV2		2nd vector.
*/
FBSDK_DLL void		FBMult				(FBTVector &pResult, FBTVector &pV1, double pV2);

/** Calculate the cross product of two vectors.
*	\retval	pResult	Resulting vector.
*	\param	pV1		1st vector.
*	\param	pV2		2nd vector.
*/
FBSDK_DLL void		FBMult				(FBTVector &pResult, FBTVector &pV1, FBTVector &pV2);

/**	Calculate the dot product of two vectors.
*	\param	pV1		1st vector.
*	\param	pV2		2nd vector.
*	\return	Dot product.
*/
FBSDK_DLL double	FBDot				(FBTVector &pV1, FBTVector &pV2);

/**	Get the length of a vector.
*	\param	pV			Vector to calculate length for.
*	\return	Length of vector \e pV.
*/
FBSDK_DLL double	FBLength			(FBTVector &pV);

////////////////////////////////////////////////////////////////////////////////////
// Quaternion operations
////////////////////////////////////////////////////////////////////////////////////
/**	Add two quaternions together (\e pResult = \e pQ1 \b + \e pQ2)
*	\retval	pResult	Resulting quaternion.
*	\param	pQ1		1st quaternion.
*	\param	pQ2		2nd quaternion.
*/
FBSDK_DLL void		FBQAdd				(FBQuaternion &pResult, FBQuaternion &pQ1, FBQuaternion &pQ2);

/**	Subtract \e pQ2 from \e pQ1 (\e pResult = \e pQ1 \b - \e pQ2)
*	\retval	pResult	Resulting quaternion.
*	\param	pQ1		1st quaternion.
*	\param	pQ2		2nd quaternion.
*/
FBSDK_DLL void		FBQSub				(FBQuaternion &pResult, FBQuaternion &pQ1, FBQuaternion &pQ2);

/**	Multiply \e pQ2 from \e pQ1 (\e pResult = \e pQ1 \b * \e pQ2)
*	\retval	pResult	Resulting quaternion.
*	\param	pQ1		1st quaternion.
*	\param	pQ2		2nd quaternion.
*/
FBSDK_DLL void		FBQMult				(FBQuaternion &pResult, FBQuaternion &pQ1, double pQ2);

/** Calculate the cross product of two quaternions.
*	\retval	pResult	Resulting quaternion.
*	\param	pQ1		1st quaternion.
*	\param	pQ2		2nd quaternion.
*/
FBSDK_DLL void		FBQMult				(FBQuaternion &pResult, FBQuaternion &pQ1, FBQuaternion &pQ2);

/**	Calculate the dot product of two quaternions.
*	\param	pQ1		1st quaternion.
*	\param	pQ2		2nd quaternion.
*	\return	Dot product.
*/
FBSDK_DLL double	FBQDot				(FBQuaternion &pQ1, FBQuaternion &pQ2);

/**	Get the length of a quaternion.
*	\param	pQ	Quaternion to calculate length for.
*	\return	Length of quaternion \e pQ.
*/
FBSDK_DLL double	FBQLength			(FBQuaternion &pQ);

////////////////////////////////////////////////////////////////////////////////////
// Vertex operations
////////////////////////////////////////////////////////////////////////////////////
/**	Get the length of a vertex (from origin)
*	\param	pV	Vertex for which length is to be measured.
*	\return	Length of vertex (from origin).
*/
FBSDK_DLL double	FBLength			(FBVertex &pV);

////////////////////////////////////////////////////////////////////////////////////
// Rotation utilities
////////////////////////////////////////////////////////////////////////////////////
/**	Interpolate a rotation in Euler space.
*	\retval	pROut	Resulting, interpolated rotation.
*	\param	pR0	1st rotation.
*	\param	pR1	2nd rotation.
*	\param	pU	Interpolation ratio.
*/
FBSDK_DLL void		FBInterpolateRotation	( FBRVector &pROut, FBRVector &pR0, FBRVector &pR1, double pU );

/**	Get a continuous rotation in Euler space.
*	This routine will help to avoid gimble locks due to interpolation.
*	\retval	pROut	Successful continuous rotation (gimble-lock free).
*	\param	pR0	Suggested next rotation.
*	\param	pR1	Previous rotation.
*/
FBSDK_DLL void		FBGetContinuousRotation	( FBRVector &pROut, FBRVector &pR0, FBRVector &pR1 );

////////////////////////////////////////////////////////////////////////////////////
// Miscellaneous utilities 
////////////////////////////////////////////////////////////////////////////////////
/**	Clamp value.
*	\param	pV			Value to clamp.
*	\param	pL			Low limit.
*	\param	pH			High limit.
*	\return	Clamped value.
*/
FBSDK_DLL double	FBClamp		(double pV, double pL, double pH);

/**	Change from big endian to native format.
*	\param	pV			Value to modify.
*/
FBSDK_DLL void	FBBigEndianToNative		(unsigned short &pV);

/**	Change from little endian to native format.
*	\param	pV			Value to modify.
*/
FBSDK_DLL void	FBLittleEndianToNative	(unsigned short &pV);


#ifdef FBSDKUseNamespace
	}
#endif
#endif
