#ifndef __FBMODEL_H__
#define __FBMODEL_H__
/**************************************************************************
 Copyright (c) 1994 - 2006 Autodesk, Inc. and/or its licensors.
 All Rights Reserved.
 
 The coded instructions, statements, computer programs, and/or related 
 material (collectively the "Data") in these files contain unpublished 
 information proprietary to Autodesk, Inc. and/or its licensors, which is 
 protected by Canada and United States of America federal copyright law 
 and by international treaties.
 
 The Data may not be disclosed or distributed to third parties, in whole 
 or in part, without the prior written consent of Autodesk, Inc. 
 ("Autodesk").
 
 THE DATA IS PROVIDED "AS IS" AND WITHOUT WARRANTY.
 ALL WARRANTIES ARE EXPRESSLY EXCLUDED AND DISCLAIMED. AUTODESK MAKES NO 
 WARRANTY OF ANY KIND WITH RESPECT TO THE DATA, EXPRESS, IMPLIED OR 
 ARISING BY CUSTOM OR TRADE USAGE, AND DISCLAIMS ANY IMPLIED WARRANTIES 
 OF TITLE, NON-INFRINGEMENT, MERCHANTABILITY OR FITNESS FOR A PARTICULAR 
 PURPOSE OR USE. WITHOUT LIMITING THE FOREGOING, AUTODESK DOES NOT 
 WARRANT THAT THE OPERATION OF THE DATA WILL BE UNINTERRUPTED OR ERROR 
 FREE.
 
 IN NO EVENT SHALL AUTODESK, ITS AFFILIATES, PARENT COMPANIES, LICENSORS 
 OR SUPPLIERS ("AUTODESK GROUP") BE LIABLE FOR ANY LOSSES, DAMAGES OR 
 EXPENSES OF ANY KIND (INCLUDING WITHOUT LIMITATION PUNITIVE OR MULTIPLE 
 DAMAGES OR OTHER SPECIAL, DIRECT, INDIRECT, EXEMPLARY, INCIDENTAL, LOSS 
 OF PROFITS, REVENUE OR DATA, COST OF COVER OR CONSEQUENTIAL LOSSES OR 
 DAMAGES OF ANY KIND), HOWEVER CAUSED, AND REGARDLESS OF THE THEORY OF 
 LIABILITY, WHETHER DERIVED FROM CONTRACT, TORT (INCLUDING, BUT NOT 
 LIMITED TO, NEGLIGENCE), OR OTHERWISE, ARISING OUT OF OR RELATING TO THE 
 DATA OR ITS USE OR ANY OTHER PERFORMANCE, WHETHER OR NOT AUTODESK HAS 
 BEEN ADVISED OF THE POSSIBILITY OF SUCH LOSS OR DAMAGE.
 
**************************************************************************/

/** \file fbmodel.h
*	Contains definitions for models.
*/

#include <kaydaradef.h>
#ifndef FBSDK_DLL 
	#define FBSDK_DLL K_DLLIMPORT
#endif

#include <fbsdk/fbcore.h>
#include <fbsdk/fbcomponent.h>
#include <fbsdk/fbshader.h>	// FBPropertyListShader

// somebody at SGI defined this in their include files
// until we find which one, we remove it
#undef Status

#ifdef FBSDKUseNamespace
	namespace FBSDKNamespace {
#endif

typedef class FBSDK_DLL FBArrayTemplate<HFBModel> FBModelList;

FB_DEFINE_COMPONENT( FBSDK_DLL, Camera			);
FB_DEFINE_COMPONENT( FBSDK_DLL, Light			);
FB_DEFINE_COMPONENT( FBSDK_DLL, Model			);
FB_DEFINE_COMPONENT( FBSDK_DLL, Geometry		);
FB_DEFINE_COMPONENT( FBSDK_DLL, FastTessellator	);
FB_DEFINE_COMPONENT( FBSDK_DLL, Video			);
FB_DEFINE_COMPONENT( FBSDK_DLL, AnimationNode   );

/**	FBModelTransactionBegin.
*	This set of functions speeds up the process of batch operations on models.
*/
FBSDK_DLL void FBModelTransactionBegin();

/**	FBModelTransactionEnd.
*	This set of functions speeds up the process of batch operations on models.
*/
FBSDK_DLL void FBModelTransactionEnd();

////////////////////////////////////////////////////////////////////////////////////
// FBPropertyListModel
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBModel ); 
__FB_FORWARD( FBPropertyListModel );
FB_DEFINE_LIST( FBSDK_DLL, Model );

typedef class FBSDK_DLL FBPropertyBaseList< HFBModel > FBPropertyBaseListModel;

//! <b>List:</b> Model
class FBSDK_DLL FBPropertyListModel : public FBPropertyBaseList< HFBModel >
{
public:
	/**	Add a model to the property list.
	*	\param	pItem	Model to add to list.
	*	\return	Number of items in list after operation.
	*/
	virtual int	 Add	( HFBModel pItem );

    /**	Remove model \e pItem from property list.
	*	\param	pItem	Model to remove from list.
	*	\return The index of the removed element. Returns -1 if the element was not part of the list.
	*/
    virtual int  Remove	( HFBModel pItem );

    /** Remove the model at \e pIndex.
	*	\param pIndex	Index of model to remove.
	*/
    virtual void RemoveAt( int pIndex );

    /**	Get the model at \e pIndex.
	*	\param	pIndex	Index of model to get.
	*	\return Model at \e pIndex.
	*/
	virtual HFBModel operator[](int pIndex);

    /** Get the number of models.
	*	\return Number of models.
	*/
	virtual int  GetCount();
};

////////////////////////////////////////////////////////////////////////////////////
// FBModel
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBModel		);
__FB_FORWARD( FBModelLimits );

//! Types of transformation vector/matrices possible.
enum FBModelTransformationMatrix {   
		kModelTransformation           = 1<<0,					//!< Transformation.
        kModelRotation                 = 1<<1,					//!< Rotation.
        kModelTranslation              = 1<<2,					//!< Translation.
        kModelScaling                  = 1<<3,					//!< Scaling.
        kModelInverse_Transformation   = 1<<4,					//!< Inverse transformation.
        kModelInverse_Rotation         = 1<<5,					//!< Inverse rotation.
        kModelInverse_Translation      = 1<<6,					//!< Inverse translation.
        kModelInverse_Scaling          = 1<<7,					//!< Inverse scaling.
        kModelCenter                   = 1<<8,					//!< Center.
        kModelAll                      = (kModelCenter<<1)-1};	//!< All.


//! Modes for model shading.
enum FBModelShadingMode {	
		kFBModelShadingDefault,		//!< Default shading.
		kFBModelShadingWire,		//!< Wireframe shading.
		kFBModelShadingFlat,		//!< Flat shading.
		kFBModelShadingHard,		//!< Hard shading.
		kFBModelShadingTexture,		//!< Textured shading.
		kFBModelShadingNone,		//!< No shading.
};

FB_DEFINE_ENUM( FBSDK_DLL, ModelShadingMode				); 
FB_DEFINE_ENUM( FBSDK_DLL, ModelTransformationMatrix	);

//! Model class.
class FBSDK_DLL FBModel : public FBBox 
{
	__FBClassDeclare( FBModel,FBBox );
public:
	/**	Constructor.
	*	\param	pName		Name of model(default=NULL).
	*	\param	pObject	For internal use only(default=NULL).
	*/
	FBModel(char *pName = NULL, HIObject pObject=NULL);
	virtual void	FBDelete();

	IObject_Declare		(Implementation);			// Interface to IObject.
    ICallback_Declare	(Implementation);			// Interface to ICallback.

	FBPropertyListModel				Children;		//!< <b>List:</b> Children for model.

    FBPropertyListShader			Shaders;		//!< <b>List:</b> Shaders for model.
									
	FBPropertyString				ShortName;		//!< <b>Read Write Property:</b> Short name for model.
	FBPropertyBool					Icon3D;			//!< <b>Read Write Property:</b> Is model a 3D icon?
	FBPropertyBool					Selected;		//!< <b>Read Write Property:</b> Is model selected?
	FBPropertyBool					IsDeformable;	//!< <b>Read Write Property:</b> Is model deformable (weights)?
	FBPropertyModel					Parent;			//!< <b>Read Write Property:</b> Parent model.
	FBPropertyModel					LookAt;			//!< <b>Read Write Property:</b> Look at model (interest point).
	FBPropertyModel					UpVector;		//!< <b>Read Write Property:</b> UpVector model.
	FBPropertyGeometry				Geometry;		//!< <b>Read Only Property:</b> Geometry for the model.
	FBPropertyScene					Scene;			//!< <b>Read Only Property:</b> Scene containing the model.
	FBPropertyModelShadingMode		ShadingMode;	//!< <b>Read Write Property:</b> Shading mode for the model.
	FBPropertyAnimationNode			AnimationNode;	//!< <b>Read Only Property:</b> Animation node of the model.

	// Animatable
	FBPropertyAnimatableBool		Visibility;		//!< <b>Read Write Property:</b> Visibility of model. This can be overriden by the 'Show' property.
	FBPropertyAnimatableVector3d	Translation;	//!< <b>Read Write Property:</b> Lcl translation.
	FBPropertyAnimatableVector3d	Rotation;		//!< <b>Read Write Property:</b> Lcl rotation.
	FBPropertyAnimatableVector3d	Scaling;		//!< <b>Read Write Property:</b> Lcl scaling.
    FBPropertyBool                  Show;           //!< <b>Read Write Property:</b> Indicate if the viewer should show the object, according to its visibility value. This has a defaul value of 'false'.

	//@{
	/**	Load/Save from/to a file.
	*	\param	pFileName	File to/from which the model will be saved.
	*	\return	\b true if successful.
	*/
	bool LoadFromFile	( char *pFileName );
	bool SaveToFile		( char *pFileName );
	//@}

	/**	Set a matrix for the model.
	*	\param	pMatrix			Information to use to set the model's matrix.
	*	\param	pWhat			Type of matrix to set (default=transformation).
	*	\param	pGlobalInfo		\b true if it is GlobalInfo, \b false if Local (default=true).
	*/
    void SetMatrix(FBMatrix pMatrix, int pWhat=kModelTransformation,	bool pGlobalInfo=true);

	/**	Get a matrix from the model.
	*	\param	pWhat			Type of information requested (default=transformation).
	*	\param	pGlobalInfo		\b true if it is GlobalInfo, \b false if Local (default=true).
	*	\retval	pMatrix			Matrix to fill with requested information.
	*/
    void GetMatrix(FBMatrix &pMatrix, int pWhat=kModelTransformation,	bool pGlobalInfo=true);

	/**	Set a vector for the model.
	*	\param	pVector			Vector to use to set values.
	*	\param	pWhat			Type of information to set (default=translation, inverses not supported).
	*	\param	pGlobalInfo		\b true if it is GlobalInfo, \b false if Local (default=true).
	*/
    void SetVector(FBVector3d pVector, int pWhat=kModelTranslation,	bool pGlobalInfo=true);

	/**	Get a vector from the model.
	*	\param	pWhat			Type of information requested (default=translation, inverses not supported).
	*	\param	pGlobalInfo		\b true if it is GlobalInfo, \b false if Local (default=true).
	*	\retval	pVector			Vector to fill with requested values.
	*/
    void GetVector(FBVector3d &pVector,	int pWhat=kModelTranslation,	bool pGlobalInfo=true);

    /** Get the bounding box of the model.
    *   \retval pMin	Minimum value of the bounding box.
    *   \retval pMax	Maximum value of the bounding box.
    */
    void GetBoundingBox( FBVector3d& pMin, FBVector3d& pMax );

	/**Get the list of the points selection state
	*	\return Pointer to the list of the points selection state
	*/
	unsigned char* GetSelectedPoints();

	/**Get the number of selected points in the model
	*	\return Number of selected points.
	*/
	int GetSelectedPointsCount();

	/**	Store and Retrieve function that can be overloaded.
	*	\param	pFbxObject	FBX Object that is used to communicate I/O operations.
	*	\param	pStoreWhat	Which attributes are currently stored/retrieved.
	*/
	virtual bool FbxStore(HFBFbxObject pFbxObject, kFbxObjectStore pStoreWhat);
	virtual bool FbxRetrieve(HFBFbxObject pFbxObject, kFbxObjectStore pStoreWhat);

	/** ResetDisplayList
	*/
	void ResetDisplayList();

    //@{
    /** These function are used when saving the objects to a file.
    */
    virtual char*	FbxGetObjectType();
    virtual char*	FbxGetObjectSubType();
	//@}

};

////////////////////////////////////////////////////////////////////////////////////
// FBModelLimits
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBModelLimits );

//! Types of rotations.
enum FBRotationType		{ 
	kFBRotationTypeEuler,		//!< Euler limits.
	kFBRotationTypeQuaternion	//!< Quaternion limits.
};
//! Types of clamping.
enum FBRotationClampType	{ 
	kFBRotationClampTypeRectangular,	//!< Rectangular clamping.	
	kFBRotationClampTypeEllipsoid		//!< Ellipsoid clamping.
};

FB_DEFINE_ENUM( FBSDK_DLL, RotationType		);
FB_DEFINE_ENUM( FBSDK_DLL, RotationClampType	);

//! Limits for model transformations.
class FBSDK_DLL FBModelLimits : public FBComponent
{
	// Open Reality declaration.
	__FBClassDeclare( FBModelLimits, FBComponent );
public:
	/**	Constructor.
	*	\param	pModel	Model owning limits.
	*/
	FBModelLimits(HFBModel pModel);

	FBPropertyBool				TranslationAuto;	//!< <b>Read Write Property:</b> Is translation limit Auto?
	FBPropertyBool				RotationAuto;		//!< <b>Read Write Property:</b> Is rotation limit Auto?
	FBPropertyBool				ScalingAuto;		//!< <b>Read Write Property:</b> Is scaling limit Auto?

	FBPropertyBool				TranslationActive;	//!< <b>Read Write Property:</b> Is translation limit Active?
	FBPropertyBool				RotationActive;		//!< <b>Read Write Property:</b> Is rotation limit Active?
	FBPropertyBool				ScalingActive;		//!< <b>Read Write Property:</b> Is scaling limit Active?

	FBPropertyVector3d			TranslationCenter;	//!< <b>Read Write Property:</b> Default translation of object.
	FBPropertyVector3d			TranslationMin;		//!< <b>Read Write Property:</b> Minimum translation value (with respect to center).
	FBPropertyVector3d			TranslationMax;		//!< <b>Read Write Property:</b> Maximum translation value (with respect to center).

	FBPropertyVector3d			RotationCenter;		//!< <b>Read Write Property:</b> Default Rotation of object.
	FBPropertyVector3d			RotationMin;		//!< <b>Read Write Property:</b> Minimum Rotation value (with respect to center).
	FBPropertyVector3d			RotationMax;		//!< <b>Read Write Property:</b> Maximum Rotation value (with respect to center).
	FBPropertyRotationType		RotationType;		//!< <b>Read Write Property:</b> Rotation Type (Quaternion or Euler).
	FBPropertyRotationClampType	RotationClampType;	//!< <b>Read Write Property:</b> Clamping Type for rotation (rectangular or ellipsoid).
	FBPropertyVector3d			RotationAxis;		//!< <b>Read Write Property:</b> Axis for rotation limits.
	FBPropertyDouble			RotationAxisLength;	//!< <b>Read Write Property:</b> Length of rotation axis.

	FBPropertyVector3d			ScalingCenter;		//!< <b>Read Write Property:</b> Default Scaling of object.
	FBPropertyVector3d			ScalingMin;			//!< <b>Read Write Property:</b> Minimum Scaling value (with respect to center).
	FBPropertyVector3d			ScalingMax;			//!< <b>Read Write Property:</b> Maximum Scaling value (with respect to center).
};
////////////////////////////////////////////////////////////////////////////////////
// FBModelNull
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBModelNull);

//! Null object class.
class FBSDK_DLL FBModelNull : public FBModel {
	__FBClassDeclare( FBModelNull,FBModel );
public:
	/**	Constructor.
	*	\param	pName		Name of null(default=NULL).
	*	\param	pObject		For internal use only(default=NULL).
	*/
	FBModelNull(char *pName = NULL, HIObject pObject=NULL);

	FBPropertyDouble	Size;	//!< <b>Read Write Property:</b> Size (not related to scaling).
};

////////////////////////////////////////////////////////////////////////////////////
// FBModelRoot
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBModelRoot);

//! Root object class.
class FBSDK_DLL FBModelRoot : public FBModel {
	__FBClassDeclare( FBModelRoot,FBModel );
public:
	/**	Constructor.
	*	\param	pName		Name of root(default=NULL).
	*	\param	pObject		For internal use only(default=NULL).
	*/
	FBModelRoot(char *pName = NULL, HIObject pObject=NULL);

	FBPropertyDouble	Size;	//!< <b>Read Write Property:</b> Size (not related to scaling).
};

////////////////////////////////////////////////////////////////////////////////////
// FBModelMarker
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBModelMarker);

//! Model marker class.
class FBSDK_DLL FBModelMarker : public FBModel {
	__FBClassDeclare( FBModelMarker,FBModel );
public:
	/**	Constructor.
	*	\param	pName		Name of model marker(default=NULL).
	*	\param	pObject		For internal use only(default=NULL).
	*/
	FBModelMarker(char *pName = NULL, HIObject pObject=NULL);

	FBPropertyDouble	Size;	//!< <b>Read Write Property:</b> Size (not related to scaling).
	FBPropertyColor		Color;	//!< <b>Read Write Property:</b> Color of model marker.
};

////////////////////////////////////////////////////////////////////////////////////
// FBModelSkeleton
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBModelSkeleton);

//! Root object class.
class FBSDK_DLL FBModelSkeleton : public FBModel {
	__FBClassDeclare( FBModelSkeleton,FBModel );
public:
	/**	Constructor.
	*	\param	pName		Name of skeleton(default=NULL).
	*	\param	pObject		For internal use only(default=NULL).
	*/
	FBModelSkeleton(char *pName = NULL, HIObject pObject=NULL);

	FBPropertyDouble	Size;	//!< <b>Read Write Property:</b> Size (not related to scaling).
	FBPropertyColor		Color;	//!< <b>Read Write Property:</b> Color of skeleton node.
};

////////////////////////////////////////////////////////////////////////////////////
// FBModelCube
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBModelCube );

//! Cube model class.
class FBSDK_DLL FBModelCube : public FBModel {
	__FBClassDeclare( FBModelCube, FBModel );
public:
	/**	Constructor.
	*	\param	pName		Name of cube(default=NULL).
	*	\param	pObject		For internal use only(default=NULL).
	*/
	FBModelCube(char *pName = NULL, HIObject pObject=NULL);
};

////////////////////////////////////////////////////////////////////////////////////
// FBModelPlane
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBModelPlane );

//! Plane model class.
class FBSDK_DLL FBModelPlane : public FBModel {
	__FBClassDeclare( FBModelPlane, FBModel );
public:
	/**	Constructor.
	*	\param	pName		Name of Plane(default=NULL).
	*	\param	pObject		For internal use only(default=NULL).
	*/
	FBModelPlane(char *pName = NULL, HIObject pObject=NULL);
};

////////////////////////////////////////////////////////////////////////////////////
// FBModelPath3D
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBModelPath3D );

//! Tangeant modes available for path keys.
enum FBPathTangeantMode {
	kFBPathTangeantInvalid,	//!< Invalid tangeant.
	kFBPathTangeantLinear,	//!< Linear tangeant.
	kFBPathTangeantCubic,	//!< Cubic tangeant.
	kFBPathTangeantAuto,	//!< Automatic-cubic tangeant.
	kFBPathTangeantUser		//!< User-cubic tangeant.
};

FB_DEFINE_ENUM(FBSDK_DLL, PathTangeantMode);

//! Path 3D model class.
class FBSDK_DLL FBModelPath3D : public FBModel
{
	__FBClassDeclare(FBModelPath3D, FBModel);

public:
	/**	Constructor.
	*	\param	pName	Name of Path 3D(default=NULL).
	*	\param	pObject	For internal use only(default=NULL).
	*/
	FBModelPath3D(char* pName=NULL, HIObject pObject=NULL);

	//--- Visual ------------------------------------------------------------------------------------------------------------------------
    void		ShowCurveControls(bool pShow);
    void		ShowCurvePoints(bool pShow);

	//--- Key Manipulation --------------------------------------------------------------------------------------------------------------
	int		PathKeyGetCount();
	FBVector4d	PathKeyGet(int pKeyIndex);
	void		PathKeySet(int pKeyIndex, FBVector4d pTLocal, bool pUpdate=true);
	void		PathKeyRemove(int pKeyIndex);
    void		PathKeyClear();

	//--- Tangeant ----------------------------------------------------------------------------------------------------------------------
	void		PathKeySetLeftTangeant(int pKeyIndex, FBVector4d pTLocal, bool pUpdate=true);
	void		PathKeySetRightTangeant(int pKeyIndex, FBVector4d pTLocal, bool pUpdate=true);
	FBVector4d	PathKeyGetLeftTangeant(int pKeyIndex);
	FBVector4d	PathKeyGetRightTangeant(int pKeyIndex);
    void		PathKeySetXDerivative(int pKeyIndex, double pDerivative, bool pUpdate);
    void		PathKeySetYDerivative(int pKeyIndex, double pDerivative, bool pUpdate);
    void		PathKeySetZDerivative(int pKeyIndex, double pDerivative, bool pUpdate);
    void		PathKeySetXYZDerivative(int pKeyIndex, FBVector4d pDerivative, bool pUpdate);
    FBVector4d	PathKeyGetXYZDerivative(int pKeyIndex);
    double		PatKeyGetLeftTangeantLength(int pKeyIndex);
    double		PatKeyGetRightTangeantLength(int pKeyIndex);

	//--- Path remove keys (editing) ----------------------------------------------------------------------------------------------------
    int		GetSelectedPathKeyCount();
	void		PathKeyRemoveSelected();

	//--- Keys Helpers ------------------------------------------------------------------------------------------------------------------
    int		InsertNewStartKey();
    int		InsertNewEndKey();
    int		PathKeyStartAdd(FBVector4d pTLocal);
	int		PathKeyEndAdd(FBVector4d pTLocal);

	//--- Curve -------------------------------------------------------------------------------------------------------------------------
	int		Total_PathKeyAdd(double pTotalPercent, FBVector4d pTLocal);
    int		Total_IsPathKey(double pTotalPercent);
	FBVector4d	Total_GlobalPathEvaluate(double pTotalPercent);
	FBVector4d	Total_LocalPathEvaluate(double pTotalPercent);
	FBVector4d	Total_GlobalPathEvaluateDerivative(double pTotalPercent);
	FBVector4d	Total_LocalPathEvaluateDerivative(double pTotalPercent);
	int		Segment_PathKeyAdd(double pSegmentPercent, FBVector4d pTLocal);
    int		Segment_IsPathKey(double pSegmentPercent);
	FBVector4d	Segment_GlobalPathEvaluate(double pSegmentPercent);
	FBVector4d	Segment_LocalPathEvaluate(double pSegmentPercent);
	FBVector4d	Segment_GlobalPathEvaluateDerivative(double pSegmentPercent);
	FBVector4d	Segment_LocalPathEvaluateDerivative(double pSegmentPercent);

	//--- Key Type Converter ------------------------------------------------------------------------------------------------------------
    double		ConvertTotalPercentToSegmentPercent(double pPercent);
    double		ConvertSegmentPercentToTotalPercent(double pPercent);
    double		ConvertToSegmentPercentFactor();
    double		ConvertToTotalPercentFactor();

	//--- Tangeant ----------------------------------------------------------------------------------------------------------------------
    bool				SetRightTangeantMode(int pKeyIndex, FBPathTangeantMode pTangeantMode);
    bool				SetLeftTangeantMode(int pKeyIndex, FBPathTangeantMode pTangeantMode);
    FBPathTangeantMode	GetRightTangeantMode(int pKeyIndex);
    FBPathTangeantMode	GetLeftTangeantMode(int pKeyIndex);
};

////////////////////////////////////////////////////////////////////////////////////
// FBTexture
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBTexture );

/**	Texture usage modes.
*	What the texture is used for.
*/
enum FBTextureUsage		{ 
	kFBTextureUsageAll,						//!< All types of texture usage.
	kFBTextureUsageColor,					//!< Color map.
	kFBTextureUsageShadowMap,				//!< Shadow map.
	kFBTextureUsageLightMap,				//!< Light map.
	kFBTextureUsageSphericalReflection,		//!< Spherical reflection map.
	kFBTextureUsageSphereMap,				//!< Spherical map.
};

/**	Texture mapping modes.
*	How the texture is mapped.
*/
enum FBTextureMapping	{
	kFBTextureMappingUV,				//!< UV mapping.
	kFBTextureMappingXY,				//!< XY mapping.
	kFBTextureMappingYZ,				//!< YZ mapping.
	kFBTextureMappingXZ,				//!< XZ mapping
	kFBTextureMappingSpherical,			//!< Spherical mapping.
	kFBTextureMappingCylindrical,		//!< Cylindrical mapping.
	kFBTextureMappingEnvironment,		//!< Environment mapping.
	kFBTextureMappingProjection			//!< Projection mapping.
};

/**	Texture blend modes.
*	How the texture is blended with another.
*/
enum FBTextureBlendMode	{ 
	kFBTextureBlendTranslucent,				//!< Layer transparency.
	kFBTextureBlendAdditive,				//!< Layer addition.
	kFBTextureBlendModulate,				//!< Layer multiplication.
	kFBTextureBlendModulate2,				//!< Layer multiplication + brightness.
};

FB_DEFINE_ENUM( FBSDK_DLL, TextureUsage	);
FB_DEFINE_ENUM( FBSDK_DLL, TextureMapping	);
FB_DEFINE_ENUM( FBSDK_DLL, TextureBlendMode	);

//! Texture class.
// This class is used to encapsulate an underlying media that is used to texture a model.
//
class FBSDK_DLL FBTexture : public FBBox {
	__FBClassDeclare( FBTexture,FBBox );
public:
	/**	Constructor.
	*	\param	pName		Name of texture media. Can be a NULL pointer. If set, this will create a FBVideo object used as the Video property.
	*	\param	pObject		For internal use only.
	*/
	FBTexture(char *pName, HIObject pObject=NULL);
	virtual void FBDelete();

    FBPropertyString			Filename;		//!< <b>Read Write Property:</b> Filename for texture.
    FBPropertyInt				Width;			//!< <b>Read Only Property:</b> Width of texture.
    FBPropertyInt				Height;			//!< <b>Read Only Property:</b> Height of texture.

	FBPropertyTextureUsage		Usage;			//!< <b>Read Write Property:</b> Texture usage.
	FBPropertyTextureMapping	Mapping;		//!< <b>Read Write Property:</b> Texture mapping.
	FBPropertyTextureBlendMode	BlendMode;		//!< <b>Read Write Property:</b> Texture blend mode.

	FBPropertyAnimatableVector2d	Translation;	//!< <b>Read Write Property:</b> Translation coordinates.
	FBPropertyAnimatableVector3d	Rotation;		//!< <b>Read Write Property:</b> Rotation coordinates.
	FBPropertyAnimatableVector2d	Scaling;		//!< <b>Read Write Property:</b> Scaling coordinates.
	FBPropertyAnimatableDouble		Alpha;			//!< <b>Read Write Property:</b> Texture alpha value.

	FBPropertyBool				SwapUV;			//!< <b>Read Write Property:</b> Swap UV coordinates?
	FBPropertyBool				UseMaterial;	//!< <b>Read Write Property:</b> Use the material color?
	FBPropertyInt				CroppingX[2];	//!< <b>Read Write Property:</b> X cropping.
	FBPropertyInt				CroppingY[2];	//!< <b>Read Write Property:</b> Y cropping.

	FBPropertyVideo				Video;			//!< <b>Read Write Property:</b> Media used for texturing.

	void	OGLInit();
	double* GetMatrix();
};

////////////////////////////////////////////////////////////////////////////////////
// FBPropertyListTexture
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBPropertyListTexture);
FB_DEFINE_LIST( FBSDK_DLL, Texture );

//! \b PropertyList: Texture
class FBSDK_DLL FBPropertyListTexture : public FBPropertyBaseList< HFBTexture >
{
public:
    /** Add a texture to the property list.
    *   @param      pItem   Texture to add to the list.
    *   @return             The index of the element in the list. Returns -1 on failure.
    *   @warning            Adding a new element will invalidate any existing iterators
    *                       as the new items may not be added at the end of the list.
    */
    virtual int Add( HFBTexture pItem );

    /** Remove a specific texture from the property list.
    *   @param  pItem   Texture to remove from list.
	*	@return         The index of the removed element. Returns -1 if the element was not part of the list.
    */
    virtual int Remove( HFBTexture pItem );

    /** Remove the texture at a specific index.
    *   @param pIndex   Index of the texture to remove.
    */
    virtual void RemoveAt( int pIndex );

    /** Get the texture at a specific index.
    *   @param  pIndex  Index of texture to get.
    *   @return         The texture at the specified index. Will return a NULL pointer on failure.
    */
    virtual HFBTexture operator[](int pIndex);

    /** Get the number of textures in the list.
    *   \return How many items are present in the list.
    */
    virtual int GetCount();
};

////////////////////////////////////////////////////////////////////////////////////
// FBMaterial
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBMaterial );

//! Material class.
class FBSDK_DLL FBMaterial : public FBBox {
	__FBClassDeclare( FBMaterial,FBBox );
public:
	/**	Constructor.
	*	\param	pName		Name of material.
	*	\param	pObject		For internal use only(default=NULL).
	*/
	FBMaterial(char *pName, HIObject pObject=NULL);

	FBPropertyAnimatableColor	Ambient;	//!< <b>Read Write Property:</b> Ambient color.
	FBPropertyAnimatableColor	Diffuse;	//!< <b>Read Write Property:</b> Diffuse color.
	FBPropertyAnimatableColor	Specular;	//!< <b>Read Write Property:</b> Specular color.
	FBPropertyAnimatableColor	Emissive;	//!< <b>Read Write Property:</b> Emissive color.
	FBPropertyAnimatableDouble	Shininess;	//!< <b>Read Write Property:</b> Shininess value.
	FBPropertyAnimatableDouble	Opacity;	//!< <b>Read Write Property:</b> Opacity value.
	FBPropertyAnimatableDouble	Reflect;	//!< <b>Read Write Property:</b> Reflection value.

	void OGLInit();
};

////////////////////////////////////////////////////////////////////////////////////
// FBPropertyListMaterial
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBPropertyListMaterial );
FB_DEFINE_LIST( FBSDK_DLL, Material );

//! \b PropertyList: Material
class FBSDK_DLL FBPropertyListMaterial : public FBPropertyBaseList< HFBMaterial >
{
public:
    /** Add a material to the property list.
    *   @param      pItem   Material to add to the list.
    *   @return             The index of the element in the list. Returns -1 on failure.
    *   @warning            Adding a new element will invalidate any existing iterators
    *                       as the new items may not be added at the end of the list.
    */
    virtual int Add( HFBMaterial pItem );

    /** Remove a specific material from the property list.
    *   @param  pItem   Material to remove from list.
    *   @return         The index of the removed element. Returns -1 if the element was not part of the list.
    */
    virtual int Remove( HFBMaterial pItem );

    /** Remove the material at a specific index.
    *   @param pIndex   Index of the material to remove.
    */
    virtual void RemoveAt( int pIndex );

    /** Get the material at a specific index.
    *   @param  pIndex  Index of material to get.
    *   @return         The material at the specified index. Will return a NULL pointer on failure.
    */
    virtual HFBMaterial operator[](int pIndex);

    /** Get the number of materials in the list.
    *   \return How many items are present in the list.
    */
    virtual int GetCount();
};

////////////////////////////////////////////////////////////////////////////////////
// FBCluster
////////////////////////////////////////////////////////////////////////////////////
//! Different clustering modes.
enum FBClusterMode	{
	kFBClusterNormalize,		//!< Normalize (values between 0.0 and 1.0 )
	kFBClusterAdditive,			//!< Add the values together.
	kFBClusterTotal100			//!< The balanced values will add up to 100 percent.
};
FB_DEFINE_ENUM( FBSDK_DLL, ClusterMode );

__FB_FORWARD( FBCluster );

/** Cluster.
*	Weighting interface for meshes.
*	\warning	This class is experimental.
*/
class FBSDK_DLL FBCluster : public FBComponent {
	__FBClassDeclare( FBCluster,FBComponent );
public:
	/** Constructor.
	*	\param	pModel	Parent model in question.
	*/
	FBCluster(HFBModel pModel);

	//--- Cluster-level operations
	/**	Begin cluster definition.
	*	\param	pIndex		Link index.
	*	\return	Index of last item(default=-1).
	*/
	int ClusterBegin(int pIndex = -1);

	/**	End cluster definition.
	*	\param	pQueryOnly	Were the operations on the cluster non-destructive?
	*	\return 0, (Not implemented).
	*/
	int ClusterEnd(bool pQueryOnly = false);

	//--- Link-level operations
	/**	Set the name of a link.
	*	\param	pLinkNumber	Number value of link to name.
	*	\param	pName		Name of the link.
	*/
	void		LinkSetName(char *pName, int pLinkNumber);
	/**	Get the name of a link.
	*	\param	pLinkNumber	Number value of link to get name from.
	*	\return Name of link number \e pLinkNumber.
	*/
	char*		LinkGetName(int pLinkNumber);
	/**	Remove a link.
	*	\param	pLinkNumber		Number value of link to rename.
	*/
	void		LinkRemove(int pLinkNumber);
	/**	Get number of links.
	*	\return Number of links.
	*/
	int			LinkGetCount();
	/**	Set model to a link.
	*	\param	pModel	Model to set.
	*/
	void		LinkSetModel(HFBModel pModel);
	/**	Get model from a link.
	*	\param	pLinkNumber		Number value of link to get model from.
	*	\return Model at link number \e pLinkNumber.
	*/
	HFBModel	LinkGetModel(int pLinkNumber);
	/**	Get model associated with link.
	*	\param	pLinkNumber		Nubmer value of link to get associated model from.
	*	\return Model associated to link number \e pLinkNumber.
	*/
	HFBModel	LinkGetAssociateModel(int pLinkNumber);
	/**	Remove all unused links.
	*	\param	pThreshold	Weight value under which links are considered unused (default=-1).
	*/
	void		LinkClearUnused( double pThreshold = -1.0 );
	/**	Link at current vertex.
	*	\param	pLinkIndex		Index of link to add vertex to.
	*	\param	pPointIndex		Index of vertex to add.
	*/
	void		LinkSetCurrentVertex(int pLinkIndex,int pPointIndex);
	/**	Get current vertex at link.
	*	\param	pIndex	Index of link to get vertex from.
	*	\return Index value of the current vertex associated to link at index number \e pIndex
	*/
	int			LinkGetVertexIndex(int pIndex);

	//--- Vertex level operations.
	/**	Set transform of a cluster set.
	*	\param	pPosition	Position transform.
	*	\param	pRotation	Rotation transform.
	*	\param	pScaling	Scaling transform.
	*/
	void		VertexSetTransform(FBVector3d pPosition,FBVector3d pRotation, FBVector3d pScaling);
	/**	Get transform  of a cluster set.
	*	\retval	pPosition	Position transform.
	*	\retval	pRotation	Rotation transform.
	*	\retval	pScaling	Scaling transform.
	*/
	void		VertexGetTransform(FBVector3d& pPosition,FBVector3d& pRotation, FBVector3d& pScaling);
	/**	Add a vertex to a cluster.
	*	\param	pVertexIndex	Index of vertex to add.
	*	\param	pWeight			Weight to give to vertex.
	*/
	void		VertexAdd(int pVertexIndex,double pWeight);
	/**	Remove a vertex from a cluster.
	*	\param	pVertexIndex	Index of vertex to remove.
	*/
	void		VertexRemove(int pVertexIndex);
	/**	Get the number of vertices.
	*	\return Number of vertices in a cluster.
	*/
	int			VertexGetCount();
	/**	Get vertex number.
	*	\param	pIndex	Index of link to get vertex from.
	*	\return Number value of vertex at link number \e pIndex
	*/
	int			VertexGetNumber(int pIndex);
	/**	Get vertex weight.
	*	\param	pIndex	Index of link to get vertex from.
	*	\return Weight of vertex found at link number \e pIndex.
	*/
	double		VertexGetWeight(int pIndex);
	/**	Set vertex weight.
	*	\param	pIndex		Index of link to get vertex from.
	*	\param	pWeight		Weight to give to vertex.
	*/
	void		VertexSetWeight(double pWeight,int pIndex);
	/**	Clear all linked vertices.
	*/
	void		VertexClear();

	FBPropertyClusterMode	ClusterMode;		//!< <b>Read Write Property:</b> Cluster mode.
	FBPropertyDouble		ClusterAccuracy;	//!< <b>Read Write Property:</b> Cluster accuracy.
};

////////////////////////////////////////////////////////////////////////////////////
// FBCamera
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBCamera );

//! Frame size modes.
enum FBCameraFrameSizeMode { 
	kFBFrameSizeWindow,					//!< Frame size of window.
	kFBFrameSizeFixedRatio,				//!< Fixed ratio.
	kFBFrameSizeFixedResolution,		//!< Fixed resolution.
	kFBFrameSizeFixedWidthResolution,	//!< Fixed width resolution.
	kFBFrameSizeFixedHeightResolution	//!< Fixed height resolution.
};
//! Resolution modes.
enum FBCameraResolutionMode { 
	kFBResolutionCustom,				//!< Custom resolution mode.
	kFBResolutionD1NTSC,				//!< D1 NTSC.
	kFBResolutionNTSC,					//!< NTSC.
	kFBResolutionPAL,					//!< PAL.
	kFBResolutionD1PAL,					//!< D1 PAL.
	kFBResolutionHD,					//!< HD 1920x1080
	kFBResolution640x480,				//!< 640x480.
	kFBResolution320x200,				//!< 320x200
	kFBResolution320x240,				//!< 320x240
	kFBResolution128x128,				//!< 128x128
	kFBResolutionFullScreen				//!< FullScreen.
};
//! Aperture modes.
enum FBCameraApertureMode { 
	kFBApertureVertical,				//!< Vertical aperture varies.
	kFBApertureHorizontal,				//!< Horizontal aperture varies.
	kFBApertureVertHoriz				//!< Vertical and horizontal aperture vary.
};
//! Filmback types.
enum FBCameraFilmBackType { 
	kFBFilmBackCustom,					//!< Custom Filmback.
	kFBFilmBack16mmTheatrical,			//!< 16mm Theatrical.
	kFBFilmBackSuper16mm,				//!< Super16mm.
	kFBFilmBack35mmAcademy,				//!< 35mm Academy.
	kFBFilmBack35mmTVProjection,		//!< 35mm TV Projection.
	kFBFilmBack35mmFullAperture,		//!< 35mm Full Aperture.
	kFBFilmBack35mm185Projection,		//!< 35mm 185 Projection.
	kFBFilmBack35mmAnamorphic,			//!< 35mm Anamorphic.
	kFBFilmBack70mmProjection,			//!< 70mm Projection.
	kFBFilmBackVistaVision,				//!< Vista Vision.
	kFBFilmBackDynavision,				//!< Dynavision.
	kFBFilmBackIMAX						//!< IMAX.
};
//! Background drawing modes.
enum FBCameraBGDrawingMode { 
	kFBBGDrawingModeBackGround,			//!< Background plane drawn in background.
	kFBBGDrawingModeForeGround,			//!< Background plane drawn in foreground.
	kFBBGDrawingModeBackAndForeGround	//!< Background plane drawn in back and foreground.
};
//! Background plane viewing modes.
enum FBCameraViewBGPlaneMode { 
	kFBViewBGPlaneDisabled,				//!< Background plane disabled.
	kFBViewBGPlaneAlways,				//!< Always draw background plane.
	kFBViewBGPlaneWhenMedia				//!< Background plane when media.
};
//! Background distance modes.
enum FBCameraBGDistanceMode { 
	kFBDistModeRelativeToInterest,		//!< Background distance relative to interest.
	kFBDistModeAbsoluteFromCamera		//!< Background distance absolute from camera.
};
//! Safe area modes.
enum FBCameraSafeAreaMode { 
	kFBSafeAreaSquare,					//!< Square safe area.
	kFBSafeAreaRound					//!< Round safe area.
};
//! Antialiasing methods.
enum FBCameraAntiAliasingMethod	{ 
	kFBAntiAliasingSoftware,			//!< Antaliasing in software.
	kFBAntialiasingMultiSamplingOnyx	//!< Multisampling (only on Onyx).
};
//! Antialiasing sampling types.
enum FBCameraSamplingType { 
	kFBSamplingUniform,					//!< Uniform sampling.
	kFBSamplingStochastic				//!< Stochastic sampling.
};
//! Focus distance sources.
enum FBCameraFocusDistanceSource { 
	kFBFocusDistanceCameraInterest,		//!< Interest as source.
	kFBFocusDistanceSpecificDistance	//!< Specific distance as source.
};

//! Focus distance types.
enum FBCameraType { 
	kFBCameraTypePerspective,			//!< Interest as source.
	kFBCameraTypeOrthogonal				//!< Specific distance as source.
};

//! Camera matrix types.
enum FBCameraMatrixType { 
	kFBProjection,						//!< Projection View matrix (OpenGL).
	kFBModelView						//!< Model View matrix (OpenGL).
};


FB_DEFINE_ENUM( FBSDK_DLL, CameraType					); 
FB_DEFINE_ENUM( FBSDK_DLL, CameraApertureMode			); 
FB_DEFINE_ENUM( FBSDK_DLL, CameraFilmBackType			); 
FB_DEFINE_ENUM( FBSDK_DLL, CameraFrameSizeMode			); 
FB_DEFINE_ENUM( FBSDK_DLL, CameraResolutionMode			); 
FB_DEFINE_ENUM( FBSDK_DLL, CameraBGDrawingMode			); 
FB_DEFINE_ENUM( FBSDK_DLL, CameraViewBGPlaneMode		); 
FB_DEFINE_ENUM( FBSDK_DLL, CameraBGDistanceMode			); 
FB_DEFINE_ENUM( FBSDK_DLL, CameraSafeAreaMode			); 
FB_DEFINE_ENUM( FBSDK_DLL, CameraAntiAliasingMethod		); 
FB_DEFINE_ENUM( FBSDK_DLL, CameraSamplingType			);
FB_DEFINE_ENUM( FBSDK_DLL, CameraFocusDistanceSource	); 
FB_DEFINE_ENUM( FBSDK_DLL, CameraMatrixType				); 

/** Camera class.
*   Cameras do come in two versions: system cameras, and user cameras.
*   System cameras are created by the application and cannot be destroyed.
*/
class FBSDK_DLL FBCamera : public FBModel {
	__FBClassDeclare( FBCamera,FBModel );
public:
	/**	Constructor.
	*	\param	pName		Name of camera(default=NULL).
	*	\param	pObject		For internal use only(default=NULL).
	*/
	FBCamera(char * pName = NULL, HIObject pObject=NULL);

	FBPropertyBool						SystemCamera;           //!< <b>Read Only Property:</b> Indicate if this a default camera or a user-created camera.

    // Camera Format
	FBPropertyCameraFrameSizeMode		FrameSizeMode;          //!< <b>Read Write Property:</b> Frame size standard mode.
	FBPropertyCameraResolutionMode		ResolutionMode;         //!< <b>Read Write Property:</b> Resolution standard mode.
	FBPropertyDouble					ResolutionWidth;        //!< <b>Read Write Property:</b> Resolution width.
	FBPropertyDouble					ResolutionHeight;       //!< <b>Read Write Property:</b> Resolution height.
	FBPropertyDouble					WindowHeight;           //!< <b>Read Only Property:</b> Window height.
	FBPropertyDouble					PixelAspectRatio;       //!< <b>Read Write Property:</b> Pixel aspect ratio.
	FBPropertyDouble					NearPlaneDistance;      //!< <b>Read Write Property:</b> Near plane distance.
	FBPropertyDouble					FarPlaneDistance;       //!< <b>Read Write Property:</b> Far plane distance.
	FBPropertyBool						MouseLockCamera;        //!< <b>Read Write Property:</b> Mouse lock for camera?
	FBPropertyCameraType				Type;                   //!< <b>Read Write Property:</b> Type of camera
	FBPropertyCameraApertureMode		ApertureMode;           //!< <b>Read Write Property:</b> Aperture mode.
	FBPropertyAnimatableDouble			FieldOfView;            //!< <b>Read Write Property:</b> Field of View (used when in horizontal or vertical aperture modes).

	FBPropertyAnimatableDouble			FieldOfViewX;           //!< <b>Read Write Property:</b> Field of View X angle (used in horizontal and vertical aperture mode).
	FBPropertyAnimatableDouble			FieldOfViewY;           //!< <b>Read Write Property:</b> Field of View Y angle (used in horizontal and vertical aperture mode).
	FBPropertyAnimatableDouble			OpticalCenterX;         //!< <b>Read Write Property:</b> Optical Center X (mm).
	FBPropertyAnimatableDouble			OpticalCenterY;         //!< <b>Read Write Property:</b> Optical Center Y (mm).
	FBPropertyAnimatableDouble			FocalLength;            //!< <b>Read Write Property:</b> Focal Length.

    FBPropertyCameraFilmBackType		FilmBackType;           //!< <b>Read Write Property:</b> Film back standard type.
	FBPropertyDouble					FilmSizeWidth;          //!< <b>Read Write Property:</b> Width of the film.
	FBPropertyDouble					FilmSizeHeight;         //!< <b>Read Write Property:</b> Height of the film.
	FBPropertyDouble					FilmAspectRatio;        //!< <b>Read Write Property:</b> Film aspect ratio.
	FBPropertyDouble					SqueezeRatio;           //!< <b>Read Write Property:</b> Squeeze ratio.
	FBPropertyDouble					OrthogonalVerticalSize;	//!< <b>Read Write Property:</b> Near plane distance.

	// Camera Background options
	FBPropertyVideo						BackGroundMedia;				//!< <b>Read Write Property:</b> BackGround Image
	FBPropertyCameraBGDrawingMode		BackGroundDrawingMode;			//!< <b>Read Write Property:</b> Drawing mode for the background image.
	FBPropertyDouble					BackGroundMaterialThreshold;	//!< <b>Read Write Property:</b> Material threshold for a transparent background.
	FBPropertyBool						ForegroundTransparent;			//!< <b>Read Write Property:</b> Is the background transparent?
	FBPropertyCameraViewBGPlaneMode		ViewBackGroundPlaneMode;		//!< <b>Read Write Property:</b> Background plane view mode
	FBPropertyBool						BackGroundImageCenter;			//!< <b>Read Write Property:</b> Center the background image?
	FBPropertyBool						BackGroundImageFit;				//!< <b>Read Write Property:</b> Fit the background image?
	FBPropertyBool						BackGroundImageKeepRatio;		//!< <b>Read Write Property:</b> Keep the background image's ratio?
	FBPropertyBool						BackGroundImageCrop;			//!< <b>Read Write Property:</b> Crop the background image?
	FBPropertyDouble					BackGroundPlaneDistance;		//!< <b>Read Write Property:</b> Set the distance for the background plane.
	FBPropertyCameraBGDistanceMode		BackGroundPlaneDistanceMode;	//!< <b>Read Write Property:</b> Select mode for the background plane's distance.

	// Camera View Options
	FBPropertyBool						ViewCameraInterest;		//!< <b>Read Write Property:</b> Show the camera interest?
	FBPropertyBool						ViewNearFarPlane;		//!< <b>Read Write Property:</b> Show near/far planes?
	FBPropertyBool						ViewShowGrid;			//!< <b>Read Write Property:</b> Show grid?
	FBPropertyBool						ViewShowAxis;			//!< <b>Read Write Property:</b> Show axis?
	FBPropertyBool						ViewShowTimeCode;		//!< <b>Read Write Property:</b> Show time code?
	FBPropertyBool						ViewDisplaySafeArea;	//!< <b>Read Write Property:</b> Display safe area?
	FBPropertyBool						ViewOpticalCenter;		//!< <b>Read Write Property:</b> View optical center?
	FBPropertyCameraSafeAreaMode		SafeAreaMode;			//!< <b>Read Write Property:</b> Select mode for safe area.

	FBPropertyAnimatableColor			BackGroundColor;		//!< <b>Read Write Property:</b> Background color for camera.
	FBPropertyBool						UseFrameColor;			//!< <b>Read Write Property:</b> Use frame color?
	FBPropertyColor						FrameColor;				//!< <b>Read Write Property:</b> Frame color for camera.

	// Camera Render Options
		// Rendering Options
	FBPropertyBool						UseAntiAliasing;		//!< <b>Read Write Property:</b> Use anti-aliasing?
	FBPropertyBool						UseDepthOfField;		//!< <b>Read Write Property:</b> Use depth of field calculations?
	FBPropertyBool						InteractiveMode;		//!< <b>Read Write Property:</b> Interactive mode?

		// Antialiasing Options
	FBPropertyDouble					AntiAliasingIntensity;	//!< <b>Read Write Property:</b> Anti-aliasing intensity.
	FBPropertyCameraAntiAliasingMethod	AntiAliasingMethod;		//!< <b>Read Write Property:</b> Anti-aliasing method.
	FBPropertyInt						NumberOfSamples;		//!< <b>Read Write Property:</b> Number of samples to oversample with.
	FBPropertyCameraSamplingType		SamplingType;			//!< <b>Read Write Property:</b> Type of over sampling.
	FBPropertyBool						UseAccumulationBuffer;	//!< <b>Read Write Property:</b> Use accumulation buffer?
	
		// Depth of Field Options
	FBPropertyCameraFocusDistanceSource	FocusDistanceSource;	//!< <b>Read Write Property:</b> Select source for focusing.
	FBPropertyDouble					FocusSpecificDistance;	//!< <b>Read Write Property:</b> Specfic distance for focusing.
	FBPropertyDouble					FocusAngle;				//!< <b>Read Write Property:</b> Focus Angle (rendering dof).

		// 2D Magnifier
	FBPropertyBool						Use2DMagnifier;			//!< <b>Read Write Property:</b> Enable/Disable the 2D Magnifier.
	FBPropertyBool						Display2DMagnifierFrame;//!< <b>Read Write Property:</b> Enable/Disable the drawing of the 2D Magnifier frame box.
	FBPropertyAnimatableDouble			MagnifierZoom;			//!< <b>Read Write Property:</b> 2D Magnifier Zoom value.
	FBPropertyAnimatableDouble			MagnifierPosX;			//!< <b>Read Write Property:</b> 2D Magnifier X Position.
	FBPropertyAnimatableDouble			MagnifierPosY;			//!< <b>Read Write Property:</b> 2D Magnifier Y Position.

	// Other Attributes
	FBPropertyBool						Active;					//!< <b>Read Write Property:</b> Is camera registered in application?
	FBPropertyModel						Interest;				//!< <b>Read Write Property:</b> Direct camera's interest. 
	FBPropertyAnimatableDouble			Roll;					//!< <b>Read Write Property:</b> Camera's roll on it's Z axis.
	FBPropertyAnimatableDouble			TurnTable;				//!< <b>Read Write Property:</b> Camera's rotation around its interest.
	FBPropertyBool						DisplayTurnTableIcon;	//!< <b>Read Write Property:</b> Enable/Disable the drawing of the Turn Table icon.

	/**	Obtain the camera's transformation matrix (either Model or Projection View).
	*	\param	pType	Matrix type to obtain.
	*	\return	Matrix of doubles (the camera's matrix in question).
	*/
	double*	GetMatrix( FBCameraMatrixType pType );
};

////////////////////////////////////////////////////////////////////////////////////
// FBPropertyListCamera
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBPropertyListCamera );
FB_DEFINE_LIST( FBSDK_DLL, Camera );

//! \b PropertyList: Camera
class FBSDK_DLL FBPropertyListCamera : public FBPropertyBaseList< HFBCamera >
{
public:
	/**	Add a Camera to the property list.
	*	\param	pItem	Camera to add to list.
	*	\return	Number of items in list after operation.
	*/
	virtual int	 Add	( HFBCamera pItem );

    /**	Remove Camera \e pItem from property list.
	*	\param	pItem	Camera to remove from list.
	*	\return The index of the removed element. Returns -1 if the element was not part of the list.
	*/
    virtual int  Remove	( HFBCamera pItem );

    /** Remove the camera at \e pIndex.
	*	\param pIndex	Index of camera to remove.
	*/
    virtual void RemoveAt( int pIndex );

    /**	Get the model at \e pIndex.
	*	\param	pIndex	Index of camera to get.
	*	\return Camera at \e pIndex.
	*/
	virtual HFBCamera operator[](int pIndex);

    /** Get the number of cameras.
	*	\return Number of cameras.
	*/
	virtual int GetCount();
};

////////////////////////////////////////////////////////////////////////////////////
// FBCameraSwitcher
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBCameraSwitcher );

/** Camera switcher.
*   This class is a wrapper around the system's camera switcher object. There can
*   only be one switcher in a given scene. Any attempts at creating a new instance
*   will return the existing one.
*/
class FBSDK_DLL FBCameraSwitcher : public FBCamera
{
	__FBClassDeclare( FBCameraSwitcher, FBCamera );
public:
	/**	Constructor.
	*	\param	pName		Name of camera(default=NULL).
	*	\param	pObject		For internal use only(default=NULL).
	*/
	FBCameraSwitcher(char * pName = NULL, HIObject pObject=NULL);

	FBPropertyCamera	CurrentCamera;  //!< <b>Read Write Property:</b> Camera currently being used by the switcher.
};


////////////////////////////////////////////////////////////////////////////////////
// FBGlobalLight
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBGlobalLight );

//! Fog falloff modes.
enum FBFogMode	{ 
	kFBFogModeLinear,				//!< Linear falloff.
	kFBFogModeExponential,			//!< Exponential falloff.
	kFBFogModeSquareExponential		//!< Squared exponential falloff.
};

FB_DEFINE_ENUM( FBSDK_DLL, FogMode );

//! Global light class.
class FBSDK_DLL FBGlobalLight : public FBBox
{
	__FBClassDeclare( FBGlobalLight, FBBox );
public:
	//! Constructor.
	FBGlobalLight();

	FBPropertyAnimatableColor	AmbientColor;		//!< <b>Read Write Property:</b> Ambient light color.

	// Fog
	FBPropertyBool				FogEnable;			//!< <b>Read Write Property:</b> Enable fog?

	FBPropertyAnimatableColor	FogColor;			//!< <b>Read Write Property:</b> Fog color.
	FBPropertyAnimatableDouble	FogBegin;			//!< <b>Read Write Property:</b> Begin fog distance.
	FBPropertyAnimatableDouble	FogEnd;				//!< <b>Read Write Property:</b> End fog distance.
	FBPropertyAnimatableDouble	FogDensity;			//!< <b>Read Write Property:</b> Fog density.

	FBPropertyFogMode			FogMode;			//!< <b>Read Write Property:</b> Fog falloff mode.
};

////////////////////////////////////////////////////////////////////////////////////
// FBLight
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBLight );

//! Light types
enum FBLightType	{ 
	kFBLightTypePoint,		//!< Point light.
	kFBLightTypeInfinite,	//!< Infinite light (plane).
	kFBLightTypeSpot		//!< Spotlight.
};

FB_DEFINE_ENUM( FBSDK_DLL, LightType	);

//! Light class.
class FBSDK_DLL FBLight : public FBModel 
{
	__FBClassDeclare( FBLight,FBModel );
public:
	/**	Constructor.
	*	\param	pName		Name of light(default=NULL).
	*	\param	pObject		For internal use only(default=NULL).
	*/
	FBLight(char *pName = NULL, HIObject pObject=NULL);

	// Light parameters
	FBPropertyLightType			LightType;					//!< <b>Read Write Property:</b> Type of light.
	FBPropertyAnimatableDouble	Intensity;					//!< <b>Read Write Property:</b> Light intensity.
	FBPropertyAnimatableDouble	ConeAngle;					//!< <b>Read Write Property:</b> Cone angle for light.
	FBPropertyAnimatableDouble	FogIntensity;				//!< <b>Read Write Property:</b> Intensity of the fog (spot light).
	FBPropertyAnimatableColor	DiffuseColor;				//!< <b>Read Write Property:</b> Color: Diffuse color.
	FBPropertyBool				CastLightOnObject;			//!< <b>Read Write Property:</b> Cast light on object?
	FBPropertyBool				CastShadow;					//!< <b>Read Write Property:</b> Cast shadow for light?

	// Effects and Gobos
	FBPropertyBool				DrawGroundProjection;		//!< <b>Read Write Property:</b> Draw ground projection of gobo?
	FBPropertyBool				DrawVolumetricLight;		//!< <b>Read Write Property:</b> Draw volumetric light with gobo?
	FBPropertyBool				DrawFrontFacingVolumetric;	//!< <b>Read Write Property:</b> Draw front facing volumetric light?
	FBPropertyVideo				GoboMedia;					//!< <b>Read Write Property:</b> Media to use as a Gobo with the light.
};

////////////////////////////////////////////////////////////////////////////////////
// FBPropertyListLight
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBPropertyListLight );
FB_DEFINE_LIST( FBSDK_DLL, Light );

//! \b PropertyList: Light
class FBSDK_DLL FBPropertyListLight : public FBPropertyBaseList< HFBLight >
{
public:
	/**	Add a Light to the property list.
	*	\param	pItem	Light to add to list.
	*	\return	Number of items in list after operation.
	*/
	virtual int	 Add	( HFBLight pItem );

    /** Remove the light at \e pIndex.
	*	\param pIndex	Index of light to remove.
	*/
    virtual void RemoveAt( int pIndex );

    /**	Get the model at \e pIndex.
	*	\param	pIndex	Index of light to get.
	*	\return Light at \e pIndex.
	*/
	virtual HFBLight operator[](int pIndex);

    /** Get the number of lights.
	*	\return Number of lights.
	*/
	virtual int GetCount();
};

////////////////////////////////////////////////////////////////////////////////////
// Property list PrimitiveVertexCount
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBPropertyListPrimitiveVertexCount );

//! \b PropertyList: NormalIndex
class FBSDK_DLL FBPropertyListPrimitiveVertexCount : public FBPropertyBaseList< int >
{
public:
	virtual int	 Add	( int pItem );
    virtual int  RemoveIt	( int pItem );
    virtual void RemoveAt( int pIndex );
	virtual int operator[](int pIndex);
	virtual int  GetCount();
};

////////////////////////////////////////////////////////////////////////////////////
// Property list VertexIndex
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBPropertyListVertexIndex );

//! \b PropertyList: NormalIndex
class FBSDK_DLL FBPropertyListVertexIndex : public FBPropertyBaseList< int >
{
public:
	virtual int	 Add	( int pItem );
    virtual int  RemoveIt	( int pItem );
    virtual void RemoveAt( int pIndex );
	virtual int operator[](int pIndex);
	virtual int  GetCount();
};

////////////////////////////////////////////////////////////////////////////////////
// Property list NormalIndex
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBPropertyListNormalIndex );

//! \b PropertyList: NormalIndex
class FBSDK_DLL FBPropertyListNormalIndex : public FBPropertyBaseList< int >
{
public:
	virtual int	 Add	( int pItem );
    virtual int  RemoveIt	( int pItem );
    virtual void RemoveAt( int pIndex );
	virtual int operator[](int pIndex);
	virtual int  GetCount();
};

////////////////////////////////////////////////////////////////////////////////////
// Property list UVIndex
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBPropertyListUVIndex );

class FBSDK_DLL FBPropertyListUVIndex : public FBPropertyBaseList< int >
{
public:
	virtual int	 Add	( int pItem );
    virtual int  RemoveIt	( int pItem );
    virtual void RemoveAt( int pIndex );
	virtual int operator[](int pIndex);
	virtual int  GetCount();
};

////////////////////////////////////////////////////////////////////////////////////
// Property list UVIndex
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBPropertyListMaterialId );

class FBSDK_DLL FBPropertyListMaterialId : public FBPropertyBaseList< int >
{
public:
	virtual int	 Add	( int pItem );
    virtual int  RemoveIt	( int pItem );
    virtual void RemoveAt( int pIndex );
	virtual int operator[](int pIndex);
	virtual int  GetCount();
};

////////////////////////////////////////////////////////////////////////////////////
// Property list UVIndex
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBPropertyListTextureId );

class FBSDK_DLL FBPropertyListTextureId : public FBPropertyBaseList< int >
{
public:
	virtual int	 Add	( int pItem );
    virtual int  RemoveIt	( int pItem );
    virtual void RemoveAt( int pIndex );
	virtual int operator[](int pIndex);
	virtual int  GetCount();
};

////////////////////////////////////////////////////////////////////////////////////
// FBFastTessellator
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBFastTessellator );

//! Fast tessellator class.
class FBSDK_DLL FBFastTessellator : public FBComponent {
	__FBClassDeclare( FBFastTessellator,FBComponent );
private:
	FBFastTessellator(HFBGeometry pGeometry, int pType);

public:
	FBPropertyBool		IsMesh;             //!< <b>Read Only Property:</b> true if it's a mesh, false if it's a surface (nurbs or patch)
	FBPropertyBool		IsDeformable;       //!< <b>Read Only Property:</b> 
	FBPropertyBool		HasChanged;         //!< <b>Read Only Property:</b> 
	FBPropertyBool		IsDrawable;         //!< <b>Read Only Property:</b> true if the tesselated geometry has been computed for this frame
	FBPropertyBool		ForceUVByVertex;    //!< <b>Read Write Property:</b> true if the texture coordinates are forced buvertex instead of by polygon-vertex

	FBPropertyInt		PrimitiveCount;     //!< <b>Read Only Property:</b> 
	FBPropertyInt		VertexCount;        //!< <b>Read Only Property:</b> 
	FBPropertyInt		NormalMode;         //!< <b>Read Only Property:</b> 
	
	FBPropertyListPrimitiveVertexCount	PrimitiveVertexCount;   //!< <b>List:</b> 
	FBPropertyListVertexIndex			VertexIndex;            //!< <b>List:</b> 
	FBPropertyListNormalIndex			NormalIndex;            //!< <b>List:</b> 
	FBPropertyListUVIndex				UVIndex;                //!< <b>List:</b> 
	FBPropertyListMaterialId			MaterialId;             //!< <b>List:</b> 
	FBPropertyListTextureId				TextureId;              //!< <b>List:</b> 

	float*	GetPositionArray();
	float*	GetNormalArray();
	float*	GetUVArray();
	double* GetTransformationMatrix();

	friend class DataFBGeometry;
};

////////////////////////////////////////////////////////////////////////////////////
// FBGeometry
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBGeometry );

//! Different types of geometry mapping possible.
enum FBGeometryMappingMode { 
	kFBMappingModeUnique,	//!< Unique mapping.
	kFBMappingModeVertex,	//!< Vertex mapping.
	kFBMappingModeSurface	//!< Surface mapping.
};

enum FBGeometryUVMappingMode {
	kFBUVMappingModeNone,
	kFBUVMappingModeVertex,
	kFBUVMappingModeSurface,
	kFBUVMappingModeRaw
};

FB_DEFINE_ENUM( FBSDK_DLL, GeometryMappingMode ); 
FB_DEFINE_ENUM( FBSDK_DLL, GeometryUVMappingMode ); 

/** Geometry class.
*   This class groups all geometry related elements which are shared across
*   the different subclasses (FBMesh, FBSurface, FBNurbs and FBPatch).
*
*   Class FBGeometry has a property 'MaterialMappingMode' which can be one
*   of three values. Depending on which is the current values there are
*   different ways of getting the "current" material.
*
*   kFBMappingModeUnique:
*       Meaning that there is only one material used for the whole model.
*       The material used is the last one in the list of FBGeometry::Materials.
*
*   kFBMappingModeVertex:
*       Meaning that each vertex has its own material. Calling
*       FBGeometry::VertexMaterialIdGet() will give you the index of the
*       material used from the list FBGeometry::Materials.
*
*   kFBMappingModeSurface:
*       Meaning that each polygon of the model has its own material. Calling
*       FBMesh::PolygonMaterialIdGet() will give you the index of the
*       material used from the list FBGeometry::Materials.
*
* By default, models created with the GUI or the SDK use the mapping mode
* kFBMappingModeUnique. Doing a call to FBGeometry::VertexMaterialIdSet()
* (with valid vertex Id and material Id) will change the mapping mode to
* kFBMappingModeVertex. The equivalent is true when doing a call to method
* FBMesh::PolygonMaterialIdSet().
*/
class FBSDK_DLL FBGeometry : public FBComponent {
	__FBClassDeclare( FBGeometry,FBComponent );
public:
	void* GetGPS();
	/**	Constructor.
	*	\param	pModel	Model to which the geometry belongs.
	*/
	FBGeometry(HFBModel pModel);

	//@{
	/**	Begin/End geometry mapping.
	*	\return	\b true if successful.
	*/
	bool		GeometryBegin();
	bool		GeometryEnd();
	//@}

	/**	Initialize vertex
	*	\param	pSize	Number of vertices to allocate.
	*/
	void		VertexInit	(int pSize);

	/**	Add a vertex.
	*	\param	pVertex		Vertex values used to add vertex.
	*	\return	Index where vertex was added.
	*/
	int			VertexAdd	(FBVertex pVertex);

	/** Add a vertex.
	*	\param	px	X coordinate of vertex to add.
	*	\param	py	Y coordinate of vertex to add.
	*	\param	pz	Z coordinate of vertex to add.
	*	\return	Index where vertex was added.
	*/
	int			VertexAdd	(double px, double py, double pz);

	/** Set a vertex
	*	\param	pVertex		Vertex values used to set vertex.
	*	\param	pIndex		Index of vertex to affect (default=-1).
	*	\return	\b true if successful.
	*/
	bool		VertexSet	(FBVertex pVertex,int pIndex=-1);

	/** Set a vertex
	*	\param	px		X coordinate to set.
	*	\param	py		Y coordinate to set.
	*	\param	pz		Z coordinate to set.		
	*	\param	pIndex	Index of vertex to set(default=-1).
	*	\return	\b true if successful.
	*/
	bool		VertexSet	(double px, double py, double pz,int pIndex=-1);

	/**	Get a vertex
	*	\param	pIndex	Index of vertex to get.
	*	\return	Vertex stored at \e pIndex.
	*/
	FBVertex	VertexGet	(int pIndex);

	/**	Get the number of vertices in the geometry.
	*	\return	Number of vertices in the geometry.
	*/
	int			VertexCount	();

	/**	Clear all vertices.
	*	\return	\b true if successful.
	*/
	bool		VertexClear	();

	/**	Set a normal at a vertex.
	*	\param	pVertex		Normal to set.
	*	\param	pIndex		Index of vertex to set Normal at(default=-1).
	*	\return	\b true if successful.
	*/
	bool		VertexNormalSet(FBNormal pVertex,int pIndex=-1);

	/**	Set a normal at a vertex.
	*	\param	px		X coordinate of normal.
	*	\param	py		Y coordinate of normal.
	*	\param	pz		Z coordinate of normal.		
	*	\param	pIndex	Index of vertex to set Normal at(default=-1).
	*	\return	\b true if successful.
	*/
	bool		VertexNormalSet(double px, double py, double pz,int pIndex=-1);

	/**	Get a normal at a vertex.
	*	\param	pIndex	Vertex to get normal at(default=-1).
	*	\return	Normal of vertex at \e pIndex.
	*/
	FBNormal	VertexNormalGet(int pIndex=-1);

	/**	Set a UV coordinate.
	*	\param	pUV				UV coordinate to set.
	*	\param	pIndex			Index of Vertex to affect with UV coordinate(default=-1).
	*	\return	\b true if successful.
	*/
	bool		VertexUVSet(FBUV pUV, int pIndex=-1);

	/**	Set a UV coordinate.
	*	\param	pU				U coordinate to set.
	*	\param	pV				V coordinate to set.
	*	\param	pIndex			Index of Vertex to affect with UV coordinate(default=-1).
	*	\return	\b true if successful.
	*/
	bool		VertexUVSet(float pU, float pV, int pIndex=-1);

	/**	Get a UV coordinate.
	*	\param	pIndex			Index of Vertex to get UV coordinate for(default=-1).
	*	\return	UV coordinate at \e UVSetIndex.
	*/
	FBUV		VertexUVGet(int pIndex=-1);

	/**	Set Material ID at a vertex
	*	\param	pMaterialId		Material ID to set.
	*	\param	pIndex			Index of vertex to set material ID for(default=-1).
	*	\return	\b true if successful.
	*/
	bool		VertexMaterialIdSet(int pMaterialId,int pIndex=-1);

	/**	Get vertex material id number.
	*	\param	pIndex	Vertex to get material ID for(default=-1).
	*	\return	Material ID for vertex at \e pIndex.
	*/
	int			VertexMaterialIdGet(int pIndex=-1);

    /** Add an existing material on the model.
    *   @param  pMaterial   The material to add.
    *   @return             The index of the material in the material list. Returns -1 on failure.
    */
    int MaterialAdd( HFBMaterial pMaterial );

    /** Remove and existing material from the model.
    *   @param  pMaterial   The material to remove.
    *   @return             The index of the removed element. Returns -1 if the element was not part of the list.
    */
    int MaterialRemove( HFBMaterial pMaterial );

    /** Add an existing texture on the model.
    *   @param  pTexture    The texture to add.
    *   @return             The index of the texture in the texture list. Returns -1 on failure.
    */
    int TextureAdd( HFBTexture pTexture );

    /** Remove and existing texture from the model.
    *   @param  pTexture    The texture to remove.
    *   @return             The index of the removed element. Returns -1 if the element was not part of the list.
    */
    int TextureRemove( HFBTexture pTexture );

	FBPropertyListMaterial			Materials;				//!< <b>List:</b> Geometry materials.
	FBPropertyListTexture			Textures;				//!< <b>List:</b> Geometry textures.

	FBPropertyGeometryMappingMode	MaterialMappingMode;	//!< <b>Read Write Property:</b> Material mapping mode.
	FBPropertyGeometryMappingMode	TextureMappingMode;		//!< <b>Read Write Property:</b> Texture mapping mode.
	FBPropertyGeometryUVMappingMode	UVMappingMode;			//!< <b>Read Only Property:</b> UV mapping mode.

	FBPropertyFastTessellator		FastTessellatorStrip;	//!< <b>Read Only Property:</b> Shader API
	FBPropertyFastTessellator		FastTessellatorPolygon;	//!< <b>Read Only Property:</b> Shader API

   /**	Get the selected state of a vertex.
	*	\param	pIndex	The index of the vertex
	*	\return	true if the vertex is selected.false if not
	*/
	bool VertexGetSelected(int pIndex);
   /**	Set the selected state of a vertex.
	*	\param	pIndex	The index of the vertex
	*	\param	pState	The true to selected, false to unselect
	*	\return	true if the vertex is selected.false if not
	*/
	bool VertexSetSelected(int pIndex,bool pState);

   /**	Get the visible state of a vertex.
	*	\param	pIndex	The index of the vertex
	*	\return	true if the vertex is visible.false if not
	*/
	bool VertexGetVisible(int pIndex);
   /**	Set the visible state of a vertex.
	*	\param	pIndex	The index of the vertex
	*	\param	pState	The true to visible, false to unselect
	*	\return	true if the vertex is visible.false if not
	*/
	bool VertexSetVisible(int pIndex,bool pState);

   /**	Get the Transformable state of a vertex.
	*	\param	pIndex	The index of the vertex
	*	\return	true if the vertex is Transformable.false if not
	*/
	bool VertexGetTransformable(int pIndex);
};

////////////////////////////////////////////////////////////////////////////////////
// FBMesh
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBMesh );

//! Mesh class.
class FBSDK_DLL FBMesh : public FBGeometry {
	__FBClassDeclare( FBMesh,FBGeometry );
public:
	/**	Constructor.
	*	\param	pModel	Model to which mesh belongs.
	*/
	FBMesh(HFBModel pModel);

	/**	Begin Polygon definition.
	*	\return	Number of existing polygons in Mesh
	*/
	int		PolygonBegin();

	/**	Add a vertex.
	*	\param	pVertex		Index in mesh of vertex to add to polygon
	*	\return	\b true if successful.
	*/
	bool	PolygonVertexAdd(int pVertex);

	/**	Set a material ID.
	*	\param	pMaterialId		Material ID to set with.
	*	\param	pIndex			Vertex to set material ID at(default=-1).
	*	\return	\b true if successful.
	*/
	bool	PolygonMaterialIdSet(int pMaterialId,int pIndex=-1);

	/**	Set a texture ID
	*	\param	pTextureId		Texture ID to set with.
	*	\param	pIndex			Vertex to set texture ID at(default=-1).
	*	\return	\b true if successful.
	*/
	bool	PolygonTextureIdSet	(int pTextureId,int pIndex=-1);

	/**	Set a surface ID
	*	\param	pSurfaceId		Surface ID to set with.
	*	\param	pIndex			Vertex to set surface ID at(default=-1).
	*	\return	\b true if successful.
	*/
	bool	PolygonSurfaceIdSet	(int pSurfaceId,int pIndex=-1);

	/**	Get a Material ID.
	*	\param	pIndex	Vertex to get material ID at(default=-1).
	*	\return ID of material of vertex at \e pIndex.
	*/
	int		PolygonMaterialIdGet(int pIndex=-1);

	/**	Get a Texture ID.
	*	\param	pIndex	Vertex to get Texture ID at(default=-1).
	*	\return ID of Texture at vertex \e pIndex.
	*/
	int		PolygonTextureIdGet	(int pIndex=-1);

	/**	Get a Surface ID.
	*	\param	pIndex	Vertex to get Surface ID at(default=-1).
	*	\return ID of Surface of vertex at \e pIndex.
	*/
	int		PolygonSurfaceIdGet	(int pIndex=-1);

	/**	End Polygon definition.
	*	Clean up and associate vertices internally.
	*	\return	Current number of polygons.
	*/
	int		PolygonEnd();

	/**	Get Polygon vertext count
	*	\param	pPolygonIndex	Index of polygon to get vertex count from.
	*	\return	Number of vertices in polygon at \e pPolygonIndex.
	*/
	int		PolygonVertexCount(int pPolygonIndex);

	/**	Get global (for the mesh) index of a vertex from a polygon
	*	\param	pPolygonIndex			Index of polygon in question.
	*	\param	pVertexPolygonIndex		Polygon vertex index.
	*	\return	Index in mesh of vertex.
	*/
	int		PolygonVertexIndex(int pPolygonIndex, int pVertexPolygonIndex);

	/**	Get number of polygons in mesh.
	*	\return	Number of polygons in mesh.
	*/
	int		PolygonCount();
};

////////////////////////////////////////////////////////////////////////////////////
// FBSurface
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBSurface );

//! Surface modes
enum FBSurfaceMode
{
	kFBSurfaceModeRaw,				//!< Raw data.
	kFBSurfaceModeLowNoNormals,		//!< Low quality, no normals.
	kFBSurfaceModeLow,				//!< Low quality.
	kFBSurfaceModeHighNoNormals,	//!< High quality, no normals.
	kFBSurfaceModeHigh				//!< High quality.
};
FB_DEFINE_ENUM( FBSDK_DLL, SurfaceMode );

//! Surface types.
enum FBSurfaceType
{
	kFBSurfaceTypeBezier,			//!< Bezier surface.
	kFBSurfaceTypeBezierQuadric,	//!< Bezier Quadric surface.
	kFBSurfaceTypeCardinal,			//!< Cardinal surface.
	kFBSurfaceTypeBspline,			//!< BSpline surface.
	kFBSurfaceTypeLinear,			//!< Linear surface.
};
FB_DEFINE_ENUM( FBSDK_DLL, SurfaceType );


//! Surface class.
class FBSDK_DLL FBSurface : public FBGeometry
{
	__FBClassDeclare( FBSurface, FBGeometry );
  public:
	/**	Constructor.
	*	\param	pModel	Model to which Nurbs belongs.
	*/
	FBSurface(HFBModel pModel);

	IObject_Declare( Implementation );

	virtual void	SurfaceBegin();
	virtual void	SurfaceEnd();
	virtual void	SurfaceEditBegin();
	virtual void	SurfaceEditEnd();
	virtual bool	GetSurfaceCapped( int pUorV, int pDirection );
	virtual int		GetVertexCount( int pUorVorGlobal = -1 );
	virtual void	ControlPointsBegin();
	virtual void	SetControlPoint( int pIndex, double pX, double pY, double pZ, double pW );
	virtual void	GetControlPoint( int pIndex, double &pX, double &pY, double &pZ, double &pW );
	virtual void	ControlPointsEnd();

   /**	Get the selected state of a vertex.
	*	\param	pU	The u index of the vertex.
	*	\param	pV	The v index of the vertex.
	*	\return	true if the vertex is selected, false if not.
	*/
	bool VertexGetSelected(int pU,int pV);
   /**	Set the selected state of a vertex.
	*	\param	pU	The u index of the vertex.
	*	\param	pV	The v index of the vertex.
    *   \param  pState  Set the select state.
	*	\return	true if the vertex is selected, false if not.
	*/
	bool VertexSetSelected(int pU,int pV,bool pState);
		
   /**	Get the visible state of a vertex.
	*	\param	pU	The u index of the vertex.
	*	\param	pV	The v index of the vertex.
	*	\return	true if the vertex is visible, false if not.
	*/
	bool VertexGetVisible(int pU,int pV);

   /**	Set the visible state of a vertex.
	*	\param	pU	The u index of the vertex.
	*	\param	pV	The v index of the vertex.
    *   \param  pState  Set the visible state.
	*	\return	true if the vertex is visible, false if not.
	*/
	bool VertexSetVisible(int pU,int pV,bool pState);
	
   /**	Get the Transformable state of a vertex.
	*	\param	pU	The u index of the vertex.
	*	\param	pV	The v index of the vertex.
	*	\return	true if the vertex is Transformable, false if not.
	*/
	bool VertexGetTransformable(int pU,int pV);

	float* GetTessellatedPointCoords(int pIndex);
	
	int GetTessellatedPointsCount();
	
	float* GetTessellatedPointUV(int pIndex);
	
	float* GetTessellatedPointNormal(int pIndex);
	
	int GetTessellatedPointMaterialID();

	void GetTessellatedTrianglePoints(int pTriangleID, int* pTrianglePoints, int* pTriangleQuadUVIndex = NULL );
	
	int GetTessellatedTriangleCount();	

	void AddToArrayAndGoNext( int pValue, int **pArray, int *pRemainArraySize, int *pArrayCount  );

	int GetPatchStripVertexIndexCount();

	int GetPatchStripCount();

	int GetStripFromPatch( int *pStripIndexCountArray, int pStripIndexCountArraySize,  int *pStripVertexIndexArray, int pStripVertexIndexArraySize );

	FBPropertyInt			Size		[2];	//!< <b>Read Write Property:</b> Size in U and V directions.
	FBPropertyInt			Step		[2];	//!< <b>Read Write Property:</b> Step in U and V directions.
	FBPropertyBool			Closed		[2];	//!< <b>Read Write Property:</b> UV Closed?

	FBPropertySurfaceMode	SurfaceMode;		//!< <b>Read Write Property:</b> Surface mode.
	FBPropertyDouble		ReductionFactor;	//!< <b>Read Write Property:</b> Reduction factor.
};

////////////////////////////////////////////////////////////////////////////////////
// FBNurbs
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBNurbs );

//! Nurbs class.
class FBSDK_DLL FBNurbs : public FBSurface
{
	__FBClassDeclare( FBNurbs, FBSurface );
  public:
	/**	Constructor.
	*	\param	pModel	Model to which Nurbs belongs.
	*/
	FBNurbs(HFBModel pModel);

	/** Begin NURBS definition.
	*/
	virtual void	SurfaceBegin();

	/** End NURBS definition.
	*/
	virtual void	SurfaceEnd();

	/**	Begin NURBS surface edition
	*/
	virtual void	SurfaceEditBegin();
	/**	End NURBS surface edition
	*/
	virtual void	SurfaceEditEnd();
	/**	Begin NURBS control points edition
	*/
	virtual void	ControlPointsBegin();
	/**	End NURBS control points edition
	*/
	virtual void	ControlPointsEnd();
	/**	Set weight of control point
	*	\param	pIndex Index of control point to set weight at.
	*	\param	pWeight Weight of control point.
	*/
	virtual void	SetControlWeight( int pIndex, double pWeight );
	/**	Get weight of control point
	*	\param	pIndex Index of control point to get weight from.
	*	\return	Weight of control point at index \e pIndex.
	*/
	virtual double	GetControlWeight( int pIndex );
	/**	Set multiplicity (number of "instances") of control point.
	*	\param	pUorV	\b 1 if V multiplicity, \b 0 if U multlipicity.
	*	\param	pIndex	Index of control point to set multiplicity for.
	*	\param	pMultiplicity Multiplicity value for control point at \e pIndex.
	*/
	virtual void	SetControlMultiplicity( int pUorV, int pIndex, int	pMultiplicity	);
	/**	Get multiplicity (number of "instances") of control point.
	*	\param	pUorV	\b 1 if V multiplicity, \b 0 if U multlipicity.
	*	\param	pIndex	Index of control point to get multiplicity for.
	*/
	virtual int		GetControlMultiplicity( int pUorV, int pIndex );
	/**	Set knot vector value of control point.
	*	\param	pUorV	\b 1 if V knot vector, \b 0 if U knot vector.
	*	\param	pIndex	Index of control point to set knot value for.
	*	\param	pKnotValue Knot value for control point at \e pIndex.
	*/
	virtual void	SetControlKnotValue( int pUorV, int pIndex, double	pKnotValue		);
	/**	Get knot vector value of control point.
	*	\param	pUorV	\b 1 if V knot vector, \b 0 if U knot vector.
	*	\param	pIndex	Index of control point to set knot value for.
	*/
	virtual double	GetControlKnotValue( int pUorV, int pIndex );
	/**	Number of knot vectors
	*	\param	pUorV	\b 1 if V knot vector, \b 0 if U knot vector.
	*	\return	Number of knot vectors on NURBS surface
	*/
	virtual int		GetKnotCount( int pUorV );

	FBPropertyInt			Order	[2];	//!< <b>Read Write Property:</b> Nurbs order.
	FBPropertyBool			Periodic[2];	//!< <b>Read Write Property:</b> Periodic nurb?
};

////////////////////////////////////////////////////////////////////////////////////
// FBPatch
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBPatch );

//! Patch class.
class FBSDK_DLL FBPatch : public FBSurface
{
	__FBClassDeclare( FBPatch, FBSurface );
  public:
	/**	Constructor.
	*	\param	pModel	Model to which Patch belongs.
	*/
	FBPatch(HFBModel pModel);
	
	/** Begin Patch definition
	*/
	virtual void SurfaceBegin();

	/** End Patch definition
	*/
	virtual void SurfaceEnd();

	/**	Begin patch surface edit.
	*/
	virtual void SurfaceEditBegin();

	/**	End patch surface edit.
	*/
	virtual void SurfaceEditEnd();

	/** Begin control points edition
	*/
	virtual void ControlPointsBegin();
	/** End control points edition
	*/
	virtual void ControlPointsEnd();


	FBPropertySurfaceType	SurfaceType[2];	//!< <b>Read Write Property:</b> Patch mode for U direction.
};

#ifdef FBSDKUseNamespace
	}
#endif
#endif
