#ifndef __FBSYSTEM_H__
#define __FBSYSTEM_H__
/**************************************************************************
 Copyright (c) 1994 - 2006 Autodesk, Inc. and/or its licensors.
 All Rights Reserved.
 
 The coded instructions, statements, computer programs, and/or related 
 material (collectively the "Data") in these files contain unpublished 
 information proprietary to Autodesk, Inc. and/or its licensors, which is 
 protected by Canada and United States of America federal copyright law 
 and by international treaties.
 
 The Data may not be disclosed or distributed to third parties, in whole 
 or in part, without the prior written consent of Autodesk, Inc. 
 ("Autodesk").
 
 THE DATA IS PROVIDED "AS IS" AND WITHOUT WARRANTY.
 ALL WARRANTIES ARE EXPRESSLY EXCLUDED AND DISCLAIMED. AUTODESK MAKES NO 
 WARRANTY OF ANY KIND WITH RESPECT TO THE DATA, EXPRESS, IMPLIED OR 
 ARISING BY CUSTOM OR TRADE USAGE, AND DISCLAIMS ANY IMPLIED WARRANTIES 
 OF TITLE, NON-INFRINGEMENT, MERCHANTABILITY OR FITNESS FOR A PARTICULAR 
 PURPOSE OR USE. WITHOUT LIMITING THE FOREGOING, AUTODESK DOES NOT 
 WARRANT THAT THE OPERATION OF THE DATA WILL BE UNINTERRUPTED OR ERROR 
 FREE.
 
 IN NO EVENT SHALL AUTODESK, ITS AFFILIATES, PARENT COMPANIES, LICENSORS 
 OR SUPPLIERS ("AUTODESK GROUP") BE LIABLE FOR ANY LOSSES, DAMAGES OR 
 EXPENSES OF ANY KIND (INCLUDING WITHOUT LIMITATION PUNITIVE OR MULTIPLE 
 DAMAGES OR OTHER SPECIAL, DIRECT, INDIRECT, EXEMPLARY, INCIDENTAL, LOSS 
 OF PROFITS, REVENUE OR DATA, COST OF COVER OR CONSEQUENTIAL LOSSES OR 
 DAMAGES OF ANY KIND), HOWEVER CAUSED, AND REGARDLESS OF THE THEORY OF 
 LIABILITY, WHETHER DERIVED FROM CONTRACT, TORT (INCLUDING, BUT NOT 
 LIMITED TO, NEGLIGENCE), OR OTHERWISE, ARISING OUT OF OR RELATING TO THE 
 DATA OR ITS USE OR ANY OTHER PERFORMANCE, WHETHER OR NOT AUTODESK HAS 
 BEEN ADVISED OF THE POSSIBILITY OF SUCH LOSS OR DAMAGE.
 
**************************************************************************/

/**	\file fbsystem.h
*	Basic system interaction.
*	The FBSystem file contains the necessary data structure to 
*	interface with the underlying system.
*/

#include <kaydaradef.h>
#ifndef FBSDK_DLL 
	#define FBSDK_DLL K_DLLIMPORT
#endif

// Essential includes
#include <fbsdk/fbcomponent.h>
#include <fbsdk/fbdata.h>	// FBPropertyListTake

#include <fbsdk/fbio.h>		// FBPropertyListCommPort
#include <fbsdk/fbshader.h>
#include <fbsdk/fbmodel.h>
#include <fbsdk/fbvideo.h>

// For 'lean version' we don't need the plugin headers.
#if !defined(K_NO_AUDIO)
	#include <fbsdk/fbaudio.h>
#endif

#if !defined(K_NO_DECK)
	#include <fbsdk/fbdeck.h>			// FBPropertyListDeck
#endif

#if !defined(K_NO_ASSETMNG)
	#include <fbsdk/fbassetmng.h>	// FBPropertyAssetMng
#endif

#if !defined(K_NO_MANIPULATOR)
	#include <fbsdk/fbmanipulator.h>	// FBPropertyListManipulator
#endif

#ifdef FBSDKUseNamespace
	namespace FBSDKNamespace {
#endif

FB_FORWARD( FBCharacter );

struct FBVideoGrabOptions;
typedef FBVideoGrabOptions* HFBVideoGrabOptions;

struct FBPickInfos;
typedef class FBSDK_DLL FBArrayTemplate<FBPickInfos> FBPickInfosList;

FB_DEFINE_COMPONENT( FBSDK_DLL, Take );
FB_DEFINE_COMPONENT( FBSDK_DLL, Renderer );

////////////////////////////////////////////////////////////////////////////////////
// Utility function
////////////////////////////////////////////////////////////////////////////////////
// For internal use only.

/**	Get the SDK object from an Internal Object.
*	\param	pObject			Object to get SDK object for.
*	\param	pAutoCreate		Create object if it doesn't exist? (default = false)
*	\return	SDK object for \e pObject.
*/
FBSDK_DLL HIObject		FBGetFBObject( HIObject pObject, bool pAutoCreate=false );

////////////////////////////////////////////////////////////////////////////////////
// FBSystem
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBSystem );

/**	System interface class.
*	This class permits the user to interact with the underlying system, as well
*	as a few operating system calls.
*/
class FBSDK_DLL FBSystem : public FBComponent {
	__FBClassDeclare( FBSystem,FBComponent );
public:
	/**	Constructor.
	*/
	FBSystem();

    FBPropertyString                        ApplicationPath;            //!< <b>Read Only Property:</b> Location where the application is installed.
	FBPropertyEventUIIdle					OnUIIdle;					//!< <b>Event:</b> User-interface idle.
	FBPropertyEventConnectionNotify			OnConnectionNotify;			//!< <b>Event:</b> A connection event occured between objects in the system.
	FBPropertyEventConnectionDataNotify		OnConnectionDataNotify;		//!< <b>Event:</b> A data event occured between objects in the system.
	FBPropertyEventConnectionStateNotify	OnConnectionStateNotify;	//!< <b>Event:</b> A state change event occured between objects in the system.

	FBPropertyString						ComputerName;				//!< <b>Read Only Property:</b> Computer name.
	FBPropertyDouble						Version;					//!< <b>Read Only Property:</b> Application version.
																		
	FBPropertyTime							SystemTime;					//!< <b>Read Only Property:</b> System time.
	FBPropertyTime							LocalTime;					//!< <b>Read Only Property:</b> Local time in take.
	FBPropertyTake							CurrentTake;				//!< <b>Read Write Property:</b> Current take.
																		
	FBPropertyScene							Scene;						//!< <b>Read Only Property:</b> Scene.
																		
	FBPropertyModel							SceneRootModel;				//!< <b>Read Only Property:</b> Scene root model. 
	FBPropertyModel							RootModel;					//!< <b>Read Only Property:</b> Root model.
	FBPropertyString						PathImages;					//!< <b>Read Only Property:</b> Path to images.
	FBPropertyString						PathMeshs;					//!< <b>Read Only Property:</b> Path to meshes
																		
	FBPropertyListMaterial					Materials;					//!< <b>List:</b> Materials for scene.
	FBPropertyListTexture					Textures;					//!< <b>List:</b> Textures for scene.
	FBPropertyListShader					Shaders;					//!< <b>List:</b> Shaders for scene.
	FBPropertyListTake						Takes;						//!< <b>List:</b> Takes for scene.
	FBPropertyListDevice					Devices;					//!< <b>List:</b> Devices for scene.
																		
#if !defined(K_NO_DECK)
	FBPropertyListDeck						Decks;						//!< <b>List:</b> Decks for scene.
#endif																	
																		
	FBPropertyListCamera					Cameras;					//!< <b>List:</b> Cameras in scene.
	FBPropertyListLight						Lights;						//!< <b>List:</b> Lights in scene.
	FBPropertyListMedia						Media;						//!< <b>List:</b> Media in scene. (Videos only)
																		
#if !defined(K_NO_AUDIO)
	FBPropertyListAudioIn					AudioInputs;				//!< <b>List:</b> Available audio inputs.
	FBPropertyListAudioOut					AudioOutputs;				//!< <b>List:</b> Available audio outputs.
	FBPropertyListAudioClip					AudioClips;					//!< <b>List:</b> Audio clips in scene.
#endif																	
																		
	FBPropertyListCommPort					CommPorts;					//!< <b>List:</b> Comm Ports available.
	FBPropertyRenderer						Renderer;					//!< <b>Read Only Property:</b> Default renderer.
																		
	FBPropertyDouble						FrameRate;					//!< <b>Read Only Property:</b> The frame rate of the viewer.
    FBPropertyVector2d						DesktopSize;				//!< <b>Read Only Property:</b> The width and height of the desktop.
    FBPropertyBool							FullScreenViewer;			//!< <b>Read Write Property:</b> Indicates that the viewer is in full screen mode.

#if !defined(K_NO_ASSETMNG)
	FBPropertyAssetMng						AssetManager;				//!< <b>Read Only Property:</b> Current asset manager.
#endif
};

////////////////////////////////////////////////////////////////////////////////////
// FBTriggerManager
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBTriggerManager );
/**	Trigger engine management.
*	This simple class interfaces the basic triggering functions.
*/
class FBSDK_DLL FBTriggerManager :public FBComponent 
{
	__FBClassDeclare( FBTriggerManager, FBComponent );
  public:
	/**	Constructor.
	*/
	FBTriggerManager();

	/**	Get the number of groups.
	*	\return Group count.
	*/
	int GetGroupCount();

	/**	Get the name of a trigger group.
	*	\param	pIndex	Index of group to get name for.
	*	\return	Name of group.
	*/
	char* GetGroupName( int pIndex );

	/**	Get the active status of a trigger group.
	*	\param	pIndex	Index of group to get status for.
	*	\return	Active status of group.
	*/
	bool GetGroupActive( int pIndex );

	/**	Disable or enable a trigger group.
	*	\param	pIndex	Index of trigger group.
	*	\param	pState	New state for group.
	*/
	void SetGroupActive( int pIndex, bool pState );

	FBPropertyBool		Active;		//!< <b>Read Write Property:</b> Active status of engine.
};

////////////////////////////////////////////////////////////////////////////////////
// FBBatchOptions
////////////////////////////////////////////////////////////////////////////////////
FB_FORWARD( FBBatchOptions );

//! Different process type for the batch.
enum FBBatchProcessType
{
	kFBBatchProcessTypeLoad,	//!< Load the files and plot the character with every take.
	kFBBatchProcessTypeSave,	//!< Save the takes in different files.
	kFBBatchProcessTypeConvert	//!< Does the load and save.
};
FB_DEFINE_ENUM( FBSDK_DLL, BatchProcessType );

//! Different file formats for the batch.
enum FBBatchFileFormat
{
	kFBBatchFileFormatTRC,	    //!< File format for Motion Analysis TRC.
	kFBBatchFileFormatC3D,		//!< File format for Vicon C3D.
	kFBBatchFileFormatAMC,		//!< File format for Acclaim AMC.
	kFBBatchFileFormatBVH,		//!< File format for Biovision BVH.
	kFBBatchFileFormatHTR,		//!< File format for Motion Analysis HTR.
	kFBBatchFileFormatFBX		//!< File format for Kaydara FBX (animation only).
};
FB_DEFINE_ENUM( FBSDK_DLL, BatchFileFormat );

//! Different actions to perform when a take already exist while in a batch process.
enum FBBatchOnTakeExist
{
	kFBBatchOnTakeExistOverwrite,	//!< Overwrite the take.
	kFBBatchOnTakeExistSkip			//!< Skip the take.
};
FB_DEFINE_ENUM( FBSDK_DLL, BatchOnTakeExist );

//! Different actions to perform when a scene already contains batch takes while in a batch process.
enum FBBatchOnContainsBatchTakes
{
	kFBBatchOnContainsBatchTakesSaveBatchTakesOnly,	//!< Save only the batch takes.
	kFBBatchOnContainsBatchTakesSaveAllTakes		//!< Save all the takes.
};
FB_DEFINE_ENUM( FBSDK_DLL, BatchOnContainsBatchTakes );

//! Different return values of the Batch process.
enum FBBatchStatus
{
	kFBBatchStatusSuccess,
	kFBBatchStatusError,
	kFBBatchStatusCharacterNotSpecified,
	kFBBatchStatusCharacterNotCharacterized,
	kFBBatchStatusCharacterHasNoReference,
	kFBBatchStatusInputActorNotSpecified,
	kFBBatchStatusActorInputMarkersetNotSpecified,
	kFBBatchStatusActorInputMarkersetHasNoReferenceModel,
	kFBBatchStatusActorInputMarkersetNotCorrectlyAssociated,
	kFBBatchStatusInputCharacterNotCharacterized,
	kFBBatchStatusInputCharacterHasNoReference,
	kFBBatchStatusInputDirectoryNotValid,
	kFBBatchStatusAsfSkeletonFileNotSpecified,
	kFBBatchStatusCantOpenAsfSkeletonFile,
	kFBBatchStatusOutputDirectoryNotValid
};

//! Option parameters for the batch process.
class FBSDK_DLL FBBatchOptions
{
  public:

	/** Constructor.
	*/
	FBBatchOptions();

	FBBatchFileFormat		mInputFileFormat;			//!< File format of the input files.	
	FBBatchFileFormat		mOutputFileFormat;			//!< File format of the output files.
														
	FBBatchProcessType		mProcessType;				//!< What process should be done? Load, Save or Both.
														
	FBString				mInputDirectory;            //!< The directory containning the input files.
	FBString				mOutputDirectory;           //!< The directory containning the output files.
	FBString				mSkeletonFile;              //!< The Skeleton file (for Acclaim AMC files).

#if !defined(K_NO_CHARACTER)
	HFBCharacter			mCharacter;					//!< The character to receive the animation.
#endif
														
	bool					mStartAnimationAtZero;		//!< Set the time of all loaded files to 0.
	bool					mFrameAnimation;			//!< Set timeline start and end time to corespond with the start and end of animation.
	bool					mOverwriteScaling;			//!< Set the scaling to a default setting of 1.0.
	bool					mKeepDummyBones;			//!< To keep dummy bones.
	bool					mWriteRate;					//!< Write frame rate in Acclaim AMC files.
	bool					mWriteTranslation;			//!< Write translation animation data included with Acclaim AMC files.
	bool					mPlotToCharacter;			//!< To plot the animation on the character.
	bool					mPlotToControlSet;			//!< To plot the animation on the control set.
	bool					mUseSingleTake;				//!< Use only one take to convert all files.
	bool					mUseBatchSuffix;			//!< Add a batch suffix to the name of the files.
	bool					mKeepCharacterConstraint;	//!< To keep the character constaint when saving.

	FBBatchOnTakeExist			mOnTakeExistAction;				//!< Action to perform when a take already exist while in a batch process.
	FBBatchOnContainsBatchTakes	mOnContainsBatchTakesAction;	//!< Action to perform when a scene already contains batch takes while in a batch process.
};


////////////////////////////////////////////////////////////////////////////////////
// FBPlotOptions
////////////////////////////////////////////////////////////////////////////////////
FB_FORWARD( FBPlotOptions );

//! Rotation filters
enum FBRotationFilter
{
	kFBRotationFilterNone,
	kFBRotationFilterGimbleKiller,
	kFBRotationFilterUnroll
};
FB_DEFINE_ENUM( FBSDK_DLL, RotationFilter );

//! Option parameters for plotting
class FBSDK_DLL FBPlotOptions
{
  public:

	/** Constructor.
	*/
	FBPlotOptions();

	bool				mPlotAllTakes;					//!< Should we plot all takes?
	bool				mPlotOnFrame;					//!< Should we plot on frame?
	FBTime				mPlotPeriod;					//!< The plot period (1/fps).
	FBRotationFilter	mRotationFilterToApply;			//!< The rotation filter to apply.
	bool				mUseConstantKeyReducer;			//!< Should we use a constant key reducer with the filter?
	bool				mConstantKeyReducerKeepOneKey;	//!< Should the constant key reducer keep at least one key?
	bool				mPlotTranslationOnRootOnly;		//!< Should we plot the translation on root only?
	bool				mPreciseTimeDiscontinuities;	//!< Should we use precise time dicontinuities?
};




////////////////////////////////////////////////////////////////////////////////////
// FBApplication
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBApplication );
__FB_FORWARD( FBCharacter );
__FB_FORWARD( FBActor );
FB_DEFINE_COMPONENT( FBSDK_DLL, Character );
FB_DEFINE_COMPONENT( FBSDK_DLL, Actor );

//! Different viewer modes for the 3D viewer.
enum FBViewerMode {	
	kFBViewerModeOneWindow,		//!< View one pane.
	kFBViewerModeTwoWindow,		//!< View two panes.
	kFBViewerModeThreeWindow,	//!< View three panes.
	kFBViewerModeFourWindow,	//!< View four panes.
	kFBViewerModeSchematic		//!< Schematic view.
};

/** Application class.
*	Permits the manipulation of the application.
*   It should be noted that the Event registration is instanced based. When a FBApplication object is destroyed, all the
*   event callbacks are unregistered. If you want to have a tool to be notified of events, it needs to have a FBApplication
*   data member.
*   Please note that the properties CurrentActor and CurrentCharacter cannot be non NULL at the same time. The character
*   tool will work on only one item at a time. When no item is selected in the tool, both properties will be null.
*/
class FBSDK_DLL FBApplication : public FBComponent 
{
	__FBClassDeclare( FBApplication,FBComponent );
public:
	/**	Constructor.
	*	\param	pObject	Internal parent object(default=NULL).
	*/
	FBApplication(HIObject pObject=NULL);

	FBPropertyEvent	OnFileNewCompleted;		//!< <b>Event:</b> A File New has been completed.
	FBPropertyEvent	OnFileNew;				//!< <b>Event:</b> A File New has been requested, nothing has been destroyed yet.
	FBPropertyEvent	OnFileOpenCompleted;	//!< <b>Event:</b> A File Open has been completed.
	FBPropertyEvent	OnFileOpen;				//!< <b>Event:</b> A File Open has been requested, nothing has been loaded yet.
	FBPropertyEvent	OnFileSaveCompleted;	//!< <b>Event:</b> A File Save has been completed.
	FBPropertyEvent	OnFileSave;				//!< <b>Event:</b> A File Save has been requested, nothing has been saved yet.
	FBPropertyEvent	OnFileExit;				//!< <b>Event:</b> A File Exit as been requested, nothing has been destroyed yet.

	/**	Minimize window.
	*	\param	pBlocking	Is the minimization blocking operation (default =\b true).
	*	\return	Operation was successful (\b true or \b false).
	*/
	bool Minimize(bool pBlocking=true);

	/**	Maximize window (minimized).
	*	\return Operation was successful (\b true or \b false).
	*/
	bool Maximize();

	/**	Command FILE->NEW in the menus.
	*	\return	\b true if successful.
	*/
	bool FileNew();

	/**	Open a file, replacing the current scene.
	*	Command File->Open in the menus.
	*	\param	pFilename	File to open.
	*	\return	\b true if successful.
	*/
	bool FileOpen( char* pFilename );

	/**	Merge a file with the current scene.
	*	Command File->Merge in the menus.
	*	\param	pFilename	File to merge.
	*	\return	\b true if successful.
	*/
	bool FileMerge( char* pFilename );

	/** Save the file under another name.
	*	Command File->SaveAs in the menus.
	*	\param	pFilename	Save file as \b pFilename. A value of NULL will use the current file name.
	*	\return	\b true if successful.
	*/
	bool FileSave( char* pFilename = NULL );

	/**	Quit application.
	*	Command FILE->EXIT in the menus.
	*	\param	pSave	\b true if file is saved on exit(default=false).
	*/
	void FileExit( bool pSave = false );

	/**	Import a motion file.
	*	Command FILE->IMPORT in the menus.
	*	\param	pFilename		The file to import.
	*	\param	pTake			The take in which the motion should be loaded. Optionnal, a take is created if not specified.
	*	\param	pMatchModels	If there is already a model in the scene with the same name, the model will not be created and we replace the animation of the given model. If there are models selected in the scene, only these models will be checked for a potential name match. If only one model is selected (ex: hips), this models and its hierarchy will be used. Every unmatched models will be created.
	*	\return					True if the import succeeded.
	*	\remark					The last two parameters are only used for motion files.
	*	\remark					For now, you cannot import custom file types.
	*	\remark					Currently, only the default import options are used.
	*	\warning				The signature of this function might change in the future to support import options.
	*/
	bool FileImport( FBString pFilename, HFBTake pTake = NULL, bool pMatchModels = false );

	/**	Export a motion file.
	*	Command FILE->EXPORT in the menus.
	*	\param	pFilename		The file to create.
	*	\param	pTake			The take from which the motion should be exported.
	*	\return					True if the export succeeded.
	*	\remark					If the file exists, it will be overwritten.
	*	\remark					The last parameter is only used for motion files.
	*	\remark					For now, you cannot export custom file types.
	*	\remark					Currently, only the default export options are used.
	*	\warning				The signature of this function might change in the future to support export options.
	*/
	bool FileExport( FBString pFilename, HFBTake pTake );

	/**	Start a batch.
	*	Command FILE->BATCH in the menus.
	*	\param	pBatchOptions	The options for the batch process (same as in the batch UI).
	*	\param	pPlotOptions	The options for plotting (same as in the plot UI)(default=NULL).
	*	\return					The status of the operation.
	*/
	FBBatchStatus FileBatch( HFBBatchOptions pBatchOptions, HFBPlotOptions pPlotOptions = NULL );

	/** Render current scene to media file.
	*	Command FILE->RENDER in the menus.
	*	\param	pRenderOptions	The options used when rendering the scene. If you don't specify them, current one are used.
	*	\remark					Render options can be changed if they are not valid.
	*	\warning				If the destination media file exist, it will be overwritten by default.
	*/
	void FileRender( HFBVideoGrabOptions pRenderOptions = NULL );

	/**	Execute a python script file.
	*	\param	pFilename		The script file to execute.
	*	\return					True if the script file was found and executed.
	*	\remark					This function can only be used in the UI thread.
	*/
	bool ExecuteScript(FBString pFilename);

	/**	Switch the current viewer's camera.
	*	\param	pCamera	Camera to switch current viewer to.
	*/
	void SwitchViewerCamera( FBCamera &pCamera );

public:
    FBPropertyString        FBXFileName;            //!< <b>Read Write Property:</b> Current scene filename.
    FBPropertyActor         CurrentActor;           //!< <b>Read Write Property:</b> Indicate the current actor, as used by the character tool. Can be NULL.
    FBPropertyCharacter     CurrentCharacter;       //!< <b>Read Write Property:</b> Indicate the current character, as used by the character tool. Can be NULL.

#if !defined(K_NO_MANIPULATOR)
	FBPropertyListManipulator	Manipulators;	//!< <b>List:</b> of manipulators.
#endif
};


////////////////////////////////////////////////////////////////////////////////////
// FBPlayerControl
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBPlayerControl );

//! Transport modes.
enum FBTransportMode {	
	kFBTransportPlay,			/**!< Play mode		*/	kFBTransportPlayPrepare,		kFBTransportPlayReady,
	kFBTransportStop,			/**!< Stop mode		*/	kFBTransportStopPost,			kFBTransportStopReady,
	kFBTransportShuttle,		/**!< Shuttle mode	*/	kFBTransportShuttlePrepare,		kFBTransportShuttleReady,
	kFBTransportPlayReverse,	/**!< Play reverse.	*/	kFBTransportPlayReversePrepare,	kFBTransportPlayReverseReady,
	kFBTransportJog,			/**!< Jog.			*/	kFBTransportJogPrepare,			kFBTransportJogReady,
	kFBTransportGoto,			/**!< Goto.			*/	kFBTransportGotoPrepare,		kFBTransportGotoReady,
	kFBTransportStepForward,	/**!< Step forward	*/	kFBTransportStepForwardPrepare,	kFBTransportStepForwardReady,
	kFBTransportStepBackward,	/**!< Step backward.*/	kFBTransportStepBackwardPrepare,kFBTransportStepBackwardReady
};
FB_DEFINE_ENUM( FBSDK_DLL, TransportMode );

//! Available transport control play speed.
enum FBTransportPlaySpeed
{
    kFBSpeed_1_10x, //!< 1/10x.
    kFBSpeed_1_5x,  //!< 1/5x
    kFBSpeed_1_4x,  //!< 1/4x
    kFBSpeed_1_3x,  //!< 1/3x
    kFBSpeed_1_2x,  //!< 1/2x
    kFBSpeed_1x,    //!< 1x
    kFBSpeed_ALL_FR, //!< All frames
    kFBSpeed_2x,    //!< 2x
    kFBSpeed_3x,    //!< 3x
    kFBSpeed_4x,    //!< 4x
    kFBSpeed_5x,    //!< 5x
    kFBSpeed_10x    //!< 10x
};
FB_DEFINE_ENUM( FBSDK_DLL, TransportPlaySpeed);

//! Available snap methods for the transport control.
enum FBTransportSnapMode
{
    kFBTransportSnapModeNoSnap,              //!< No snapping is applied.
    kFBTransportSnapModeSnapOnFrames,        //!< Snaps to an exact frame when modifying the current time.
    kFBTransportSnapModePlayOnFrames,        //!< When playing, plays to exact frames.
    kFBTransportSnapModeSnapAndPlayOnFrames, //!< Combines both Snap and Play on frames modes.
};
FB_DEFINE_ENUM( FBSDK_DLL, TransportSnapMode );


/**	Player control.
*	Interface to use the transport controls.
*/
class FBSDK_DLL FBPlayerControl : public FBComponent {
	__FBClassDeclare( FBPlayerControl,FBComponent );
public:

	/**	Constructor.
	*	\param	pObject		Internal parent object(default=NULL).
	*/
	FBPlayerControl(HIObject pObject=NULL);

	/**	Play button.
	*	\param	pUseMarkers	Play until next marker if \b true, ignore markers otherwise.
	*	\return	\b true if successful.
	*/
	bool Play(bool pUseMarkers = false);

	/**	Play Reverse button.
	*	\param	pUseMarkers	Play until next marker if \b true, ignore markers otherwise.
	*	\return	\b true if successful.
	*/
	bool PlayReverse(bool pUseMarkers = false);

    /**	Set Play Speed .
	*	\param	pPlaySpeed	indicate the play speed when a play command occur.
	*/
    void SetPlaySpeed (FBTransportPlaySpeed pPlaySpeed );

    /**	Get Play Speed .
    *	\return	\b transport current playback speed.
	*/
    FBTransportPlaySpeed GetPlaySpeed();

	/**	Stop button.
	*	\return	\b true if successful.
	*/
	bool Stop();

	/**	GotoStart button (Rewind).
	*	\return	\b true if successful.
	*/
	bool GotoStart();

	/**	GotoEnd button (FastForward).
	*	\return	\b true if successful.
	*/
	bool GotoEnd();

	/**	Goto a time specified by \e pTime.
	*	\param	pTime	Time to jump to.
	*	\return	\b true if successful.
	*/
	bool Goto( FBTime pTime );

	/**	Step one frame ahead.
	*	\return	\b true if successful.
	*/
	bool StepForward();

	/**	Step one frame backward.
	*	\return	\b true if successful.
	*/
	bool StepBackward();

	/**	Begin recording.
	*	\param	pOverrideTake	Write over current take?(default=false)
	*	\param	pCopyData		Unused. Necessary for compatibility(default=true).
	*	\return	\b true if successful.
	*/
	bool Record( bool pOverrideTake=false,bool pCopyData=true );

	/** Get Transport Mode.
	*	\return Current mode of the transport controls.
	*/
	FBTransportMode	GetTransportMode();
	
    /**	Set the system frame rate use for display.
	*	\param	pTimeMode	Indicate the frame rate value to use base on the FBTimeMode values enum.
    *   \param  pCustom     Should the time mode be kFBTimeModeCustom, this is used to specify the custom framerate.
	*/
    void       SetTransportFps ( FBTimeMode pTimeMode, double pCustom=0.0);

    /**	Get the UI frame rate use for display configure in the system
	*	\return current FrameRate selected for the system.
    */
    FBTimeMode GetTransportFps ();

    /**	Get the UI frame rate value
	*	\return current FrameRate value based on the selected FBTimeMode for the system.
    */
    double     GetTransportFpsValue ();
    
	/** Key default data.
	*	Key all selected data.
	*/
	void Key();

	/** Go to the next key.
	*/
	void GotoNextKey();

	/** Go to the previous key.
	*/
	void GotoPreviousKey();

	FBPropertyBool	LoopActive;		//!< <b>Read Write Property:</b> Is looping active?
	FBPropertyTime	LoopStart;		//!< <b>Read Write Property:</b> Loop begin time.
	FBPropertyTime	LoopStop;		//!< <b>Read Write Property:</b> Loop end time.

    FBPropertyTime	ZoomWindowStart;	//!< <b>Read Write Property:</b> Starting time of the transport control zoom window.
	FBPropertyTime	ZoomWindowStop;		//!< <b>Read Write Property:</b> Stopping time of the transport control zoom window.

	FBPropertyTime	NextMarker;		//!< <b>Read Only Property:</b> Next marked time.
	FBPropertyTime	PreviousMarker;	//!< <b>Read Only Property:</b> Previous marked time.

	FBPropertyBool	IsRecording;	//!< <b>Read Only Property:</b> Is there a recording in progress?
	FBPropertyBool	IsPlaying;		//!< <b>Read Only Property:</b> Is the transport control playing?

    FBPropertyTransportSnapMode SnapMode;   //!< <b>Read Write Property:</b> Set the transport control snap mode.
};

////////////////////////////////////////////////////////////////////////////////////
// FBKeyControl
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBKeyControl );

/**	Key control.
*	Interface to use the key controls tool.
*/
class FBSDK_DLL FBKeyControl : public FBComponent
{
	__FBClassDeclare(FBKeyControl, FBComponent);

public:
	/**	Constructor.
	*	\param	pObject		Internal parent object(default=NULL).
	*/
	FBKeyControl(HIObject pObject=NULL);

	FBPropertyBool	AutoKey;	//!< <b>Read Write Property:</b> Enable/Disable Auto Key feature (key when moving 3D objects).
};

////////////////////////////////////////////////////////////////////////////////////
// FBCriticalSection
////////////////////////////////////////////////////////////////////////////////////
FB_FORWARD( FBCriticalSection );

/**	High priority critical section class.
*	This class permits the user to operate critical operations, freezing the system for 
*	a high priority thread.
*/
class FBSDK_DLL FBCriticalSection
{
	void *mPtr;					// \internal Internal data.
public:
	//! Constructor.
	FBCriticalSection();

	//! Destructor.
	~FBCriticalSection();

	/**	Initialize critical section operations.
	*	\return	\b true if successful.
	*/
	bool Init();

	//! Enter section.
	void Enter();

	//! Leave section.
	void Leave();
};

////////////////////////////////////////////////////////////////////////////////////
// FBReferenceTime
////////////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBReferenceTime );

//! Reference time class.
class FBSDK_DLL FBReferenceTime : public FBComponent {
	__FBClassDeclare( FBReferenceTime,FBComponent );
public:
	/**	Constructor.
	*	\param	pObject	For internal use only(default=NULL).
	*/
	FBReferenceTime(HIObject pObject=NULL);

	/**	Add a reference time to list.
	*	\param	pName	Name of time to add the list.
	*	\return Number of reference times after operation.
	*/
	int    Add( char *pName );

	/**	Remove a reference time from the list.
	*	\param	pIndex	Index of reference time to remove.
	*/
	void   Remove( int pIndex );

	/**	Set a reference time.
	*	\param	pIndex			Index of reference time set.
	*	\param	pReferenceTime	Time to use as reference time.
	*	\param	pSystem			System time.
	*/
	void   SetTime( int pIndex, FBTime pReferenceTime, FBTime pSystem );
	/**	Get a reference time.
	*	\param	pIndex		Index of reference to get.
	*	\param	pSystem		System time.
	*	\return	Reference time at \e pIndex.
	*/
	FBTime GetTime( int pIndex, FBTime pSystem );

	/**	Overloaded [] operator
	*	\param	pIndex
	*	\return	Name of reference time at \e pIndex.
	*/
	char *operator[](int pIndex);

	FBPropertyInt				Count;			//!< <b>Read Only Property:</b> Number of reference times.
	FBPropertyInt				ItemIndex;		//!< <b>Read Write Property:</b> Current reference time index.
};

////////////////////////////////////////////////////////////////////////////////////
// Global utility functions
////////////////////////////////////////////////////////////////////////////////////


/**	Create a FBModelList object.
*	This is to get around memory management issues when mixing debug and release runtime
*   libraties on Windows. By using the FBCreateModelList/FBDestroyModelList
*   duo of functions, you ensure that the memory used in the array is allocated,
*   reallocated and freed in the same memory pool.
*/
FBSDK_DLL FBModelList* FBCreateModelList();

/** Delete a FBModelList object.
*   \param  pModelList  pointer to the object created with a call to FBCreateModelList().
*/
FBSDK_DLL void FBDestroyModelList( FBModelList* pModelList );

/**	Create a FBPickInfosList object.
*	This is to get around memory management issues when mixing debug and release runtime
*   libraties on Windows. By using the FBCreatePickInfosList/FBDestroyPickInfosList
*   duo of functions, you ensure that the memory used in the array is allocated,
*   reallocated and freed in the same memory pool.
*/
FBSDK_DLL FBPickInfosList* FBCreatePickInfosList();

/** Delete a FBPickInfosList object.
*   \param  pPickInfosList  pointer to the object created with a call to FBCreatePickInfosList().
*/
FBSDK_DLL void FBDestroyPickInfosList( FBPickInfosList* pPickInfosList );

/**	Sleep function
*	Puts system to sleep for specified time.
*	\param	MilliSeconds	Time to sleep for.
*/
FBSDK_DLL void FBSleep( unsigned long MilliSeconds );

/**	Find a model in the hierarchy.
*	Searches the scene for a model, based on the model's 
*	unique name.
*	\param	pModelName		Name of model to search for.
*	\param	pParent			Root model to search from (default=NULL(root)).
*	\return	A handle onto the model that was found, returns 
*	NULL if no model was found by the search.
*/
FBSDK_DLL HFBModel FBFindModelByName( char* pModelName, HFBModel pParent=NULL  );

/**	Find all models of a certain type in the scene.
*	Searches recursively from a root model for models of a certain type,
*	and adds them to a list of models.
*	\retval	pList		List to add found models to.
*	\param	pTypeInfo	Model type to look for.
*	\param	pParent		Root model to look from (default=NULL(root)).
*/
FBSDK_DLL void FBFindModelsOfType( FBModelList& pList, int pTypeInfo, HFBModel pParent=NULL );

/**	Find all models that are selected (if \b pSelected is true)
*	Searches recursively from a root model for models that are selected,
*	and adds them to a list of models.
*	\retval	pList		List to add found models to.
*	\param	pParent		Root model to look from (default=NULL(root)).
*	\param	pSelected	\b true to find selected models, \b false to find unselected models(default=true).
*/
FBSDK_DLL void FBGetSelectedModels( FBModelList& pList, HFBModel pParent=NULL, bool pSelected=true );

/** FBSelectObjectsByNamespace.
*   This function will select objects in the current scene according to the parameters you provide
*	\param	pNameSpaceName	        Indicate the name to search for selection/deselection.
*	\param	pSelect                 Indicate if we select or unselect the objects
*	\param	pSearchExclusive        "true" will select object that are exclusive to that namespace(pNameSpaceName), "false" will select object that could be part of that namespace as well as others namespace.
*/
FBSDK_DLL void FBSelectObjectsByNamespace ( char* pNameSpaceName, bool pSelect, bool pSearchExclusive = true );

/**	This function prints useful debugging strins in the console.
*   On Windows, setting the environment variable 'FILMBOX_CONSOLE' to '1', it is possible to
*   print formatted messages, as a printf would. On Mac OSX, the strings are simply sent to stderr.
*   \param pFormatString    A printf-style format string, to use the following arguments in the list.
*   \warning    There is currently a limitation which sets the maximum length of the resulting
*               string to be limited to 2048 bytes.
*   \warning    Not thread safe, as an static array is used internally.
*/
FBSDK_DLL void FBTrace( char* pFormatString, ... );

#if !defined(K_NO_UI)
/**	This function is used to bring up a specific tool in the GUI.
*   \param pToolName    The name of the tool as shown in the Open Reality menu.
*   \param pSetFocus    Indicate if the tool will have the focus.
*   \return             If the tool was brought up successfully.
*/
FBSDK_DLL bool FBPopNormalTool( char* pToolName, bool pSetFocus = true );

#endif

#ifdef FBSDKUseNamespace
	}
#endif
#endif

