//***************************************************************************************
//
// File supervisor: Softimage 3D Games & 3D Bridge team
//
// (c) Copyright 1999-2005 Avid Technology, Inc. . All rights reserved.
//
//***************************************************************************************

//****************************************************************************************
/*!
   \file DX10View_API.h
   \brief Defines the DirectX 10 View Mode API

THIS CODE IS PUBLISHED AS A SAMPLE ONLY AND IS PROVIDED "AS IS".
IN NO EVENT SHALL SOFTIMAGE, AVID TECHNOLOGY, INC. AND/OR THEIR RESPECTIVE
SUPPLIERS BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY
DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
CONNECTION WITH THE USE OR PERFORMANCE OF THIS CODE .

COPYRIGHT NOTICE. Copyright  2006-2006 Avid Technology Inc. . All rights reserved.

SOFTIMAGE is a registered trademark of Avid Technology Inc. or its subsidiaries
or divisions. Windows NT is a registered trademark of Microsoft Corp. All other
trademarks contained herein are the property of their respective owners.
*/
//****************************************************************************************/

#ifndef DIRECTX10_VIEW_H
#define DIRECTX10_VIEW_H


#include <d3d10.h>
#include <d3dx10.h>
#include <d3dx10tex.h>
#include <SIBCArray.h>
#include <iostream>
#include <fstream>

struct SimpleVertex
{
    D3DXVECTOR3 Pos;
};

// this is the currently used structure, we could probably
// update it one day to be more flexible
typedef struct tagDX10View_PhatVertex
{
	XSI_RTS_Vector3			m_Pos;
	XSI_RTS_Vector3			m_Normal;
	XSI_RTS_ColorRGBAf		m_Col[2];
	XSI_RTS_Vector4			m_TC[8];
} DX10View_PhatVertex;

#define PhatVertexLayoutNumElems 12
static D3D10_INPUT_ELEMENT_DESC PhatVertexLayout[PhatVertexLayoutNumElems] =
{
    { "POSITION", 0, DXGI_FORMAT_R32G32B32_FLOAT, 0,	(12*0) + (16*0), D3D10_INPUT_PER_VERTEX_DATA, 0 },
    { "NORMAL", 0, DXGI_FORMAT_R32G32B32_FLOAT, 0,		(12*1) + (16*0), D3D10_INPUT_PER_VERTEX_DATA, 0 },
    { "COLOR", 0, DXGI_FORMAT_R32G32B32A32_FLOAT, 0,		(12*2) + (16*0), D3D10_INPUT_PER_VERTEX_DATA, 0 },
    { "COLOR", 1, DXGI_FORMAT_R32G32B32A32_FLOAT, 0,		(12*2) + (16*1), D3D10_INPUT_PER_VERTEX_DATA, 0 },
    { "TEXCOORD", 0, DXGI_FORMAT_R32G32B32A32_FLOAT, 0,	(12*2) + (16*2), D3D10_INPUT_PER_VERTEX_DATA, 0 },
    { "TEXCOORD", 1, DXGI_FORMAT_R32G32B32A32_FLOAT, 0,	(12*2) + (16*3), D3D10_INPUT_PER_VERTEX_DATA, 0 },
    { "TEXCOORD", 2, DXGI_FORMAT_R32G32B32A32_FLOAT, 0,	(12*2) + (16*4), D3D10_INPUT_PER_VERTEX_DATA, 0 },
    { "TEXCOORD", 3, DXGI_FORMAT_R32G32B32A32_FLOAT, 0,	(12*2) + (16*5), D3D10_INPUT_PER_VERTEX_DATA, 0 },
    { "TEXCOORD", 4, DXGI_FORMAT_R32G32B32A32_FLOAT, 0,	(12*2) + (16*6), D3D10_INPUT_PER_VERTEX_DATA, 0 },
    { "TEXCOORD", 5, DXGI_FORMAT_R32G32B32A32_FLOAT, 0,	(12*2) + (16*7), D3D10_INPUT_PER_VERTEX_DATA, 0 },
    { "TEXCOORD", 6, DXGI_FORMAT_R32G32B32A32_FLOAT, 0,	(12*2) + (16*8), D3D10_INPUT_PER_VERTEX_DATA, 0 },
    { "TEXCOORD", 7, DXGI_FORMAT_R32G32B32A32_FLOAT, 0,	(12*2) + (16*9), D3D10_INPUT_PER_VERTEX_DATA, 0 }
};

typedef struct tagDX10View_TexCoordMap
{
	int		nbActiveUVs;
	int		target[256];
	int		UVSet[256];
} DX10View_TexCoordMap;


#define NB_VTX_BUFFERS	8
#define VTX_BUFFER_SIZE	49152


/*! \struct tagDX10ViewGlobals DX10View_API.h
	\brief The DX10ViewGlobals structure.

	This structure allows you to render to a DirectX 10 device from the Realtime Shaders API.
	You access it by calling the \ref DX10View_GetContext "DX10View_GetContext"  callback
	function, which returns this structure.

	\sa \ref DX10View_Callbacks, \xt cus_rtshad_DirectXUtilityFunctions DirectX Utility Functions \endxt
	\since 7.0
*/
typedef struct tagDX10ViewGlobals
{
	//****************************************************************************************
	// 	Windows
	//****************************************************************************************
																	/*! The window handle 			*/
	HWND							m_hwnd;							/*! The parent window handle	*/
	HWND							m_ParentHwnd;					/*! The parent window rect		*/
	RECT							m_ParentRect;					/*! The window class			*/
	WNDCLASSEX						m_wc;							/*! HINSTANCE handle			*/
	HINSTANCE						m_hInst;

	//****************************************************************************************
	// 	Core and display buffers
	//****************************************************************************************
																	/*! Pointer to the global \c ID3D10Device device for XSI. Resources should be managed through that single device. */
	ID3D10Device*           		m_pd3dDevice;					/*! Pointer to the current view's swap chain */
	IDXGISwapChain*         		m_pSwapChain;					/*! Pointer to the current view's primary render target */
	ID3D10RenderTargetView* 		m_pRenderTargetView;			/*! Pointer to the current view's primary depth/stencil buffer */
	ID3D10DepthStencilView*			m_pDSView;						/*! Provides a global \c IDXGIFactory object you can use to create \c DXGI objects */
	IDXGIFactory*					m_pFactory;

	//****************************************************************************************
	// 	Grid
	//****************************************************************************************
																	/*! Default effect (Grid)			*/
	ID3D10Effect*           		m_pEffect;						/*! Default technique (Grid)		*/
	ID3D10EffectTechnique*  		m_pTechnique;					/*! Default vertex layout (Grid)	*/
	ID3D10InputLayout*      		m_pGridVertexLayout;			/*! Default vertex buffer (Grid)	*/
	ID3D10Buffer*           		m_pGridVertexBuffer;

	//****************************************************************************************
	// 	Font
	//****************************************************************************************
																	/*! Display font		*/
	ID3DX10Font*					m_pFont;						/*! Display sprite		*/
	ID3DX10Sprite*					m_pSprite;						/*! Font blend state	*/
	ID3D10BlendState*				m_pFontBlendState;

	//****************************************************************************************
	//	States and buffers
	//****************************************************************************************
																	/*! Default blend state			*/
	ID3D10BlendState*				m_pDefaultBlendState;			/*! Default rasterizer state	*/
	ID3D10RasterizerState*			m_pDefaultRasterizerState;		/*! Current projection matrix	*/
	ID3D10EffectMatrixVariable*		m_pProjectionMatrix;			/*! Current model matrix		*/
    ID3D10EffectMatrixVariable*		m_pModelMatrix;					/*! Current view matrix			*/
    ID3D10EffectMatrixVariable*		m_pViewMatrix;

	//****************************************************************************************
	//	\par Current states
	//****************************************************************************************
																			/*! The current GS or VS input signature		*/
	BYTE*							m_pIACurrentInputSignature;				/*! The current GS or VS input signature size	*/
	SIZE_T							m_nIACurrentInputSignatureSize;			/*! The current vertex buffers					*/
	ID3D10Buffer*					m_pCurrentVertexBuffer[NB_VTX_BUFFERS];	/*! The current vertex buffers size				*/
	int								m_nCurrentVertexBufferSize;				/*! The current vertex buffer id				*/
	int								m_nCurrentVertexBufferId;				/*! The current GS reflection					*/
	ID3D10ShaderReflection*			m_pCurrentGS;							/*! The current VS reflection					*/
	ID3D10ShaderReflection*			m_pCurrentVS;							/*! The current PS reflection					*/
	ID3D10ShaderReflection*			m_pCurrentPS;

	//****************************************************************************************
	//	RTS API Stuff
	//****************************************************************************************
																	/*! Texture coordinate maps		*/
	DX10View_TexCoordMap			m_TexCoordMap;					/*! Semantics data				*/
	XSI_RTS_FXSemantics				m_Semantics;

} DX10ViewGlobals;

//****************************************************************************************
// DX10View CustomView API
//****************************************************************************************
/*! \page DX10View_Callbacks DirectX10 Callbacks
	The following callbacks are available in the DX10View CustomView API:

	\section Context Management
	<TABLE BORDER="0">
	<TR><TD>&nbsp;</TD>		<TD>\ref DX10View_GetContext "DX10View_GetContext"</TD> 				<TD>Gets the context (tagDX10ViewGlobals data structure)</TD></TR>
	</TABLE>

	\section Context Initialization and Cleanup
	<TABLE BORDER="0">
	<TR><TD>&nbsp;</TD>		<TD>\ref DX10View_IsSupported "DX10View_IsSupported"</TD> 				<TD>Returns whether or not DirectX 10 is supported on the current machine</TD></TR>
	<TR><TD>&nbsp;</TD>		<TD>\ref DX10View_Initialize "DX10View_Initialize"</TD> 				<TD>Initializes the global DirectX 10 View Mode API library</TD></TR>
	<TR><TD>&nbsp;</TD>		<TD>\ref DX10View_InitializeWindow "DX10View_InitializeWindow"</TD>		<TD>Creates and initializes a DirectX 10 swap chain, a render target, and depth/stencil view bound to the window handle parameter</TD></TR>
	<TR><TD>&nbsp;</TD>		<TD>\ref DX10View_InitViewport "DX10View_InitViewport"</TD> 			<TD>Helper for \ref DX10View_InitViewport "DX10View_InitViewport"</TD></TR>
	<TR><TD>&nbsp;</TD>		<TD>\ref DX10View_Cleanup "DX10View_Cleanup"</TD> 						<TD>Performs a global cleanup of the DirectX 10 View Mode API library</TD></TR>
	</TABLE>

	\section Rendering
	<TABLE BORDER="0">
	<TR><TD>&nbsp;</TD>		<TD>\ref DX10View_BeginRender "DX10View_BeginRender"</TD>				<TD>Sets the current render targets and clears them with the given color</TD></TR>
	<TR><TD>&nbsp;</TD>		<TD>\ref DX10View_SetInitialState "DX10View_SetInitialState"</TD>		<TD>Reset blend state and the rasterizer state to the default states</TD></TR>
	<TR><TD>&nbsp;</TD>		<TD>\ref DX10View_EndRender "DX10View_EndRender"</TD>					<TD>Updates the window with all drawing calls since the last call to \ref DX10View_BeginRender "DX10View_BeginRender"</TD></TR>
	<TR><TD>&nbsp;</TD>		<TD>\ref DX10View_DrawGeometry "DX10View_DrawGeometry"</TD>				<TD>Draws the specified geometry in the DirectX 10 view</TD></TR>
	</TABLE>

	\section Texture Upload
	<TABLE BORDER="0">
	<TR><TD>&nbsp;</TD>		<TD>\ref DX10View_UploadTexture1D "DX10View_UploadTexture1D"</TD>		<TD>Uploads an \c ID3D10Texture1D texture to the DirectX 10 device</TD></TR>
	<TR><TD>&nbsp;</TD>		<TD>\ref DX10View_UploadTexture2D "DX10View_UploadTexture2D"</TD>		<TD>Uploads an \c ID3D10Texture2D to the DirectX 10 device.</TD></TR>
	<TR><TD>&nbsp;</TD>		<TD>\ref DX10View_UploadTexture3D "DX10View_UploadTexture3D"</TD>		<TD>Uploads an \c ID3D10Texture3D to the DirectX 10 device</TD></TR>
	<TR><TD>&nbsp;</TD>		<TD>\ref DX10View_UploadTextureCube "DX10View_UploadTextureCube"</TD>	<TD>Uploads an \c ID3D10Texture2D cube to the DirectX 10 device</TD></TR>
	</TABLE>

	\sa tagDX10ViewGlobals, \xt cus_rtshad_DirectXUtilityFunctions DirectX Utility Functions \endxt
*/
//****************************************************************************************
// Helpers
//****************************************************************************************
#define RELEASE( ptr ) \
	{ if( ptr ) { if(ptr->Release() > 0) assert(0); ptr = NULL; } }

#define SAFE_RELEASE( ptr ) \
	{ if( ptr ) ptr->Release(); }

#define NULLPTR( ptr ) \
	{ ptr = NULL; }

RTSHADER_API void							DX10View_Printf(XSI::siSeverityType in_lErrCode, wchar_t *in_pFormat, ...);

//****************************************************************************************
// Context management
//****************************************************************************************
/*! \page DX10View_GetContext DX10View_GetContext DirectX 10 Callback Reference
	\code
	RTSHADER_API DX10ViewGlobals* DX10View_GetContext();
	\endcode
	Use this callback function to get the DirectX 10 context, which is a pointer to the
	tagDX10ViewGlobals data structure.
	\sa tagDX10ViewGlobals, \ref DX10View_Callbacks
	\since 7.0
*/
RTSHADER_API DX10ViewGlobals*				DX10View_GetContext();

//****************************************************************************************
// Context initialization and cleanup
//****************************************************************************************
/*! \page DX10View_IsSupported DX10View_IsSupported DirectX 10 Callback Reference
	\code
	RTSHADER_API HRESULT DX10View_IsSupported();
	\endcode
	Returns whether or not DirectX 10 is supported on the current machine.
	\sa tagDX10ViewGlobals, \ref DX10View_Callbacks
	\since 7.0
*/
RTSHADER_API HRESULT						DX10View_IsSupported();

/*! \page DX10View_Initialize DX10View_Initialize DirectX 10 Callback Reference
	\code
	RTSHADER_API HRESULT DX10View_Initialize();
	\endcode
	Initializes the global DirectX 10 View Mode API library. It creates a global DirectX 10 device
	(tagDX10ViewGlobals::m_pd3dDevice) and a DXGI factory object (tagDX10ViewGlobals::m_pFactory).
	It loads the default FX file used to display text and the grid, allocates and initializes
	the vertex buffers used for drawing and creates all the other resources that are not
	render-target dependent.
	\sa tagDX10ViewGlobals, \ref DX10View_Callbacks
	\since 7.0
*/
RTSHADER_API HRESULT						DX10View_Initialize();

/*! \page DX10View_InitializeWindow DX10View_InitializeWindow DirectX 10 Callback Reference
	\code
	RTSHADER_API HWND DX10View_InitializeWindow
	(
		HWND 	hWnd
	);
	\endcode
	Creates and initializes a DirectX 10 swap chain (tagDX10ViewGlobals::m_pSwapChain), a
	render target (tagDX10ViewGlobals::m_pRenderTargetView) and depth/stencil view
	(tagDX10ViewGlobals::m_pDSView) that is bound to the window handle parameter (\c hWnd).
	This needs to be called for every window that will be set to DirectX10 mode.
	\param hWnd Handle to the window.
	\sa tagDX10ViewGlobals, \ref DX10View_Callbacks
	\since 7.0
*/
RTSHADER_API HWND						DX10View_InitializeWindow(HWND hWnd);

/*! \page DX10View_InitViewport DX10View_InitViewport DirectX 10 Callback Reference
	\code
	RTSHADER_API HRESULT DX10View_InitViewport
	(
		int 	x,
		int 	y,
		int 	width,
		int 	height
	);
	\endcode
	Instructs \ref DX10View_Initialize "DX10View_Initialize" how to create and initialize
	the render target (tagDX10ViewGlobals::m_pRenderTargetView) using the specified parameters.
	\param x 		X coordinate for viewport location
	\param y 		Y coordinate for viewport location
	\param width 	Width of the viewport
	\param height 	Height of the viewport
	\sa tagDX10ViewGlobals, \ref DX10View_Callbacks
	\since 7.0
*/
RTSHADER_API HRESULT						DX10View_InitViewport(int x, int y, int width, int height);

/*! \page DX10View_Cleanup DX10View_Cleanup DirectX 10 Callback Reference
	\code
	RTSHADER_API void DX10View_Cleanup();
	\endcode
	Performs a global cleanup of the DirectX 10 View Mode API library. The global device and all associated
	resources are released and are no longer valid.
	\sa tagDX10ViewGlobals, \ref DX10View_Callbacks
	\since 7.0
*/
RTSHADER_API void							DX10View_Cleanup();

//****************************************************************************************
// Rendering
//****************************************************************************************

/*! \page DX10View_BeginRender DX10View_BeginRender DirectX 10 Callback Reference
	\code
	RTSHADER_API void DX10View_BeginRender
	(
		float 		red,
		float 		green,
		float 		blue
	);
	\endcode
	Sets the current render targets and clears them with the given color. It also prepares
	the device so that drawing calls can be issued.
	\param red		R value for the new color
	\param green	G value for the new color
	\param blue		B value for the new color
	\sa \ref DX10View_EndRender "DX10View_EndRender", tagDX10ViewGlobals, \ref DX10View_Callbacks
	\since 7.0
*/
RTSHADER_API void							DX10View_BeginRender(float red, float green, float blue);

/*! \page DX10View_SetInitialState DX10View_SetInitialState DirectX 10 Callback Reference
	\code
	RTSHADER_API void DX10View_SetInitialState();
	\endcode
	This function should be called after \ref DX10View_BeginRender "DX10View_BeginRender"
	so that the blend state and the rasterizer state are set to the default states
	(tagDX10ViewGlobals::m_pDefaultBlendState and tagDX10ViewGlobals::m_pDefaultRasterizerState).

	It also intializes the current shaders to NULL.
	\sa tagDX10ViewGlobals, \ref DX10View_Callbacks
	\since 7.0
*/
RTSHADER_API void							DX10View_SetInitialState();

/*! \page DX10View_EndRender DX10View_EndRender DirectX 10 Callback Reference
	\code
	RTSHADER_API void DX10View_EndRender();
	\endcode
	This function should be called when all drawing calls are issued and the view is ready
	to be presented. This swaps the back and front buffers of the current render target such
	that the window is updated with all the drawing calls that occurred since the last call to
	\ref DX10View_BeginRender "DX10View_BeginRender".
	\sa \ref DX10View_BeginRender "DX10View_BeginRender", tagDX10ViewGlobals, \ref DX10View_Callbacks
	\since 7.0
*/
RTSHADER_API void							DX10View_EndRender();

/*	These callbacks are used internally by the DX10 display mode.
	Do not doc.
*/
RTSHADER_API void DX10View_Display(wchar_t *in_pString);
RTSHADER_API void							DX10View_DrawGrid();

/*! \page DX10View_DrawGeometry DX10View_DrawGeometry DirectX 10 Callback Reference
	\code
	RTSHADER_API bool DX10View_DrawGeometry
	(
		XSI_RTS_Primitive				*in_pPrimitive,
		void							*in_pAttributes,
		XSI_RTS_Attribute_Size			*in_pAttributesSize,

		bool							in_bNormals,
		bool							in_bColors,
		bool							in_bUVs,
		DX10View_TexCoordMap*			in_pTexCoordMap
	);
	\endcode
	Draws the specified geometry in the DirectX 10 view.
	\param in_pPrimitive 		The geometry information
	\param in_pAttributes 		The attribute data
	\param in_pAttributesSize	The size of the attribute data
	\param in_bNormals 			Use normals in the drawing call
	\param in_bColors 			Use colors in the drawing call
	\param in_bUVs 				Use UVs in the drawing call
	\param in_pTexCoordMap 		Texture coordinate map
	\sa tagDX10ViewGlobals, \ref DX10View_Callbacks, \ref Execute
	\since 7.0
*/
RTSHADER_API bool							DX10View_DrawGeometry
(
	XSI_RTS_Primitive				*in_pPrimitive,
	void							*in_pAttributes,
	XSI_RTS_Attribute_Size			*in_pAttributesSize,

	bool							in_bNormals,
	bool							in_bColors,
	bool							in_bUVs,
	DX10View_TexCoordMap*			in_pTexCoordMap
);

//****************************************************************************************
// Texture upload
//****************************************************************************************

/*! \page DX10View_UploadTexture1D DX10View_UploadTexture1D DirectX 10 Callback Reference
	\code
	RTSHADER_API HRESULT DX10View_UploadTexture1D
	(
		XSI_RTS_Texture* 				in_pImage,
		ID3D10Texture1D** 				out_pT1d,
		ID3D10ShaderResourceView** 		out_pTView
	);
	\endcode
	Uploads an \c ID3D10Texture1D texture to the DirectX 10 device.
	\param in_pImage	Texture information coming from XSI to be uploaded to DX10
	\retval out_pT1d	1D texture interface to the newly uploaded texture that will be used to manipulate the DX10 texture
	\retval out_pTView	Shader resource interface to the newly uploaded texture that will be used to manipulate the texture as a shader resource
	\sa tagDX10ViewGlobals, \ref DX10View_Callbacks, \ref Execute
	\since 7.0
*/
RTSHADER_API HRESULT	DX10View_UploadTexture1D(XSI_RTS_Texture* in_pImage, ID3D10Texture1D** out_pT1d, ID3D10ShaderResourceView** out_pTView);

/*! \page DX10View_UploadTexture2D DX10View_UploadTexture2D DirectX 10 Callback Reference
	\code
	RTSHADER_API HRESULT DX10View_UploadTexture2D
	(
		XSI_RTS_Texture* 				in_pImage,
		ID3D10Texture2D** 				out_pT2d,
		ID3D10ShaderResourceView** 		out_pTView
	);
	\endcode
	Uploads an \c ID3D10Texture2D to the DirectX 10 device.
	\param in_pImage	Texture information coming from XSI to be uploaded to DX10
	\retval out_pT2d	2D texture interface to the newly uploaded texture that will be used to manipulate the DX10 texture
	\retval out_pTView	Shader resource interface to the newly uploaded texture that will be used to manipulate the texture as a shader resource
	\sa tagDX10ViewGlobals, \ref DX10View_Callbacks, \ref Execute
	\since 7.0
*/
RTSHADER_API HRESULT	DX10View_UploadTexture2D(XSI_RTS_Texture* in_pImage, ID3D10Texture2D** out_pT2d, ID3D10ShaderResourceView** out_pTView);

/*! \page DX10View_UploadTexture3D DX10View_UploadTexture3D DirectX 10 Callback Reference
	\code
	RTSHADER_API HRESULT DX10View_UploadTexture3D
	(
		XSI_RTS_Texture* 				in_pImage,
		ID3D10Texture3D** 				out_pT3d,
		ID3D10ShaderResourceView** 		out_pTView
	);
	\endcode
	Uploads an \c ID3D10Texture3D to the DirectX 10 device.
	\param in_pImage	Texture information coming from XSI to be uploaded to DX10
	\retval out_pT3d	3D texture interface to the newly uploaded texture that will be used to manipulate the DX10 texture
	\retval out_pTView	Shader resource interface to the newly uploaded texture that will be used to manipulate the texture as a shader resource
	\sa tagDX10ViewGlobals, \ref DX10View_Callbacks, \ref Execute
	\since 7.0
*/
RTSHADER_API HRESULT	DX10View_UploadTexture3D(XSI_RTS_Texture* in_pImage, ID3D10Texture3D** out_pT3d, ID3D10ShaderResourceView** out_pTView);

/*! \page DX10View_UploadTextureCube DX10View_UploadTextureCube DirectX 10 Callback Reference
	\code
	RTSHADER_API HRESULT DX10View_UploadTextureCube
	(
		XSI_RTS_Texture* 				in_pImage,
		ID3D10Texture2D** 				out_pTCube,
		ID3D10ShaderResourceView** 		out_pTView
	);
	\endcode
	Uploads an \c ID3D10Texture2D cube to the DirectX 10 device.
	\param in_pImage	Texture information coming from XSI to be uploaded to DX10
	\retval out_pTCube	2D texture interface to the newly uploaded texture that will be used to manipulate the DX10 texture
	\retval out_pTView	Shader resource interface to the newly uploaded texture that will be used to manipulate the texture as a shader resource
	\sa tagDX10ViewGlobals, \ref DX10View_Callbacks, \ref Execute
	\since 7.0
*/
RTSHADER_API HRESULT	DX10View_UploadTextureCube(XSI_RTS_Texture* in_pImage, ID3D10Texture2D** out_pTCube, ID3D10ShaderResourceView** out_pTView);

extern "C" RTSHADER_API void*	GetInterface();


#endif // DIRECTX10_VIEW_H