//*****************************************************************************
/*!
   \file xsi_dataarray.h
   \brief CDataArray classes declaration.

    Copyright 1998-2007 Avid Technology, Inc. and its licensors. All rights
   reserved. This file contains confidential and proprietary information of
   Avid Technology, Inc., and is subject to the terms of the SOFTIMAGE|XSI
   end user license agreement (or EULA).
*/
//*****************************************************************************

#if (_MSC_VER > 1000) || defined(SGI_COMPILER)
#pragma once
#endif

#ifndef __XSIDATAARRAY_H__
#define __XSIDATAARRAY_H__

#include <sicppsdk.h>
#include <xsi_icenodecontext.h>
#include <xsi_indexset.h>

#include <xsi_vector2f.h>
#include <xsi_vector3f.h>
#include <xsi_vector4f.h>
#include <xsi_quaternionf.h>
#include <xsi_rotationf.h>
#include <xsi_matrix3f.h>
#include <xsi_matrix4f.h>
#include <xsi_color4f.h>
#include <xsi_shape.h>

namespace XSI {

class CBaseDataArray
{
	public:
	friend class ICENodeContext;
	
	CBaseDataArray() :
		m_nHandle(UINT_MAX),
		m_pData(NULL),
		m_bIsConstant(true),
		m_nCount(1)
	{}

	protected:
	SICPPSDK_INLINE ICENodeContext& GetContextRef();
	SICPPSDK_INLINE ULONG& GetHandleRef();
	SICPPSDK_INLINE void*& GetDataRef();
	SICPPSDK_INLINE bool& GetConstantRef();
	SICPPSDK_INLINE ULONG& GetCountRef();
	
	virtual SICPPSDK_INLINE CStatus AcquireInputDataArray( XSI::siICENodeDataType in_arrayDataType, ULONG in_nInputPortID, ULONG in_nInstanceIndex );
	virtual SICPPSDK_INLINE CStatus AcquireOutputDataArray( XSI::siICENodeDataType in_arrayDataType );
	virtual SICPPSDK_INLINE CStatus ReleaseDataArray( );

	ICENodeContext m_ctxt;
	ULONG m_nHandle;
	void* m_pData;
	bool m_bIsConstant;
	ULONG m_nCount;

	private:
	CBaseDataArray( const CBaseDataArray& );
};

SICPPSDK_INLINE CStatus CBaseDataArray::AcquireInputDataArray( XSI::siICENodeDataType in_arrayDataType, ULONG in_nInputPortID, ULONG in_nInstanceIndex )
{
	return m_ctxt.AcquireInputDataArray( *this, in_arrayDataType, in_nInputPortID, in_nInstanceIndex );
}

SICPPSDK_INLINE CStatus CBaseDataArray::AcquireOutputDataArray( XSI::siICENodeDataType in_arrayDataType )
{
	return m_ctxt.AcquireOutputDataArray( *this, in_arrayDataType );
}

SICPPSDK_INLINE CStatus CBaseDataArray::ReleaseDataArray( )
{
	return m_ctxt.ReleaseDataArray( *this );
}

SICPPSDK_INLINE ICENodeContext& CBaseDataArray::GetContextRef()
{
	return m_ctxt;
}

SICPPSDK_INLINE ULONG& CBaseDataArray::GetHandleRef()
{
	return m_nHandle;
}

SICPPSDK_INLINE void*& CBaseDataArray::GetDataRef()
{
	return m_pData;
}

SICPPSDK_INLINE bool& CBaseDataArray::GetConstantRef()
{
	return m_bIsConstant;
}

SICPPSDK_INLINE ULONG& CBaseDataArray::GetCountRef()
{
	return m_nCount;
}

//*****************************************************************************
/*! \class CDataArray xsi_dataarray.h
	\brief This template class encapsulates the 1D data arrays passed to custom node plug-ins
	during a graph evaluation. The underlying data arrays are zero-based, the following types
	are available:

	\li \ref CDataArrayFloat "CDataArrayFloat"
	\li \ref CDataArrayLong "CDataArrayLong"
	\li \ref CDataArrayBool "CDataArrayBool"
	\li \ref CDataArrayVector2f "CDataArrayVector2f"
	\li \ref CDataArrayVector3f "CDataArrayVector3f"
	\li \ref CDataArrayVector4f "CDataArrayVector4f"
	\li \ref CDataArrayQuaternionf "CDataArrayQuaternionf"
	\li \ref CDataArrayRotationf "CDataArrayRotationf"
	\li \ref CDataArrayMatrix3f "CDataArrayMatrix3f"
	\li \ref CDataArrayMatrix4f "CDataArrayMatrix4f"
	\li \ref CDataArrayColor4f "CDataArrayColor4f"
	\li \ref CDataArrayShape "CDataArrayShape"

	CDataArray objects are created based on the current evaluation context and a specific port
	type (input ports only). Data are usually indexed with a CIndexSet which contains the array
	index set for the whole evaluation graph.

	\note If the class type doesn't match the port data type being evaluated, an error occurs at
	run-time and a class instance is created with an empty array.

	\eg
	\code
		using namespace XSI;

		CDataArray outData( in_ctxt );
		CDataArray inData( in_ctxt, ID_IN_vector3 );
		CDataArrayFloat scalarData( in_ctxt, ID_IN_factor );

		CIndexSet indexSet( in_ctxt );
		for(CIndexSet::Iterator it = indexSet.Begin(); it.HasNext(); it.Next())
		{
			outData[it] = inData[it];
			outData[it].ScaleInPlace( scalarData[it] );
		}
	\endcode

	\since 7.0
	\sa CDataArray2D, CIndexSet, ICENodeDef, \ref CDataArrayTypedefs "Type Definitions for CDataArray"
 */
//*****************************************************************************

template <class T>
class CDataArray : public CBaseDataArray
{
public:
	/*! \typedef T TData
		\brief TData is the underlying data type of a CDataArray instance. Possible values are:

	 	\li \ref CDataArrayFloat "float"
		\li \ref CDataArrayLong "LONG"
		\li \ref CDataArrayBool "bool"
		\li \ref CDataArrayVector2f "CVector2f"
		\li \ref CDataArrayVector3f "CVector3f"
		\li \ref CDataArrayVector4f "CVector4f"
		\li \ref CDataArrayQuaternionf "CQuaternionf"
		\li \ref CDataArrayRotationf "CRotationf"
		\li \ref CDataArrayMatrix3f "CMatrix3f"
		\li \ref CDataArrayMatrix4f "CMatrix4f"
		\li \ref CDataArrayColor4f "CColor4f"
		\li \link MATH::CShape CShape\endlink
	*/
	typedef T TData;

	/*! Constructor for data types bound to input ports.
	\param in_ctxt ICENode evaluation context.
	\param in_nInputPortID Input port identifier.
	\param in_nInstanceIndex The group instance of the port.
	*/
	SICPPSDK_INLINE CDataArray( ICENodeContext& in_ctxt, ULONG in_nInputPortID, ULONG in_nInstanceIndex=0 ) ;

	/*! Constructor for data types bound to output ports.
	\param in_ctxt ICENode evaluation context.
	*/
	SICPPSDK_INLINE CDataArray( ICENodeContext& in_ctxt ) ;

	/*! Destructor.
	*/
	SICPPSDK_INLINE ~CDataArray( );

	/*! Accessor to the encapsulated array. This operator is called when reading the data so the return value is read-only.
	\param in_index Index in the array. The index must be smaller than the number of elements in the array, otherwise the
	results are unpredicted.
	\return A read-only reference to the indexed item.
	*/
	SICPPSDK_INLINE const TData& operator[]( ULONG in_index ) const;

	/*! Accessor to elements at a given index.
	\param in_index Index in this zero based array. The index must be smaller than the number of elements in the array,
	otherwise the results are unpredicted.
	\return A reference to the indexed item.
	*/
	SICPPSDK_INLINE TData& operator[]( ULONG in_index ) ;

	/*! Returns the number of elements in the array.
	\return Number of elements.
	*/
	SICPPSDK_INLINE ULONG GetCount() const;

	/*! Returns true if the array is constant or false otherwise. A constant array has only one value.
	\return True if constant, false otherwise.
	*/
	SICPPSDK_INLINE bool IsConstant() const;

private:
	SICPPSDK_INLINE void Clear();
	static SICPPSDK_INLINE XSI::siICENodeDataType GetDefaultType( );

#ifdef _DEBUG
	TData* m_pDebugData;
#endif

};

template< class T >
SICPPSDK_INLINE CDataArray<T>::CDataArray( ICENodeContext& in_ctxt, ULONG in_nInputPortID, ULONG in_nInstanceIndex )
{
	m_ctxt = in_ctxt;

	CStatus st = AcquireInputDataArray( GetDefaultType(), in_nInputPortID, in_nInstanceIndex );
	st.AssertSucceeded( );

#ifdef _DEBUG
	m_pDebugData = (T*)GetDataRef();
#endif
}

template< class T >
SICPPSDK_INLINE CDataArray<T>::CDataArray( ICENodeContext& in_ctxt )
{
	m_ctxt = in_ctxt;

	CStatus st = AcquireOutputDataArray( GetDefaultType() );
	st.AssertSucceeded( );
#ifdef _DEBUG
	m_pDebugData = (T*)GetDataRef();
#endif
}

template< class T >
SICPPSDK_INLINE CDataArray<T>::~CDataArray( )
{
	ReleaseDataArray( );
}

template< class T >
SICPPSDK_INLINE const T& CDataArray<T>::operator[]( ULONG i ) const
{
	assert( !m_bIsConstant ? i < m_nCount : true );
	T* pData = (T*)GetDataRef();
	assert( pData != NULL );

	static T defVal;
	return pData ? ( m_bIsConstant ? *pData : pData[i] ) : defVal;
}

template< class T >
SICPPSDK_INLINE T& CDataArray<T>::operator[]( ULONG i )
{
	assert( !m_bIsConstant ? i < m_nCount : true );
	T* pData = (T*)GetDataRef();
	assert( pData != NULL );

	static T defVal;
	return pData ? ( m_bIsConstant ? *pData : pData[i] ) : defVal;
}

template< class T >
SICPPSDK_INLINE void CDataArray<T>::Clear()
{
	ReleaseDataArray( );
	m_nHandle = UINT_MAX;
	m_nCount = 0;
	m_bIsConstant = false;
}

template< class T >
SICPPSDK_INLINE ULONG CDataArray<T>::GetCount() const
{
	return m_nCount;
}

template< class T >
SICPPSDK_INLINE bool CDataArray<T>::IsConstant() const
{
	return m_bIsConstant;
}

//*****************************************************************************
// NB: Do not use the Doxygen \class tag for this template!!!
/*!
	\brief This class is a specialization of a CDataArray class of type \c bool.

	\since 7.0
	\sa CDataArray2D, CIndexSet, ICENodeDef, \ref CDataArrayBool "CDataArrayBool"
 */
//*****************************************************************************
template<>
class CDataArray< bool > : public CBaseDataArray
{
	public:

	/*! Constructor for \c bool array types bound to input ports.
	\param in_ctxt ICENode evaluation context.
	\param in_nInputPortID Input port identifier.
	\param in_nInstanceIndex The group instance of the port.
	*/
	SICPPSDK_INLINE CDataArray< bool >( ICENodeContext& in_ctxt, ULONG in_nInputPortID, ULONG in_nInstanceIndex=0 ) ;

	/*! Constructor for \c bool array types bound to output ports.
	\param in_ctxt ICENode evaluation context.
	*/
	SICPPSDK_INLINE CDataArray< bool >( ICENodeContext& in_ctxt ) ;

	/*! Destructor.
	*/
	SICPPSDK_INLINE ~CDataArray< bool >( );

	/*! Accessor to the \c bool array. This operator is called when reading the data so the return value is read-only.
	\param in_index Index in the array. The index must be smaller than the number of elements in the array, otherwise the
		results are unpredicted.
	\return A read-only value to the indexed item.
	*/
	SICPPSDK_INLINE const bool operator[]( ULONG in_index ) const;

	/*! Sets the array at a given index with a value.
	\param in_index Index in the array. The index must be smaller than the number of elements in the array, otherwise the
		results are unpredicted.
	\return CStatus::OK Success
	*/
	SICPPSDK_INLINE CStatus Set( ULONG in_index, bool in_bVal ) ;

	/*! Returns the number of elements in the array.
	\return Number of elements.
	*/
	SICPPSDK_INLINE ULONG GetCount() const;

	/*! Returns true if the array is constant or false otherwise. A constant array has only one value.
	\return True if constant, false otherwise.
	*/
	SICPPSDK_INLINE bool IsConstant() const;

private:
	SICPPSDK_INLINE void Clear();
	static SICPPSDK_INLINE XSI::siICENodeDataType GetDefaultType( );

	CBitsetHelper m_bitset;
};

SICPPSDK_INLINE CDataArray<bool>::CDataArray( ICENodeContext& in_ctxt, ULONG in_nInputPortID, ULONG in_nInstanceIndex )
{
	m_ctxt = in_ctxt;

	CStatus st = AcquireInputDataArray( GetDefaultType(), in_nInputPortID, in_nInstanceIndex );
	st.AssertSucceeded( );

	m_bitset.GetCountRef() = GetCountRef();
	m_bitset.GetArrayRef() = (ULONG*)GetDataRef();
}

SICPPSDK_INLINE CDataArray<bool>::CDataArray( ICENodeContext& in_ctxt )
{
	m_ctxt = in_ctxt;
	m_bIsConstant = true;
	m_nCount = 1;

	CStatus st = AcquireOutputDataArray( GetDefaultType() );
	st.AssertSucceeded( );

	m_bitset.GetCountRef() = m_nCount;
	m_bitset.GetArrayRef() = (ULONG*)GetDataRef();
}

SICPPSDK_INLINE CDataArray<bool>::~CDataArray( )
{
	ReleaseDataArray( );
}

SICPPSDK_INLINE CStatus CDataArray<bool>::Set( ULONG in_index, bool in_bVal )
{
	return m_bitset.SetBit( in_index, in_bVal );
}

SICPPSDK_INLINE const bool CDataArray<bool>::operator[] ( ULONG in_index ) const
{
	ULONG nIndex = m_bIsConstant ? 0 : in_index;
	return m_bitset.GetBit( nIndex );
}

SICPPSDK_INLINE void CDataArray<bool>::Clear()
{
	ReleaseDataArray( );
	m_nHandle = UINT_MAX;
	m_bIsConstant = true;
	m_nCount = 1;
	m_bitset.Clear();
}

SICPPSDK_INLINE XSI::siICENodeDataType CDataArray<bool>::GetDefaultType()
{
	return siICENodeDataBool;
}

SICPPSDK_INLINE ULONG CDataArray<bool>::GetCount() const
{
	return m_nCount;
}

SICPPSDK_INLINE bool CDataArray<bool>::IsConstant() const
{
	return m_bIsConstant;
}

/*! \page CDataArrayTypedefs Type Definitions for CDataArray
	The CDataArray class is a template class that encapsulates the following 1D data arrays:

	\li \ref CDataArrayFloat "CDataArrayFloat"
	\li \ref CDataArrayLong "CDataArrayLong"
	\li \ref CDataArrayBool "CDataArrayBool"
	\li \ref CDataArrayVector2f "CDataArrayVector2f"
	\li \ref CDataArrayVector3f "CDataArrayVector3f"
	\li \ref CDataArrayVector4f "CDataArrayVector4f"
	\li \ref CDataArrayQuaternionf "CDataArrayQuaternionf"
	\li \ref CDataArrayRotationf "CDataArrayRotationf"
	\li \ref CDataArrayMatrix3f "CDataArrayMatrix3f"
	\li \ref CDataArrayMatrix4f "CDataArrayMatrix4f"
	\li \ref CDataArrayColor4f "CDataArrayColor4f"
	\li \ref CDataArrayShape "CDataArrayShape"

	\section CDataArrayFloat CDataArrayFloat Type
	\code typedef CDataArray< float > CDataArrayFloat \endcode
	A CDataArray class of type \c float.
	\since 7.0

	\section CDataArrayLong CDataArrayLong Type
	\code typedef CDataArray< LONG > CDataArrayLong \endcode
	A CDataArray class of type \c LONG.
	\since 7.0

	\section CDataArrayBool CDataArrayBool Type
	\code typedef CDataArray< bool > CDataArrayBool \endcode
	The CDataArray< bool > class.
	\since 7.0

	\section CDataArrayVector2f CDataArrayVector2f Type
	\code typedef CDataArray< CVector2f > CDataArrayVector2f
	\endcode A CDataArray class of type \link MATH::CVector2f CVector2f\endlink.
	\since 7.0

	\section CDataArrayVector3f CDataArrayVector3f Type
	\code typedef CDataArray< CVector3f > CDataArrayVector3f \endcode
	A CDataArray class of type \link MATH::CVector3f CVector3f\endlink.
	\since 7.0

	\section CDataArrayVector4f CDataArrayVector4f Type
	\code typedef CDataArray< CVector4f > CDataArrayVector4f \endcode
	A CDataArray class of type \link MATH::CVector4f CVector4f\endlink.
	\since 7.0

	\section CDataArrayQuaternionf CDataArrayQuaternionf Type
	\code typedef CDataArray< Quaternionf > CDataArrayQuaternionf \endcode
	A CDataArray class of type \link MATH::CQuaternionf CQuaternionf\endlink.
	\since 7.0

	\section CDataArrayRotationf CDataArrayRotationf Type
	\code typedef CDataArray< CRotationf > CDataArrayRotationf \endcode
	A CDataArray class of type \link MATH::CRotationf CRotationf\endlink.
	\since 7.0

	\section CDataArrayMatrix3f CDataArrayMatrix3f Type
	\code typedef CDataArray< CMatrix3f > CDataArrayMatrix3f \endcode
	A CDataArray class of type \link MATH::CMatrix3f CMatrix3f\endlink.
	\since 7.0

	\section CDataArrayMatrix4f CDataArrayMatrix4f Type
	\code typedef CDataArray< CMatrix4f > CDataArrayMatrix4f \endcode
	A CDataArray class of type \link MATH::CMatrix4f CMatrix4f\endlink.
	\since 7.0

	\section CDataArrayColor4f CDataArrayColor4f Type
	\code typedef CDataArray< CColor4f > CDataArrayColor4f \endcode
	A CDataArray class of type \link MATH::CColor4f CColor4f\endlink.
	\since 7.0

	\section CDataArrayShape CDataArrayShape Type
	\code typedef CDataArray< CShape > CDataArrayShape \endcode
	A CDataArray class of type \link MATH::CShape CShape\endlink.
	\since 7.0
*/
typedef XSI::CDataArray< float > CDataArrayFloat;
typedef XSI::CDataArray< LONG > CDataArrayLong;
typedef XSI::CDataArray< bool > CDataArrayBool;
typedef XSI::CDataArray< XSI::MATH::CVector2f > CDataArrayVector2f;
typedef XSI::CDataArray< XSI::MATH::CVector3f > CDataArrayVector3f;
typedef XSI::CDataArray< XSI::MATH::CVector4f > CDataArrayVector4f;
typedef XSI::CDataArray< XSI::MATH::CQuaternionf > CDataArrayQuaternionf;
typedef XSI::CDataArray< XSI::MATH::CRotationf > CDataArrayRotationf;
typedef XSI::CDataArray< XSI::MATH::CMatrix3f > CDataArrayMatrix3f;
typedef XSI::CDataArray< XSI::MATH::CMatrix4f > CDataArrayMatrix4f;
typedef XSI::CDataArray< XSI::MATH::CColor4f > CDataArrayColor4f;
typedef XSI::CDataArray< XSI::MATH::CShape> CDataArrayShape;

template<> SICPPSDK_INLINE XSI::siICENodeDataType CDataArrayFloat::GetDefaultType( ){return siICENodeDataFloat;}
template<> SICPPSDK_INLINE XSI::siICENodeDataType CDataArrayLong::GetDefaultType( ){return siICENodeDataLong;}
template<> SICPPSDK_INLINE XSI::siICENodeDataType CDataArrayVector2f::GetDefaultType( ){return siICENodeDataVector2;}
template<> SICPPSDK_INLINE XSI::siICENodeDataType CDataArrayVector3f::GetDefaultType( ){return siICENodeDataVector3;}
template<> SICPPSDK_INLINE XSI::siICENodeDataType CDataArrayVector4f::GetDefaultType( ){return siICENodeDataVector4;}
template<> SICPPSDK_INLINE XSI::siICENodeDataType CDataArrayQuaternionf::GetDefaultType( ){return siICENodeDataQuaternion;}
template<> SICPPSDK_INLINE XSI::siICENodeDataType CDataArrayMatrix3f::GetDefaultType( ){return siICENodeDataMatrix33;}
template<> SICPPSDK_INLINE XSI::siICENodeDataType CDataArrayMatrix4f::GetDefaultType( ){return siICENodeDataMatrix44;}
template<> SICPPSDK_INLINE XSI::siICENodeDataType CDataArrayColor4f::GetDefaultType( ){return siICENodeDataColor4;}
template<> SICPPSDK_INLINE XSI::siICENodeDataType CDataArrayRotationf::GetDefaultType( ){return siICENodeDataRotation;}
template<> SICPPSDK_INLINE XSI::siICENodeDataType CDataArrayShape::GetDefaultType( ){return siICENodeDataShape;}
};

#endif // __XSIDATAARRAY_H__
