//*****************************************************************************
/*!
   \file xsi_dataarray2D.h
   \brief CDataArray2D classes declaration.

    Copyright 1998-2007 Avid Technology, Inc. and its licensors. All rights
   reserved. This file contains confidential and proprietary information of
   Avid Technology, Inc., and is subject to the terms of the SOFTIMAGE|XSI
   end user license agreement (or EULA).
*/
//*****************************************************************************

#if (_MSC_VER > 1000) || defined(SGI_COMPILER)
#pragma once
#endif

#ifndef __XSIDATAARRAY2D_H__
#define __XSIDATAARRAY2D_H__

#include <sicppsdk.h>
#include <xsi_icenodecontext.h>
#include <xsi_indexset.h>
#include <xsi_dataarray.h>

#include <xsi_vector2f.h>
#include <xsi_vector3f.h>
#include <xsi_vector4f.h>
#include <xsi_quaternionf.h>
#include <xsi_rotationf.h>
#include <xsi_matrix3f.h>
#include <xsi_matrix4f.h>
#include <xsi_color4f.h>

namespace XSI {

class ICENodeContext;

class CBaseDataArray2D : public CBaseDataArray
{
	public:
	CBaseDataArray2D() {}

	protected:
	SICPPSDK_INLINE CStatus AcquireInputDataArray( XSI::siICENodeDataType in_arrayDataType, ULONG in_nInputPortID, ULONG in_nInstanceIndex );
	SICPPSDK_INLINE CStatus AcquireOutputDataArray( XSI::siICENodeDataType in_arrayDataType );
	SICPPSDK_INLINE CStatus ReleaseDataArray( );
	SICPPSDK_INLINE CStatus GetSubArray( ULONG in_nArrayIndex, void*& out_ppData, ULONG& out_nCount, ULONG& out_nStartBit ) const;
	SICPPSDK_INLINE CStatus ResizeSubArray( ULONG in_nArrayIndex, ULONG in_nSize, void*& out_ppData, ULONG& out_nCount, ULONG& out_nStartBit );

	private:
	CBaseDataArray2D( const CBaseDataArray2D& );
};

SICPPSDK_INLINE CStatus CBaseDataArray2D::AcquireInputDataArray( XSI::siICENodeDataType in_arrayDataType, ULONG in_nInputPortID, ULONG in_nInstanceIndex )
{
	return m_ctxt.AcquireInputDataArray2D( *this, in_arrayDataType, in_nInputPortID, in_nInstanceIndex );
}

SICPPSDK_INLINE CStatus CBaseDataArray2D::AcquireOutputDataArray( XSI::siICENodeDataType in_arrayDataType )
{
	return m_ctxt.AcquireOutputDataArray2D( *this, in_arrayDataType );
}

SICPPSDK_INLINE CStatus CBaseDataArray2D::ReleaseDataArray( )
{
	return m_ctxt.ReleaseDataArray( *this );
}

SICPPSDK_INLINE CStatus CBaseDataArray2D::ResizeSubArray( ULONG in_nArrayIndex, ULONG in_nSize, void*& out_ppData, ULONG& out_nCount, ULONG& out_nStartBit )
{
	return m_ctxt.ResizeSubArray( *this, in_nArrayIndex, in_nSize, out_ppData, out_nCount, out_nStartBit );
}

SICPPSDK_INLINE CStatus CBaseDataArray2D::GetSubArray( ULONG in_nArrayIndex, void*& out_ppData, ULONG& out_nCount, ULONG& out_nStartBit ) const
{
	return m_ctxt.GetSubArray( (CBaseDataArray2D&)*this, in_nArrayIndex, out_ppData, out_nCount, out_nStartBit );
}

//*****************************************************************************
/*! \class CDataArray2D xsi_dataarray2D.h
	\brief This template class encapsulates 2D data arrays that are passed to custom node plug-ins
	during a graph evaluation. The underlying data arrays are zero-based, the following types are
	available:

	\li \ref CDataArray2DFloat "CDataArray2DFloat"
	\li \ref CDataArray2DLong "CDataArray2DLong"
	\li \ref CDataArray2DBool "CDataArray2DBool"
	\li \ref CDataArray2DVector2f "CDataArray2DVector2f"
	\li \ref CDataArray2DVector3f "CDataArray2DVector3f"
	\li \ref CDataArray2DVector4f "CDataArray2DVector4f"
	\li \ref CDataArray2DQuaternionf "CDataArray2DQuaternionf"
	\li \ref CDataArray2DRotationf "CDataArray2DRotationf"
	\li \ref CDataArray2DMatrix3f "CDataArray2DMatrix3f"
	\li \ref CDataArray2DMatrix4f "CDataArray2DMatrix4f"
	\li \ref CDataArray2DColor4f "CDataArray2DColor4f"
	\li \ref CDataArray2DShape "CDataArray2DShape"

	As for CDataArray, instances of CDataArray2D are created based on the current evaluation context
	and a specific port type (input ports only). CDataArray2D objects are also indexed with CIndexSet
	objects. Unlike CDataArray, CDataArray2D contains sub-arrays where data can be accessed  through
	the CDataArray2D::Accessor class.

	\note If the class type doesn't match the port data type being evaluated, an error occurs at run-time
		and a class instance is created with an empty array.

	\sa CDataArray2D, CIndexSet, ICENodeDef, \ref CDataArray2DTypedefs "Type Definitions for CDataArray2D"
	\since 7.0

	\eg
	\code
		using namespace XSI;

		XSIPLUGINCALLBACK CStatus Custom2DArray_Evaluate( ICENodeContext& in_ctxt )
		{
			// The current output port being evaluated...
			ULONG out_portID = in_ctxt.GetEvaluatedOutputPortID( );

			switch( out_portID )
			{
				case ID_OutPort:
				{
					// Here's how we access the input data buffers for each port
					CDataArray2DVector3f InPortData( in_ctxt, ID_InPort );

					// Declare the output port array ...
					CDataArray2DVector3f outData( in_ctxt );

					// We need a CIndexSet to iterate over the data
					CIndexSet indexSet( in_ctxt );

					for(CIndexSet::Iterator it = indexSet.Begin(); it.HasNext(); it.Next())
					{
						// Creates an Accessor object to iterate over the 2D array data
						CDataArray2DVector3f::Accessor floatArray = InPortData[it];

						// Output arrays must always be initialized first
						CDataArray2DVector3f::Accessor outAccessor = outData.Resize( it, floatArray.GetCount( ) );

						for (ULONG i=0; i<floatArray.GetCount( ); i++)
						{
							outAccessor[i].Scale( floatArray[i].GetX(), floatArray[i] );
						}
					}
				}
				break;
			};

			return CStatus::OK;
		}
	\endcode
 */
//*****************************************************************************

template <class T>
class CDataArray2D : public CBaseDataArray2D
{
	public:

	/*! \typedef T TData
	 *  \brief TData is the underlying data type of a CDataArray2D instance. Possible values are:

	 	\li \ref CDataArray2DFloat "float"
		\li \ref CDataArray2DLong "LONG"
		\li \ref CDataArray2DBool "bool"
		\li \ref CDataArray2DVector2f "CVector2f"
		\li \ref CDataArray2DVector3f "CVector3f"
		\li \ref CDataArray2DVector4f "CVector4f"
		\li \ref CDataArray2DQuaternionf "CQuaternionf"
		\li \ref CDataArray2DRotationf "CRotationf"
		\li \ref CDataArray2DMatrix3f "CMatrix3f"
		\li \ref CDataArray2DMatrix4f "CMatrix4f"
		\li \ref CDataArray2DColor4f "CColor4f"
		\li \ref CDataArray2DShape "CShape"
	*/
	typedef T TData;

	//*****************************************************************************
	/*! \class Accessor xsi_dataarray2D.h
		\brief A class for accessing the sub-array data of CDataArray2D objects.

		\since 7.0
		\sa CIndexSet, CDataArray, CDataArray2D
	 */
	//*****************************************************************************

	class Accessor
	{
		public:
		friend class CDataArray2D<T>;

		/*! Constructor.
		*/
		Accessor( ) : m_pData(NULL), m_nCount(0)
		{
		}

		/*! Constructs an Accessor object from another Accessor object.
		\param in_accessor constant Accessor object.*/
		Accessor( const Accessor& in_accessor )
		{
			*this = in_accessor;
		}

		/*! Assignment operator.
		\param in_accessor constant class object.
		\return A reference to this Accessor.
		*/
		Accessor& operator=( const Accessor& in_accessor )
		{
			m_pData = in_accessor.m_pData;
			m_nCount = in_accessor.m_nCount;
			return *this;
		}

		/*! Returns the number of elements in the array.
		\return Number of elements.
		*/
		ULONG GetCount() const
		{
			return m_nCount;
		}

		/*! Accessor to elements at a given index. This operator is called when reading the data so the return value is read-only.
		\param in_index Index in the array. The index must be smaller than the number of elements in the array, otherwise the
			results are unpredicted.
		\return A read-only reference to the indexed item.
		*/
		const T& operator[]( ULONG in_index ) const
		{
			assert( in_index < m_nCount );
			return m_pData[ in_index ];
		}

		/*! Accessor to elements at a given index.
		\param in_index Index in this zero-based array. The index must be smaller than the number of elements in the array,
			otherwise the results are unpredicted.
		\return A reference to the indexed item.
		*/
		T& operator[]( ULONG in_index )
		{
			assert( in_index < m_nCount );
			return m_pData[ in_index ];
		}

		private:
		void*& GetDataRef()
		{
			return (void*&)m_pData;
		}

		ULONG& GetCountRef()
		{
			return m_nCount;
		}

		T* m_pData;
		ULONG m_nCount;
	};

	/*! Constructor for data types bound to input ports.
	\param in_ctxt ICENode evaluation context.
	\param in_nInputPortID Input port identifier.
	\param in_nInstanceIndex The group instance of the port.
	*/
	SICPPSDK_INLINE CDataArray2D( ICENodeContext& in_ctxt, ULONG in_nInputPortID, ULONG in_nInstanceIndex=0 ) ;

	/*! Constructor for data types bound to output ports.
	\param in_ctxt ICENode evaluation context.
	*/
	SICPPSDK_INLINE CDataArray2D( ICENodeContext& in_ctxt ) ;

	/*! Destructor.
	*/
	SICPPSDK_INLINE ~CDataArray2D( );

	/*! Returns an accessor at a given index. This operator is called when reading the data so the return value is read-only.
	\param in_nArrayIndex Index in the array. The index must be smaller than the number of elements in the array, otherwise the
		results are unpredicted.
	\return Accessor object.
	*/
	SICPPSDK_INLINE Accessor operator[]( ULONG in_nArrayIndex ) const;

	/*! Changes the size of the sub-array at a given position and returns an accessor pointing to the resized sub-array.
	\param in_nArrayIndex Index in the array. The index must be smaller than the number of elements in the array, otherwise the
		results are unpredicted.
	\param in_nSize The new size.
	\return Accessor object.
	*/
	SICPPSDK_INLINE Accessor Resize( ULONG in_nArrayIndex, ULONG in_nSize );

private:
	SICPPSDK_INLINE Accessor Get( ULONG in_nArrayIndex ) const;
	SICPPSDK_INLINE void Clear();
	static SICPPSDK_INLINE XSI::siICENodeDataType GetDefaultType( );
};

template< class T >
SICPPSDK_INLINE CDataArray2D<T>::CDataArray2D( ICENodeContext& in_ctxt, ULONG in_nInputPortID, ULONG in_nInstanceIndex )
{
	m_ctxt = in_ctxt;

	CStatus st = AcquireInputDataArray( GetDefaultType(), in_nInputPortID, in_nInstanceIndex );
	st.AssertSucceeded( );
}

template< class T >
SICPPSDK_INLINE CDataArray2D<T>::CDataArray2D( ICENodeContext& in_ctxt )
{
	m_ctxt = in_ctxt;

	CStatus st = AcquireOutputDataArray( GetDefaultType() );
	st.AssertSucceeded( );
}

template< class T >
SICPPSDK_INLINE CDataArray2D<T>::~CDataArray2D( )
{
	ReleaseDataArray( );
}

template< class T >
SICPPSDK_INLINE typename CDataArray2D<T>::Accessor CDataArray2D<T>::Get( ULONG in_nArrayIndex ) const
{
	assert( m_bIsConstant ? in_nArrayIndex < m_nCount : true );

	Accessor a;
	ULONG nNotused;
	CStatus st = GetSubArray( in_nArrayIndex, a.GetDataRef(), (ULONG&)a.GetCountRef(), nNotused );
	st.AssertSucceeded();

	return a;
}

template< class T >
SICPPSDK_INLINE typename CDataArray2D<T>::Accessor CDataArray2D<T>::operator[]( ULONG in_nArrayIndex ) const
{
	return Get( in_nArrayIndex );
}

template< class T >
SICPPSDK_INLINE typename CDataArray2D<T>::Accessor CDataArray2D<T>::Resize( ULONG in_nArrayIndex, ULONG in_nSize )
{
	assert( m_bIsConstant ? in_nArrayIndex < m_nCount : true );

	ULONG nNotused;
	Accessor a;
	CStatus st = ResizeSubArray( in_nArrayIndex, in_nSize, a.GetDataRef(), (ULONG&)a.GetCountRef(), nNotused );
	st.AssertSucceeded();

	return a;
}

template< class T >
SICPPSDK_INLINE void CDataArray2D<T>::Clear()
{
	ReleaseDataArray( );
	m_nHandle = UINT_MAX;
	m_nCount = 0;
	m_bIsConstant = false;
}

//*****************************************************************************
// NB: Do not use the Doxygen \class tag for this template!!!
/*!
	\brief This class is a specialization of a CDataArray2D class of type \c bool.

	\since 7.0
	\sa CDataArray, CIndexSet, ICENodeDef, Accessor, \ref CDataArray2DBool "CDataArray2DBool typedef"
 */
//*****************************************************************************

template <>
class CDataArray2D<bool> : public CBaseDataArray2D
{
	public:
	//*****************************************************************************
	/*! \class Accessor xsi_dataarray2D.h
		\brief A class for accessing the sub-array data of CDataArray2D< bool > objects.

		\since 7.0
		\sa CIndexSet, CDataArray, CDataArray2D
	 */
	//*****************************************************************************
	class Accessor
	{
		public:
		friend class CDataArray2D<bool>;

		/*! Constructor.
		*/
		Accessor( ) : m_nStartbit(0)
		{
		}

		/*! Constructs an Accessor object from another Accessor object.
		\param in_accessor constant Accessor object.*/
		Accessor( const Accessor& in_accessor )
		{
			*this = in_accessor;
		}

		/*! Assignment operator.
		\param in_accessor constant class object.
		\return A reference to this Accessor.
		*/
		Accessor& operator=( const Accessor& in_accessor )
		{
			m_bitset = in_accessor.m_bitset;
			m_nStartbit = in_accessor.m_nStartbit;
			return *this;
		}

		/*! Accessor to elements at a given index. This operator is called when reading the data so the return value is read-only.
		\param in_index Index in the array. The index must be smaller than the number of elements in the array, otherwise the
			results are unpredicted.
		\return A read-only reference to the indexed item.
		*/
		const bool operator[] ( ULONG in_index ) const
		{
			ULONG nIndex = m_nStartbit + m_bitset.GetCount() == UINT_MAX ? 0 : in_index;
			return m_bitset.GetBit( nIndex );
		}

		/*! Sets the array at a given index with a value.
		\param in_index Index in the array. The index must be smaller than the number of elements in the array, otherwise the
			results are unpredicted.
		\param in_bVal	Value to set.
		\return CStatus::OK Success
		*/
		CStatus Set( ULONG in_index, bool in_bVal )
		{
			return m_bitset.SetBit( m_nStartbit + in_index, in_bVal );
		}

		/*! Returns the number of elements in the array.
		\return Number of elements.
		*/
		ULONG GetCount() const
		{
			return m_bitset.GetCount( );
		}

		private:
		ULONG m_nStartbit;
		CBitsetHelper m_bitset;
	};

	/*! Constructor for data types bound to input ports.
	\param in_ctxt ICENode evaluation context.
	\param in_nInputPortID Input port identifier.
	\param in_nInstanceIndex The group instance of the port.
	*/
	SICPPSDK_INLINE CDataArray2D( ICENodeContext& in_ctxt, ULONG in_nInputPortID, ULONG in_nInstanceIndex=0 ) ;

	/*! Constructor for data types bound to output ports.
	\param in_ctxt ICENode evaluation context.
	*/
	SICPPSDK_INLINE CDataArray2D( ICENodeContext& in_ctxt ) ;

	/*! Destructor.
	*/
	SICPPSDK_INLINE ~CDataArray2D( );

	/*! Returns an accessor at a given index. This operator is called when reading the data so the return value is read-only.
	\param in_nArrayIndex Index in the array. The index must be smaller than the number of elements in the array, otherwise the
		results are unpredicted.
	\return Accessor object.
	*/
	SICPPSDK_INLINE Accessor operator[]( ULONG in_nArrayIndex ) const;

	/*! Changes the size of the sub-array at a given position and returns an accessor pointing to the resized sub-array.
	\param in_nArrayIndex Index in the array. The index must be smaller than the number of elements in the array, otherwise the
		results are unpredicted.
	\param in_nSize The new size.
	\return Accessor object.
	*/
	SICPPSDK_INLINE Accessor Resize( ULONG in_nArrayIndex, ULONG in_nSize );

private:
	SICPPSDK_INLINE Accessor Get( ULONG in_nArrayIndex ) const;
	static SICPPSDK_INLINE XSI::siICENodeDataType GetDefaultType( );
};

SICPPSDK_INLINE CDataArray2D<bool>::CDataArray2D( ICENodeContext& in_ctxt, ULONG in_nInputPortID, ULONG in_nInstanceIndex )
{
	m_ctxt = in_ctxt;

	CStatus st = AcquireInputDataArray( GetDefaultType(), in_nInputPortID, in_nInstanceIndex );
	st.AssertSucceeded( );
}

SICPPSDK_INLINE CDataArray2D<bool>::CDataArray2D( ICENodeContext& in_ctxt )
{
	m_ctxt = in_ctxt;

	CStatus st = AcquireOutputDataArray( GetDefaultType() );
	st.AssertSucceeded( );
}

SICPPSDK_INLINE CDataArray2D<bool>::~CDataArray2D( )
{
	ReleaseDataArray( );
}

SICPPSDK_INLINE CDataArray2D<bool>::Accessor CDataArray2D<bool>::Get( ULONG in_nArrayIndex ) const
{
	assert( !m_bIsConstant ? in_nArrayIndex < m_nCount : true );

	Accessor a;
	CStatus st = GetSubArray( in_nArrayIndex, (void*&)a.m_bitset.GetArrayRef(), (ULONG&)a.m_bitset.GetCountRef(), a.m_nStartbit );
	st.AssertSucceeded();

	return a;
}

SICPPSDK_INLINE CDataArray2D<bool>::Accessor CDataArray2D<bool>::operator[]( ULONG in_nArrayIndex ) const
{
	return Get( in_nArrayIndex );
}

SICPPSDK_INLINE CDataArray2D<bool>::Accessor CDataArray2D<bool>::Resize( ULONG in_nArrayIndex, ULONG in_nSize )
{
	assert( !m_bIsConstant ? in_nArrayIndex < m_nCount : true );

	Accessor a;
	CStatus st = ResizeSubArray( in_nArrayIndex, in_nSize, (void*&)a.m_bitset.GetArrayRef(), (ULONG&)a.m_bitset.GetCountRef(), a.m_nStartbit );
	st.AssertSucceeded();

	return a;
}

// Explicit template instances
/*! \page CDataArray2DTypedefs Type Definitions for CDataArray2D
	The CDataArray2D class is a template class that encapsulates the following 2D data arrays:

	\li \ref CDataArray2DFloat "CDataArray2DFloat"
	\li \ref CDataArray2DLong "CDataArray2DLong"
	\li \ref CDataArray2DBool "CDataArray2DBool"
	\li \ref CDataArray2DVector2f "CDataArray2DVector2f"
	\li \ref CDataArray2DVector3f "CDataArray2DVector3f"
	\li \ref CDataArray2DVector4f "CDataArray2DVector4f"
	\li \ref CDataArray2DQuaternionf "CDataArray2DQuaternionf"
	\li \ref CDataArray2DRotationf "CDataArray2DRotationf"
	\li \ref CDataArray2DMatrix3f "CDataArray2DMatrix3f"
	\li \ref CDataArray2DMatrix4f "CDataArray2DMatrix4f"
	\li \ref CDataArray2DColor4f "CDataArray2DColor4f"
	\li \ref CDataArray2DShape "CDataArray2DShape"

	\section CDataArray2DFloat CDataArray2DFloat Type
	\code typedef CDataArray2D< float > CDataArray2DFloat \endcode
	A CDataArray2D class of type \c float.
	\since 7.0

	\section CDataArray2DLong CDataArray2DLong Type
	\code typedef CDataArray2D< LONG > CDataArray2DLong \endcode
	A CDataArray2D class of type \c LONG.
	\since 7.0

	\section CDataArray2DBool CDataArray2DBool Type
	\code typedef CDataArray2D< bool > CDataArray2DBool \endcode
	The CDataArray2D< bool > class.
	\since 7.0

	\section CDataArray2DVector2f CDataArray2DVector2f Type
	\code typedef CDataArray2D< CVector2f > CDataArray2DVector2f \endcode
	A CDataArray2D class of type \link MATH::CVector2f CVector2f\endlink.
	\since 7.0

	\section CDataArray2DVector3f CDataArray2DVector3f Type
	\code typedef CDataArray2D< CVector3f > CDataArray2DVector3f \endcode
	A CDataArray2D class of type \link MATH::CVector3f CVector3f\endlink.
	\since 7.0

	\section CDataArray2DVector4f CDataArray2DVector4f Type
	\code typedef CDataArray2D< CVector4f > CDataArray2DVector4f \endcode
	A CDataArray2D class of type \link MATH::CVector4f CVector4f\endlink.
	\since 7.0

	\section CDataArray2DQuaternionf CDataArray2DQuaternionf Type
	\code typedef CDataArray2D< CQuaternionf > CDataArray2DQuaternionf \endcode
	A CDataArray2D class of type \link MATH::CQuaternionf CQuaternionf\endlink.
	\since 7.0

	\section CDataArray2DRotationf CDataArray2DRotationf Type
	\code typedef CDataArray2D< CRotationf > CDataArray2DRotationf \endcode
	A CDataArray2D class of type \link MATH::CRotationf CRotationf\endlink.
	\since 7.0

	\section CDataArray2DMatrix3f CDataArray2DMatrix3f Type
	\code typedef CDataArray2D< CMatrix3f > CDataArray2DMatrix3f \endcode
	A CDataArray2D class of type \link MATH::CMatrix3f CMatrix3f\endlink.
	\since 7.0

	\section CDataArray2DMatrix4f CDataArray2DMatrix4f Type
	\code typedef CDataArray2D< CMatrix4f > CDataArray2DMatrix4f \endcode
	A CDataArray2D class of type \link MATH::CMatrix4f CMatrix4f\endlink.
	\since 7.0

	\section CDataArray2DColor4f CDataArray2DColor4f Type
	\code typedef CDataArray2D< CColor4f > CDataArray2DColor4f \endcode
	A CDataArray2D class of type \link MATH::CColor4f CColor4f\endlink.
	\since 7.0

	\section CDataArray2DShape CDataArray2DShape Type
	\code typedef CDataArray2D< CShape> CDataArray2DShape \endcode
	A CDataArray2D class of type \link MATH::CShape CShape\endlink.
	\since 7.0
*/
typedef CDataArray2D< float > CDataArray2DFloat;
typedef CDataArray2D< LONG > CDataArray2DLong;
typedef CDataArray2D< bool > CDataArray2DBool;
typedef CDataArray2D< XSI::MATH::CVector2f > CDataArray2DVector2f;
typedef CDataArray2D< XSI::MATH::CVector3f > CDataArray2DVector3f;
typedef CDataArray2D< XSI::MATH::CVector4f > CDataArray2DVector4f;
typedef CDataArray2D< XSI::MATH::CQuaternionf > CDataArray2DQuaternionf;
typedef CDataArray2D< XSI::MATH::CRotationf > CDataArray2DRotationf;
typedef CDataArray2D< XSI::MATH::CMatrix3f > CDataArray2DMatrix3f;
typedef CDataArray2D< XSI::MATH::CMatrix4f > CDataArray2DMatrix4f;
typedef CDataArray2D< XSI::MATH::CColor4f > CDataArray2DColor4f;
typedef XSI::CDataArray2D< XSI::MATH::CShape> CDataArray2DShape;

template<> SICPPSDK_INLINE siICENodeDataType CDataArray2DFloat::GetDefaultType( ){return siICENodeDataFloat;}
SICPPSDK_INLINE siICENodeDataType CDataArray2DBool::GetDefaultType( ){return siICENodeDataBool;}
template<> SICPPSDK_INLINE siICENodeDataType CDataArray2DLong::GetDefaultType( ){return siICENodeDataLong;}
template<> SICPPSDK_INLINE siICENodeDataType CDataArray2DVector2f::GetDefaultType( ){return siICENodeDataVector2;}
template<> SICPPSDK_INLINE siICENodeDataType CDataArray2DVector3f::GetDefaultType( ){return siICENodeDataVector3;}
template<> SICPPSDK_INLINE siICENodeDataType CDataArray2DVector4f::GetDefaultType( ){return siICENodeDataVector4;}
template<> SICPPSDK_INLINE siICENodeDataType CDataArray2DQuaternionf::GetDefaultType( ){return siICENodeDataQuaternion;}
template<> SICPPSDK_INLINE siICENodeDataType CDataArray2DMatrix3f::GetDefaultType( ){return siICENodeDataMatrix33;}
template<> SICPPSDK_INLINE siICENodeDataType CDataArray2DMatrix4f::GetDefaultType( ){return siICENodeDataMatrix44;}
template<> SICPPSDK_INLINE siICENodeDataType CDataArray2DColor4f::GetDefaultType( ){return siICENodeDataColor4;}
template<> SICPPSDK_INLINE siICENodeDataType CDataArray2DRotationf::GetDefaultType( ){return siICENodeDataRotation;}
template<> SICPPSDK_INLINE XSI::siICENodeDataType CDataArray2DShape::GetDefaultType( ){return siICENodeDataShape;}
}; // XSI

#endif // __XSIDATAARRAY2D_H__
