//*****************************************************************************
/*!
   \file xsi_iceattributedataarray.h
   \brief CICEAttributeDataArray classes declaration.

    Copyright 1998-2007 Avid Technology, Inc. and its licensors. All rights
   reserved. This file contains confidential and proprietary information of
   Avid Technology, Inc., and is subject to the terms of the SOFTIMAGE|XSI
   end user license agreement (or EULA).
*/
//*****************************************************************************

#if (_MSC_VER > 1000) || defined(SGI_COMPILER)
#pragma once
#endif

#ifndef __XSIICEATTRIBUTEDATAARRAY_H__
#define __XSIICEATTRIBUTEDATAARRAY_H__

#include <sicppsdk.h>
#include <xsi_iceattributedataarray.h>
#include <xsi_vector2f.h>
#include <xsi_vector3f.h>
#include <xsi_vector4f.h>
#include <xsi_quaternionf.h>
#include <xsi_rotationf.h>
#include <xsi_matrix3f.h>
#include <xsi_matrix4f.h>
#include <xsi_color4f.h>
#include <xsi_indexset.h>

namespace XSI {

class ICEAttribute;

class CBaseICEAttributeDataArray
{
public:
	friend class ICEAttribute;

	CBaseICEAttributeDataArray( XSI::siICENodeDataType in_dataType, XSI::siICENodeStructureType in_structType )
		:	m_dataType( in_dataType ),
			m_structType( in_structType ),
			m_nCount(0),
			m_nHandle(0),
			m_pData(NULL),
			m_bIsConstant(false)
	{}

	virtual ~CBaseICEAttributeDataArray()
	{
		m_attrib.ReleaseDataArray( *this );
	}

protected:
	const void* GetDataPtr( ) const
	{
		return (const void*)m_pData;
	}

	ULONG GetHandle() const
	{
		return m_nHandle;
	}

	XSI::siICENodeDataType GetDataType() const
	{
		return m_dataType;
	}

	XSI::siICENodeStructureType GetStructureType() const
	{
		return m_structType;
	}

	void*& GetDataRef( )
	{
		return m_pData;
	}

	ULONG& GetHandleRef()
	{
		return m_nHandle;
	}

	ULONG& GetCountRef()
	{
		return m_nCount;
	}

	ICEAttribute& GetICEAttributeRef()
	{
		return m_attrib;
	}

	CBitsetHelper& GetBitsetRef()
	{
		return m_bitset;
	}

	bool& GetConstantRef()
	{
		return m_bIsConstant;
	}

protected:
	CStatus GetSubDataArray( ULONG in_index, CBaseICEAttributeDataArray& out_dataArray ) const
	{
		CStatus st = m_attrib.GetSubDataArray( *this, in_index, out_dataArray );
		assert( st == CStatus::OK );
		return st;
	}

	ULONG m_nCount;
	CBitsetHelper m_bitset;
	bool m_bIsConstant;

private:
	CBaseICEAttributeDataArray( const CBaseICEAttributeDataArray& );

	ULONG m_nHandle;
	void* m_pData;
	XSI::siICENodeStructureType m_structType;
	XSI::siICENodeDataType m_dataType;
	ICEAttribute m_attrib;
};

//*****************************************************************************
/*! \class CICEAttributeDataArray xsi_iceattributedataarray.h
	\brief This template class encapsulates ICEAttribute data as a 1D array. %CICEAttributeDataArray
	objects are read-only and can be filled with the methods supplied with the ICEAttribute class.
	%CICEAttributeDataArray is zero-based and can be one of the following types:

	\li \ref CICEAttributeDataArrayFloat "CICEAttributeDataArrayFloat"
	\li \ref CICEAttributeDataArrayLong "CICEAttributeDataArrayLong"
	\li \ref CICEAttributeDataArrayBool "CICEAttributeDataArrayBool"
	\li \ref CICEAttributeDataArrayVector2f "CICEAttributeDataArrayVector2f"
	\li \ref CICEAttributeDataArrayVector3f "CICEAttributeDataArrayVector3f"
	\li \ref CICEAttributeDataArrayVector4f "CICEAttributeDataArrayVector4f"
	\li \ref CICEAttributeDataArrayQuaternionf  "CICEAttributeDataArrayQuaternionf"
	\li \ref CICEAttributeDataArrayRotationf "CICEAttributeDataArrayRotationf"
	\li \ref CICEAttributeDataArrayMatrix3f "CICEAttributeDataArrayMatrix3f"
	\li \ref CICEAttributeDataArrayMatrix4f "CICEAttributeDataArrayMatrix4f"
	\li \ref CICEAttributeDataArrayColor4f "CICEAttributeDataArrayColor4f"
	\li \ref CICEAttributeDataArrayShape "CICEAttributeDataArrayShape"

	All types are associated to specific ICEAttribute types. Therefore, you need to declare the right
	array object type that matches the ICEAttribute data type you want to access. Otherwise a runtime
	error will occur and the returned array will be empty.

	\sa ICEAttribute::GetDataArray, ICEAttribute::GetDataArrayChunk,
		\ref CICEAttributeDataArrayTypedefs "Type Definitions for CICEAttributeDataArray"
	\since 7.0

	\eg This example demonstrates how to iterate over the PointPosition attribute data on a grid primitive.
	\code
		using namespace XSI;
		CValue CreatePrim( const CString& in_presetobj, const CString& in_geometrytype, const CString& in_name, const CString& in_parent );

		X3DObject grid = CreatePrim( L"Grid", L"MeshSurface", L"", L"");

		ICEAttribute attr = grid.GetActivePrimitive().GetGeometry().GetICEAttributeFromName( L"PointPosition" );

		CICEAttributeDataArrayVector3f points;
		attr.GetDataArray( points );

		Application xsi;
		for( ULONG i=0; i<points.GetCount( ); i++ )
		{
			xsi.LogMessage( CString( points[ i ] ) );
		}

		// Helper
		CValue CreatePrim( const CString& in_presetobj, const CString& in_geometrytype, const CString& in_name, const CString& in_parent )
		{
			CValueArray args(4);
			CValue retval;
			args[0]= in_presetobj;
			args[1]= in_geometrytype;
			args[2]= in_name;
			args[3]= in_parent;

			Application app;
			app.ExecuteCommand( L"CreatePrim", args, retval );
			return retval;
		}

	\endcode
 */
//*****************************************************************************
template<class T>
class CICEAttributeDataArray : public CBaseICEAttributeDataArray
{
public:

	/*! Constructor.*/
	CICEAttributeDataArray( ) : CBaseICEAttributeDataArray( GetDefaultType( ), XSI::siICENodeStructureSingle ) {}

	/*! Destructor.	*/
	~CICEAttributeDataArray( ) {}

	/*! Accessor to the encapsulated array. This operator is called when reading the data so the return value is read-only.
	\param in_index Index in the array. The index must be smaller than the number of elements in the array, otherwise the
	results are unpredictable. If the array is constant, the function always return the first item's value.
	\return A read-only reference to the indexed item.
	*/
	const T& operator[]( ULONG in_index ) const
	{
		const T* pData = (const T*)GetDataPtr();
		assert( pData != NULL || in_index < GetCount() );

		static T defVal;
		return pData ? ( m_bIsConstant ? *pData : pData[in_index] ) : defVal;
	}

	/*! Accessor to the number of elements in the array.
	\return Number of elements in the array.
	*/
	ULONG GetCount() const
	{
		return m_nCount;
	}

	/*! Returns true if the array is constant in which case the array values are the same.
	\return True if array is constant or false otherwise.
	*/
	bool IsConstant() const
	{
		return m_bIsConstant;
	}

private:
	static XSI::siICENodeDataType GetDefaultType( );

};

//*****************************************************************************
// NB: Do not use the Doxygen \class tag for this template!!!
/*!
	\brief This class is a specialization of a CICEAttributeDataArray class of type \c bool.

	\sa CICEAttributeDataArray2D, ICEAttribute, \ref CICEAttributeDataArrayBool "CICEAttributeDataArrayBool typedef"
	\since 7.0
 */
//*****************************************************************************
template<>
class CICEAttributeDataArray<bool> : public CBaseICEAttributeDataArray
{
public:
	/*! Constructor.*/
	CICEAttributeDataArray<bool>( ) :
		CBaseICEAttributeDataArray( GetDefaultType( ), XSI::siICENodeStructureSingle )
	{}

	/*! Destructor.	*/
	~CICEAttributeDataArray<bool>( ) {}

	/*! Accessor to the bool array. This operator is called when reading the data so the return value is read-only.
	\param in_index Index in the array. The index must be smaller than the number of elements in the array, otherwise the
	results are unpredictable.
	\return A read-only value to the indexed item.
	*/
	const bool operator[]( ULONG in_index ) const
	{
		return m_bitset.GetBit( in_index );
	}

	/*! Accessor to the number of elements in the array.
	\return Number of elements in the array.
	*/
	ULONG GetCount() const
	{
		return m_nCount;
	}

private:
	static SICPPSDK_INLINE XSI::siICENodeDataType GetDefaultType( )
	{
		return siICENodeDataBool;
	}
};

/*! \page CICEAttributeDataArrayTypedefs Type Definitions for CICEAttributeDataArray
	The CICEAttributeDataArray class is a template class that encapsulates the following data arrays:

	\li \ref CICEAttributeDataArrayFloat "CICEAttributeDataArrayFloat"
	\li \ref CICEAttributeDataArrayLong "CICEAttributeDataArrayLong"
	\li \ref CICEAttributeDataArrayBool "CICEAttributeDataArrayBool"
	\li \ref CICEAttributeDataArrayVector2f "CICEAttributeDataArrayVector2f"
	\li \ref CICEAttributeDataArrayVector3f "CICEAttributeDataArrayVector3f"
	\li \ref CICEAttributeDataArrayVector4f "CICEAttributeDataArrayVector4f"
	\li \ref CICEAttributeDataArrayQuaternionf "CICEAttributeDataArrayQuaternionf"
	\li \ref CICEAttributeDataArrayRotationf "CICEAttributeDataArrayRotationf"
	\li \ref CICEAttributeDataArrayMatrix3f "CICEAttributeDataArrayMatrix3f"
	\li \ref CICEAttributeDataArrayMatrix4f "CICEAttributeDataArrayMatrix4f"
	\li \ref CICEAttributeDataArrayColor4f "CICEAttributeDataArrayColor4f"
	\li \ref CICEAttributeDataArrayShape "CICEAttributeDataArrayShape"


	\section CICEAttributeDataArrayFloat CICEAttributeDataArrayFloat Type
	\code typedef XSI::CICEAttributeDataArray< float > CICEAttributeDataArrayFloat \endcode
	A CICEAttributeDataArray class of type \c float.
	\since 7.0

	\section CICEAttributeDataArrayLong CICEAttributeDataArrayLong Type
	\code typedef XSI::CICEAttributeDataArray< LONG > CICEAttributeDataArrayLong \endcode
	A CICEAttributeDataArray class of type \c LONG.
	\since 7.0

	\section CICEAttributeDataArrayBool CICEAttributeDataArrayBool Type
	\code typedef XSI::CICEAttributeDataArray< bool > CICEAttributeDataArrayBool \endcode
	The CICEAttributeDataArray< bool > class.
	\since 7.0

	\section CICEAttributeDataArrayVector2f CICEAttributeDataArrayVector2f Type
	\code typedef XSI::CICEAttributeDataArray< CVector2f > CICEAttributeDataArrayVector2f \endcode
	A CICEAttributeDataArray class of type \link MATH::CVector2f CVector2f\endlink.
	\since 7.0

	\section CICEAttributeDataArrayVector3f CICEAttributeDataArrayVector3f Type
	\code typedef XSI::CICEAttributeDataArray< CVector3f > CICEAttributeDataArrayVector3f \endcode
	A CICEAttributeDataArray class of type \link MATH::CVector3f CVector3f\endlink.
	\since 7.0

	\section CICEAttributeDataArrayVector4f CICEAttributeDataArrayVector4f Type
	\code typedef XSI::CICEAttributeDataArray< CVector4f > CICEAttributeDataArrayVector4f \endcode
	A CICEAttributeDataArray class of type \link MATH::CVector4f CVector4f\endlink.
	\since 7.0

	\section CICEAttributeDataArrayQuaternionf CICEAttributeDataArrayQuaternionf Type
	\code typedef XSI::CICEAttributeDataArray< Quaternionf > CICEAttributeDataArrayQuaternionf \endcode
	A CICEAttributeDataArray class of type \link MATH::CQuaternionf CQuaternionf\endlink.
	\since 7.0

	\section CICEAttributeDataArrayRotationf CICEAttributeDataArrayRotationf Type
	\code typedef XSI::CICEAttributeDataArray< CRotationf > CICEAttributeDataArrayRotationf \endcode
	A CICEAttributeDataArray class of type \link MATH::CRotationf CRotationf\endlink.
	\since 7.0

	\section CICEAttributeDataArrayMatrix3f CICEAttributeDataArrayMatrix3f Type
	\code typedef XSI::CICEAttributeDataArray< CMatrix3f > CICEAttributeDataArrayMatrix3f \endcode
	A CICEAttributeDataArray class of type \link MATH::CMatrix3f CMatrix3f\endlink.
	\since 7.0

	\section CICEAttributeDataArrayMatrix4f CICEAttributeDataArrayMatrix4f Type
	\code typedef XSI::CICEAttributeDataArray< CMatrix4f > CICEAttributeDataArrayMatrix4f \endcode
	A CICEAttributeDataArray class of type \link MATH::CMatrix4f CMatrix4f\endlink.
	\since 7.0

	\section CICEAttributeDataArrayColor4f CICEAttributeDataArrayColor4f Type
	\code typedef XSI::CICEAttributeDataArray< CColor4f > CICEAttributeDataArrayColor4f \endcode
	A CICEAttributeDataArray class of type \link MATH::CColor4f CColor4f\endlink.
	\since 7.0

	\section CICEAttributeDataArrayShape CICEAttributeDataArrayShape Type
	\code \typedef XSI::CICEAttributeDataArray< CShape > CICEAttributeDataArrayShape \endcode
	A CICEAttributeDataArray class of type \link MATH::CShape CShape\endlink.
	\since 7.0
*/
typedef XSI::CICEAttributeDataArray< float > CICEAttributeDataArrayFloat;
typedef XSI::CICEAttributeDataArray< LONG > CICEAttributeDataArrayLong;
typedef XSI::CICEAttributeDataArray< bool > CICEAttributeDataArrayBool;
typedef XSI::CICEAttributeDataArray< XSI::MATH::CVector2f > CICEAttributeDataArrayVector2f;
typedef XSI::CICEAttributeDataArray< XSI::MATH::CVector3f > CICEAttributeDataArrayVector3f;
typedef XSI::CICEAttributeDataArray< XSI::MATH::CVector4f > CICEAttributeDataArrayVector4f;
typedef XSI::CICEAttributeDataArray< XSI::MATH::CQuaternionf > CICEAttributeDataArrayQuaternionf;
typedef XSI::CICEAttributeDataArray< XSI::MATH::CRotationf > CICEAttributeDataArrayRotationf;
typedef XSI::CICEAttributeDataArray< XSI::MATH::CMatrix3f > CICEAttributeDataArrayMatrix3f;
typedef XSI::CICEAttributeDataArray< XSI::MATH::CMatrix4f > CICEAttributeDataArrayMatrix4f;
typedef XSI::CICEAttributeDataArray< XSI::MATH::CColor4f > CICEAttributeDataArrayColor4f;
typedef XSI::CICEAttributeDataArray< XSI::MATH::CShape> CICEAttributeDataArrayShape;

template<> SICPPSDK_INLINE XSI::siICENodeDataType CICEAttributeDataArrayFloat::GetDefaultType( ){return siICENodeDataFloat;}
template<> SICPPSDK_INLINE XSI::siICENodeDataType CICEAttributeDataArrayLong::GetDefaultType( ){return siICENodeDataLong;}
template<> SICPPSDK_INLINE XSI::siICENodeDataType CICEAttributeDataArrayVector2f::GetDefaultType( ){return siICENodeDataVector2;}
template<> SICPPSDK_INLINE XSI::siICENodeDataType CICEAttributeDataArrayVector3f::GetDefaultType( ){return siICENodeDataVector3;}
template<> SICPPSDK_INLINE XSI::siICENodeDataType CICEAttributeDataArrayVector4f::GetDefaultType( ){return siICENodeDataVector4;}
template<> SICPPSDK_INLINE XSI::siICENodeDataType CICEAttributeDataArrayQuaternionf::GetDefaultType( ){return siICENodeDataQuaternion;}
template<> SICPPSDK_INLINE XSI::siICENodeDataType CICEAttributeDataArrayMatrix3f::GetDefaultType( ){return siICENodeDataMatrix33;}
template<> SICPPSDK_INLINE XSI::siICENodeDataType CICEAttributeDataArrayMatrix4f::GetDefaultType( ){return siICENodeDataMatrix44;}
template<> SICPPSDK_INLINE XSI::siICENodeDataType CICEAttributeDataArrayColor4f::GetDefaultType( ){return siICENodeDataColor4;}
template<> SICPPSDK_INLINE XSI::siICENodeDataType CICEAttributeDataArrayRotationf::GetDefaultType( ){return siICENodeDataRotation;}
template<> SICPPSDK_INLINE XSI::siICENodeDataType CICEAttributeDataArrayShape::GetDefaultType( ){return siICENodeDataShape;}

};

#endif // __XSIICEATTRIBUTEDATAARRAY_H__
