//*****************************************************************************
/*!
   \file xsi_icenodecontext.h
   \brief ICENodeContext class declaration.

    Copyright 1998-2003 Avid Technology, Inc. and its licensors. All rights
   reserved. This file contains confidential and proprietary information of
   Avid Technology, Inc., and is subject to the terms of the SOFTIMAGE|XSI
   end user license agreement (or EULA).
*/
//*****************************************************************************

#if (_MSC_VER > 1000) || defined(SGI_COMPILER)
#pragma once
#endif

#ifndef __XSIICENODECONTEXT_H__
#define __XSIICENODECONTEXT_H__

#include <xsi_context.h>
#include <xsi_time.h>

namespace XSI {

class CStatus;
class CBaseDataArray;
class CBaseDataArray2D;
class CIndexSet;

//*****************************************************************************
/*! \class ICENodeContext xsi_icenodecontext.h
	\brief %ICENodeContext is used for accessing/handling custom node graph data. Instances
	of this object can pass information to the C++ API callbacks that implement the custom
	node plug-in item.

	\sa ICENode, PluginRegistrar::RegisterICENode, \xt cb_icenodes %ICENode Callbacks \endxt
	\since 7.0
*/
//*****************************************************************************

class SICPPSDKDECL ICENodeContext : public Context
{
public:

	friend class CBaseDataArray;
	friend class CBaseDataArray2D;
	friend class CIndexSet;

	/*! Default constructor. */
	ICENodeContext();

	/*! Default destructor. */
	~ICENodeContext();

	/*! Constructor.
	\param in_ref constant reference object.
	*/
	ICENodeContext(const CRef& in_ref);

	/*! Copy constructor.
	\param in_obj constant class object.
	*/
	ICENodeContext(const ICENodeContext& in_obj);

	/*! Returns true if a given class type is compatible with this API class.
	\param in_ClassID class type.
	\return true if the class is compatible, false otherwise.
	*/
	bool IsA( siClassID in_ClassID) const;

	/*! Returns the type of the API class.
	\return The class type.
	*/
	siClassID GetClassID() const;

	/*! Creates an object from another object.
	\param in_obj constant class object.
	\return The new ICENodeContext object.
	*/
	ICENodeContext& operator=(const ICENodeContext& in_obj);

	/*! Creates an object from a reference object. The newly created object is
	set to empty if the input reference object is not compatible.
	\param in_ref constant class object.
	\return The new ICENodeContext object.
	*/
	ICENodeContext& operator=(const CRef& in_ref);

	/*! Returns the number of instances for a given group.
	\param in_grpUniqID %Group index to query
	\retval out_inst_count Number of instances
	\return CStatus::OK success
	*/
	CStatus GetGroupInstanceCount( ULONG in_grpUniqID, ULONG& out_inst_count ) const;

	/*! Returns the number of threads used during the \xt cb_ICENode_Evaluate Evaluate \endxt callback.
	\return Thread count.
	*/
	ULONG GetEvaluationThreadCount( ) const;

	/*! Returns the current evaluation thread index. If called from \xt cb_ICENode_Evaluate Evaluate \endxt,
	the index ranges between 0 and GetEvaluationThreadCount()-1. Returns 0 if called from either the
	\xt cb_ICENode_BeginEvaluate BeginEvaluate \endxt or \xt cb_ICENode_EndEvaluate EndEvaluate \endxt callback.
	\return ULONG Thread index.
	*/
	ULONG GetCurrentThreadIndex( ) const;

	/*! Returns the total number of elements to process during the evaluation of an ICENode. Returns 0 if the
	operation fails. The return value can be set by the custom ICENode if an output port is set with the
	::siICENodeContextElementGenerator context type.
	\return ULONG Number of elements.
	\sa ICENodeContext::PutNumberOfElementsToProcess, ICENodeDef::AddOutputPort
	*/
	ULONG GetNumberOfElementsToProcess( ) const;

	/*! Sets the number elements to generate for an element generator node. The new value will affect the current
	output port being evaluated.
	\param in_nElements Number of elements. The value is ignored if the current port has not been set with the
	::siICENodeContextElementGenerator context type.
	\return CStatus::OK Success.
	\return CStatus::Fail Operation failed.
	\sa ICENodeContext::GetNumberOfElementsToProcess, ICENodeDef::AddOutputPort

	\eg %This example shows how to set the number of elements to generate for an element generator type of ICENode.
	The \xt cb_ICENode_BeginEvaluate BeginEvaluate \endxt callback must be used for calling
	ICENodeContext::PutNumberOfElementsToProcess.
	\code

		XSIPLUGINCALLBACK CStatus MyICENodeGenerator_BeginEvaluate( ICENodeContext& in_ctxt )
		{
			// Get some values from an input port
			CDataArrayLong inSize( in_ctxt, ID_Size );

			// Total number of elements to generate
			ULONG nSize = inSize[ 0 ];
			ULONG nElements = nSize * nSize;
			in_ctxt.PutNumberOfElementsToProcess( nElements );

			return CStatus::OK;
		}
	\endcode
	*/
	CStatus PutNumberOfElementsToProcess( ULONG in_nElements );

	/*! Returns the connection information for the specified ICENodePort.
	\param in_nPortID Index of the port to query
	\retval out_type Data type for this port (see ICENodePort::GetDataType)
	\retval out_struct Structure for this port (see ICENodePort::GetStructureType)
	\retval out_context Context for this port (see ICENodePort::GetContextType)
	\return CStatus::OK success
	*/
	CStatus GetPortInfo( ULONG in_nPortID, siICENodeDataType& out_type, siICENodeStructureType& out_struct, siICENodeContextType& out_context ) const;

	/*! Returns the unique identifier of the output port being evaluated.
	\return CStatus::OK success
	*/
	ULONG GetEvaluatedOutputPortID( ) const;

	/*! Returns the evaluation time. For a custom ICENode, the evaluation time is not necessarily the same as the current
	scene time or that it is evaluated just once per frame.
	\return CTime Evaluation time
	*/
	CTime GetTime() const;

	private:

	// API for handling the data array
	CStatus AcquireIndexSet( CIndexSet& io_set ) const;
	CStatus ReleaseIndexSet( CIndexSet& io_set );
	CStatus RemoveElementFromIndexSet( CIndexSet& io_set, LONG in_index, LONG& out_nNewIndex, bool& out_bEnd );
	CStatus AcquireInputDataArray( CBaseDataArray& io_dataArray, siICENodeDataType in_requiredDataType, ULONG in_port_uniq_index, ULONG in_grp_instance_index ) const;
	CStatus AcquireOutputDataArray( CBaseDataArray& io_dataArray, siICENodeDataType in_requiredDataType ) const;
	CStatus ReleaseDataArray( CBaseDataArray& io_dataArray );

	// API for handling the 2D data array
	CStatus AcquireInputDataArray2D( CBaseDataArray2D& io_dataArray, siICENodeDataType in_requiredDataType, ULONG in_port_uniq_index, ULONG in_grp_instance_index ) const;
	CStatus AcquireOutputDataArray2D( CBaseDataArray2D& io_dataArray, siICENodeDataType in_requiredDataType ) const;
	CStatus ResizeSubArray( CBaseDataArray2D& io_dataArray, ULONG in_nArrayIndex, ULONG in_nSize, void*& out_ppData, ULONG& out_nCount, ULONG& out_nStartBit ) ;
	CStatus GetSubArray( CBaseDataArray2D& io_dataArray, ULONG in_nArrayIndex, void*& out_ppData, ULONG& out_nCount, ULONG& out_nStartBit ) const;

	ICENodeContext * operator&() const;
	ICENodeContext * operator&();
};

};
#endif // __XSIICENODECONTEXT_H__
