using System;
using System.Collections.Generic;
using System.Text;
using System.Xml;
using StatsParse;

namespace StatsCharts
{
	class CAttachmentStats
	{
		public int m_used;
		public int m_shots;
		public int m_hits;
		public int m_kills;
		public int m_deaths;
		public int m_reloads;
		public double m_damage;
	}

	class CAttachmentContext
	{
		public int m_beginTime, m_endTime;
		public ITimelineValue[] m_attachmentTL;
		public ITimelineValue[] m_weaponTL;
	};

	public class CAttachmentInfoWorker : IStatsChartsWorker
	{
		public EChartKind GetChartKind()
		{
			return EChartKind.Attachment_info;
		}

		string GetCurrentAttachment( string sock, int time, CAttachmentContext context )
		{
			foreach ( CPrmTimelineValue wtv in context.m_weaponTL )
				if ( time >= wtv.GetTime() )
				{
					string weaponName = wtv.m_prm;
					foreach ( CAttachmentTimelineValue atv in context.m_attachmentTL )
						if ( time >= atv.GetTime()  )
							if ( atv.m_socket == sock && atv.m_target == weaponName )
								return atv.m_name;
				}

			return "none";
		}

		int GetAttachmentUsedTime(string sock, string att, CAttachmentContext context)
		{
			int time = 0, attCur = 0, attBgnTime = 0, attEndTime = 0;
			int attLen = context.m_attachmentTL.Length;

			while( attCur < attLen )
			{
				CAttachmentTimelineValue atv = (CAttachmentTimelineValue)context.m_attachmentTL[attCur];
				string longAttName = MakeLongAttName( sock, atv.m_name );
				if (atv.m_socket == sock && longAttName == att)
				{
					attBgnTime = atv.m_time;

					bool endFound = false;
					for (int i = attCur; i < attLen; ++i)
					{
						CAttachmentTimelineValue tmpV = (CAttachmentTimelineValue)context.m_attachmentTL[i];
						string longTmpName = MakeLongAttName( sock, tmpV.m_name );
						if (tmpV.m_socket == sock && longTmpName != att)
						{
							attCur = i;
							attEndTime = tmpV.m_time;
							endFound = true;
							break;
						}
					}

					if (!endFound)
					{
						attEndTime = context.m_endTime;
						attCur = attLen;//last iteration
					}
				
					string weapon = atv.m_target;
					bool weapInUse = false;
					int weapBgnTime = 0, weapEndTime = 0;
					foreach (CPrmTimelineValue wtv in context.m_weaponTL)
					{
						//TODO: can be optimized through earlier cut with (attBgnTime, attEndTime)
						if (!weapInUse)
						{
							if (wtv.m_prm == weapon)
							{
								weapBgnTime = wtv.m_time;
								weapInUse = true;
							}
						}
						else
						{
							if (wtv.m_prm != weapon)
							{
								weapEndTime = wtv.m_time;
								weapInUse = false;

								time+= StatsChartsUtils.GetIntersectionLength(attBgnTime, attEndTime, weapBgnTime, weapEndTime);
							}
						}
					}
					if (weapInUse)
						time += StatsChartsUtils.GetIntersectionLength(attBgnTime, attEndTime, weapBgnTime, context.m_endTime);
			
				}//end of foreach (CPrmTimelineValue wtv...
				++attCur;
			}//end of while( attCur...

			return time;
		}

		private string MakeLongAttName(string sock, string shortAttName)
		{
			return "(" + sock + ")" + shortAttName;
		}

		private string GetAndAddCurrentAttachment( string sock, int time, CAttachmentContext context, Dictionary<string, CAttachmentStats> stats )
		{
			string curAtt = MakeLongAttName( sock, GetCurrentAttachment(sock, time, context) );
			if (!stats.ContainsKey(curAtt))
				stats[curAtt] = new CAttachmentStats();
			return curAtt;
		}

		public void MakeChartXml(Dictionary<string, CSessionRoot> sessions, SFiterData filter, string outFileName)
		{
			string[] socketNames =
			{
				"barrel",
				"scope",
				"muzzle",
				"clip",
				"butt",
				"underbarrel"
			};
			Dictionary<string, CAttachmentStats> stats = new Dictionary<string, CAttachmentStats>();

			foreach ( KeyValuePair<string, CSessionRoot> snKvp in sessions )
			{
				if (snKvp.Value.m_roundData == null)
					continue;

				foreach (CRoundData rd in snKvp.Value.m_roundData)
				{
					if (rd.m_playerLifes == null)
						continue;

					foreach ( CPlayerLife pl in rd.m_playerLifes )
					{
						ITimelineValue[] weaponTL = StatsChartsUtils.GetTimeline("weapon", pl);
						if (weaponTL.Length == 0)
						{
							//System.Diagnostics.Debug.Assert( "Unexpected: empty weapon timeline".Length == 0 );
							continue;
						}

						int ltBegin = 0, ltEnd = 0;
						if ( !pl.GetLifetime( ref ltBegin, ref ltEnd ) )
						{
							//System.Diagnostics.Debug.Assert( "Unexpected: incorrect lifetime timeline".Length == 0 );
							continue;
						}

						ITimelineValue[] attTL			= StatsChartsUtils.GetTimeline("attachment",		pl);
						ITimelineValue[] shotTL			= StatsChartsUtils.GetTimeline("shot",					pl);
						ITimelineValue[] hitTL			= StatsChartsUtils.GetTimeline("hit",					pl);
						ITimelineValue[] killTL			= StatsChartsUtils.GetTimeline("kill",					pl);
						ITimelineValue[] deathTL		= StatsChartsUtils.GetTimeline("death",				pl);
						ITimelineValue[] reloadTL		= StatsChartsUtils.GetTimeline("reload",				pl);

						CAttachmentContext ctxt = new CAttachmentContext();
						ctxt.m_beginTime = ltBegin;
						ctxt.m_endTime	 = ltEnd;
						ctxt.m_attachmentTL = attTL;
						ctxt.m_weaponTL			= weaponTL;

						foreach ( string sock in socketNames )
						{
							string curAtt;
							foreach ( CShotTimelineValue val in shotTL )
							{
								curAtt = GetAndAddCurrentAttachment(sock, val.GetTime(), ctxt, stats);
								stats[curAtt].m_shots+= 1;
							}

							foreach (CHitTimelineValue val in hitTL)
							{
								curAtt = GetAndAddCurrentAttachment(sock, val.GetTime(), ctxt, stats);
								stats[curAtt].m_hits+= 1;

								stats[curAtt].m_damage+= val.m_damage;
							}

							foreach (CKillTimelineValue val in killTL)
							{
								curAtt = GetAndAddCurrentAttachment(sock, val.GetTime(), ctxt, stats);
								stats[curAtt].m_kills+= 1;
							}

							foreach (CDeathTimelineValue val in deathTL)
							{
								curAtt = GetAndAddCurrentAttachment(sock, val.GetTime(), ctxt, stats);
								stats[curAtt].m_deaths+= 1;
							}

							foreach (CPrmTimelineValue val in reloadTL)
							{
								curAtt = GetAndAddCurrentAttachment(sock, val.GetTime(), ctxt, stats);
								stats[curAtt].m_reloads+= 1;
							}

							//used time should be collected at the end, to ensure all attachments present in stats
							foreach ( KeyValuePair<string, CAttachmentStats> kvp in stats )
							{
								curAtt = kvp.Key;
								stats[curAtt].m_used+= GetAttachmentUsedTime(sock, curAtt, ctxt);
							}
						}//end of foreach ( string sock...
					}
				}
			}

			XmlDocument doc = new XmlDocument();
			XmlElement root = doc.CreateElement("stats");
			
			SortedDictionary<string, CAttachmentStats> sortedStats = new SortedDictionary<string, CAttachmentStats>(stats);
			foreach( KeyValuePair<string, CAttachmentStats> kvp in sortedStats )
			{
				CAttachmentStats val = kvp.Value;
				XmlElement child = doc.CreateElement("attachment");
				StatsChartsUtils.AddAttribute( child, "a_name",			kvp.Key,				doc );
				StatsChartsUtils.AddAttribute( child, "b_used",			val.m_used,			doc );
				StatsChartsUtils.AddAttribute( child, "c_shots",		val.m_shots,		doc );
				StatsChartsUtils.AddAttribute( child, "d_hits",			val.m_hits,			doc );
				StatsChartsUtils.AddAttribute( child, "e_kills",		val.m_kills,		doc );
				StatsChartsUtils.AddAttribute( child, "f_deaths",		val.m_deaths,		doc );
				StatsChartsUtils.AddAttribute( child, "g_reloads",	val.m_reloads,	doc );
				double noTailDamage = Math.Round(val.m_damage);//to remove troubles with floating point delimiter
				StatsChartsUtils.AddAttribute( child, "h_damage",		noTailDamage,		doc );
				root.AppendChild(child);
			}
			doc.AppendChild(root);
			doc.Save(outFileName);
		}

	}
}
