﻿using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
using System.Reflection;
using System.Diagnostics;
using System.Xml;
using StatsDataSource.Storage;
using StatsDataSource.ObjectModel;

namespace StatsDataSource.Core
{
	public class StatsCore
	{
		public readonly string Build;

		readonly List<string> exceptions = new List<string>();

		public List<IStatsPlugin> Plugins;

		public StatsRepository Repository;

		string[] m_args;
		public string[] CmdArgs { get { return m_args; } }

		//////////////////////////////////////////////////////////////////////////

		private List<GameScope> m_currentScopes = new List<GameScope>();
		public List<GameScope> CurrentScopes 
		{ 
			get { return m_currentScopes; }
			set
			{
				m_currentScopes = value;

				if (ScopesChanged != null)
					ScopesChanged(this, m_currentScopes);

				CurrentNodes.Clear();
				foreach (GameScope s in value)
					CurrentNodes.Add(s);

				if (NodesChanged != null)
					NodesChanged(this, m_currentNodes);
			}
		}
		public delegate void ScopesChangedDeleg(StatsCore sender, List<GameScope> scopes);
		public event ScopesChangedDeleg ScopesChanged;

		//////////////////////////////////////////////////////////////////////////

		private List<GameElement> m_currentElements = new List<GameElement>();
		public List<GameElement> CurrentElements
		{
			get { return m_currentElements; }
			set
			{
				m_currentElements = value;
				if(ElementsChanged != null)
					ElementsChanged(this, m_currentElements);

				CurrentNodes.Clear();
				foreach (GameElement e in value)
					CurrentNodes.Add(e);

				if (NodesChanged != null)
					NodesChanged(this, m_currentNodes);
			}
		}
		public delegate void ElementsChangedDeleg(StatsCore sender, List<GameElement> elements);
		public event ElementsChangedDeleg ElementsChanged;

		//////////////////////////////////////////////////////////////////////////

		private List<GameNode> m_currentNodes = new List<GameNode>();
		public List<GameNode> CurrentNodes
		{
			get { return m_currentNodes; }
			set
			{
				m_currentNodes = value;
				if (NodesChanged != null)
					NodesChanged(this, m_currentNodes);
			}
		}
		public delegate void NodesChangedDeleg(StatsCore sender, List<GameNode> nodes);
		public event NodesChangedDeleg NodesChanged;

		//////////////////////////////////////////////////////////////////////////

		public StatsCore(string build, XmlReader format, string processingDef, string dataAccDef, out List<string> errors)
		{
			Build = build;

			exceptions.Add("StatsDataSource.dll");
			exceptions.Add("ZzzzRangeBar.dll");
			exceptions.Add("squishinterface_Win32.dll");
			exceptions.Add("squishinterface_x64.dll");
			exceptions.Add("SlimDX.dll");
			exceptions.Add("Microsoft.Office.Interop.Excel.dll");
			exceptions.Add("Microsoft.Vbe.Interop.dll");
			exceptions.Add("office.dll");

			Plugins = new List<IStatsPlugin>();
			StatsRegistry Registry = new StatsRegistry(format, processingDef, dataAccDef, out errors);
			Repository = new StatsRepository(Registry);
		}

		//////////////////////////////////////////////////////////////////////////

		#region Lifetime

		public void Start(IMainForm mainForm, string[] args)
		{
			m_args = args;
			int bias = 0;
			foreach (IStatsPlugin p in Plugins)
			{
				p.Start(this, mainForm, bias);
				bias += p.getUsedTagCount();
			}
		}

		public void Stop()
		{
			foreach (IStatsPlugin p in Plugins)
				p.Stop();
		}

		public void OnIdle()
		{
			foreach (IStatsPlugin p in Plugins)
				p.HandleCoreMessage(ECoreMessage.Idle);
		}

		#endregion

		//////////////////////////////////////////////////////////////////////////

		#region Plugin loading

		private bool ImplementsPluginInterface(Type t)
		{
			Type[] interfaces = t.GetInterfaces();

			foreach (Type i in interfaces)
			{
				if (i.Equals(typeof(IStatsPlugin)) || ImplementsPluginInterface(i))
					return true;
			}

			return false;
		}

		private void LoadPluginsFromAssembly(Assembly assembly)
		{
			Type[] types = assembly.GetExportedTypes();

			foreach (Type t in types)
			{
				TypeAttributes attrs = t.Attributes;
				
				if ( (attrs & TypeAttributes.Interface) != TypeAttributes.Interface
					&& (attrs & TypeAttributes.Abstract) != TypeAttributes.Abstract
					&& ImplementsPluginInterface(t))
				{
					try
					{
						IStatsPlugin p = (IStatsPlugin)assembly.CreateInstance(t.FullName);
						Plugins.Add(p);
					}
					catch
					{						
					}
				}
			}
		}

		public void LoadPlugins()
		{
			string dir = Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location);
			string[] dlls = Directory.GetFiles(dir, "*.dll");

			foreach (string dll in dlls)
			{
				string name = Path.GetFileName(dll);
				if(exceptions.Contains(name))
					continue;

				Assembly assembly = Assembly.LoadFrom(dll);
				LoadPluginsFromAssembly(assembly);
			}
		}

		public void UnloadPlugin(IStatsPlugin p)
		{
			Plugins.Remove(p);
		}
		#endregion

		//////////////////////////////////////////////////////////////////////////

	}
}
