﻿using System;
using System.Collections.Generic;
using System.Text;
using StatsDataSource.Parsing;
using StatsDataSource.Storage;

namespace StatsDataSource.DataAccumulation
{
	class CallbackImpl : IDataAccParseCallback
	{
		#region Members

		private DataAccParser m_parser;

		private StatsRegistry m_registry;

		private Dictionary<string, ContextBuilder[]> StatToContextList;

		public Dictionary<string, StatProcSettings> StatsProcSettings;

		public CallbackImpl(DataAccParser parser, StatsRegistry registry)
		{
			m_parser = parser;
			m_registry = registry;
			StatToContextList = new Dictionary<string, ContextBuilder[]>();
			StatsProcSettings = new Dictionary<string, StatProcSettings>();
		}

		#endregion

		#region IDataAccParseCallback

		public Token OnDefinitionList(object list, object definition)
		{
			return new Token();
		}

		#region Context

		public Token OnContextSet(Token name, object contextlist)
		{
			if (StatToContextList.ContainsKey(name.sVal))
				m_parser.SetError("Multiple context sets with same names found");

			StatToContextList.Add(name.sVal, ((List<ContextBuilder>)contextlist).ToArray());
			
			return new Token();
		}

		public Token OnContextList(object list, object context)
		{
			List<ContextBuilder> l = (List<ContextBuilder>)list;
			if (l == null)
				l = new List<ContextBuilder>();

			l.Add((ContextBuilder)context);

			return new Token(l);
		}

		public Token OnContext(object elements)
		{
			List<KeyValuePair<string, DataPath>> elems = (List<KeyValuePair<string, DataPath>>)elements;
			ContextBuilder context = new ContextBuilder(elements != null ? elems.ToArray() : null);
			return new Token(context);
		}

		public Token OnContextElementList(object list, object element)
		{
			List<KeyValuePair<string, DataPath>> l = (List<KeyValuePair<string, DataPath>>)list;
			if (l == null)
				l = new List<KeyValuePair<string, DataPath>>();

			l.Add((KeyValuePair<string, DataPath>)element);

			return new Token(l);
		}

		public Token OnContextElement(Token identifier, Token path)
		{
			string id = identifier.sVal;
			List<IDataPathElem> pathelems = (List<IDataPathElem>)path.oVal;
			DataPath respath = new DataPath(pathelems.ToArray());

			if (!respath.Validate(PathType.Num))
				m_parser.SetError("Invalid path construction");

			return new Token(new KeyValuePair<string, DataPath>(id, respath));
		}

		#endregion

		#region Stats
		private EOperationType GetOperationType(Tokens type)
		{
			switch (type)
			{
				case Tokens.OP_SET: return EOperationType.Set;
				case Tokens.OP_MIN: return EOperationType.Min;
				case Tokens.OP_MAX: return EOperationType.Max;
				case Tokens.OP_AVG: return EOperationType.Avg;
				case Tokens.OP_SUM: return EOperationType.Sum;
				case Tokens.OP_FIRST: return EOperationType.First;
				case Tokens.OP_LAST: return EOperationType.Last;
				default: throw new NotImplementedException();
			}
		}

		public Token OnStatDef(Token name, Token cntset, object source, object operation)
		{
			StatProcSettings settings = (StatProcSettings)source;

			ContextBuilder[] ctxs;
			if (!StatToContextList.TryGetValue(cntset.sVal, out ctxs))
				m_parser.SetError("Unresolved context '" + cntset.sVal + '\'');

			settings.Contexts = ctxs;
			settings.StatName = name.sVal;
			settings.DBOperation = (EOperationType)operation;
			StatsProcSettings.Add(settings.EventName, settings);
			return new Token(settings);
		}

		public Token OnStatSource(Tokens operation, Token path)
		{
			EOperationType op = GetOperationType(operation);

			List<IDataPathElem> pathelems = (List<IDataPathElem>)path.oVal;

			if (!(new DataPath(pathelems.ToArray()).Validate(PathType.Player)))
				m_parser.SetError("Invalid path construction");

			while (pathelems.Count != 0 && pathelems[0].Type != PathType.Events)
				pathelems.RemoveAt(0);

			if (pathelems.Count == 0)
				m_parser.SetError("Invalid path construction");

			string eventName = ((PathEvents)pathelems[0]).Selectors[0];
			pathelems.RemoveAt(0);

			DataPath respath = new DataPath(pathelems.ToArray());
			
			return new Token(new StatProcSettings(null, null, eventName, respath, op, EOperationType.Set));
		}

		public Token OnStatOperation(Tokens operation)
		{
			return new Token(GetOperationType(operation));
		}

		#endregion

		#region Path

		private IDataPathElem CreatePathElem(Tokens type, List<string> selectorList)
		{
			switch (type)
			{
				case Tokens.PLAYER: return new PathPlayer();
				case Tokens.SCOPES: return new PathScopes(selectorList);
				case Tokens.ELEMENTS: return new PathElements(selectorList);
				case Tokens.STATES: return new PathStates(selectorList);
				case Tokens.EVENTS: return new PathEvents(selectorList);
				default: throw new NotImplementedException();
			}
		}

		public Token OnPath(object path, Token element)
		{
			List<IDataPathElem> l = (List<IDataPathElem>)path;
			if (l == null)
				l = new List<IDataPathElem>();

			l.Add((IDataPathElem)element.oVal);

			return new Token(l);
		}

		public Token OnPathElement(Token name)
		{
			return new Token(new PathLiteral(name.sVal));
		}

		public Token OnPathElement(Tokens predefinedId, object selectorList)
		{
			return new Token(CreatePathElem(predefinedId, (List<string>)selectorList));
		}

		public Token OnCollectionSelectorList(object list, object selector)
		{
			List<string> l = (List<string>)list;
			if (l == null)
				l = new List<string>();

			l.Add((string)selector);

			return new Token(l);
		}

		public Token OnCollectionSelector(Token identifier)
		{
			return new Token((object)identifier.sVal);
		}

		#endregion

		#endregion
	}
}
