﻿using System;
using System.Collections.Generic;
using System.Text;
using StatsDataSource.ObjectModel;
using System.Diagnostics;

namespace StatsDataSource.DataAccumulation
{
	enum PathType
	{
		Player,
		Scopes,
		Elements,
		States,
		Events,
		Literal,

		Num,
	}

	////////////////////////////////////////////////////////////

	abstract class IDataPathElem
	{
		public readonly PathType Type;

		public IDataPathElem(PathType type) { Type = type; }

		public abstract bool Validate(PathType preceding);

		public abstract object Step(object preceding);
	}

	////////////////////////////////////////////////////////////

	abstract class IDataCollectionPathElem : IDataPathElem
	{
		public List<string> Selectors = new List<string>();

		public IDataCollectionPathElem(PathType type, List<string> selectors) 
			: base(type) 
		{ 
			Selectors = selectors; 
		}
	}


	////////////////////////////////////////////////////////////

	class DataPath
	{
		public IDataPathElem[] Elements;

		public DataPath(IDataPathElem[] elems)
		{
			Elements = elems;
		}

		public bool Validate(PathType startType)
		{
			PathType prev = startType;

			for (int i = 0; i != Elements.Length; ++i)
			{
				if (!Elements[i].Validate(prev))
					return false;

				prev = Elements[i].Type;
			}

			return prev == PathType.Literal || prev == PathType.States;
		}

		public string GetValue(object context)
		{
			for (int i = 0; i != Elements.Length; ++i)
				context = Elements[i].Step(context);

			return context != null ? context.ToString() : "";
		}
	}

	////////////////////////////////////////////////////////////

	#region Paths

	class PathPlayer : IDataPathElem
	{
		public PathPlayer() : base(PathType.Player) { }

		public override bool Validate(PathType preceding)
		{
			return preceding == PathType.Num;
		}

		public override object Step(object preceding)
		{
			Debug.Assert(preceding is GameElement);
			return preceding;
		}
	}

	////////////////////////////////////////////////////////////

	class PathScopes : IDataCollectionPathElem
	{
		public PathScopes(List<string> selector) : base(PathType.Scopes, selector) { }

		public override bool Validate(PathType preceding)
		{
			return (preceding == PathType.Num || preceding == PathType.Player) && Selectors.Count == 1;
		}

		public override object Step(object preceding)
		{
			Debug.Assert(preceding is GameNode);

			if (preceding == null)
				return null;

			GameScope sc = preceding is GameScope ? (GameScope)preceding : ((GameElement)preceding).Parent;

			while (sc != null && sc.Desc.Name != Selectors[0])
				sc = sc.Parent;

			return sc;
		}
	}

	////////////////////////////////////////////////////////////

	class PathElements : IDataCollectionPathElem
	{
		public PathElements(List<string> selector) : base(PathType.Elements, selector) { }

		public override bool Validate(PathType preceding)
		{
			return preceding == PathType.Scopes;
		}

		public override object Step(object preceding)
		{
			throw new NotImplementedException();
		}
	}

	////////////////////////////////////////////////////////////

	class PathStates : IDataCollectionPathElem
	{
		public PathStates(List<string> selectors) : base(PathType.States, selectors) { }

		public override bool Validate(PathType preceding)
		{
			return (preceding == PathType.Scopes || preceding == PathType.Elements || preceding == PathType.Player) && Selectors.Count > 0;
		}

		public override object Step(object preceding)
		{
			Debug.Assert(preceding is GameNode);

			if (preceding == null)
				return null;

			GameNode n = (GameNode)preceding;

			GameState st = n.FindState(Selectors[0]);

			for (int i = 1; st != null && i < Selectors.Count; ++i)
			{
				st = st.Children.Find(x => x.Desc.Name == Selectors[i]);
			}

			return st != null ? st.Value : null;
		}
	}

	////////////////////////////////////////////////////////////

	class PathEvents : IDataCollectionPathElem
	{
		public PathEvents(List<string> selector) : base(PathType.Events, selector) { }

		public override bool Validate(PathType preceding)
		{
			return preceding == PathType.Scopes || preceding == PathType.Elements || preceding == PathType.Player;
		}

		public override object Step(object preceding)
		{
			throw new NotImplementedException();
		}
	}

	////////////////////////////////////////////////////////////

	class PathLiteral : IDataPathElem
	{
		public readonly string Literal;

		public PathLiteral(string lt) : base(PathType.Literal) 
		{
			Literal = lt;
		}

		public override bool Validate(PathType preceding)
		{
			return preceding == PathType.Events;
		}

		public override object Step(object preceding)
		{
			Debug.Assert(preceding is GameEvent);

			if (preceding == null)
				return null;

			return ((GameEvent)preceding).Parameters[Literal];
		}
	}

	////////////////////////////////////////////////////////////

	#endregion
}
