﻿using System;
using System.Collections.Generic;
using System.Text;

namespace StatsDataSource.DataAccumulation
{
	public enum EOperationType
	{
		Set,
		Min,
		Max,
		Sum,
		Avg,
		First,
		Last,
	}

	abstract class IOperation
	{
		public readonly EOperationType Type;
		public IOperation(EOperationType type) { Type = type; Reset(); }
		public abstract void Reset();
		public abstract void Evaluate(string val);
		public abstract string GetResult();

		static public IOperation Create(EOperationType type)
		{
			switch (type)
			{
				case EOperationType.Min: return new OperationMin();
				case EOperationType.Max: return new OperationMax();
				case EOperationType.Sum: return new OperationSum();
				case EOperationType.Avg: return new OperationAvg();
				case EOperationType.First: return new OperationFirst();
				case EOperationType.Last: return new OperationLast();
				default: return null;
			}
		}
	}

	#region Operations

	class OperationMin : IOperation
	{
		private float m_val;

		public OperationMin() : base(EOperationType.Min) { }

		public override void Reset()
		{
			m_val = float.MaxValue;
		}

		public override void Evaluate(string val)
		{
			float v = float.Parse(val);
			m_val = Math.Min(m_val, v);
		}

		public override string GetResult()
		{
			return m_val.ToString();
		}
	}

	class OperationMax : IOperation
	{
		private float m_val;

		public OperationMax() : base(EOperationType.Max) { }

		public override void Reset()
		{
			m_val = float.MinValue;
		}

		public override void Evaluate(string val)
		{
			float v = float.Parse(val);
			m_val = Math.Max(m_val, v);
		}

		public override string GetResult()
		{
			return m_val.ToString();
		}
	}

	class OperationSum : IOperation
	{
		private float m_sum;

		public OperationSum() : base(EOperationType.Sum) { }

		public override void Reset()
		{
			m_sum = 0;
		}

		public override void Evaluate(string val)
		{
			float v = float.Parse(val);
			m_sum += v;
		}

		public override string GetResult()
		{
			return m_sum.ToString();
		}
	}

	class OperationAvg : IOperation
	{
		private float m_sum;
		private int m_num;

		public OperationAvg() : base(EOperationType.Avg) { }

		public override void Reset()
		{
			m_sum = 0;
			m_num = 0;
		}

		public override void Evaluate(string val)
		{
			float v = float.Parse(val);
			m_sum += v;
			++m_num;
		}

		public override string GetResult()
		{
			return (m_sum / m_num).ToString();
		}
	}

	class OperationFirst : IOperation
	{
		private string m_val;

		public OperationFirst() : base(EOperationType.First) { }

		public override void Reset()
		{
			m_val = null;
		}

		public override void Evaluate(string val)
		{
			if (m_val != null)
				return;
			m_val = val;
		}

		public override string GetResult()
		{
			return m_val;
		}
	}

	class OperationLast : IOperation
	{
		private string m_val;

		public OperationLast() : base(EOperationType.Last) { }

		public override void Reset()
		{
			m_val = null;
		}

		public override void Evaluate(string val)
		{
			m_val = val;
		}

		public override string GetResult()
		{
			return m_val;
		}
	}

	#endregion
}
