﻿using System;
using System.Collections.Generic;
using System.Text;
using StatsDataSource.Storage;
using StatsDataSource.Preprocessing;
using StatsDataSource.DataAccumulation;

namespace StatsDataSource.ObjectModel
{
	public class GameEventGroup : Taggable
	{
		public readonly StatDesc Desc;

		WeakReference m_node;
		public GameNode Node { get { return (GameNode)m_node.Target; } private set { m_node = new WeakReference(value); } }

		public List<GameEvent> Events;

		public EFilterState FilterState;

		internal CrawlerContext CrawlerContext;

		internal Accumulator DataAccumulator;

		internal GameEventGroup(StatDesc desc, GameNode node)
		{
			FilterState = EFilterState.Unaffected;
			Desc = desc;
			Node = node;
			Events = new List<GameEvent>();
		}

		public GameEvent AddEvent(Int64 time, KeyValuePair<string, string>[] parameters)
		{
			GameEvent ev = new GameEvent(this, time, parameters);

			if (Events.Count == 0 || time >= Events[Events.Count - 1].TimeMillisecs)
			{
				Events.Add(ev);
			}
			else
			{
				int index = UpperBound(time);
				Events.Insert(index, ev);
			}

			Node.Repository.OnEventAdded(ev);
			return ev;
		}

		public GameEvent FindClosest(Int64 time)
		{
			int i = Find(time);
			if(Events[i].TimeMillisecs < time)
				++i;

			if (i == Events.Count)
				return Events.Count == 0 ? null : Events[i - 1];

			if (i == 0)
				return Events[i];

			GameEvent e1 = Events[i - 1];
			GameEvent e2 = Events[i];
			Int64 t1 = time - e1.TimeMillisecs;
			Int64 t2 = e2.TimeMillisecs - time;
			return t1 > t2 ? e2 : e1;
		}

		public int Find(Int64 time)
		{
			int high, i, low;
			for (low = -1, high = Events.Count - 1;  high - low > 1; )
			{
				i = (high + low) / 2;
        if (time <= Events[i].TimeMillisecs)
					high = i;
        else
					low  = i;
      }

			return high;
		}

		public int LowerBound(Int64 time)
		{
			int i = Find(time);

			if (Events[i].TimeMillisecs < time)
				return i + 1;

			while (i > 1 && Events[i - 1].TimeMillisecs == time)
				--i;

			return i;
		}

		public int UpperBound(Int64 time)
		{
			int i = Find(time);

			if (Events[i].TimeMillisecs < time)
				return i + 1;

			while (i != Events.Count && Events[i].TimeMillisecs == time)
				++i;

			return i;
		}

		public void EqualRange(Int64 time, out int lower, out int upper)
		{
			lower = Find(time);
			upper = lower;

			if (Events[lower].TimeMillisecs < time)
			{
				upper = ++lower;
				return;
			}

			while (lower > 1 && Events[lower - 1].TimeMillisecs == time)
				--lower;

			while (upper < Events.Count - 1 && Events[upper].TimeMillisecs == time)
				++upper;
		}

		public void AcceptVisitor(IStatsVisitor visitor)
		{
			if (visitor.VisitEventGroup(this))
			{
				foreach (GameEvent e in Events)
					visitor.VisitEvent(e);
			}
			visitor.LeaveEventGroup(this);
		}

		public void AcceptVisitorFiltered(IStatsVisitor visitor)
		{
			if (visitor.VisitEventGroup(this))
			{
				foreach (GameEvent e in Events)
					if (e.FilterState != EFilterState.Failed)
						visitor.VisitEvent(e);
			}
			visitor.LeaveEventGroup(this);
		}

		//////////////////////////////////////////////////////////////////////////

		public class EventTimeComparer : IComparer<GameEvent>
		{
			public int Compare(GameEvent e1, GameEvent e2)
			{
				if (e1.Time < e2.Time) return -1;
				if (e1.Time == e2.Time) return 0;
				return 1;
			}
		}

		//////////////////////////////////////////////////////////////////////////
	}
}
