﻿using System;
using System.Collections.Generic;
using System.Text;
using StatsDataSource.Storage;
using System.Diagnostics;

namespace StatsDataSource.ObjectModel
{
	////////////////////////////////////////////////////////////

	public class Taggable
	{
		internal static int DefaultTagNum = 1;

		private object[] m_tags;

		public object GetTag(int id)
		{
			return (m_tags != null && id < m_tags.Length)
				? m_tags[id]
				: null;
		}

		public void SetTag(int id, object value)
		{
			if (m_tags == null)
			{
				m_tags = new object[Math.Max(DefaultTagNum, id + 1)];
			}
			else if (id >= m_tags.Length)
			{
				object[] nt = new object[id + 1];
				Array.Copy(m_tags, nt, m_tags.Length);
				m_tags = nt;
			}

			m_tags[id] = value;
		}
	}


	////////////////////////////////////////////////////////////


	public abstract class GameNode : Taggable
	{
		public abstract string Name { get; }

		public readonly int Depth;

		public readonly Locator Locator;

		WeakReference m_parent;
		public GameScope Parent { get { return (GameScope)m_parent.Target; } internal set { m_parent = new WeakReference(value); } }

		public EFilterState FilterState;

		public Dictionary<int, GameState> States;

		public Dictionary<int, GameEventGroup> EventGroups;

		public abstract void AcceptVisitor(IStatsVisitor visitor);

		public abstract void AcceptVisitorFiltered(IStatsVisitor visitor);

		internal readonly StatsRepository Repository;

		////////////////////////////////////////////////////////////

		protected GameNode(Locator loc, GameScope parent, StatsRepository repo)
		{
			FilterState = EFilterState.Unaffected;
			Locator = loc;
			Parent = parent;
			Depth = parent == null ? 0 : (parent.Depth + 1);
			Repository = repo;
			States = new Dictionary<int, GameState>();
			EventGroups = new Dictionary<int, GameEventGroup>();
		}

		public GameState AddState(StatDesc desc, string value)
		{
			Debug.Assert(desc != null);
			Debug.Assert(FindState(desc) == null);

			GameState state = new GameState(desc, value, this, null);
			States[desc.ID] = state;
			Repository.OnStateAdded(state);
			return state;
		}

		public GameState FindState(StatDesc desc)
		{
			GameState s = null;
			States.TryGetValue(desc.ID, out s);
			return s;
		}

		public GameState FindState(string name)
		{
			StatDesc desc = Repository.Registry.GetStateDesc(name);
			if (desc == null)
				return null;

			GameState s = null;
			States.TryGetValue(desc.ID, out s);
			return s;
		}

		public GameEventGroup AddEventGroup(StatDesc desc)
		{
			Debug.Assert(desc != null);
			if (desc == null)
				return null;

			Debug.Assert(FindEventGroup(desc) == null);

			GameEventGroup evntGr = new GameEventGroup(desc, this);
			EventGroups[desc.ID] = evntGr;
			Repository.OnEventGroupAdded(evntGr);
			return evntGr;
		}

		public GameEventGroup FindEventGroup(StatDesc desc)
		{
			GameEventGroup eg = null;
			EventGroups.TryGetValue(desc.ID, out eg);
			return eg;
		}

		protected void VisitStates(IStatsVisitor visitor)
		{
			foreach (KeyValuePair<int, GameState> st in States)
				if (st.Value.Children.Count == 0)
					st.Value.AcceptVisitor(visitor);

			foreach (KeyValuePair<int, GameState> st in States)
				if (st.Value.Children.Count > 0)
					st.Value.AcceptVisitor(visitor);
		}

		protected void VisitStatesFiltered(IStatsVisitor visitor)
		{
			foreach (KeyValuePair<int, GameState> st in States)
				if (st.Value.Children.Count == 0)
					st.Value.AcceptVisitorFiltered(visitor);

			foreach (KeyValuePair<int, GameState> st in States)
				if (st.Value.Children.Count > 0)
					st.Value.AcceptVisitorFiltered(visitor);
		}

		protected void VisitEvents(IStatsVisitor visitor)
		{
			if (visitor.StartEventGroups(this))
			{
				foreach (KeyValuePair<int, GameEventGroup> g in EventGroups)
					g.Value.AcceptVisitor(visitor);
			}
			visitor.EndEventGroups(this);
		}

		protected void VisitEventsFiltered(IStatsVisitor visitor)
		{
			if (visitor.StartEventGroups(this))
			{
				foreach (KeyValuePair<int, GameEventGroup> g in EventGroups)
					g.Value.AcceptVisitorFiltered(visitor);
			}
			visitor.EndEventGroups(this);
		}

		////////////////////////////////////////////////////////////
	}

	////////////////////////////////////////////////////////////
	////////////////////////////////////////////////////////////

}
