﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Diagnostics;
using StatsDataSource.Storage;

namespace StatsDataSource.ObjectModel
{
	public class GameScope : GameNode
	{
		////////////////////////////////////////////////////////////

		public override string Name { get { return Desc.Name; } }
		public readonly ScopeDesc Desc;
		public Dictionary<Locator, GameElement> Elements;
		public Dictionary<Locator, GameScope> ChildScopes;

		////////////////////////////////////////////////////////////

		internal GameScope(Locator loc, GameScope parent, ScopeDesc desc, StatsRepository repo)
			: base(loc, parent, repo)
		{
			Debug.Assert(loc.IsScope());
			Debug.Assert(desc != null);

			Desc = desc;
			Elements = new Dictionary<Locator, GameElement>();
			ChildScopes = new Dictionary<Locator, GameScope>();
		}

		public override void AcceptVisitor(IStatsVisitor visitor)
		{
			if (visitor.VisitScope(this))
			{
				VisitStates(visitor);
				VisitEvents(visitor);

				foreach (GameElement el in Elements.Values)
					el.AcceptVisitor(visitor);

				foreach (GameScope sc in ChildScopes.Values)
					sc.AcceptVisitor(visitor);
			}
			visitor.LeaveScope(this);
		}

		public override void AcceptVisitorFiltered(IStatsVisitor visitor)
		{
			if (FilterState == EFilterState.Failed)
				return;

			if (visitor.VisitScope(this))
			{
				VisitStatesFiltered(visitor);
				VisitEventsFiltered(visitor);

				foreach (GameElement el in Elements.Values)
					el.AcceptVisitorFiltered(visitor);

				foreach (GameScope sc in ChildScopes.Values)
					sc.AcceptVisitorFiltered(visitor);
			}
			visitor.LeaveScope(this);
		}

		public GameNode AddChild(Locator locator)
		{
			Debug.Assert(FindChild(locator) == null);

			GameNode child;
			if (locator.IsScope())
			{
				child = new GameScope(locator, this, Repository.Registry.GetScopeDesc(locator.ScopeID), Repository);
				ChildScopes.Add(child.Locator, (GameScope)child);
			}
			else
			{
				child = new GameElement(locator, this, Repository.Registry.GetElementDesc(locator.ElementID), Repository);
				Elements.Add(child.Locator, (GameElement)child);
			}

			Repository.OnNodeAdded(child);
			return child;
		}

		public GameNode FindChild(Locator locator)
		{
			GameElement el; GameScope sc;
			if (Elements.TryGetValue(locator, out el))
				return el;
			else if (ChildScopes.TryGetValue(locator, out sc))
				return sc;
			return null;
		}

		internal void RemoveChild(GameNode child)
		{
			if (child.Locator.IsScope())
				ChildScopes.Remove(child.Locator);
			else
				Elements.Remove(child.Locator);
		}

		////////////////////////////////////////////////////////////
	}
}
