﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Diagnostics;
using StatsDataSource.Storage;
using StatsDataSource.Parsing;

namespace StatsDataSource.Preprocessing
{
	/// <summary>
	/// Implementation of parser callback
	/// which constructs the data crawlers.
	/// </summary>
	public class CallbackImpl : IEventParseCallback
	{
		////////////////////////////////////////////////////////////

		#region Members

		public Dictionary<string, Crawler> Crawlers;

		public Dictionary<string, object> CrawlerTargets;

		private EProcParser m_parser;

		private StatsRegistry m_registry;

		public CallbackImpl(EProcParser parser, StatsRegistry registry)
		{
			m_parser = parser;
			m_registry = registry;
			Crawlers = new Dictionary<string, Crawler>();
			CrawlerTargets = new Dictionary<string, object>();
		}

		#endregion

		////////////////////////////////////////////////////////////

		#region Definition

		////////////////////////////////////////////////////////////

		public Token OnDefinitionList(object list, object definition) 
		{
			if (definition == null)
				return new Token();

			if (list == null)
				list = Crawlers;

			Dictionary<string, Crawler> l = (Dictionary<string, Crawler>)list;
			Crawler c = (Crawler)definition;

			l.Add(c.TargetEvent, c);

			return new Token(list); 
		}

		////////////////////////////////////////////////////////////

		public Token OnDefinition(Token header, object actions) 
		{
			List<ICrawlerAction> act = (List<ICrawlerAction>)actions;
			return new Token(
				new Crawler(
					header.sVal, 
					act != null ? act.ToArray() : null
				)); 
		}

		////////////////////////////////////////////////////////////

		public Token OnDefinitionHeader(Tokens type, string name) 
		{
			if (type != Tokens.EVENT)
				m_parser.SetError("unexpected type");

			return new Token(name);
		}

		////////////////////////////////////////////////////////////

		#endregion

		////////////////////////////////////////////////////////////

		#region Actions

		////////////////////////////////////////////////////////////

		public Token OnActionList(object list, Token action) 
		{
			if (list == null)
				list = new List<ICrawlerAction>();

			List<ICrawlerAction> l = (List<ICrawlerAction>)list;
			l.Add((ICrawlerAction)action.oVal);

			return new Token(list); 
		}

		////////////////////////////////////////////////////////////

		public Token OnAction(Tokens specialAction) 
		{ 
			return new Token(new CActionInsertAllOwn());
		}

		////////////////////////////////////////////////////////////

		public Token OnAction(Token target, Token path) 
		{
			List<IPathElement> p = (List<IPathElement>)path.oVal;
			IPathElement[] pa = p.ToArray();

			if (!CrawlerPath.ValidatePath(pa))
				m_parser.SetError("the path is invalid");

			return new Token(new CActionExtract( target.sVal, new CrawlerPath(pa) ));
		}

		////////////////////////////////////////////////////////////

		public Token OnActionTarget(Token target)
		{
			return target;
		}

		////////////////////////////////////////////////////////////

		#endregion

		////////////////////////////////////////////////////////////

		#region Path

		////////////////////////////////////////////////////////////

		private IPathElement CreatePathElement(Tokens tk, CollSelector selector, string literal)
		{
			switch (tk)
			{
				case Tokens.THIS:					return new PathThis();
				case Tokens.OWNER:				return new PathOwner();
				case Tokens.SCOPES:				return selector != null ? new PathScopes(selector) : null;
				case Tokens.ELEMENTS:			return selector != null ? new PathElements(selector) : null;
				case Tokens.STATES:				return selector != null ? new PathStates(selector) : null;
				case Tokens.EVENTS:				return selector != null ? new PathEvents(selector) : null;
				case Tokens.IDENTIFIER:		return new PathLiteral(literal);
				default: m_parser.SetError("invalid token found in path"); return null;
			}
		}

		////////////////////////////////////////////////////////////

		public Token OnPath(object path, Token element) 
		{
			if (path == null)
				path = new List<IPathElement>();

			List<IPathElement> p = (List<IPathElement>)path;
			p.AddRange((List<IPathElement>)element.oVal);

			return new Token(path);
		}

		////////////////////////////////////////////////////////////

		public Token OnPathElement(Token name) 
		{
			if(name.Type != TokenType.STRING)
				m_parser.SetError("invalid token found in path, string expected");

			List<IPathElement> elems = new List<IPathElement>();
			elems.Add(new PathLiteral(name.sVal));

			return new Token(elems);
		}

		////////////////////////////////////////////////////////////

		public Token OnPathElement(Tokens predefinedId, object selectorList) 
		{
			List<IPathElement> elems = new List<IPathElement>();
			List<CollSelector> selectors = (List<CollSelector>)selectorList;

			if (selectors == null)
			{
				IPathElement el = CreatePathElement(predefinedId, null, null);
				elems.Add(el);
			}
			else
			{
				foreach (CollSelector s in selectors)
				{
					IPathElement el = CreatePathElement(predefinedId, s, null);
					elems.Add(el);
				}
			}

			if (predefinedId == Tokens.EVENTS)
				foreach (CollSelector s in selectors)
					foreach (string t in s.item_names)
						if (!CrawlerTargets.ContainsKey(t))
							CrawlerTargets.Add(t, null);

			return new Token(elems);
		}

		////////////////////////////////////////////////////////////

		#endregion

		////////////////////////////////////////////////////////////

		#region Collection selector

		////////////////////////////////////////////////////////////

		public Token OnCollectionSelectorList(object list, object selector) 
		{
			if (list == null)
				list = new List<CollSelector>();

			List<CollSelector> l = (List<CollSelector>)list;
			l.Add((CollSelector)selector);

			return new Token(l); 
		}

		////////////////////////////////////////////////////////////

		public Token OnCollectionSelector(object types, object condition) 
		{
			return new Token(
				new CollSelector(
					((List<string>)types).ToArray(),
					null));
		}

		////////////////////////////////////////////////////////////

		public Token OnSelectedTypeList(object list, Token typeName) 
		{
			if (typeName.Type != TokenType.STRING)
				m_parser.SetError("invalid item name type in selector");

			if (list == null)
				list = new List<string>();

			List<string> l = (List<string>)list;
			l.Add(typeName.sVal);

			return new Token(list);
		}

		////////////////////////////////////////////////////////////

		#endregion

		////////////////////////////////////////////////////////////

		#region ConditionalExpr

		////////////////////////////////////////////////////////////

		public Token OnConditionalExpr(Token term) 
		{
			throw new NotImplementedException(); 
		}

		////////////////////////////////////////////////////////////

		public Token OnConditionalExpr(Token expr, Tokens relOp, Token term) 
		{ 
			throw new NotImplementedException(); 
		}

		////////////////////////////////////////////////////////////

		#endregion

		////////////////////////////////////////////////////////////
	}
}
