﻿using System;
using System.Collections.Generic;
using System.Text;
using StatsDataSource.ObjectModel;
using StatsDataSource.Storage;

namespace StatsDataSource.Preprocessing
{
	/// <summary>
	/// Crawler class is responsible for processing
	/// specific event group by applying a set of 
	/// actions to each event.
	/// 
	/// Crawler instances are shared between all 
	/// instances of one event group type so should
	/// be stateless.
	/// 
	/// All cached data and state info should be
	/// stored in CrawlerContext.
	/// </summary>
	public class Crawler
	{
		public string TargetEvent;

		public ICrawlerAction[] Actions;

		public Crawler(string target, ICrawlerAction[] actions)
		{
			TargetEvent = target;
			Actions = actions;
		}

		public void EstablishConnections(CrawlerContext context)
		{
			GameEventGroup g = context.Group;

			for (int i = 0; i != Actions.Length; ++i)
				if (context.CachedActionInfo[i] == null)
					context.CachedActionInfo[i] = Actions[i].Connect(context, g);
		}

		public void Process(CrawlerContext context)
		{
			for (int i = 0; i != Actions.Length; ++i)
				Actions[i].Execute(context, context.CachedActionInfo[i]);
		}
	}



	/// <summary>
	/// Context object defines the state
	/// where crawler should operate.
	/// 
	/// It can also be used to cache data.
	/// </summary>
	public class CrawlerContext
	{
		public StatsRegistry Registry;

		public Crawler Crawler;

		public object[] CachedActionInfo;

		public SortedList<string, string> ProcessedParameters;


		private WeakReference m_group;

		public GameEventGroup Group { get { return (GameEventGroup)m_group.Target; } }

		public int Position;

		public GameEvent Event { get { return (Position >= 0 && Position < Group.Events.Count) ? Group.Events[Position] : null; } }


		public Int64 Timestamp { get { return Position != -1 ? Group.Events[Position].TimeMillisecs : -1; } }
		
		public bool CanAdvance { get { return Group.Events.Count > (Position + 1); } }
		
		public Int64 NextTimestamp
		{
			get
			{
				GameEventGroup g = Group;
				int i = Position + 1;
				return g.Events.Count > i ? g.Events[i].TimeMillisecs : -1;
			}
		}


		public CrawlerContext(Crawler crawler, GameEventGroup group)
		{
			Crawler = crawler;
			Registry = group.Node.Repository.Registry;
			
			if(crawler != null)
				CachedActionInfo = new object[Crawler.Actions.Length];

			Position = -1;
			m_group = new WeakReference(group);
		}

		public bool Advance(Int64 time)
		{
			GameEventGroup group = Group;
			int pos = Position + 1;
			
			if (group.Events.Count <= pos || group.Events[pos].TimeMillisecs > time)
				return false;

			ProcessedParameters = new SortedList<string, string>();

			Position = pos;
			return true;
		}

		public void Process()
		{
			if (Crawler == null)
				return;

			Crawler.Process(this);

			Event.Parameters = ProcessedParameters;
		}
	}

}
