﻿using System;
using System.Collections.Generic;
using System.Text;
using StatsDataSource.ObjectModel;

namespace StatsDataSource.Preprocessing
{
	enum PathType
	{
		This,
		Owner,
		Scopes,
		Elements,
		States,
		Events,
		Literal,

		Num,
	}

	////////////////////////////////////////////////////////////

	class CollSelector
	{
		public CollSelector(string[] items, object rules)
		{
			item_names = items;
			item_ids = new int[items.Length];
			for (int i = 0; i != item_ids.Length; ++i)
				item_ids[i] = -1;
			selection_rules = rules;
		}

		public string[] item_names;
		public int[] item_ids;
		public object selection_rules;
	}

	////////////////////////////////////////////////////////////

	abstract class IPathElement
	{
		public readonly PathType Type;

		public IPathElement(PathType type)
		{
			Type = type;
		}

		public abstract bool Cacheable();

		public abstract bool Validate(PathType preceding);

		public abstract object Step(CrawlerContext context, object preceding);
	}

	////////////////////////////////////////////////////////////

	abstract class IPathElementCol : IPathElement
	{
		public readonly CollSelector Selector;

		public IPathElementCol(CollSelector selector, PathType type)
			: base(type)
		{
			Selector = selector;
		}

		protected int GetStatID(CrawlerContext context, string name, PathType type)
		{
			StatDesc desc = null;

			switch (type)
			{
				case PathType.Scopes:
					desc = context.Registry.GetScopeDesc(name);
					break;
				case PathType.Elements:
					desc = context.Registry.GetElementDesc(name);
					break;
				case PathType.States:
					desc = context.Registry.GetStateDesc(name);
					break;
				case PathType.Events:
					desc = context.Registry.GetEventDesc(name);
					break;
			}

			return desc != null ? desc.ID : -1;
		}
	}

	////////////////////////////////////////////////////////////

	#region Paths

	class PathThis : IPathElement
	{
		public PathThis() : base(PathType.This) { }

		public override bool Cacheable() { return true; }

		public override bool Validate(PathType preceding)
		{
			return preceding == PathType.Num;
		}

		public override object Step(CrawlerContext context, object preceding)
		{
			return preceding is GameEvent ? preceding : null;
		}
	}

	////////////////////////////////////////////////////////////

	class PathOwner : IPathElement
	{
		public PathOwner() : base(PathType.Owner) { }

		public override bool Cacheable() { return true; }

		public override bool Validate(PathType preceding)
		{
			return preceding == PathType.This || preceding == PathType.Num;
		}

		public override object Step(CrawlerContext context, object preceding)
		{
			GameEvent ge = preceding as GameEvent;
			if (ge != null) return ge.Group.Node;

			GameEventGroup geg = preceding as GameEventGroup;
			if (geg != null) return geg.Node;
				
			return null;
		}
	}

	////////////////////////////////////////////////////////////

	class PathScopes : IPathElementCol
	{
		public PathScopes(CollSelector selector) : base(selector, PathType.Scopes) 
		{ }

		public override bool Cacheable() { return true; }

		public override bool Validate(PathType preceding)
		{
			return preceding == PathType.Owner;
		}

		public override object Step(CrawlerContext context, object preceding)
		{
			if (Selector.item_ids[0] == -1)
				Selector.item_ids[0] = GetStatID(context, Selector.item_names[0], Type);

			int id = Selector.item_ids[0];
			if (id == -1) return null;

			GameNode node = preceding as GameNode;

			while (node != null)
			{
				if(!node.Locator.IsScope())
					node = node.Parent;

				if (node.Locator.ScopeID == id)
					break;

				node = node.Parent;
			}

			return node.Locator.ScopeID == id ? node : null;
		}
	}

	////////////////////////////////////////////////////////////

	class PathElements : IPathElementCol
	{
		public PathElements(CollSelector selector)
			: base(selector, PathType.Elements)
		{ }

		public override bool Cacheable() { return true; }

		public override bool Validate(PathType preceding)
		{
			return preceding == PathType.Scopes;
		}

		public override object Step(CrawlerContext context, object preceding)
		{
			throw new NotImplementedException();
		}
	}

	////////////////////////////////////////////////////////////

	class PathStates : IPathElementCol
	{
		public PathStates(CollSelector selector)
			: base(selector, PathType.States)
		{ }

		public override bool Cacheable() { return true; }

		public override bool Validate(PathType preceding)
		{
			return preceding == PathType.Owner || preceding == PathType.Elements || preceding == PathType.Scopes || preceding == PathType.States;
		}

		public override object Step(CrawlerContext context, object preceding)
		{
			GameState state = null;

			if (Selector.item_ids[0] == -1)
				Selector.item_ids[0] = GetStatID(context, Selector.item_names[0], Type);

			int id = Selector.item_ids[0];
			if (id == -1)
				return null;

			GameNode gn = preceding as GameNode;
			if (gn != null)
			{
				gn.States.TryGetValue(id, out state);
				return state;
			}

			state = preceding as GameState;
			if (state != null)
			{
				state = state.Children.Find(X => X.Desc.ID == id);
				return state;
			}

			return state;
		}
	}

	////////////////////////////////////////////////////////////

	class PathEvents : IPathElementCol
	{
		public PathEvents(CollSelector selector)
			: base(selector, PathType.Events)
		{ }

		public override bool Cacheable() { return true; }

		public override bool Validate(PathType preceding)
		{
			return preceding == PathType.Owner || preceding == PathType.Elements || preceding == PathType.Scopes;
		}

		public override object Step(CrawlerContext context, object preceding)
		{
			GameNode gn = preceding as GameNode;
			if (gn != null)
			{
				if (Selector.item_ids[0] == -1)
					Selector.item_ids[0] = GetStatID(context, Selector.item_names[0], Type);

				int id = Selector.item_ids[0];
				if (id == -1) return null;

				GameEventGroup g;
				gn.EventGroups.TryGetValue(id, out g);
				return g;
			}
			return null;
		}
	}

	////////////////////////////////////////////////////////////

	class PathLiteral : IPathElement
	{
		public readonly string Literal;

		public PathLiteral(string literal)
			: base(PathType.Literal)
		{
			Literal = literal;
		}

		public override bool Cacheable() { return false; }

		public override bool Validate(PathType preceding)
		{
			return preceding == PathType.Events || preceding == PathType.States || preceding == PathType.This || preceding == PathType.Num;
		}

		public override object Step(CrawlerContext context, object preceding)
		{
			string val = null;
			GameEvent ev = null;
			
			GameEventGroup eg = preceding as GameEventGroup;
			if (eg != null)
				ev = eg.CrawlerContext.Event;
			
			if(ev == null)
				ev = preceding as GameEvent;

			if (ev != null)
				ev.Parameters.TryGetValue(Literal, out val);

			return val;
		}
	}

	////////////////////////////////////////////////////////////

	#endregion

	////////////////////////////////////////////////////////////

	/// <summary>
	/// Path objects are used to locate the data
	/// for precessing by crawler actions.
	/// </summary>
	class CrawlerPath
	{
		struct CachedConnection
		{
			public object cachedValue;
			public int position;
		}

		public IPathElement[] Elements;

		public CrawlerPath(IPathElement[] elems)
		{
			Elements = elems;
		}

		static public bool ValidatePath(IPathElement[] path)
		{
			PathType prev = PathType.Num;

			for (int i = 0; i != path.Length; ++i)
			{
				if (!path[i].Validate(prev))
					return false;

				prev = path[i].Type;
			}

			return prev == PathType.Literal || prev == PathType.States;
		}

		public object Connect(CrawlerContext context, GameEventGroup group)
		{
			object curElem = group;

			int i;
			for (i = 0; i != Elements.Length; ++i)
			{
				object nextElem = Elements[i].Cacheable() ? Elements[i].Step(context, curElem) : null;

				if (nextElem == null)
					break;

				curElem = nextElem;
			}

			CachedConnection con = new CachedConnection();
			con.cachedValue = curElem == group ? null : curElem;
			con.position = i;
			return con;
		}

		public string FetchValue(CrawlerContext context, object cachedConnection)
		{
			CachedConnection connection = (CachedConnection)cachedConnection;

			if (connection.cachedValue == null)
				connection.cachedValue = context.Event;

			for (int i = connection.position; i != Elements.Length; ++i)
			{
				connection.cachedValue = Elements[i].Step(context, connection.cachedValue);
				if (connection.cachedValue == null)
					break;
			}

			return connection.cachedValue != null ? connection.cachedValue.ToString() : null;
		}
	}

}
