﻿using System;
using System.Collections.Generic;
using System.Text;
using StatsDataSource.ObjectModel;
using StatsDataSource.DataAccumulation;
using StatsDataSource.Preprocessing;

namespace StatsDataSource.Storage
{
	public class DataProcessor
	{
		WeakReference m_repository;

		StatsRepository Repository { get { return (StatsRepository)m_repository.Target; } }

		internal DataProcessor(StatsRepository repo)
		{
			m_repository = new WeakReference(repo);
		}

		internal void OnEventGroupAdded(GameEventGroup evntGroup)
		{
			StatsRegistry reg = Repository.Registry;

			// Associate crawler
			Crawler crawler;
			reg.Crawlers.TryGetValue(evntGroup.Desc.Name, out crawler);

			if (crawler != null || reg.CrawlersTargets.ContainsKey(evntGroup.Desc.Name))
				evntGroup.CrawlerContext = new CrawlerContext(crawler, evntGroup);

			// Associate accumulator
			StatProcSettings sps;
			if (ShouldAccumulate(evntGroup.Node) && reg.StatsProcSettings.TryGetValue(evntGroup.Desc.Name, out sps))
				evntGroup.DataAccumulator = new Accumulator(sps);
		}

		static internal bool ShouldAccumulate(GameNode node)
		{
			// TODO: make game-independent
			return node.Name == "player";
		}

		internal void UpdateAllProcessors()
		{
			ProcesoorUpdater upd = new ProcesoorUpdater();
			Repository.AcceptVisitor(upd);
		}

		internal void UpdateProcessorsIncremental(List<GameEventGroup> commonGroups)
		{
			ProcesoorUpdater upd = new ProcesoorUpdater();
			foreach (GameEventGroup g in commonGroups)
				g.Node.AcceptVisitor(upd);
		}

		public List<DataUpdate> FinalizeAccumulation(GameScope rootScope)
		{
			FinalizeAccVisitor v = new FinalizeAccVisitor(Repository.Registry);
			rootScope.AcceptVisitor(v);
			return v.Updates;
		}
	}


	public struct DataUpdate
	{
		public string Stat;
		public string Context;
		public int ProfileID;
		public string Value;
		public EOperationType DBOperation;

		public DataUpdate(string stat, string context, int profileId, string value, EOperationType op)
		{
			Stat = stat;
			Context = context;
			ProfileID = profileId;
			Value = value;
			DBOperation = op;
		}
	}


	class FinalizeAccVisitor : IStatsVisitor
	{
		StatsRegistry m_registry;

		public List<DataUpdate> Updates = new List<DataUpdate>();

		public FinalizeAccVisitor(StatsRegistry reg)
		{
			m_registry = reg;
		}

		string[] BuildContexts(GameElement elem, ContextBuilder[] contexts)
		{
			string[] cts = new string[contexts.Length];

			for (int i = 0; i != contexts.Length; ++i)
				cts[i] = contexts[i].GetContextName(elem);

			return cts;
		}

		void FinalizeAccumulators(GameElement elem)
		{
			// TODO: make game-independent
			GameState pids = elem.FindState("profile_id");
			int playerID = int.Parse(pids.Value);

			foreach (GameEventGroup eg in elem.EventGroups.Values)
			{
				Accumulator acc = eg.DataAccumulator;
				if (acc != null && eg.Events.Count != 0)
				{
					string res = acc.GetResult();
					eg.DataAccumulator = null;

					string[] contexts = BuildContexts(elem, acc.ProcessingSettings.Contexts);

					for (int i = 0; i != contexts.Length; ++i)
						Updates.Add(new DataUpdate(
							acc.ProcessingSettings.StatName, 
							contexts[i], 
							playerID, 
							res, 
							acc.ProcessingSettings.DBOperation ));
				}
			}
		}

		public bool VisitScope(GameScope scope)
		{
			return true;
		}
		public void LeaveScope(GameScope scope) { }
		public bool VisitElement(GameElement elem)
		{
			if (DataProcessor.ShouldAccumulate(elem))
				FinalizeAccumulators(elem);

			return false;
		}
		public void LeaveElement(GameElement elem) { }
		public bool VisitState(GameState state) { return false; }
		public void LeaveState(GameState state) { }
		public bool StartEventGroups(GameNode node) { return false; }
		public bool VisitEventGroup(GameEventGroup evntGroup) { return false; }
		public void LeaveEventGroup(GameEventGroup evntGroup) { }
		public void EndEventGroups(GameNode node) { }
		public void VisitEvent(GameEvent evnt) { }
	}
}
