﻿using System;
using System.Collections.Generic;
using System.Text;
using StatsDataSource.ObjectModel;
using System.Xml;
using System.IO;
using StatsDataSource.Preprocessing;
using StatsDataSource.DataAccumulation;
using System.Threading;

namespace StatsDataSource.Storage
{
	public class StatsRegistry
	{
		private Dictionary<int, ScopeDesc> Scopes;
		private Dictionary<string, ScopeDesc> ScopeDict;

		private Dictionary<int, ElementDesc> Elements;
		private Dictionary<string, ElementDesc> ElementDict;

		private Dictionary<int, StatDesc> States;
		private Dictionary<string, StatDesc> StateDict;

		private Dictionary<int, StatDesc> Events;
		private Dictionary<string, StatDesc> EventDict;

		internal Dictionary<string, Crawler> Crawlers;
		internal Dictionary<string, object> CrawlersTargets;

		internal Dictionary<string, StatProcSettings> StatsProcSettings;

		////////////////////////////////////////////////////////////

		public StatsRegistry(XmlReader format, string processingDef, string dataAccDef, out List<string> errors)
		{
			Scopes = new Dictionary<int, ScopeDesc>();
			ScopeDict = new Dictionary<string, ScopeDesc>();
			Elements = new Dictionary<int, ElementDesc>();
			ElementDict = new Dictionary<string, ElementDesc>();
			States = new Dictionary<int, StatDesc>();
			StateDict = new Dictionary<string, StatDesc>();
			Events = new Dictionary<int, StatDesc>();
			EventDict = new Dictionary<string, StatDesc>();

			Crawlers = new Dictionary<string, Crawler>();
			CrawlersTargets = new Dictionary<string, object>();

			StatsProcSettings = new Dictionary<string, StatProcSettings>();

			errors = new List<string>();

			InitFormat(format);

			if (!string.IsNullOrEmpty(processingDef) && File.Exists(processingDef))
				InitCrawlers(processingDef, errors);

			if (!string.IsNullOrEmpty(dataAccDef) && File.Exists(dataAccDef))
				InitDataProcessors(dataAccDef, errors);
		}

		////////////////////////////////////////////////////////////

		#region Format parsing
		private void InitFormat(XmlReader fmt)
		{
			XmlDocument xml = new XmlDocument();
			xml.Load(fmt);

			XmlNode format = xml.ChildNodes[1];

			if (format.Name != "format")
				throw new FormatException("Invalid format of file");

			for (int i = 0; i != format.ChildNodes.Count; ++i)
			{
				XmlNode group = format.ChildNodes[i];
				if (group.Name == "scopes")
					ParseScopes(group);
				else if (group.Name == "elements")
					ParseElements(group);
				else
					throw new FormatException("Invalid format of file");
			}
		}

		private void ParseScopes(XmlNode scopes)
		{
			for (int i = 0; i != scopes.ChildNodes.Count; ++i)
			{
				XmlNode s = scopes.ChildNodes[i];

				int id = int.Parse(s.Attributes["id"].Value);
				string name = s.Attributes["name"].Value;

				ScopeDesc sc = RegisterScope(id, name);
				ParseKeyStates(s, sc.KeyStates);
			}
		}

		private void ParseElements(XmlNode elements)
		{
			for (int i = 0; i != elements.ChildNodes.Count; ++i)
			{
				XmlNode e = elements.ChildNodes[i];

				int id = int.Parse(e.Attributes["id"].Value);
				string name = e.Attributes["name"].Value;

				ElementDesc elem = RegisterElement(id, name);
				ParseKeyStates(e, elem.KeyStates);
			}
		}

		private void ParseKeyStates(XmlNode node, List<string> to)
		{
			XmlAttribute attr = node.Attributes["pk"];
			if (attr != null)
				to.Add(attr.Value);

			for (int i = 0; i != node.ChildNodes.Count; ++i)
			{
				XmlNode c = node.ChildNodes[i];
				if (c.Name == "pk")
					to.Add(c.Attributes["name"].Value);
			}
		}
		#endregion

		#region Crawler parsing

		void InitCrawlers(string file, List<string> errors)
		{
			using (FileStream fs = new FileStream(file, FileMode.Open, FileAccess.Read))
			{
				EProcScanner scanner = new EProcScanner(fs);
				EProcParser parser = new EProcParser(scanner);
				Preprocessing.CallbackImpl callback = new Preprocessing.CallbackImpl(parser, this);
				parser.Callback = callback;

				parser.Parse();

				Crawlers = callback.Crawlers;
				CrawlersTargets = callback.CrawlerTargets;

				errors.AddRange(scanner.Errors);
				string f = Path.GetFileName(file);

				for (int i = 0; i != errors.Count; ++i)
					errors[i] = errors[i].Replace("<filename>", f);
			}
		}

		#endregion

		#region Data accumulation parsing

		void InitDataProcessors(string file, List<string> errors)
		{
			using (FileStream fs = new FileStream(file, FileMode.Open, FileAccess.Read))
			{
				DataAccScanner scanner = new DataAccScanner(fs);
				DataAccParser parser = new DataAccParser(scanner);
				DataAccumulation.CallbackImpl callback = new DataAccumulation.CallbackImpl(parser, this);
				parser.Callback = callback;

				parser.Parse();
				StatsProcSettings = callback.StatsProcSettings;

				errors.AddRange(scanner.Errors);
				string f = Path.GetFileName(file);

				for (int i = 0; i != errors.Count; ++i)
					errors[i] = errors[i].Replace("<filename>", f);
			}
		}

		#endregion

		////////////////////////////////////////////////////////////

		#region Registration

		public ScopeDesc RegisterScope(int scopeID, string name)
		{
			lock (Scopes)
			{
				ScopeDesc desc = new ScopeDesc(scopeID, name);
				Scopes.Add(scopeID, desc);
				ScopeDict.Add(name, desc);
				return desc;
			}
		}

		public ElementDesc RegisterElement(int elemID, string name)
		{
			lock (Elements)
			{
				ElementDesc desc = new ElementDesc(elemID, name);
				Elements.Add(elemID, desc);
				ElementDict.Add(name, desc);
				return desc;
			}
		}

		private int m_lastStateId = 0;
		public StatDesc RegisterState(int stateID, string name)
		{
			lock (States)
			{
				StatDesc desc = new StatDesc(stateID, name);
				States.Add(stateID, desc);
				StateDict.Add(name, desc);
				m_lastStateId = Math.Max(stateID, m_lastStateId);
				return desc;
			}
		}

		public StatDesc RegisterState(string name)
		{
			lock (States)
			{
				return RegisterState(++m_lastStateId, name);
			}
		}

		private int m_lastEventId = 0;
		public StatDesc RegisterEvent(int eventID, string name)
		{
			lock (Events)
			{
				StatDesc desc = new StatDesc(eventID, name);
				Events.Add(eventID, desc);
				EventDict.Add(name, desc);
				m_lastEventId = Math.Max(m_lastEventId, eventID);
				return desc;
			}
		}

		public StatDesc RegisterEvent(string name)
		{
			lock (Events)
			{
				return RegisterEvent(++m_lastEventId, name);
			}
		}

		#endregion

		#region Search

		private T GetValueOrDefault<T, K>(Dictionary<K, T> dict, K key)
		{
			lock (dict)
			{
				T ret;
				if (dict.TryGetValue(key, out ret))
					return ret;
				return default(T);
			}
		}

		public ScopeDesc GetScopeDesc(int scopeID)
		{
			return GetValueOrDefault(Scopes, scopeID);
		}

		public ScopeDesc GetScopeDesc(string scopeName)
		{
			return GetValueOrDefault(ScopeDict, scopeName);
		}

		public ElementDesc GetElementDesc(int elementID)
		{
			return GetValueOrDefault(Elements, elementID);
		}

		public ElementDesc GetElementDesc(string elementName)
		{
			return GetValueOrDefault(ElementDict, elementName);
		}

		public StatDesc GetStateDesc(int stateID)
		{
			return GetValueOrDefault(States, stateID);
		}

		public StatDesc GetStateDesc(string stateName)
		{
			return GetValueOrDefault(StateDict, stateName);
		}

		public StatDesc GetEventDesc(int eventID)
		{
			return GetValueOrDefault(Events, eventID);
		}

		public StatDesc GetEventDesc(string eventName)
		{
			return GetValueOrDefault(EventDict, eventName);
		}

		#endregion

		#region Data access

		public ScopeDesc[] GetAllScopes()
		{
			lock (Scopes)
			{
				ScopeDesc[] scopes = new ScopeDesc[Scopes.Values.Count];
				Scopes.Values.CopyTo(scopes, 0);
				return scopes;
			}
		}

		public ElementDesc[] GetAllElements()
		{
			lock (Elements)
			{
				ElementDesc[] elements = new ElementDesc[Elements.Values.Count];
				Elements.Values.CopyTo(elements, 0);
				return elements;
			}
		}

		public StatDesc[] GetAllStates()
		{
			lock (States)
			{
				StatDesc[] states = new StatDesc[States.Values.Count];
				States.Values.CopyTo(states, 0);
				return states;
			}
		}

		public StatDesc[] GetAllEvents()
		{
			lock (Events)
			{
				StatDesc[] events = new StatDesc[Events.Values.Count];
				Events.Values.CopyTo(events, 0);
				return events;
			}
		}

		#endregion

		////////////////////////////////////////////////////////////
	}
}
