﻿using System;
using System.Collections.Generic;
using System.Text;
using StatsDataSource.ObjectModel;
using System.Diagnostics;
using System.Xml;
using StatsDataSource.Utils;
using StatsDataSource.Preprocessing;
using StatsDataSource.DataAccumulation;
using StatsDataSource.Filters;

namespace StatsDataSource.Storage
{
	public class IncrementalUpdate
	{
		public List<GameNode> AddedNodes = new List<GameNode>();
		public List<GameState> AddedStates = new List<GameState>();
		public List<GameEventGroup> AddedEventGroups = new List<GameEventGroup>();
		public List<GameEvent> AddedEvents = new List<GameEvent>();
	}

	public class StatsRepository
	{
		////////////////////////////////////////////////////////////

		public StatsRegistry Registry;

		private CompoundFilter ActiveFilter;

		public DataProcessor DataProcessor;

		public List<GameScope> RootScopes { get { return m_rootScopes; } }

		////////////////////////////////////////////////////////////

		public StatsRepository(StatsRegistry registry)
		{
			ActiveFilter = new CompoundANDFilter();
			Registry = registry;
			DataProcessor = new DataProcessor(this);
		}

		public void LoadFromXML(XmlReader reader)
		{
			StartModificationBatch(false);
			
			XMLStatsParser p = new XMLStatsParser(reader, this, m_lastTimeStamp);
			p.Parse();

			EndModificationBatch();
		}

		public void SaveToXML(string file)
		{
			using (XMLStatsSerializer s = new XMLStatsSerializer(file))
			{
				AcceptVisitor(s);
			}
		}

		////////////////////////////////////////////////////////////

		public CompoundFilter GetActiveFilter()
		{
			return ActiveFilter;
		}

		public void AddSubFilter(FilterBase filter)
		{
			ActiveFilter.AddChild(filter);
			UpdateFilter(EFilterUpdateType.Regular, null);
		}

		public void RemoveSubFilter(FilterBase filter)
		{
			ActiveFilter.RemoveChild(filter);
			UpdateFilter(EFilterUpdateType.Regular, null);
		}

		public void SetNewRootFilterType(CompoundFilter root)
		{
			root.ClearChildren();
			
			foreach (FilterBase f in ActiveFilter.SubFilters)
				root.AddChild(f);

			ActiveFilter = root;
			UpdateFilter(EFilterUpdateType.Regular, null);
		}

		////////////////////////////////////////////////////////////

		public GameScope AddRootScope(Locator locator)
		{
			GameScope scope = new GameScope(locator, null, Registry.GetScopeDesc(locator.ScopeID), this);
			m_rootScopes.Add(scope);
			OnNodeAdded(scope);
			return scope;
		}

		public void DropRootScope(GameScope scope)
		{
			if (scope.Parent == null)
			{
				m_rootScopes.Remove(scope);
				scope.AcceptVisitor(new RemovalVisitor(this));
			}
		}

		public GameNode FindNode(Locator locator)
		{
			GameNode node;
			if (m_allNodes.TryGetValue(locator, out node))
				return node;
			return null;
		}

		public void AcceptVisitor(IStatsVisitor visitor)
		{
			foreach (GameScope sc in m_rootScopes)
				sc.AcceptVisitor(visitor);
		}

		public void AcceptVisitorFiltered(IStatsVisitor visitor)
		{
			foreach (GameScope sc in m_rootScopes)
				sc.AcceptVisitorFiltered(visitor);
		}

		public void UpdateFilter(EFilterUpdateType type, IStatsVisitor updateVisitor)
		{
			DoUpdateFilter(type, updateVisitor);

			foreach (IFilteredStatsListener v in m_filteredListeners)
				v.RepositoryFilterChanged(ActiveFilter, type);
		}

		////////////////////////////////////////////////////////////

		public void RegisterListener(IStatsListener listener) { m_listeners.Add(listener); }
		public void UnRegisterListener(IStatsListener listener) { m_listeners.Remove(listener); }
		public void RegisterFilteredListener(IFilteredStatsListener listener) { m_listeners.Add(listener); m_filteredListeners.Add(listener); }
		public void UnRegisterFilteredListener(IFilteredStatsListener listener) { m_listeners.Add(listener); m_filteredListeners.Remove(listener); }

		////////////////////////////////////////////////////////////

		internal void OnNodeAdded(GameNode node)
		{
			if (m_inModificationBatchDepth == 0)
				throw new InvalidOperationException("All updates should be performed in batches");

			m_lastTimeStamp = Math.Max(m_lastTimeStamp, node.Locator.TimeStamp);
			m_allNodes.Add(node.Locator, node);

			if (m_incUpdate != null)
				m_incUpdate.AddedNodes.Add(node);

			foreach (IStatsListener ln in m_listeners)
				ln.RepositoryNodeAdded(node);
		}

		internal void OnStateAdded(GameState state)
		{
			if (m_inModificationBatchDepth == 0)
				throw new InvalidOperationException("All updates should be performed in batches");

			if (m_incUpdate != null)
				m_incUpdate.AddedStates.Add(state);

			foreach (IStatsListener ln in m_listeners)
				ln.RepositoryStateAdded(state);
		}

		internal void OnEventGroupAdded(GameEventGroup evntGroup)
		{
			if (m_inModificationBatchDepth == 0)
				throw new InvalidOperationException("All updates should be performed in batches");

			DataProcessor.OnEventGroupAdded(evntGroup);

			if (m_incUpdate != null)
				m_incUpdate.AddedEventGroups.Add(evntGroup);

			foreach (IStatsListener ln in m_listeners)
				ln.RepositoryEventGroupAdded(evntGroup);
		}

		internal void OnEventAdded(GameEvent ev)
		{
			if (m_inModificationBatchDepth == 0)
				throw new InvalidOperationException("All updates should be performed in batches");

			if (m_incUpdate != null)
				m_incUpdate.AddedEvents.Add(ev);

			foreach (IStatsListener ln in m_listeners)
				ln.RepositoryEventAdded(ev);
		}

		////////////////////////////////////////////////////////////

		void DoUpdateFilter(EFilterUpdateType type, IStatsVisitor updateVisitor)
		{
			if (updateVisitor == null)
				updateVisitor = new GeneralFilterUpdater(ActiveFilter);

			AcceptVisitor(updateVisitor);
		}

		void DoIncrementalUpdateFilter(List<GameNode> commonNodes, List<GameEventGroup> commonGroups)
		{
			if (ActiveFilter == null)
				return;

			GeneralFilterUpdater updater = new GeneralFilterUpdater(ActiveFilter);

			foreach (GameNode node in commonNodes)
			{
				node.AcceptVisitor(updater);
				updater.Reset();
			}

			foreach (GameEventGroup eg in commonGroups)
			{
				if (!commonNodes.Contains(eg.Node))
				{
					eg.AcceptVisitor(updater);
					updater.Reset();
				}
			}
		}

		////////////////////////////////////////////////////////////

		void PostBatchUpdateAll()
		{
			DataProcessor.UpdateAllProcessors();

			DoUpdateFilter(EFilterUpdateType.Regular, null);
		}

		void PostBatchUpdateIncremental()
		{
			List<GameNode> commonNodes = new List<GameNode>(m_incUpdate.AddedNodes);
			foreach (GameState state in m_incUpdate.AddedStates)
				if (!commonNodes.Contains(state.Node))
					commonNodes.Add(state.Node);

			List<GameEventGroup> commonGroups = new List<GameEventGroup>(m_incUpdate.AddedEventGroups);
			foreach (GameEvent e in m_incUpdate.AddedEvents)
				if (!commonGroups.Contains(e.Group))
					commonGroups.Add(e.Group);

			DataProcessor.UpdateProcessorsIncremental(commonGroups);

			DoIncrementalUpdateFilter(commonNodes, commonGroups);
		}

		////////////////////////////////////////////////////////////

		IncrementalUpdate m_incUpdate = null;

		public void StartModificationBatch(bool incremental)
		{
			if (++m_inModificationBatchDepth != 1)
			{
				if (m_incUpdate != null && !incremental)
					m_incUpdate = null;
				return;
			}

			if (incremental)
				m_incUpdate = new IncrementalUpdate();

			foreach (IStatsListener v in m_listeners)
				v.ModificationBatchStarted();
		}

		public void EndModificationBatch()
		{
			if (--m_inModificationBatchDepth != 0)
				return;

			Debug.Assert(m_inModificationBatchDepth >= 0);

			if (m_incUpdate == null)
				PostBatchUpdateAll();
			else
				PostBatchUpdateIncremental();

			IncrementalUpdate u = m_incUpdate;
			m_incUpdate = null;

			foreach (IStatsListener v in m_listeners)
				v.ModificationBatchEnded(u);
		}

		////////////////////////////////////////////////////////////

		int m_lastTimeStamp = 0;
		int m_inModificationBatchDepth = 0;
		List<GameScope> m_rootScopes = new List<GameScope>();
		Dictionary<Locator, GameNode> m_allNodes = new Dictionary<Locator, GameNode>();
		List<IStatsListener> m_listeners = new List<IStatsListener>();
		List<IFilteredStatsListener> m_filteredListeners = new List<IFilteredStatsListener>();

		////////////////////////////////////////////////////////////

		class RemovalVisitor : IStatsVisitor
		{
			StatsRepository m_repo;

			public RemovalVisitor(StatsRepository repo)
			{
				m_repo = repo;
			}

			public bool VisitScope(GameScope scope)
			{
				m_repo.m_allNodes.Remove(scope.Locator);
				return true;
			}
			public void LeaveScope(GameScope scope) { }

			public bool VisitElement(GameElement elem) 
			{
				m_repo.m_allNodes.Remove(elem.Locator);
				return false; 
			}

			public void LeaveElement(GameElement elem) { }
			public bool VisitState(GameState state) { return false; }
			public void LeaveState(GameState state) { }
			public bool StartEventGroups(GameNode node) {	return false; }
			public bool VisitEventGroup(GameEventGroup evntGroup)	{ return false; }
			public void LeaveEventGroup(GameEventGroup evntGroup) { }
			public void EndEventGroups(GameNode node) { }
			public void VisitEvent(GameEvent evnt) { }
		}
	}
	
}
