﻿using System;
using System.Collections.Generic;
using StatsDataSource.ObjectModel;
using StatsDataSource.Preprocessing;
using StatsDataSource.DataAccumulation;

namespace StatsDataSource.Storage
{
	////////////////////////////////////////////////////////////
	////////////////////////////////////////////////////////////

	class ProcesoorUpdater : IStatsVisitor
	{
		struct CrawlerUpdateContext
		{
			public CrawlerContext context;
			public bool stepped;
		}

		void UpdateCrawlers(GameNode node)
		{
			if (node.EventGroups.Count == 0)
				return;

			CrawlerUpdateContext[] contexts = new CrawlerUpdateContext[node.EventGroups.Count];

			int i = -1;
			foreach (GameEventGroup gr in node.EventGroups.Values)
			{
				CrawlerContext context = gr.CrawlerContext;
				
				contexts[++i].context = context;

				if (context != null && context.Crawler != null)
					context.Crawler.EstablishConnections(context);
			}

			int advancable;

			for(;;)
			{
				advancable = 0;
				Int64 smallestStep = Int64.MaxValue;

				// Calculate smallest step
				for (i = 0; i != contexts.Length; ++i)
				{
					if (contexts[i].context != null)
					{
						Int64 next = contexts[i].context.NextTimestamp;
						if (next >= 0)
						{
							++advancable;
							if (smallestStep > next)
								smallestStep = next;
						}
						else
						{
							contexts[i].context = null;
						}
					}
				}

				if (advancable == 0)
					break;

				// Advance
				for (i = 0; i != contexts.Length; ++i)
					if (contexts[i].context != null)
						contexts[i].stepped = contexts[i].context.Advance(smallestStep);

				// Process
				for (i = 0; i != contexts.Length; ++i)
					if (contexts[i].context != null && contexts[i].stepped)
						contexts[i].context.Process();
			}
		}

		void UpdateAccumulators(GameElement elem)
		{
			foreach (GameEventGroup group in elem.EventGroups.Values)
			{
				Accumulator acc = group.DataAccumulator;

				if (acc != null)
				{
					// Accumulator should not go ahead crawler
					int maxPos = group.CrawlerContext != null ? group.CrawlerContext.Position : group.Events.Count;
					acc.Process(group, maxPos);
				}
			}
		}

		public bool VisitScope(GameScope scope)
		{
			UpdateCrawlers(scope);
			return true;
		}
		public void LeaveScope(GameScope scope) { }
		public bool VisitElement(GameElement elem)
		{
			UpdateCrawlers(elem);
			UpdateAccumulators(elem);
			return false;
		}
		public void LeaveElement(GameElement elem) { }
		public bool VisitState(GameState state) { return false; }
		public void LeaveState(GameState state) { }
		public bool StartEventGroups(GameNode node) { return false; }
		public bool VisitEventGroup(GameEventGroup evntGroup) { return false; }
		public void LeaveEventGroup(GameEventGroup evntGroup) { }
		public void EndEventGroups(GameNode node) { }
		public void VisitEvent(GameEvent evnt) { }
	}

	////////////////////////////////////////////////////////////
	////////////////////////////////////////////////////////////

	public class GeneralFilterUpdater : IStatsVisitor
	{
		const int NO_PROPAGATION = -1;
		IStatsFilter m_filter;
		int m_propagate;

		public GeneralFilterUpdater(IStatsFilter filter)
		{
			m_filter = filter;
			m_propagate = NO_PROPAGATION;
		}

		public void Reset()
		{
			m_propagate = NO_PROPAGATION;
		}

		public bool VisitScope(GameScope scope)
		{
			if (m_propagate != NO_PROPAGATION)
			{
				scope.FilterState = EFilterState.Failed;
				return true;
			}

			EFilterState oldstate = scope.FilterState;
			scope.FilterState = m_filter != null ? m_filter.TestScope(scope) : EFilterState.Unaffected;

			if (scope.FilterState == EFilterState.Failed)
				m_propagate = scope.Depth;

			return true;
		}

		public void LeaveScope(GameScope scope)
		{
			if (m_propagate == scope.Depth)
				m_propagate = NO_PROPAGATION;
		}

		public bool VisitElement(GameElement elem)
		{
			if (m_propagate != NO_PROPAGATION)
			{
				elem.FilterState = EFilterState.Failed;
				return true;
			}

			EFilterState oldstate = elem.FilterState;
			elem.FilterState = m_filter != null ? m_filter.TestElement(elem) : EFilterState.Unaffected;

			if (elem.FilterState == EFilterState.Failed)
				m_propagate = elem.Depth;

			return true;
		}

		public void LeaveElement(GameElement elem)
		{
			if (m_propagate == elem.Depth)
				m_propagate = NO_PROPAGATION;
		}

		public bool VisitState(GameState state)
		{
			if (m_propagate != NO_PROPAGATION)
			{
				state.FilterState = EFilterState.Failed;
				return true;
			}

			EFilterState oldstate = state.FilterState;
			state.FilterState = m_filter != null ? m_filter.TestState(state) : EFilterState.Unaffected;

			if (state.FilterState == EFilterState.Failed)
				m_propagate = state.Depth;

			return true;
		}

		public void LeaveState(GameState state)
		{
			if (m_propagate == state.Depth)
				m_propagate = NO_PROPAGATION;
		}

		public bool StartEventGroups(GameNode node) { return true; }

		public bool VisitEventGroup(GameEventGroup evntGroup)
		{
			if (m_propagate != NO_PROPAGATION)
			{
				evntGroup.FilterState = EFilterState.Failed;
				return true;
			}

			EFilterState oldstate = evntGroup.FilterState;
			evntGroup.FilterState = m_filter != null ? m_filter.TestEventGroup(evntGroup) : EFilterState.Unaffected;

			if (evntGroup.FilterState == EFilterState.Failed)
				m_propagate = evntGroup.Node.Depth + 1;

			return true;
		}

		public void LeaveEventGroup(GameEventGroup evntGroup)
		{
			if (m_propagate == evntGroup.Node.Depth + 1)
				m_propagate = NO_PROPAGATION;
		}

		public void EndEventGroups(GameNode node) { }

		public void VisitEvent(GameEvent evnt)
		{
			if (m_propagate != NO_PROPAGATION)
			{
				evnt.FilterState = EFilterState.Failed;
			}
			else
			{
				evnt.FilterState = m_filter != null ? m_filter.TestEvent(evnt) : EFilterState.Unaffected;
			}
		}
	}

	////////////////////////////////////////////////////////////
	////////////////////////////////////////////////////////////
}
